/*-
 * Copyright (c) 1991 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1991 The Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "from @(#)dmesg.c	5.9 (Berkeley) 5/2/91";
#endif /* not lint */

/*
 * bpatch: binary patch
 * Patches a.out files, including kernel, or /dev/kmem for running kernel.
 * Usage:
 *	bpatch [ type-opt ] [ -r ] location [ value ]
 *
 *	type-options (chose at most one, 'l' is default):
 *	    -b	byte as number
 *	    -c	byte as character
 *	    -w	short/u_short
 *	    -l	long/u_long
 *	    -s	string
 *
 *	Other options:
 *	    -r		patch running kernel rather than a.out
 *	    -N namelist	name for kernel namelist file
 *	    -M memfile	name for memory file, implies -r
 *
 *	location
 *	    a symbol name, or a decimal/hex/octal number
 *
 *	value
 *	    depending on the type, a character, string,
 *	    or a decimal/hex/octal number.
 *	    If none, the current value is printed.
 */

#include <sys/param.h>
#include <sys/cdefs.h>
#include <sys/errno.h>

#include <a.out.h>
#include <ctype.h>
#include <err.h>
#include <fcntl.h>
#include <kvm.h>
#include <paths.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#ifndef	_PATH_KERNEL
#define	_PATH_KERNEL	"/bsd"	/* XXX */
#endif

struct nlist nl[] = {
#define	X_SYSBASE	0
	{ "SYSTEM" },
#define	X_VAR		1
	{ "" },
#define	X_ETEXT		2
	{ "_etext" },
#define	X_EDATA		3
	{ "_edata" },
#define	X_VERSION	4
	{ "_version" },
	{ NULL },
};

void usage();

enum { UNSPEC, BYTE, CHAR, WORD, LONG, STRING } type = UNSPEC;

#define	MAXSTRING	1024

u_char	cbuf;
u_short	wbuf;
u_long	lbuf;
u_char	sbuf[MAXSTRING + 1];
u_char	vbuf[MAXSTRING + 1];
int	checkversion = 1;	/* check version strings, namelist vs kmem */

off_t	execoff __P((u_long addr, int segment, struct exec *ep));

int
main(argc, argv)
	int argc;
	char **argv;
{
	register int ch;
	char *namelist, *core, *file, *locname, *valname;
	struct exec exec;
	int writing, fd, nfd, seg;
	u_long loc;
	void *p;
	off_t foff;
	char *ep;
	size_t len;

	namelist = _PATH_KERNEL;
	core = NULL;
	writing = 0;
	while ((ch = getopt(argc, argv, "bcwlsrM:N:")) != EOF)
		switch (ch) {
		case 'b':
			if (type != UNSPEC)
				usage();
			type = BYTE;
			p = &cbuf;
			len = sizeof(cbuf);
			break;
		case 'c':
			if (type != UNSPEC)
				usage();
			type = CHAR;
			p = &cbuf;
			len = sizeof(cbuf);
			break;
		case 'w':
			if (type != UNSPEC)
				usage();
			type = WORD;
			p = &wbuf;
			len = sizeof(wbuf);
			break;
		case 'l':
			if (type != UNSPEC)
				usage();
			type = LONG;
			p = &lbuf;
			len = sizeof(lbuf);
			break;
		case 's':
			if (type != UNSPEC)
				usage();
			type = STRING;
			p = sbuf;
			len = MAXSTRING;
			break;
		case 'r':
			if (core == NULL)
				core = _PATH_KMEM;
			break;
		case 'M':
			core = optarg;
			checkversion = 0;
			break;
		case 'N':
			namelist = optarg;
			break;
		case '?':
		default:
			usage();
		}
	argc -= optind;
	argv += optind;
	if (type == UNSPEC) {
		type = LONG;
		p = &lbuf;
		len = sizeof(lbuf);
	}

	if (argc < 1 || argc > 2)
		usage();
	writing = (argc == 2);
	locname = argv[0];
	valname = argv[1];	/* NULL if not writing */

	if (core)
		file = core;
	else
		file = namelist;
	if (writing) {
		if ((fd = open(file, O_RDWR)) == -1)
			err(1, "%s", file);
	} else {
		if ((fd = open(file, O_RDONLY)) == -1)
			err(1, "%s", file);
		checkversion = 0;
	}

	nl[X_VAR].n_un.n_name = locname;
	if (nlist(namelist, nl) == -1)
		err(1, "nlist(%s)", namelist);

	/*
	 * See whether the given location is numeric or symbolic.
	 */
	errno = 0; 
	loc = strtoul(locname, &ep, 0);
	if (errno != ERANGE && *ep == '\0') {
		/* value is number; check whether text/data/otherwise */
		if (nl[X_ETEXT].n_type && loc <= nl[X_ETEXT].n_value)
			seg = N_TEXT;
		else if (nl[X_EDATA].n_type && loc <= nl[X_EDATA].n_value)
			seg = N_DATA;
		else
			errx(1, "%s: can't determine segment for offset",
			    namelist);
	} else {
		/*
		 * Value must be symbolic.  If not found,
		 * try prepending underscore.
		 */
		if (nl[X_VAR].n_type == 0) {
			nl[X_VAR].n_un.n_name = malloc(strlen(locname + 2));
			if (nl[X_VAR].n_un.n_name)
				sprintf(nl[X_VAR].n_un.n_name, "_%s", locname);
			(void) nlist(namelist, nl);
		}
		if (nl[X_VAR].n_type == 0)
			errx(1, "%s not defined in namelist", locname);
		seg = nl[X_VAR].n_type & N_TYPE;
		if (seg != N_TEXT && seg != N_DATA)
			errx(1, "%s: offset not in file", locname);
        	loc = nl[X_VAR].n_value;
	}

	/*
	 * If patching the binary file, need to figure out file offsets.
	 * If checking version, also need exec header.
	 */
	if (core == NULL || checkversion) {
		if (core == NULL)
			nfd = fd;
		else if ((nfd = open(namelist, O_RDONLY)) == -1)
			err(1, "%s: open", namelist);
		(void) lseek(nfd, 0, SEEK_SET);
		if (read(nfd, &exec, sizeof(exec)) != sizeof(exec))
			errx(1, "%s: can't read exec header", namelist);
		if (N_BADMAG(exec))
			errx(1, "%s: bad number in exec header", namelist);
	}
 
	foff = core ? loc : execoff(loc, seg, &exec);
 	if (!writing) {
 		if (lseek(fd, foff, SEEK_SET) == -1)
 			err(1, "%s: lseek", file);
        	if (read(fd, p, len) != len)
 			errx(1, "%s: can't read at %s", file, locname);
        	switch (type) {
        	case BYTE:
        		(void) printf("0x%x = %d\n", cbuf, cbuf);
        		break;
        	case CHAR:
        		(void) printf("%c\n", cbuf);
        		break;
        	case WORD:
        		(void) printf("0x%x = %d\n", wbuf, wbuf);
        		break;
        	case LONG:
        		(void) printf("0x%lx = %ld\n", lbuf, lbuf);
        		break;
        	case STRING:
        		(void) printf("%s\n", sbuf);
        		break;
        	}
	} else {
		if (checkversion && core && nl[X_VERSION].n_type) {
			off_t nloc;

			/* check version strings in a.out and memory */
			nloc = execoff(nl[X_VERSION].n_value, N_DATA, &exec);
			if (lseek(nfd, nloc, SEEK_SET) == -1)
				err(1, "%s: lseek to version", namelist);
			if (read(nfd, vbuf, MAXSTRING) != MAXSTRING)
				errx(1, "%s: can't read version", namelist);
			if (lseek(fd, nl[X_VERSION].n_value, SEEK_SET) == -1)
				err(1, "%s: lseek to version", core);
			if (read(fd, sbuf, MAXSTRING) != MAXSTRING)
				errx(1, "%s: can't read version", core);
			if (strcmp(vbuf, sbuf))
				errx(1, "version string mismatch in %s and %s",
				    namelist, core);
		}
		errno = 0;
        	switch (type) {
        	case BYTE:
        		lbuf = strtol(valname, NULL, 0);
        		if (lbuf > UCHAR_MAX || errno == ERANGE)
        			errx(1, "%s: value out of range", valname);
        		cbuf = lbuf;
        		break;
        	case CHAR:
        		cbuf = valname[0];
        		break;
        	case WORD:
        		lbuf = strtol(valname, NULL, 0);
        		if (lbuf > USHRT_MAX || errno == ERANGE)
        			errx(1, "%s: value out of range", valname);
        		wbuf = lbuf;
        		break;
        	case LONG:
        		lbuf = strtol(valname, NULL, 0);
        		if (errno == ERANGE)
        			errx(1, "%s: value out of range", valname);
        		break;
        	case STRING:
        		p = valname;
        		len = strlen(p) + 1;
        		break;
        	}
 		if (lseek(fd, foff, SEEK_SET) == -1)
 			err(1, "%s: lseek", file);
        	if (write(fd, p, len) != len)
 			err(1, "%s: can't write at %s", file, locname);
	}
	exit(0);
}

/*
 * Convert address in kernel to offset in exec (a.out) file.
 * The segment argument should be either N_TEXT or N_DATA.
 */
off_t
execoff(addr, segment, ep)
	u_long addr;
	int segment;
	struct exec *ep;
{
	off_t foff;
#ifdef DEBUG
	char *s;
#endif

	/*
	 * Convert address to offset from "zero" by subtracting
	 * the address in which the first byte of the executable
	 * resides ("SYSTEM", aka nl[X_SYSBASE].n_value).
	 */
	foff = addr;
	if (nl[X_SYSBASE].n_type)
		foff -= nl[X_SYSBASE].n_value;
	switch (segment) {

	case N_TEXT:
		/*
		 * Text is easy: just add the text offset.
		 */
#ifdef DEBUG
		s = "text";
#endif
		foff += N_TXTOFF(*ep);
		break;

	case N_DATA:
		/*
		 * Data is clumsy, because the kernel does not behave
		 * quite like ordinary binaries.  Ordinary binaries
		 * are loaded at N_TXTADDR and their data appears at
		 * N_DATADDR, but the kernel is loaded such that it
		 * appears to run at "SYSTEM".  Subtracting N_TXTADDR
		 * from N_DATADDR gives the data address we would have
		 * if the binary were loaded at address 0.  We can
		 * then add N_DATOFF to get the correct data offset in
		 * the a.out image.
		 */
#ifdef DEBUG
		s = "data";
#endif
		foff -= N_DATADDR(*ep) - N_TXTADDR(*ep);
		foff += N_DATOFF(*ep);
		break;

	default:
		warn("bug! bad segment type %d\n", segment);
		abort();
		/* NOTREACHED */
	}
#ifdef DEBUG
	printf("%s addr %lx, txtaddr %x, dataddr %x, datoff %x; foff %qx\n",
	    s, addr, N_TXTADDR(*ep), N_DATADDR(*ep), N_DATOFF(*ep), foff);
#endif
	return (foff);
}

void
usage()
{
	(void) fprintf(stderr,
"usage: bpatch [-b|c|w|l|s] [-r] [-M core] [-N system] var [ val ]\n");
	exit(1);
}
