/*	BSDI $Id: igmp.c,v 2.3 1995/10/18 04:51:22 karels Exp $	*/

/*
 * Internet Group Management Protocol (IGMP) routines.
 *
 * Written by Steve Deering, Stanford, May 1988.
 * Modified by Rosen Sharma, Stanford, Aug 1994.
 * Modified by Bill Fenner, Xerox PARC, Feb 1995.
 *
 * MULTICAST Revision: 1.3
 */

#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/protosw.h>

#include <net/if.h>
#include <net/route.h>

#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/igmp.h>
#include <netinet/igmp_var.h>

#if BSD - 0 >= 199207
#define IP_MULTICASTOPTS 0
#define m_off m_data
#define M_HASCL(m) ((m)->m_flags & M_EXT)
#else
#define MGETHDR MGET
#define M_NOWAIT M_DONTWAIT
#define M_MRTABLE MT_MRTABLE
#define malloc(size, type, flag) mcast_alloc(size, flag, type)
#define free(p, type) m_free(dtom(p))
static caddr_t
mcast_alloc(int size, int flag, int type)
{
	struct mbuf *m;
	MGET(m, type, flag);
	return ((m == 0) ? 0 : mtod(m, caddr_t));
}
#endif

extern struct ifnet loif;

static int                 igmp_timers_are_running;
static u_long		   igmp_all_hosts_group;
static u_long              igmp_local_group;
static u_long		   igmp_local_group_mask;
static struct router_info *Head;

static void igmp_sendpkt();
static void igmp_sendleave();

void
igmp_init()
{
	/*
	 * To avoid byte-swapping the same values over and over again.
	 */
	igmp_all_hosts_group = htonl(INADDR_ALLHOSTS_GROUP);
	igmp_local_group = htonl(0xe0000000);			/* 224.0.0.0 */
	igmp_local_group_mask = htonl(0xffffff00);		/* ........^ */

	igmp_timers_are_running = 0;

	Head = (struct router_info *) 0;

	/*
	 * We need to seed the random number generator, but the host
	 * doesn't have an address yet.
	 */
}

static int
fill_rti(inm)
struct in_multi *inm;
{
	register struct router_info *rti = Head;

#ifdef IGMP_DEBUG
	printf("[igmp.c, _fill_rti] --> entering \n");
#endif
	while (rti) {
		if (rti->ifp == inm->inm_ifp) {
			inm->inm_rti  = rti;
#ifdef IGMP_DEBUG
			printf("[igmp.c, _fill_rti] --> found old entry \n");
#endif
			if (rti->type == IGMP_OLD_ROUTER) 
				return IGMP_HOST_MEMBERSHIP_REPORT;
			else
				return IGMP_HOST_NEW_MEMBERSHIP_REPORT;
		}
		rti = rti->next;
	}
	rti = (struct router_info *)malloc(sizeof(*rti), M_MRTABLE, M_NOWAIT);
	rti->ifp = inm->inm_ifp;
	rti->type = IGMP_NEW_ROUTER;
	rti->time = IGMP_AGE_THRESHOLD;
	rti->next = Head;
	Head = rti;
	inm->inm_rti = rti;
#ifdef IGMP_DEBUG
	printf("[igmp.c, _fill_rti] --> created new entry \n");
#endif
	return IGMP_HOST_NEW_MEMBERSHIP_REPORT;
}

static struct router_info *
find_rti(ifp)
struct ifnet *ifp;
{
        register struct router_info *rti = Head;

#ifdef IGMP_DEBUG
	printf("[igmp.c, _find_rti] --> entering \n");
#endif
        while (rti) {
                if (rti->ifp == ifp) {
#ifdef IGMP_DEBUG
			printf("[igmp.c, _find_rti] --> found old entry \n");
#endif
                        return rti;
                }
                rti = rti->next;
        }
	rti = (struct router_info *)malloc(sizeof(*rti), M_MRTABLE, M_NOWAIT);
        rti->ifp = ifp;
        rti->type = IGMP_NEW_ROUTER;
        rti->time = IGMP_AGE_THRESHOLD;
        rti->next = Head;
        Head = rti;
#ifdef IGMP_DEBUG
	printf("[igmp.c, _find_rti] --> created an entry \n");
#endif
        return rti;
}

#if BSD >= 199207
void
igmp_input(m, iphlen)
	register struct mbuf *m;
	register int iphlen;
{
	register struct ifnet *ifp = m->m_pkthdr.rcvif;
	struct ip *ip = mtod(m, struct ip *);
#else
igmp_input(m, ifp)
	register struct mbuf *m;
	register struct ifnet *ifp;
{
	struct ip *ip = mtod(m, struct ip *);
	int iphlen = ip->ip_hl << 2;
#endif
	register struct igmp *igmp;
	register int igmplen;
	register int minlen;
	struct in_multi *inm;
	struct in_multistep step;
	struct router_info *rti;
	register struct in_ifaddr *ia;

	int timer; /** timer value in the igmp query header **/

	++igmpstat.igps_rcv_total;

	igmplen = ip->ip_len;

	/*
	 * Validate lengths
	 */
	if (igmplen < IGMP_MINLEN) {
		++igmpstat.igps_rcv_tooshort;
		m_freem(m);
		return;
	}
	minlen = iphlen + IGMP_MINLEN;
	if ((
#if BSD - 0 >= 199207
	     m->m_flags & M_EXT
#else
	     m->m_off > MMAXOFF
#endif
	     || m->m_len < minlen) &&
	    (m = m_pullup(m, minlen)) == 0) {
		++igmpstat.igps_rcv_tooshort;
		return;
	}

	/*
	 * Validate checksum
	 */
	m->m_off += iphlen;
	m->m_len -= iphlen;
	igmp = mtod(m, struct igmp *);
	if (in_cksum(m, igmplen)) {
		++igmpstat.igps_rcv_badsum;
		m_freem(m);
		return;
	}
	m->m_off -= iphlen;
	m->m_len += iphlen;
	ip = mtod(m, struct ip *);

	timer = igmp->igmp_code * PR_FASTHZ / IGMP_TIMER_SCALE;
	  rti = find_rti(ifp);

	switch (igmp->igmp_type) {

	case IGMP_HOST_MEMBERSHIP_QUERY:
		++igmpstat.igps_rcv_queries;

		if (ifp == &loif)
			break;

		if (igmp->igmp_code == 0) {
		    
		    rti->type = IGMP_OLD_ROUTER; rti->time = 0;

		    /* 
		    ** Do exactly as RFC 1112 says
		    */

		    if (ip->ip_dst.s_addr != igmp_all_hosts_group) {
			++igmpstat.igps_rcv_badqueries;
			m_freem(m);
			return;
		    }

		    /*
		     * Start the timers in all of our membership records for
		     * the interface on which the query arrived, except those
		     * that are already running and those that belong to a
		     * "local" group (224.0.0.X).
		     */
		    IN_FIRST_MULTI(step, inm);
		    while (inm != NULL) {
			if (inm->inm_ifp == ifp && inm->inm_timer == 0 &&
			    (inm->inm_addr.s_addr & igmp_local_group_mask) !=
				igmp_local_group) {

		    	inm->inm_state = IGMP_DELAYING_MEMBER;
		    	inm->inm_timer = IGMP_RANDOM_DELAY(
				IGMP_MAX_HOST_REPORT_DELAY * PR_FASTHZ );

			igmp_timers_are_running = 1;
		        }
			IN_NEXT_MULTI(step, inm);
		    }
		}
		else {
		    /*
		    ** New Router
		    */
		    
		    if (!IN_MULTICAST(ip->ip_dst.s_addr)) {
			++igmpstat.igps_rcv_badqueries;
			m_freem(m);
			return;
		    }

		    /*
		     * - Start the timers in all of our membership records
		     *   that the query applies to for the interface on
		     *   which the query arrived excl. those that belong
		     *   to a "local" group (224.0.0.X)
		     * - For timers already running check if they need to
		     *   be reset.
		     * - Use the igmp->igmp_code field as the maximum 
		     *   delay possible
		     */
		    IN_FIRST_MULTI(step, inm);
		    while (inm != NULL) {
			if (inm->inm_ifp == ifp &&
			    (inm->inm_addr.s_addr & igmp_local_group_mask) !=
				igmp_local_group &&
			    (ip->ip_dst.s_addr == igmp_all_hosts_group ||
			     ip->ip_dst.s_addr == inm->inm_addr.s_addr)) {
			    switch (inm->inm_state) {
			      case IGMP_IDLE_MEMBER:
			      case IGMP_LAZY_MEMBER:
			      case IGMP_AWAKENING_MEMBER:
				inm->inm_timer = IGMP_RANDOM_DELAY(timer);
				igmp_timers_are_running = 1;
				inm->inm_state = IGMP_DELAYING_MEMBER;
				break;
			      case IGMP_DELAYING_MEMBER:
				if (inm->inm_timer > timer) {
				  inm->inm_timer = IGMP_RANDOM_DELAY(timer);
				  igmp_timers_are_running = 1;
				  inm->inm_state = IGMP_DELAYING_MEMBER;
				}
				break;
			      case IGMP_SLEEPING_MEMBER:
				inm->inm_state = IGMP_AWAKENING_MEMBER;
				break;
			    }
			}
			IN_NEXT_MULTI(step, inm);
		      }
		  }

		break;

	      case IGMP_HOST_MEMBERSHIP_REPORT:
		/*
		 *  an old report
		 */
		++igmpstat.igps_rcv_reports;

		if (ifp == &loif)
			break;

		if (!IN_MULTICAST(ntohl(igmp->igmp_group.s_addr)) ||
		    igmp->igmp_group.s_addr != ip->ip_dst.s_addr) {
			++igmpstat.igps_rcv_badreports;
			m_freem(m);
			return;
		}

		/*
		 * KLUDGE: if the IP source address of the report has an
		 * unspecified (i.e., zero) subnet number, as is allowed for
		 * a booting host, replace it with the correct subnet number
		 * so that a process-level multicast routing daemon can
		 * determine which subnet it arrived from.  This is necessary
		 * to compensate for the lack of any way for a process to
		 * determine the arrival interface of an incoming packet.
		 */
		if ((ntohl(ip->ip_src.s_addr) & IN_CLASSA_NET) == 0) {
			IFP_TO_IA(ifp, ia);
			if (ia) ip->ip_src.s_addr = htonl(ia->ia_subnet);
		}

		/*
		 * If we belong to the group being reported, stop
		 * our timer for that group.
		 */
		IN_LOOKUP_MULTI(igmp->igmp_group, ifp, inm);

		if (inm != NULL) {
		  inm->inm_timer = 0;
		  ++igmpstat.igps_rcv_ourreports;
		  
		  switch(inm->inm_state){
		  case IGMP_IDLE_MEMBER:
		  case IGMP_LAZY_MEMBER:
		  case IGMP_AWAKENING_MEMBER:
		  case IGMP_SLEEPING_MEMBER:
		    inm->inm_state = IGMP_SLEEPING_MEMBER;
		    break;
		  case IGMP_DELAYING_MEMBER:
		    if (inm->inm_rti->type == IGMP_OLD_ROUTER)
			inm->inm_state = IGMP_LAZY_MEMBER;
		    else
			inm->inm_state = IGMP_SLEEPING_MEMBER;
		    break;
		  }
		}
	      
		break;

	      case IGMP_HOST_NEW_MEMBERSHIP_REPORT:
#ifdef MROUTING
		/*
		 * We can get confused and think there's someone
		 * else out there if we are a multicast router.
		 * For fast leave to work, we have to know that
		 * we are the only member.
		 */
		IFP_TO_IA(ifp, ia);
		if (ia && ip->ip_src.s_addr == IA_SIN(ia)->sin_addr.s_addr)
			break;
#endif

		/*
		 * an new report
		 */
		++igmpstat.igps_rcv_reports;
    
		if (ifp == &loif)
		  break;
		
		if (!IN_MULTICAST(ntohl(igmp->igmp_group.s_addr)) ||
		    igmp->igmp_group.s_addr != ip->ip_dst.s_addr) {
		  ++igmpstat.igps_rcv_badreports;
		  m_freem(m);
		  return;
		}
		
		/*
		 * KLUDGE: if the IP source address of the report has an
		 * unspecified (i.e., zero) subnet number, as is allowed for
		 * a booting host, replace it with the correct subnet number
		 * so that a process-level multicast routing daemon can
		 * determine which subnet it arrived from.  This is necessary
		 * to compensate for the lack of any way for a process to
		 * determine the arrival interface of an incoming packet.
		 */
		if ((ntohl(ip->ip_src.s_addr) & IN_CLASSA_NET) == 0) {
#ifndef MROUTING
		  IFP_TO_IA(ifp, ia);
#endif
		  if (ia) ip->ip_src.s_addr = htonl(ia->ia_subnet);
		}
		
		/*
		 * If we belong to the group being reported, stop
		 * our timer for that group.
		 */
		IN_LOOKUP_MULTI(igmp->igmp_group, ifp, inm);
		if (inm != NULL) {
		  inm->inm_timer = 0;
		  ++igmpstat.igps_rcv_ourreports;
		  
		  switch(inm->inm_state){
		  case IGMP_DELAYING_MEMBER:
		  case IGMP_IDLE_MEMBER:
		    inm->inm_state = IGMP_LAZY_MEMBER;
		    break;
		  case IGMP_AWAKENING_MEMBER:
		    inm->inm_state = IGMP_LAZY_MEMBER;
		    break;
		  case IGMP_LAZY_MEMBER:
		  case IGMP_SLEEPING_MEMBER:
		    break;
		  }
		}
	}

	/*
	 * Pass all valid IGMP packets up to any process(es) listening
	 * on a raw IGMP socket.
	 */
	rip_input(m);
}

void
igmp_joingroup(inm)
	struct in_multi *inm;
{
	int s = splnet();

	inm->inm_state = IGMP_IDLE_MEMBER;

	if ((inm->inm_addr.s_addr & igmp_local_group_mask)
		== igmp_local_group ||
	    inm->inm_ifp == &loif)
		inm->inm_timer = 0;
	else {
		igmp_sendpkt(inm,fill_rti(inm));
		inm->inm_timer = IGMP_RANDOM_DELAY(
					IGMP_MAX_HOST_REPORT_DELAY*PR_FASTHZ);
		inm->inm_state = IGMP_DELAYING_MEMBER;
		igmp_timers_are_running = 1;
	}
	splx(s);
}

void
igmp_leavegroup(inm)
	struct in_multi *inm;
{
         switch (inm->inm_state) {
	 case IGMP_DELAYING_MEMBER:
	 case IGMP_IDLE_MEMBER:
	   if ((inm->inm_addr.s_addr & igmp_local_group_mask) !=
			igmp_local_group &&
		inm->inm_ifp != &loif)
	       if (inm->inm_rti->type != IGMP_OLD_ROUTER)
		   igmp_sendleave(inm);
	   break;
	 case IGMP_LAZY_MEMBER:
	 case IGMP_AWAKENING_MEMBER:
	 case IGMP_SLEEPING_MEMBER:
	   break;
	 }
}

void
igmp_fasttimo()
{
	register struct in_multi *inm;
	struct in_multistep step;
	int s;

	/*
	 * Quick check to see if any work needs to be done, in order
	 * to minimize the overhead of fasttimo processing.
	 */

	if (!igmp_timers_are_running)
		return;

	s = splnet();
	igmp_timers_are_running = 0;
	IN_FIRST_MULTI(step, inm);
	while (inm != NULL) {
		if (inm->inm_timer == 0) {
			/* do nothing */
		}
		else if (--inm->inm_timer == 0) {
		  if (inm->inm_state == IGMP_DELAYING_MEMBER) {
		    if (inm->inm_rti->type == IGMP_OLD_ROUTER)
			igmp_sendpkt(inm, IGMP_HOST_MEMBERSHIP_REPORT);
		    else
			igmp_sendpkt(inm, IGMP_HOST_NEW_MEMBERSHIP_REPORT);
		    inm->inm_state = IGMP_IDLE_MEMBER;
		  }
		}
		else {
			igmp_timers_are_running = 1;
		}
		IN_NEXT_MULTI(step, inm);
	}
	splx(s);
}

void
igmp_slowtimo()
{
	int s = splnet();
	register struct router_info *rti =  Head;

#ifdef IGMP_DEBUG
	printf("[igmp.c,_slowtimo] -- > entering \n");
#endif
	while (rti) {
		rti->time ++;
		if (rti->time >= IGMP_AGE_THRESHOLD){
			rti->type = IGMP_NEW_ROUTER;
			rti->time = IGMP_AGE_THRESHOLD;
		}
		rti = rti->next;
	}
#ifdef IGMP_DEBUG	
	printf("[igmp.c,_slowtimo] -- > exiting \n");
#endif
	splx(s);
}

static void
igmp_sendpkt(inm, type)
     struct in_multi *inm;  int type;
{
	struct mbuf *m;
	struct igmp *igmp;
	struct ip *ip;
	struct ip_moptions imo;
#ifdef MROUTING
	extern struct socket *ip_mrouter;
#endif /* MROUTING */

	MGETHDR(m, M_DONTWAIT, MT_HEADER);
	if (m == NULL)
		return;
#if BSD - 0 >= 199207
	m->m_data += max_linkhdr + sizeof(struct ip);
	m->m_pkthdr.len = sizeof(struct ip) + IGMP_MINLEN;
#else
	m->m_off = roundup(MMINOFF, 8) + 16 + sizeof(struct ip);
#endif
	m->m_len = IGMP_MINLEN;
	igmp = mtod(m, struct igmp *);
	igmp->igmp_type   = type;
	igmp->igmp_code   = 0;
	igmp->igmp_group  = inm->inm_addr;
	igmp->igmp_cksum  = 0;
	igmp->igmp_cksum  = in_cksum(m, IGMP_MINLEN);

	m->m_off -= sizeof(struct ip);
	m->m_len += sizeof(struct ip);
	ip = mtod(m, struct ip *);
	ip->ip_tos = 0;
	ip->ip_len = sizeof(struct ip) + IGMP_MINLEN;
	ip->ip_off = 0;
	ip->ip_p = IPPROTO_IGMP;
	ip->ip_src.s_addr = INADDR_ANY;
	ip->ip_dst = igmp->igmp_group;

	imo.imo_multicast_ifp = inm->inm_ifp;
	imo.imo_multicast_ttl = 1;
#ifdef RSVP_ISI
	imo.imo_multicast_vif = -1;
#endif
	/*
	 * Request loopback of the report if we are acting as a multicast
	 * router, so that the process-level routing daemon can hear it.
	 */
#ifdef MROUTING
	imo.imo_multicast_loop = (ip_mrouter != NULL);
#else
	imo.imo_multicast_loop = 0;
#endif /* MROUTING */

	ip_output(m, (struct mbuf *)0, (struct route *)0,
			IP_MULTICASTOPTS, &imo);

	++igmpstat.igps_snd_reports;
}

static void
igmp_sendleave(inm)
     struct in_multi *inm;
{
  igmp_sendpkt(inm, IGMP_HOST_LEAVE_MESSAGE);
}
