/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: ppp_frame.c,v 2.6 1995/12/05 20:49:45 prb Exp $
 */

/*
 * This code is partially derived from CMU PPP.
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * PPP frame encapsulation protocol routines
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_llc.h>
#include <net/if_dl.h>

#include <machine/cpu.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#ifdef ISO
#include <netiso/argo_debug.h>
#include <netiso/iso.h>
#include <netiso/iso_var.h>
#include <netiso/iso_snpac.h>
#endif

#include <net/ppp_proto.h>
#include <net/if_p2p.h>
#include <net/slcompress.h>
#include <net/pppvar.h>
#include <net/slip.h>

extern struct timeval time;

#ifdef GATEWAY
/*
 * We want to put "interactive" traffic on a high priority queue.
 * Packets with IPTOS_LOWDELAY are queued separately and sent
 * preferentially.  For now, we retain the following disgusting hack
 * to check TCP port numbers, as we might be forwarding packets
 * for a system that does not set the IP TOS field correctly.
 * To decide if that traffic is interactive, we check that
 * a) it is TCP and b) one of its ports is telnet, rlogin or ftp control.
 */
static u_short interactive_ports[8] = {
	0,      513,    0,      0,
	0,      21,     0,      23,
};
#define INTERACTIVE(p) (interactive_ports[(p) & 7] == (p))
#endif

#define RU(x)	(((x) + sizeof(long) - 1) & ~(sizeof(long) - 1))

/*
 * Output a paket to point-to-point interface with PPP encapsulation
 */
int
ppp_output(ifp, m, dst, rt)
	register struct ifnet *ifp;
	struct mbuf *m;
	struct sockaddr *dst;
	struct rtentry *rt;
{
	u_short type;
	int hl;
	int s;
	struct p2pcom *pp = (struct p2pcom *) ifp;
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	struct ppp_header phdr;
	struct ifqueue *outq;
	struct ip *ip;
	u_char *p;
	struct mbuf *m0;
	struct slcompress_buf cb;

	if (!(ifp->if_flags & IFF_UP)) {
		m_freem(m);
		return (ENETDOWN);
	}
	ifp->if_lastchange = time;

	cb.chlen = 0;
	cb.uhlen = 0;
	cb.bpflen = pp->p2p_if.if_bpf ? SLC_BPFHDRLEN : 0;

	outq = &ifp->if_snd;
	switch (dst->sa_family) {
#ifdef INET
	case AF_INET:
		/*
		 * Compute how much space we will need at the front
		 * of the packet.  Need to leave space if we have
		 * a BPF header as well.
		 * We know that all packets will have a compressable
		 * protocol field at this point.
		 */
		cb.lllen = (ppp->ppp_txflags & PPP_ACFC) ? 0 : 2;
		cb.lllen += (ppp->ppp_txflags & PPP_PFC) ? 1 : 2;

		if (ppp->ppp_ipcp.fsm_state != CP_OPENED) {
			if (ppp->ppp_ipcp.fsm_state == CP_CLOSED ||
			    ppp->ppp_ipcp.fsm_state == CP_CLOSING) {
				m_freem(m);
				return (EAFNOSUPPORT);
			}

			/* If there was a daemon queue to ipdefq */
			s = splimp();
			if (ppp->ppp_wdrop || ppp->ppp_ipdefq.ifq_len > 0) {
				m0 = m;
				m = 0;
				if (ppp->ppp_ipdefq.ifq_len > 10)
					IF_DEQUEUE(&ppp->ppp_ipdefq, m);
				IF_ENQUEUE(&ppp->ppp_ipdefq, m0);
			}

			/* Wake up the daemon */
			if (ppp->ppp_wdrop) {
				wakeup((caddr_t) &ppp->ppp_wdrop);
				ppp->ppp_wdrop = 0;
			}
			splx(s);
			if (m)
				m_freem(m);
			return (0);     /* simply drop on the floor */
		}
		type = PPP_IP;
		ip = mtod(m, struct ip *);

		/*
		 * Push "interactive" packets
		 * to the priority queue
		 */
		if (ip->ip_tos == IPTOS_LOWDELAY)
			outq = &pp->p2p_isnd;
#ifdef GATEWAY
		else if (ip->ip_p == IPPROTO_TCP && ip->ip_off == 0 &&
			 (ppp->ppp_dflt.ppp_flags & PPP_FTEL)) {

			/* get src & dst port numbers */
			hl = ntohl(((int *)ip)[ip->ip_hl]);
			if (INTERACTIVE(hl & 0xffff) || INTERACTIVE(hl >> 16))
				outq = &pp->p2p_isnd;
		}
#endif

		/*
		 * Handle TCP compression here
		 */
		if (ip->ip_p == IPPROTO_TCP && (ppp->ppp_txflags & PPP_TCPC)) {
			switch (sl_compress_tcp(&m, ip, ppp->ppp_ctcp,
			    SL_COMP_NOCID, &cb)) {
			case 0:
				return (ENOBUFS);
			case TYPE_COMPRESSED_TCP:
				type = PPP_VJC;
				break;
			case TYPE_UNCOMPRESSED_TCP:
				type = PPP_VJNC;
				break;
			case TYPE_IP:
				break;
			default:
				panic("Unknown type returned from sl_compress");
				break;
			}
		} else {
			if (cb.bpflen)
				hl = SLC_BPFHDR;
			else
				hl = cb.lllen;
			M_PREPEND(m, hl, M_DONTWAIT);
			if (m == 0)
				return (ENOBUFS);
		}
		break;
#endif
	default:
		printf("%s%d: af%d not supported\n", ifp->if_name,
				ifp->if_unit, dst->sa_family);
		m_freem(m);
		return (EAFNOSUPPORT);
	}

	/* Reset the idle time counter */
	ppp->ppp_idlecnt = 1;

	/*
	 * Add ppp serial line header
	 */
	p = mtod(m, u_char *) + cb.lllen + cb.bpflen;

	*--p = type & 0xff;

	if ((ppp->ppp_txflags & PPP_PFC) == 0 || (type & 0xff00))
		*--p = (type >> 8);

	if ((ppp->ppp_txflags & PPP_ACFC) == 0) {
		*--p = PPP_CONTROL;
		*--p = PPP_ADDRESS;
	}
	hl = (mtod(m, u_char *) + cb.lllen + cb.bpflen) - p;

	if (hl != cb.lllen)
		panic("ppp_output: header length changed");

	/*
	 * Queue packet on interface and
	 * start output if not active.
	 */
	s = splimp();
	if (IF_QFULL2(&ifp->if_snd, &pp->p2p_isnd)) {
		IF_DROP(&ifp->if_snd);
		splx(s);
		m_freem(m);
		return (ENOBUFS);
	}

	if (cb.bpflen) {
		/*
		 * At this point the mbuf still has the orignal data
		 * we need to prepend the the link level and compressed
		 * header as well as the 3 byte "bpf" header
		 */
		*--p = cb.chlen;
		*--p = hl;
		*--p = SLIPDIR_OUT;
		bpf_mtap(pp->p2p_if.if_bpf, m);
		cb.bpflen += cb.chlen + cb.lllen;
		/*	
		 * Now move the link level data and compressed header up.
		 * If we are lucky we can just trim mbufs, else we copy
		 */
		if (m->m_len == SLC_BPFHDR &&
		    m->m_next && m->m_next->m_len >= cb.uhlen) {
			m->m_data += SLC_BPFHDRLEN;
			m->m_pkthdr.len -= SLC_BPFHDR - cb.chlen - cb.lllen;
			m->m_len = cb.chlen + cb.lllen;

			m->m_next->m_len -= cb.uhlen;
			m->m_pkthdr.len -= cb.uhlen;
			m->m_next->m_data += cb.uhlen;
		} else {
			m->m_data += SLC_BPFHDRLEN;
			m->m_len -= SLC_BPFHDRLEN;
			m->m_pkthdr.len -= SLC_BPFHDRLEN;

			hl = SLC_BPFHDR + cb.uhlen - cb.chlen - cb.lllen;
			if (hl) {
				struct mbuf *m1 = m;
				m->m_pkthdr.len -= hl;
				while (m1 && hl) {
					if (hl >= m1->m_len) {
						hl -= m1->m_len;
						m1->m_len = 0;
						if ((m1 = m1->m_next) == 0)
							panic("overran mbuf");
					} else {
						m1->m_len -= hl;
						m1->m_data += hl;
						hl = 0;
					}
				}
				hl = cb.lllen + cb.chlen;
				if (m1->m_len < hl)
					panic("IP hdr in mbuf not contiguous");
				bcopy(p + SLC_BPFHDRLEN, mtod(m1, caddr_t), hl);
			}
		}
	}

	ifp->if_obytes += m->m_pkthdr.len;

	IF_ENQUEUE(outq, m);
	if ((ifp->if_flags & IFF_OACTIVE) == 0)
		(*ifp->if_start)(ifp);
	splx(s);
	return (0);
}

/*
 * Process a received PPP packet
 *
 * Make the assumption that the mbuf chain looks like:
 *
 *      MBUF0 (permanent and should not be freed or forwarded)
 *              [ 3 bytes of empty space ]
 *              Link Level
 *              VJ Compressed Header [if any]
 *
 *      MBUF1 (Uses external data of one cluster, 2K)
 *              [ m_data points 120 bytes into data ]
 *              Uncompressed Header [if any]
 *              Data
 */
int
ppp_input(pp, m)
	struct p2pcom *pp;
	struct mbuf *m;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;
	struct ifqueue *inq;
	int s, o;
	u_short type;
	u_char *p, *q;
	struct mbuf *m1;
	int llen, clen;

	/*
	 * There are three ways we can be called:
	 *
	 * m == 0: The internal mbuf has been filled by ppp_hdr
    	 *
	 * m != 0: ppp_hdr was called for exactly the header, m
	 *	   contains the data of the mbuf.  PPP_CSKIP bytes
	 *	   are free at the front of m.
	 *
	 * m != 0: ppp_hdr was not called. PPP_CSKIP bytes are free at
	 *	   the front of m.  This is currently not supported.
	 */

	if (pp->p2p_ibuf->m_len == 0)
		panic("ppp_hdr has not been called");

	if (m == 0)
		m = pp->p2p_ibuf;
	else if (m->m_flags & (M_PKTHDR | M_EXT))
		panic("ppp_input called with small or non pkthdr mbuf");
	else {
		m1 = m;
		m = pp->p2p_ibuf;
		m->m_pkthdr.len = m->m_len + m1->m_pkthdr.len; 
	}

	llen = 0;
	clen = 0;

	m1 = m->m_next;		/* This is the actual data */

	pp->p2p_if.if_lastchange = time;
	pp->p2p_if.if_ibytes += m->m_pkthdr.len;

	/* Skip address/control bytes */
	p = mtod(m, u_char *);

	if (p[0] == PPP_ADDRESS && p[1] == PPP_CONTROL)
		llen = 2;

	/* Retrieve the protocol type */
	if (p[llen] & 01)	/* Compressed protocol field */
		type = p[llen++];
	else {
		type = (p[llen] << 8) | p[llen+1];
		llen += 2;
	}

	if (m->m_len < llen) {
		if (m1)
			m_freem(m1);
		m->m_next = 0;
		m->m_len = 0;
		return (0);
	}

	s = splimp();
	switch (type) {
	default:
		if (pp->p2p_if.if_bpf) {
			m->m_pkthdr.len += SLC_BPFHDR - m->m_len;
			m->m_len = SLC_BPFHDR;
			m->m_data -= SLC_BPFHDRLEN;
			p = mtod(m, u_char *);
			p[0] = SLIPDIR_IN;
			p[1] = llen;
			p[2] = 0;
			bpf_mtap(pp->p2p_if.if_bpf, m);
		}
			
		m1->m_pkthdr.len = m->m_pkthdr.len - m->m_len;

		ppp_cp_in(pp, m1, type);
		goto out;

#ifdef INET
	case PPP_IP:
	raw_ip:
		m1 = m->m_next;
		schednetisr(NETISR_IP);
		inq = &ipintrq;
		break;

	case PPP_VJC:
		if (ppp->ppp_ctcp &&
		    sl_muncompress_tcp(&m, TYPE_COMPRESSED_TCP, ppp->ppp_ctcp,
		    llen) == 0)
			goto raw_ip;
		m_freem(m1);	/* m_freem handles NULL */
		goto out;

	case PPP_VJNC:
		/*
		 * We assume that sl_muncompress_tcp will always succeed
		 * on uncompressed packets.
		 */
		if (ppp->ppp_ctcp == 0) {
			m_freem(m);
			goto out;
		}
		(void) sl_muncompress_tcp(&m, TYPE_UNCOMPRESSED_TCP,
		    ppp->ppp_ctcp, llen);
		goto raw_ip;
#endif
	}

	/*
	 * Queue the packet to incoming list
	 */
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		if (m1)
			m_freem(m1);
	} else {
		if (pp->p2p_if.if_bpf) {
			clen = m->m_len - llen;
			m->m_data -= SLC_BPFHDRLEN;
			m->m_pkthdr.len += SLC_BPFHDR - m->m_len;
			m->m_len = SLC_BPFHDR;

			p = mtod(m, u_char *);
			p[0] = SLIPDIR_IN;
			p[1] = llen;
			p[2] = clen;
			bpf_mtap(pp->p2p_if.if_bpf, m);
		}
			
		m1->m_pkthdr.len = m->m_pkthdr.len - m->m_len;

		IF_ENQUEUE(inq, m1);
	}
out:
	splx(s);
	m->m_next = 0;
	m->m_len = 0;
	return (0);
}

/*
 * Put nbytes of data starting at cp into the input mbuf.
 * May be called multiple times as more data in the packet
 * arrives.
 */
int
ppp_hdr(pp, cp, nbytes)
	struct p2pcom *pp;
	u_char *cp;
	int nbytes;
{
	struct ppp_sc *ppp = (struct ppp_sc *)pp->p2p_private;

	return(vjc_hdr(&ppp->ppp_vjc, cp, nbytes));
}

/*
 * Polling the idle time counters
 */
void
ppp_idlepoll(arg)
	void *arg;
{
	struct ppp_sc *ppp;
	extern struct p2pcom *ppp_ifs;
	extern int ppp_idletimer;
	register struct p2pcom *pp, *ppn;
	int s;

	s = splimp();
	if (ppp_ifs == 0) {
		ppp_idletimer = 0;
		splx(s);
		return;
	}
	for (pp = ppp_ifs; pp; pp = ppn) {
		ppn = pp->p2p_next;
		ppp = (struct ppp_sc *)pp->p2p_private;
		if (ppp->Ppp_idletime == 0)
			continue;
		if ((pp->p2p_if.if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
			continue;
		if (ppp->ppp_idlecnt == 0)
			continue;
		if (ppp->ppp_idlecnt++ >= ppp->Ppp_idletime) {
			dprintf(("IDLE TIMER EXPIRED\n"));
			ppp_cp_close(pp, FSM_LCP);
		}
	}
	timeout(ppp_idlepoll, 0, hz);
	splx(s);
}
