/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_p2psubr.c,v 2.8 1995/12/05 20:45:22 prb Exp $
 */

/*
 * Common code for point-to-point synchronous serial links
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/proc.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>
#include <net/if_dl.h>
#include <net/if_types.h>

#include <net/if_p2p.h>

extern int np2pproto;

/*
 * Point-to-point interface attachment.
 * Checks the interface type and calls the
 * corresponding link-layer protocol attach routine.
 */
int
p2p_attach(pp)
	struct p2pcom *pp;
{
	register struct ifnet *ifp = &pp->p2p_if;
	short type;
	int s;
	int error = 0;

	s = splimp();

#ifdef	DEBUG
	if (pp->p2p_private)
		panic("p2p_attach: protocol did not clean up p2p_private");
#endif

	for (type = 0; type < np2pproto; ++type)
		if (ifp->if_type == p2pprotosw[type].pp_type)
			break;

	if (type == np2pproto) {
		error = EPFNOSUPPORT;
		goto out;
	}

	if (pp->p2p_proto != &p2pprotosw[type]) {
		ifp->if_output = p2pprotosw[type].pp_output;
		pp->p2p_input = p2pprotosw[type].pp_input;
		pp->p2p_modem = p2pprotosw[type].pp_modem;
		pp->p2p_hdrinput = p2pprotosw[type].pp_hdrinput;
		pp->p2p_proto = &p2pprotosw[type];
	}
	if (pp->p2p_proto->pp_attach)
		error = ((*pp->p2p_proto->pp_attach)(pp));
	if (error == 0)
		bpfattach(&ifp->if_bpf, ifp, p2pprotosw[type].pp_bpftype,
		    p2pprotosw[type].pp_hdrlen);

out:
	splx(s);
	return (error);
}

/*
 * Point-to-point protocol detach routine
 */
void
p2p_detach(pp)
	struct p2pcom *pp;
{
	int s;

	s = splimp();

	if (pp->p2p_proto && pp->p2p_proto->pp_detach)
		(*pp->p2p_proto->pp_detach)(pp);
#ifdef	DEBUG
	if (pp->p2p_private)
		panic("p2p_detach: protocol did not clean up p2p_private");
#endif
	splx(s);
}

/*
 * Point-to-point interface initialization.
 */
int
p2p_init(pp)
	struct p2pcom *pp;
{
	int error = 0;

	if (pp->p2p_proto && pp->p2p_proto->pp_init)
		error = (*pp->p2p_proto->pp_init)(pp);
	return (error);
}

/*
 * Point-to-point interface shutdown.
 */
void
p2p_shutdown(pp)
	struct p2pcom *pp;
{
	if (pp->p2p_proto && pp->p2p_proto->pp_shutdown)
		(*pp->p2p_proto->pp_shutdown)(pp);
}

/*
 * Handle protocol-specific ioctl
 */
int
p2p_ioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct proc *p = curproc;	/* XXX should be parameter */
	struct p2pcom *pp = (struct p2pcom *) ifp;
	struct ifreq *ifr;
	struct ifaddr *ifa;
	struct sockaddr_dl *sdl;
	short type, otype;
	int error;
	int s;

	/*
	 * Call the protocol-specific ioctl handler
	 */
	s = splimp();
	if (pp->p2p_proto && pp->p2p_proto->pp_ioctl) {
		error = (*pp->p2p_proto->pp_ioctl)(pp, cmd, data);
		if (error != EINVAL)
			goto out;
	}

	otype = type = pp->p2p_if.if_type;

	/*
	 * Process the common ioctls
	 */
	error = 0;
	switch (cmd) {

	default:
		error = EINVAL;
		goto out;

	case SIOCSIFADDR:
		if (p && (error = suser(p->p_ucred, &p->p_acflag)))
			goto out;

		if (type == IFT_NONE)
			error = ENETDOWN;
		else
			ifp->if_flags |= IFF_UP;
		break;

	case _SIOCSLINK:
		/*
		 * Internal-use version of SIOCSIFADDR for AF_LINK only;
		 * used to prevent type from changing accidentally
		 * if driver has default case for setting protocol addr.
		 */
		if (p && (error = suser(p->p_ucred, &p->p_acflag)))
			goto out;

		ifa = (struct ifaddr *)data;
		sdl = (struct sockaddr_dl *)(ifa->ifa_addr);
		for (type = 0; type < np2pproto; ++type)
			if (sdl->sdl_type == p2pprotosw[type].pp_type)
				break;
		if (type == np2pproto) {
			error = EAFNOSUPPORT;
			goto out;
		}
		type = sdl->sdl_type;
		break;

	case SIOCSIFDSTADDR:
	case SIOCSIFFLAGS:
		break;

	/*
	 * These should probably be done at a lower level,
	 * but this should be ok for now.
	 */
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		ifr = (struct ifreq *)data;
		switch (ifr->ifr_addr.sa_family) {
#ifdef INET
		case AF_INET:
			break;
#endif
		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;
	}

	/*
	 * An interface always goes down when changing types.
	 * An interface with type IFT_NONE can never be up.
	 */
	if (type != otype || type == IFT_NONE)
		ifp->if_flags &= ~IFF_UP;

	/*
	 * On a change from UP to DOWN call shutdown.
	 */
	if ((pp->p2p_oldflags & IFF_UP) && !(ifp->if_flags & IFF_UP))
		p2p_shutdown(pp);

	/*
	 * If changing types detach the old protocol and attach the new one.
	 */
	if (type != otype) {
		p2p_detach(pp);
		pp->p2p_if.if_type = type;
		if (error = p2p_attach(pp)) {
			/*
			 * Hmm, we failed.  Try to go back to the
			 * old type.  If we can't do that, default
			 * to the IFT_NONE case as a last resort.
			 */
			pp->p2p_if.if_type = otype;
			if (p2p_attach(pp)) {
				pp->p2p_if.if_type = IFT_NONE;
				p2p_attach(pp);
			}
			goto out;
		}
	}

	/*
	 * On a change from DOWN to UP call init.
	 */
	if (!(pp->p2p_oldflags & IFF_UP) && (ifp->if_flags & IFF_UP))
		error = p2p_init(pp);

	pp->p2p_oldflags = ifp->if_flags;
out:
	splx(s);
	return (error);
}

/*
 * Put nbytes of data starting at cp into the input mbuf.
 * May be called multiple times as more data in the packet
 * arrives.
 */
int
p2p_hdr(pp, cp, nbytes)
	struct p2pcom *pp;
	u_char *cp;
	int nbytes;
{
	struct mbuf *m = pp->p2p_ibuf;
	struct mbuf *m1;
	int r;

	if (nbytes == -1) {
		if (m) {
			if (m->m_next)
				m_freem(m->m_next);
			m->m_next = 0;
			m->m_len = 0;
		}
		return (0);
	}

	if (m == 0) {
		MGETHDR(m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			return (-1);
		pp->p2p_ibuf = m;
		m->m_len = 0;
	}

	/*
	 * Since we know nothing about the headers,
	 * the first mbuf will always be empty.
	 * If we do not have a second mbuf yet then
	 * this must be the first byte going in
	 * and it is safe to reset everything.
	 */
	if (m->m_next == 0) {
		m->m_pkthdr.rcvif = &pp->p2p_if;
		m->m_pkthdr.len = 0;
		m->m_data = m->m_pktdat;
		m->m_len = 0;
	}

	while (nbytes-- > 0) {
	 	r = ((m->m_flags & M_EXT) ? MCLBYTES :
		     (m->m_flags & M_PKTHDR) ? MHLEN : MLEN) - m->m_len;

		if (m == pp->p2p_ibuf || r <= 0) {
			if (m == pp->p2p_ibuf) {
				MGETHDR(m1, M_DONTWAIT, MT_DATA);
				if (m1)
					m1->m_pkthdr.rcvif = &pp->p2p_if;
				r = MHLEN;
			} else {
				MGET(m1, M_DONTWAIT, MT_DATA);
				r = MLEN;
			}

			if (m1 == 0) {	/* No space available */
				printf("MGET -- no space\n");
				m = pp->p2p_ibuf;
				if (m->m_next)
					m_freem(m->m_next);
				m->m_len = 0;
				m->m_next = 0;
				return (-1);
			}

			if (nbytes >= r || m != pp->p2p_ibuf) {
				MCLGET(m1, M_DONTWAIT);
				if (m1->m_flags & M_EXT)
					r = MCLBYTES;
			}
			m1->m_len = 0;
			m->m_next = m1;
			m = m1;
		}
		if (nbytes > 0) {
			int n = nbytes < r ? nbytes + 1 : r;
			bcopy(cp, mtod(m, u_char *) + m->m_len, n);

			m->m_len += n;
			nbytes -= n;
			cp += n;
			pp->p2p_ibuf->m_pkthdr.len += n;
		} else {
			m->m_data[m->m_len++] = *cp++;
			pp->p2p_ibuf->m_pkthdr.len++;
		}
	}
	return (1);
}
