/*
 * Copyright (c) 1991, 1993, 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	$Id: subr_rmap.c,v 2.1 1995/02/03 07:53:29 polk Exp $
 */

/*-
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	from: @(#)subr_rmap.c	7.9 (Berkeley) 5/11/91
 *
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/map.h>
#include <sys/proc.h>
#include <sys/kernel.h>

/*
 * Resource maps.
 *
 * [ In this implementation, m_size has two additional meanings:
 *	1. if zero, indicates an empty entry that is ignored. If
 *	   we get tight on entries, the function condense() digs
 *	   them out the hard way.
 *	2. if -1, indicates we are at the end of active entries.
 *	   this is a weak attempt to not search the table each
 *	   time, but debatable in value because it just indicates
 *	   a failure of rmalloc() sooner.
 *
 *	In addition, an address of zero is tolerated, unlike what
 *	the map.h header file alludes to. -wfj]
 */

void
rminit(rmap, length, base, ident, nentries)
	register struct map *rmap;
	long length, base;
	char *ident;
	int nentries;
{
	register struct mapent *me;

	/* zero table (empty entries are 0,0) */
	bzero(rmap, nentries * sizeof(struct map));

	/* initialize header */
	rmap->m_name = ident;
	rmap->m_limit = ((struct mapent *) rmap) + nentries - 1;

	/* put all of it in first entry */
	me = (struct mapent *) rmap + 1;
	me->m_addr = base;
	me->m_size = length;

	/* mark next entry as tail of list of entries */
	(++me)->m_size = 0xffffffff;
	return;
}

long
rmalloc(rmap, length)
	register struct map *rmap;
	long length;
{
	register struct mapent *me;
	long base;
	int zeros = 0;

	/* walk list, find first chunk big enough */
	/* XXX add a rover to break up congestion? */
	for (me = (struct mapent *) rmap + 1;
	     me <= rmap->m_limit && me->m_size != 0xffffffff;
	     ++me) {
		if (me->m_size >= length) {
			/* Found it -- allocate tail of block */
			me->m_size -= length;
			base = me->m_addr + me->m_size;
			if (zeros > 5 && zeros > (me - (struct mapent *)rmap) / 3)
				(void) condense(rmap);
			return (base);
		}
		if (me->m_size == 0)
			zeros++;
	}

	return (0);
}


/*
 * Put item back in list. Attempt to coalesce with other
 * items to minimize fragmentation. If we can't, insert and
 * shuffle the list. If too many pieces, log occurance and
 * drop fragment. We maintain a sorted list that we let rmalloc
 * leave empty entries in. 
 */
void
rmfree(rmap, length, base)
	struct map *rmap;
	long length, base;
{
	register struct mapent *me;
	struct mapent *pme, *nme;

	/* walk sorted list, find an entry to merge with, or insert before */
redo:
	for (me = (struct mapent *)rmap + 1; me <= rmap->m_limit; ++me) {

		/* are we a empty entry? ignore, condense will deal with them */
		if (me->m_size == 0)
			continue;

		/* check for sentinel */
		if (me->m_size == 0xffffffff) {
			me->m_addr = base;
			me->m_size = length;
			if (++me <= rmap->m_limit)
				me->m_size = 0xffffffff;
			return;
		}

		/* at head of current entry? coalesce */
		if (base + length == me->m_addr) {
			me->m_addr = base;
			me->m_size += length;

			/* can we coalesce the previous list entry as well? */
			for (pme = me - 1; pme > (struct mapent *) rmap; --pme)
				if (pme->m_size) {
					if (pme->m_addr + pme->m_size ==
					    me->m_addr) {
						pme->m_size += me->m_size;
						me->m_size = 0;
					}
					return;
				}
			return;
		}

		/* at tail of current entry? coalesce */
		if (me->m_addr + me->m_size == base) {
			me->m_size += length;

			/* can we coalesce the next list entry as well? */
			for (nme = me + 1;
			     nme <= rmap->m_limit && nme->m_size != 0xffffffff;
			     ++nme)
				if (nme->m_size) {
					if (nme->m_addr == base + length) {
						me->m_size += nme->m_size;
						nme->m_size = 0;
					}
					return;
				}
			return;
		}

		/* insert a fragment in front of current */
		if (me->m_addr > base + length) {

			/* empty entry ahead of us? */
			pme = me - 1;
			if (pme > (struct mapent *) rmap && pme->m_size == 0) {
				pme->m_addr = base;
				pme->m_size = length;
				return;
			}

			/* nope, insert in dense list */
			for (nme = me + 1; nme <= rmap->m_limit; ++nme) {
				if (nme->m_size == 0xffffffff) {
					nme->m_size = 0;
					if (nme < rmap->m_limit)
						(nme + 1)->m_size = 0xffffffff;
				}
				/* find next hole */
				if (nme->m_size == 0) {
					/* make space available */
					ovbcopy(me, me + 1,
					 (nme - me) * sizeof *me);

					/* insert our entry */
					me->m_addr = base;
					me->m_size = length;
					return;
				}
			}

			if (condense(rmap))
				goto redo;
			break;
		}
	}
	/* panic("rmfree"); */
	printf("rmfree: %s too fragmented, dropping storage\n", rmap->m_name);
}

/*
 * Squeeze out empty entries in map.
 * Done when list is too long.
 */
int
condense(map)
	struct map *map;
{
	struct mapent *hole, *start, *end;
	int anyfound = 0;

	/* find next empty entry */
	for (hole = (struct mapent *)map+1;
	     hole <= map->m_limit && hole->m_size;
	     ++hole)
		;

	for (; hole <= map->m_limit; hole = end - anyfound) {

		/* find next full entry */ 
		for (start = hole + anyfound;
		     start <= map->m_limit && start->m_size == 0;
		     ++start)
			;
		anyfound = start - hole;

		/* just empty till end? */
		if (start > map->m_limit) {
			hole->m_size = 0xffffffff;
			break;
		}

		/* and the hole after that */
		for (end = start + 1; end <= map->m_limit && end->m_size; ++end)
			;

		/* condense */
		ovbcopy (start, hole, (end - start) * sizeof (struct mapent));
	}

	return (anyfound);
}
