/*
 * Copyright (c) 1991, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: kern_acct.c,v 2.4 1995/09/18 18:00:08 karels Exp $
 */

/*-
 * Copyright (c) 1982, 1986, 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	from: @(#)kern_acct.c	8.1 (Berkeley) 6/14/93
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/namei.h>
#include <sys/resourcevar.h>
#include <sys/proc.h>
#include <sys/mount.h>
#include <sys/vnode.h>
#include <sys/file.h>
#include <sys/acct.h>
#include <sys/tty.h>
#include <sys/syslog.h>
#include <sys/kernel.h>

/*
 * SHOULD REPLACE THIS WITH A DRIVER THAT CAN BE READ TO SIMPLIFY.
 */
struct	vnode *acctp;		/* file of accounting records */
struct	ucred *acctcred;	/* credentials for writing acctp */

/*
 * Values associated with enabling and disabling accounting
 */
int	acctsuspend = 2;	/* stop accounting when < 2% free space left */
int	acctresume = 4;		/* resume when free space risen to > 4% */
int	acctchkfreq = 15;	/* frequency (in seconds) to check space */

static int accounting_disabled;
static void check_acct_fs_space __P((void));
static comp_t tv_to_comp __P((struct timeval *));

/* XXX */
extern void calcru __P((struct proc *, struct timeval *, struct timeval *,
    struct timeval *));
extern void timevalsub __P((struct timeval *, struct timeval *));

/*
 * Enable or disable process accounting.
 *
 * If a non-null filename is given, that file is used to store accounting
 * records on process exit. If a null filename is given process accounting
 * is terminated. If accounting is enabled, the system checks the amount
 * of freespace on the filesystem at timeval intervals. If the amount of
 * freespace is below acctsuspend percent, accounting is suspended. If
 * accounting has been suspended, and freespace rises above acctresume,
 * accounting is resumed.
 */
struct acct_args {
	char	*path;
};
int
acct(p, uap, retval)
	struct proc *p;
	struct acct_args *uap;
	int *retval;
{
	struct nameidata nd;
	struct vnode *vp = NULL;
	struct ucred *cr = NULL;
	struct vnode *oacctp;
	struct ucred *oacctcred;
	int error;

	if (error = suser(p->p_cred->pc_ucred, (short *)0))
		return (error);

	if (uap->path) {
		/*
		 * Allocate a vnode by 'opening' the file.
		 */
		NDINIT(&nd, LOOKUP, FOLLOW, UIO_USERSPACE, uap->path, p);
		if ((error = vn_open(&nd, FFLAGS(O_WRONLY), 0)) != 0)
			return (error);
		vp = nd.ni_vp;
		VOP_UNLOCK(vp);
		cr = p->p_ucred;
		crhold(cr);
	}

	/*
	 * Atomically swap accounting files.
	 * This guarantees that we don't get into a race with acct_process(),
	 * and that we don't lose any accounting.
	 * This could be simplified, but it's clearer this way.
	 */
	oacctp = acctp;
	oacctcred = acctcred;
	acctp = vp;
	acctcred = cr;

	if (oacctp) {
		(void)vn_close(oacctp, FWRITE, oacctcred, p);
		crfree(oacctcred);
	}

	if (acctp)
		check_acct_fs_space();

	return (0);
}

/*
 * Write an accounting record to the accounting file.
 * This code doesn't need a lock since crucial structures are on the stack.
 */
void
acct_process(p)
	struct proc *p;
{
	struct acct a;
	struct uio uio;
	struct iovec iov;
	struct timeval tv, utime, stime;
	struct pstats *psp = p->p_stats;
	struct rusage *rup;
	struct ucred *cr;
	struct vnode *vp;
	unsigned int i;

	check_acct_fs_space();

	if (acctp == NULL || accounting_disabled)
		return;

	vp = acctp;
	vref(vp);
	cr = acctcred;
	crhold(cr);

	tv.tv_usec = 0;

	strncpy(a.ac_comm, p->p_comm, sizeof a.ac_comm);
	calcru(p, &utime, &stime, NULL);
	a.ac_utime = tv_to_comp(&utime);
	a.ac_stime = tv_to_comp(&stime);
	tv = time;
	timevalsub(&tv, &psp->p_start);
	a.ac_etime = tv_to_comp(&tv);
	a.ac_btime = psp->p_start.tv_sec;
	a.ac_uid = p->p_cred->p_ruid;
	a.ac_gid = p->p_cred->p_rgid;
	rup = &psp->p_ru;
	/* XXX should we bother to preserve low order time bits? */
	if (i = (tv.tv_sec * AHZ) + (tv.tv_usec * AHZ) / 1000000)
		a.ac_mem = ((rup->ru_ixrss + rup->ru_idrss + rup->ru_isrss)
			* AHZ) / i;
	else
		a.ac_mem = 0;
	/* the magic 64 comes from section 3.9 of the BSD book */
	tv.tv_sec = (rup->ru_inblock + rup->ru_oublock) * 64;
	tv.tv_usec = 0;
	a.ac_io = tv_to_comp(&tv);
	if (p->p_pgrp->pg_session->s_ttyp)
		a.ac_tty = p->p_pgrp->pg_session->s_ttyp->t_dev;
	else
		a.ac_tty = NODEV;
	a.ac_flag = p->p_acflag;

	iov.iov_base = (caddr_t)&a;
	iov.iov_len = sizeof a;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = 0;
	uio.uio_resid = sizeof a;
	uio.uio_segflg = UIO_SYSSPACE;
	uio.uio_rw = UIO_WRITE;
	uio.uio_procp = p;

	/*
	 * If the write fails, we assume we're out of space.
	 */
	VOP_LOCK(vp);
	if (VOP_WRITE(vp, &uio, IO_APPEND, cr))
		accounting_disabled = 1;
	vput(vp);
	crfree(cr);
}

/*
 * The first macro tests true if we have run out of space on the accounting
 * file's filesystem; the second tests true if we can turn it back on.
 */
#define	ACCT_FS_INSUFFICIENT_SPACE(sfp) \
	((sfp)->f_bavail <= acctsuspend * (sfp)->f_blocks / 100)
#define	ACCT_FS_ENOUGH_SPACE(sfp) \
	((sfp)->f_bavail > acctresume * (sfp)->f_blocks / 100)

/*
 * Check the space available to the accounting file.
 * Disable accounting if there's too little, re-enable if it comes back.
 * Also disable accounting if the vnode has been revoked or we can't
 * do a statfs on its filesystem.
 */
static void
check_acct_fs_space()
{
	struct statfs sb;
	static long nextcheck;

	/* If not doing acounting, or we checked recently, just return. */
	if (acctp == NULL || time.tv_sec < nextcheck)
		return;
	nextcheck = time.tv_sec + acctchkfreq;
	if (acctp->v_type == VBAD ||
	    VFS_STATFS(acctp->v_mount, &sb, (struct proc *)0)) {
		accounting_disabled = 1;
		return;
	}
	if (accounting_disabled) {
		if (ACCT_FS_ENOUGH_SPACE(&sb)) {
			accounting_disabled = 0;
			log(LOG_NOTICE, "Accounting resumed\n");
		}
	} else {
		if (ACCT_FS_INSUFFICIENT_SPACE(&sb)) {
			accounting_disabled = 1;
			log(LOG_NOTICE, "Accounting suspended\n");
		}
	}
}

/*
 * ASSUMES TWO'S COMPLEMENT ARITHMETIC.
 */
static comp_t
tv_to_comp(tvp)
	struct timeval *tvp;
{
	register int exp;
	u_long mant;
#define MANTBITS 13
#define EXPBITS 3

	/*
	 * Raise exponent and drop bits off the right of the mantissa
	 * until the mantissa fits in its field.  If we run out of exponent
	 * space, return the maximum possible value.  With an AHZ value
	 * of 64, this is good for up to 8.5 years.
	 * (8191 * (1 << (3*7)) / 64 / 60 / 60 / 24 / 365.242 ~= 8.5)
	 */
	mant = tvp->tv_sec * AHZ + (tvp->tv_usec * AHZ) / 1000000;
	for (exp = 0; exp < (1 << EXPBITS); exp++, mant >>= EXPBITS)
		if (mant < (1 << MANTBITS))
			return (mant | (exp << MANTBITS));
	return (~0);
}
