/*-
 * Copyright (c) 1992, 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: boot.c,v 2.14 1996/01/11 16:37:02 karels Exp $
 */

/*-
 * Copyright (c) 1990, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * William Jolitz.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright (c) 1990, 1993\n\
        The Regents of the University of California.  All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)boot.c      8.1 (Berkeley) 6/11/93";
#endif /* not lint */

#include <sys/param.h>
#include <sys/reboot.h>
#include <a.out.h>
#include <sys/device.h>
#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <i386/isa/rtc.h>
#include <stand/stand.h>
#include <i386/stand/stand.h>

#include "pathnames.h"


/*
 * Boot program... arguments from lower-level bootstrap determine
 * whether boot stops to ask for system name and which device
 * boot comes from.
 */

char	line[100];
char	default_boot[100] = DEFAULTBOOT;
extern	int opendev, bootdev, cyloffset;
extern	int bootdebug;
int	echo = 1;
int	compressed;
int	pausedone;		/* pause() gave user a chance to abort */

struct bootparams {
	struct	bootparamhdr hdr;
	u_char	data[BOOT_MAXPARAMS];
} bootparams = {
	{ BOOT_MAGIC, sizeof(struct bootparamhdr) },
};
caddr_t ebootparams = (caddr_t) &bootparams + sizeof(struct bootparamhdr);

void echocmd __P((char *));
void load_file __P((char *, struct boot_options *, int));
int boot_file __P((int io, struct boot_options *o));
void start_prog __P((struct boot_options *));
int dobootopts __P((char *cp, struct boot_options *o));
void makebootline __P((void));
void copy_lowmem __P((char *s, char *d, int n));
int funzip_process __P((char *, int));
int funzip __P((int));
int fload __P((int));

void
main(howto, dev, off, bhdr)
	int howto, dev, off;
	struct bootparamhdr bhdr;
{
	int io;
	int n;
	char *cp;
	char *defboot = default_boot, defbuf[512];
	char *fp = NULL, *ep, *lp;
	struct bootparam *bp;
	struct boot_options o;

	trapinit();

	bzero(&o, sizeof(o));
	o.o_howto = howto;
	if ((dev&B_MAGICMASK) == B_DEVMAGIC) {
		bootdev = dev;
		cyloffset = off;
		makebootline();
#ifdef CHANGEFLOPPY
		if (B_TYPE(bios2realdev(bootdev)) == FD_MAJORDEV)
			o.o_howto = RB_SINGLE|RB_ASKNAME;
#endif
	} else {
		o.o_howto = RB_SINGLE|RB_ASKNAME;
		cyloffset = 0;
	}
	if (bhdr.b_magic == BOOT_MAGIC) {
		setbootparams(&bhdr);
		for (bp = B_FIRSTPARAM(&bootparams.hdr); bp;
		    bp = B_NEXTPARAM(&bootparams.hdr, bp))
		    	switch (bp->b_type) {
			case B_BIOSINFO:
				n = ((struct biosinfo *) B_DATA(bp))->kbflags;
				if (n & BIOS_KB_LSHIFT) {
					bootdebug++;
					printf("Debugging enabled\n");
				}
				break;
			case B_BIOSGEOM:
				setbiosgeom((struct biosgeom *) B_DATA(bp));
				break;
#ifdef BSDGEOM
			case B_BSDGEOM:
				setbsdgeom((struct bsdgeom *) B_DATA(bp));
				break;
#endif
			default:
				break;
		}
	}
	configure();

	/*
	 * If the file _PATH_BOOTDEFAULT exists, we read it for the default
	 * boot path.  We do minimal checking; if the file is not empty, we
	 * do what it says.
	 */
	if ((io = open(_PATH_BOOTDEFAULT, 0)) >= 0) {
		if ((n = read(io, defbuf, sizeof(defbuf))) > 0) {
			fp = defbuf;
			ep = fp + n;
		}
		close(io);
	}

	for (;;) {
		line[0] = '\0';
		lp = line;
		if (fp) {
			lp = fp;
			if (*fp != '-' && *fp != '#') {
				defboot = fp;
				/* o.o_howto = RB_AUTOBOOT; */
			}
			for (; fp < ep; fp++) {
				if (*fp == '\n') {
					*fp++ = 0;
					break;
				}
			}
			if (fp >= ep)
				fp = NULL;
			if (*lp == '#')
				continue;
			if (strncmp(lp, "-echo", sizeof("-echo") - 1) == 0) {
				echocmd(lp);
				continue;
			}
			if (echo)
				printf("Boot: %s\n", lp);
		} else if (o.o_howto & RB_ASKNAME) {
			printf("Boot: ");
			gets(line);
			cyloffset = 0;

		} else if (pausedone == 0) {
			/*
			 * default action: pause 5 sec, then load default.
			 * We do this if there is no _PATH_BOOTDEFAULT or
			 * if the file does not do a pause (e.g. just sets
			 * console, etc).
			 */
			if (pause(5)) {
				o.o_howto = RB_SINGLE|RB_ASKNAME;
				continue;
			}
		}

		if (*lp == 0) {
			lp = defboot;
			printf("Boot: %s\n", lp);
		}

		if (*lp == '-') {
			/*
			 * If a pause is interrupted, abort commands
			 * from boot.default.
			 */
			if (dobootopts(lp, &o) < 0) {
				fp = NULL;
				o.o_howto = RB_SINGLE|RB_ASKNAME;
			}
			continue;
		}
		load_file(lp, &o, 1);

		o.o_howto = RB_SINGLE|RB_ASKNAME;
		cyloffset = 0; 
	}
}

void
load_file(lp, o, start)
	char *lp;
	struct boot_options *o;
	int start;
{
	char *cp;
	int io, savehowto;

	savehowto = o->o_howto;

	/* process additional flags if any */
	if (cp = (char *)index(lp, ' ')) {
		if ((bootflags(cp, &o->o_howto, "boot-file")) == -1) {
			o->o_howto |= RB_ASKNAME;
			return;
		}
		*cp = '\0';
	}

	if ((io = open(lp, 0)) < 0)
		printf("can't open %s: %s\n", lp, ges(errno));
	else {
		sizemem(o);
		/*
		 * Check for name ending in ".gz",
		 * note that file is compressed if so.
		 */
		cp = lp;
		while (*cp)
			++cp;
		compressed = ((cp - 3) > lp && cp[-1] == 'z'
					    && cp[-2] == 'g'
					    && cp[-3] == '.');
		if (boot_file(io, o) && start)
			start_prog(o);
	}
	/*
	 * If doing "-load file -flags", remember the flags;
	 * otherwise restore the previous default.
	 */
	if (start)
		o->o_howto = savehowto;
}

/* get current seconds from real-time clock (if possible) */
int
getrtsec()
{
	int sa;
	int lim = 10000;

	sa = rtcin(RTC_STATUSA);
	if (sa == 0xff || sa == 0)
		return (-1);

	/* ready for a read? */
	while ((sa&RTCSA_TUP) == RTCSA_TUP && --lim >= 0)
		sa = rtcin(RTC_STATUSA);

	return (rtcin(RTC_SEC));
}

pause(sec)
	int sec;
{
	int toolong, now, tmp;

	/* clear any pending keycodes, e.g. function-key up from bootany */
	(void) cnpoll();
	printf("press any key to interrupt boot sequence");
	/*
	 * toolong came from 50mhz 486. This means that
	 * an unattended 50 mhz machine with a broken
	 * clock could take 5 minutes to boot.
	 */
	toolong = 0;
	pausedone = 1;
	now = getrtsec();
	while (sec) {
		if (cnpoll()) {
			printf("\n");
			return (1);
		}
		if ((tmp = getrtsec()) == now && toolong++ < (175000 * 60))
			continue;
		now = tmp;
		toolong = 0;
		sec--;
		printf(" %d", sec);
	}
	printf("\n");
	return (0);
}

/*
 * Process boot flags.
 * Supports either getopt-style boolean flags, or a single numeric parameter,
 * with either specifying the complete flags if present (ignoring previous).
 * Use original flags if no flags are actually present.
 * Could use getopt, but would have to construct argc/argv and do most of
 * these checks anyway.
 */
bootflags(cp, pflags, arg0)
	char *cp;
	int *pflags;
	char *arg0;
{
	int first = 1;
	int flags = 0;

	while (*cp) {
		while (*cp == ' ')
			cp++;
		if (*cp == '\0')
			break;
		if (*cp == '-') {
			first = 0;
			while (*++cp)
				switch (*cp) {
				case ' ':
					goto nextarg;
				case 'a':
					flags |= RB_ASKNAME;
					break;
				case 'd':
					flags |= RB_KDB;
					break;
				case 'r':
					flags |= RB_DFLTROOT;
					break;
				case 's':
					flags |= RB_SINGLE;
					break;
				case 'w':
					flags |= RB_RWROOT;
					break;
				default:
					goto usage;
			}
			continue;
		}
		if (first && *cp >= '0' && *cp <= '9') {
			*pflags = strtol(cp, 0, 0);
			return (0);
		}
		goto usage;

	nextarg: ;
	}
	if (first == 0)
		*pflags = flags;
	return (0);

usage:
	printf("usage: %s [ -adrsw ]\n", arg0);
	return (-1);
}

/*
 * Note: these names are 1-to-1 with the locators
 * passed to the kernel in the boot_devspec structure.
 */
static char *lfields[] = {
	"port",		/* LOC_IOBASE */
	"iosiz",	/* LOC_IOSIZE */
	"maddr",	/* LOC_MADDR */
	"msize",	/* LOC_MSIZE */
	"irq",		/* LOC_IRQ */
	"drq",		/* LOC_DRQ */
	"bustype",	/* LOC_BUSTYPE */
#if DSLOC_FLAGS != 7
#error "bootstrap assumes DSLOC_FLAGS is 7"
#endif
	"flags",	/* DSLOC_FLAGS */
	NULL
};

static void
devspecusage(char *str)
{
	char **p;

	if (str)
		printf("dev specification error: %s\n", str);
	printf("usage: -dev devname ioconf_field=value ...\n");
	printf("valid ioconf fields:");
	for (p = lfields; *p != NULL; p++)
		if (**p)
			printf(" %s", *p);
	printf("\nexample: -dev aha0 port=0x330 irq=12\n");
}

static int
rootdevusage()
{
	printf("usage: -rootdev {sd,fd,wd,sr}(adaptor, controller, unit, partition)\n");
	printf("  Example: to use the `h' partion of wd1 as root\n");
	printf("           -rootdev wd(0,0,1,7)\n");
	return (0);
}

static char *
dolfield(cpp, ds)
	char **cpp;
	struct boot_devspec *ds;
{
	char *cp, *cp2;
	int i, j;
	int len;

	cp = *cpp;
	for (i = 0; lfields[i] != NULL; i++) {
		len = strlen(lfields[i]);
		if (len == 0 || strncmp(cp, lfields[i], len) != 0)
			continue;
		cp += len + 1;		/* skip over = sign */

		if (*cp == '?') {
			cp++;
			if (i == LOC_IRQ || i == LOC_DRQ)
				ds->ds_loc[i] = -1;
			else
				return ("only irq and drq can be '?'");
		} else {
			j = strtol(cp, &cp2, 0);
			if (i == LOC_IRQ) {
				if (cp2 == cp && strncmp(cp, "IRQNONE",
				    sizeof("IRQNONE") - 1) == 0) {
					j = IRQNONE;
					cp2 += sizeof("IRQNONE") - 1;
				} else if (j > 15)
					return ("irq must be <= 15");
			}
			if (cp2 == cp)
				return ("missing or unrecognized value");
			cp = cp2;
			ds->ds_loc[i] = j;
		}
		ds->ds_validmask |= (1 << i);
		*cpp = cp;
		return (NULL);
	}
	printf("%s: ", cp);
	return ("unknown field");
}

/*
 * Process echo commands:
 *	-echoon
 *	-echooff
 *	-echo [-n] string
 * The syntax is extremely simplistic, supporting only single spaces.
 * "string"
 */
void
echocmd(cp)
	char *cp;
{
	int nl = 1;

	if (strcmp(cp, "-echoon") == 0) {
		echo = 1;
		return;
	}
	if (strcmp(cp, "-echooff") == 0) {
		echo = 0;
		return;
	}
	if ((cp = (char *)index(cp, ' ')) == NULL)
		printf("usage: -echo [-n] string\n");
	for (; *cp; cp++) {
		switch (*cp) {
		case ' ':
			continue;

		case '-':
			if (*++cp == 'n')
				nl = 0;
			cp++;
			continue;

		case '\\':
			cp++;
			goto print;

		default:
			goto print;
		}
	}
print:
	printf("%s%s", cp, nl ? "\n" : "");
}

static int
memsize(cp)
	char *cp;
{
	char *ecp;
	int size;

	size = strtol(cp, &ecp, 0);
	switch (*ecp) {
	case 'm':
	case 'M':
		size *= 1024 * 1024;
		break;
	case 'k':
	case 'K':
		size *= 1024;
		break;
	}
	return (size & ~(NBPG - 1));
}

int
autodebug(cp)
	char *cp;
{
	int flags = 0;

	for (; *cp; ++cp) {
		switch (*cp) {
		case '-':
		case ' ':
			continue;
		case 'd':
			flags |= AC_DEBUG | AC_PAGE;
			break;
		case 'a':
			flags |= AC_ASK | AC_DEBUG;
			break;
		case 'q':
			flags |= AC_QUIET;
			break;
		case 'v':
			flags |= AC_VERBOSE;
			break;
		case 'p':
			flags |= AC_PAGE;
			break;
		default:
			printf("usage: -autodebug [-a] [-d] [-p] [-q] [-v]\n");
			return (-1);
		}
	}
	return (flags);
}

/*
 * find pointer to first word after space or tab following input pointer,
 * returning NULL if none.
 */
char *
arg(cp)
	char *cp;
{

	if ((cp = (char *)index(cp, ' ')) == NULL)
		return (NULL);
	while (*cp == ' ' || *cp == '\t')
		cp++;
	if (*cp == '\0')
		return (NULL);
	return (cp);
}

static
dobootopts(cp, o)
	char *cp;
	struct boot_options *o;
{
	struct bootparam param;
	char *index();
	int val;

	if (*cp == '#')
		return (0);
	if (strncmp(cp, "-autodebug", sizeof("-autodebug") - 1) == 0) {
		if (cp = arg(cp)) {
			if (*cp == '-') {
				if ((val = autodebug(cp)) == -1)
					return (0);
			} else {
				val = strtol(cp, 0, 0);
				if (val == AC_DEBUG)
					val |= AC_PAGE;    /* XXX compat */
				else if (val == AC_ASK)
					val |= AC_DEBUG;   /* XXX compat */
			}
			param.b_type = B_AUTODEBUG;
			param.b_len = sizeof(param) + sizeof(val);
			addbootparam(&param, (void *) &val);
		} else
			printf("usage: -autodebug [-a] [-d] [-p] [-q] [-v]\n");
		return (0);
	}
	if (strncmp(cp, "-bootdebug", sizeof("-bootdebug") - 1) == 0) {
		if (cp = arg(cp))
			bootdebug = strtol(cp, 0, 0);
		else
			printf("usage: -bootdebug number\n");
		return (0);
	}
	if (strncmp(cp, "-bootflags", sizeof("-bootflags") - 1) == 0) {
		if (cp = arg(cp))
			(void) bootflags(cp, &o->o_howto, "-bootflags");
		else
			printf("usage: -bootflags [-adrsw]\n");
		return (0);
	}
	if (strncmp(cp, "-default_kernel", sizeof("-default_kernel") - 1) == 0){
		if (cp = arg(cp))
			if (strlen(cp) >= sizeof(default_boot))
				printf("usage: pathname too long\n");
			else
				strcpy(default_boot, cp);
		else
			printf("usage: -default_kernel boot-file\n");
		return (0);
	}
	if (strncmp(cp, "-load", sizeof("-load") - 1) == 0) {
		if (cp = arg(cp))
			load_file(cp, o, 0);
		else
			printf("usage: -load boot-file [-adrsw]\n");
		return (0);
	}
	if (strncmp(cp, "-start", sizeof("-start") - 1) == 0) {
		start_prog(o);
		return (0);
	}
	if (strncmp(cp, "-pause", sizeof("-pause") - 1) == 0) {
		if (cp = arg(cp)) {
			if (pause(strtol(cp, 0, 0)))
				return (-1);		/* abort script */
		} else
			printf("usage: -pause number\n");
		return (0);
	}
	if (strncmp(cp, "-waitnl", sizeof("-waitnl") - 1) == 0) {
		while (getchar() != '\n')
			;
		return (0);
	}
	if (strncmp(cp, "-basemem", sizeof("-basemem") - 1) == 0) {
		if (cp = arg(cp))
			o->o_basemem = memsize(cp);
		else
			printf("usage: -basemem size\n");
		return (0);
	}
	/*
	 * End of extended memory, not amount of extended memory.
	 * No checks will be done to determine if this is correct.
	 */
	if (strncmp(cp, "-memsize", sizeof("-memsize") - 1) == 0) {
		if (cp = arg(cp))
			o->o_extmem = memsize(cp) - IOM_END;
		 else
			printf("usage: -memsize size\n");
		return (0);
	}
	/*
	 * When doing auto sizing of memory stop here.
	 */
	if (strncmp(cp, "-extendend", sizeof("-extendend") - 1) == 0) {
		if (cp = arg(cp))
			o->o_searchend = memsize(cp);
		else
			printf("usage: -extendend number\n");
		return (0);
	}
	/*
	 * Use CMOS value to set limit on memory size
	 */
	if (strncmp(cp, "-cmosmem", sizeof("-cmosmem") - 1) == 0) {
		o->o_searchend = -1;
		return (0);
	}
	/*
	 * Inhibit use of cache flush instruction on 486+
	 */
	if (strncmp(cp, "-noflushcache", sizeof("-noflushcache") - 1) == 0) {
		extern int noflushcache;

		noflushcache = 1;
		return (0);
	}
	if (strncmp(cp, "-changedisk", sizeof("-changedisk") - 1) == 0) {
		o->o_changedisk = 1;
		return (0);
	}
	if (strncmp(cp, "-dev", sizeof("-dev") - 1) == 0) {
		struct boot_devspec ds;
		int i;
		char *cp1, *cp2;

		bzero(&ds, sizeof(ds));
		cp += sizeof("-dev") - 1;
		while (*cp == ' ')
			cp++;
		i = strcspn(cp, "0123456789 =*");
		if (i == 0) {
			devspecusage(NULL);
			return (0);
		}
		if (i > sizeof(ds.ds_driver) - 1) {
			devspecusage("device name too long");
			return (0);
		}
		cp1 = cp + i;
		if (*cp1 == 0) {
			devspecusage("unexpected end of line");
			return (0);
		}
		if (*cp1 == '=' || *cp1 == ' ') {
			devspecusage("device name not followed by unit number");
			return (0);
		}
		if (*cp1 == '*') {
			ds.ds_unit = B_WILDDEV;
			cp2 = cp1 + 1;
		} else
			ds.ds_unit = strtol(cp1, &cp2, 0);
		*cp1 = 0;
		strcpy(ds.ds_driver, cp);
		cp = cp2;
		while (*cp == ' ')
			cp++;
		while (*cp != 0) {
			if ((cp2 = dolfield(&cp, &ds)) != NULL) {
				devspecusage(cp2);
				return (0);
			}
			while (*cp == ' ')
				cp++;
		}
		param.b_type = B_DEVSPEC;
		param.b_len = sizeof(param) + sizeof(ds);
		addbootparam(&param, (void *) &ds);
		return (0);
	}
	if (strncmp(cp, "-console", sizeof("-console") - 1) == 0) {
		cp += sizeof("-console") - 1;
		while (*cp == ' ')
			cp++;
		setconsole(cp);
		return (0);
	}
	if (strncmp(cp, "-rootdev", sizeof("-rootdev") - 1) == 0) {
		int i;

		cp += sizeof("-rootdev") - 1;
		while (*cp == ' ')
			cp++;
		if ((i = dev_decode(cp, NULL)) == -1)
			return (rootdevusage());
		/*
		 * Check for open paren, as dev_decode returns bootdev
		 * if only a name is present.
		 */
		while (*cp && *cp != '(')
			++cp;
		if (*cp != '(')
			return (rootdevusage());
		o->o_rootdev = i;
		o->o_rootdevset = 1;
		return (0);
	}
	if (strncmp(cp, "-kernspace", sizeof("-kernspace") - 1) == 0) {
		cp += sizeof("-kernspace") - 1;
		if (cp = arg(cp)) {
			val = memsize(cp);
			param.b_type = B_KERNSPACE;
			param.b_len = sizeof(param) + sizeof(val);
			addbootparam(&param, (void *) &val);
		} else
			printf("usage: -kernspace size\n");
		return (0);
	}
	printf("unknown command: %s\n", cp);
	return (0);
}

setbootparams(bhp)
	struct bootparamhdr *bhp;
{

	if (bhp->b_len <= sizeof(bootparams)) {
		bcopy(bhp, &bootparams, bhp->b_len);
		ebootparams = (caddr_t) &bootparams + bootparams.hdr.b_len;
	} else
		printf("setbootparams: parameters too large\n");
}

addbootparam(bp, val)
	struct bootparam *bp;
	void *val;
{

	if (ebootparams + bp->b_len <=
	    (caddr_t) &bootparams + sizeof(bootparams)) {
		bcopy(bp, ebootparams, sizeof(*bp));
		bcopy(val, ebootparams + sizeof(*bp), bp->b_len - sizeof(*bp));
		ebootparams += bp->b_len;
		bootparams.hdr.b_len += bp->b_len;
	} else
		printf("boot parameters too large\n");
}

/*
 * Fetch a boot parameter of the specified type.
 * If bp is null, fetch the first; otherwise, fetch the next one
 * after the one specified.
 */
struct bootparam *
getbootparam(type, bp)
	int type;
	struct bootparam *bp;
{

	if (bp)
		bp = B_NEXTPARAM(&bootparams.hdr, bp);
	else
		bp = B_FIRSTPARAM(&bootparams.hdr);
	for (; bp; bp = B_NEXTPARAM(&bootparams.hdr, bp))
		if (bp->b_type == type)
			return (bp);
	return (NULL);
}

/*
 * Add boot option with an alternate bootdev value usable by the kernel,
 * which does not use the BIOS numbering.
 */
addrealdev(dev)
	int dev;
{
	struct bootparam param;

	param.b_type = B_REALBOOTDEV;
	param.b_len = sizeof(param) + sizeof(dev);
	addbootparam(&param, (void *) &dev);
}

/*
 * Translate a device number for the BIOS disk driver into a
 * device number usable by the kernel (floppy, wd or scsi disk).
 * The "adaptor" field is nonzero for hard disks, zero for floppy.
 */
int
bios2realdev(dev)
	int dev;
{
	int types;

	if (B_TYPE(dev) != BD_MAJORDEV)
		return (dev);

	if (B_ADAPTOR(dev) == 0) {
		dev &= ~(B_TYPEMASK << B_TYPESHIFT); 
		dev |= (FD_MAJORDEV << B_TYPESHIFT);
		return (dev);
	}
	/*
	 * It is some sort of hard disk.
	 */
	dev &= ~((B_TYPEMASK << B_TYPESHIFT) |
	    (B_ADAPTORMASK << B_ADAPTORSHIFT));
	types = rtcin(RTC_DISKTYPE);

	/*
	 * We have to handle first and second units differently.
	 * We check the drives known to the CMOS, which must be wd-style
	 * drives.  Drives not known to the CMOS must be SCSI.
	 * If one wd-type hard disk exists, drive 0 is wd0,
	 * and drive 1 is sd0.
	 */
	if (B_UNIT(dev) == 0) {
		if ((types & 0xf0) != 0)	/* wd drive 0 present? */
			dev |= (WD_MAJORDEV << B_TYPESHIFT);
		else
			dev |= (SD_MAJORDEV << B_TYPESHIFT);
		return (dev);
	}

	if (types & 0x0f) {			/* wd drive 1 present? */
		dev |= (WD_MAJORDEV << B_TYPESHIFT);
#if 0
		/* is there ever a drive 1 present with no drive 0? */
		if ((types & 0xf0) == 0)
			dev &= ~(B_UNITMASK << B_UNITSHIFT);	/* unit 0??? */
#endif
		return (dev);
	}
	dev |= (SD_MAJORDEV << B_TYPESHIFT);
	/* if wd drive 0 is present, this is sd0 */ 
	if ((types & 0xf0) != 0)
		dev &= ~(B_UNITMASK << B_UNITSHIFT);
	return (dev);
}

#if 0
dread(int io, char *addr, size_t size)
{
	printf("read addr = %x, size = %x\n",
		(u_int)addr, size);
	while (getchar() != '\n')
		;
	return (read(io, addr, size));
}

dbzero(char *addr, size_t size)
{
	printf("bzero addr = %x, size = %x\n",
		(u_int)addr, size);
	while (getchar() != '\n')
		;
	bzero(addr, size);
}

#define DREAD(io, addr, size) dread((io), (addr), (size))
#define DBZERO(addr, size) dbzero((addr), (size))
#else
#define DREAD(io, addr, size) read((io), (addr), (size))
#define DBZERO(addr, size) bzero((addr), (size))
#endif

static char lowmem[LOWMEM];		/* bounce buffer for first LOWMEM */

#define MAXBOOT		(BOOTREL - (IOM_END - iom_begin) - 8*1024)
#define	XLOW_SIZE	(iom_begin - x.a_entry)

int
fload(io)
	int io;
{
	static char buf[32*1024];
	int count, ret;

	while ((count = DREAD(io, buf, sizeof(buf))) > 0)
		if ((ret = funzip_process(buf, count)) < 0)
			return (ret);
	funzip_process(buf, count);	/* one last time? */
	return (count);	/* ??? */
}

static int loadstate;
static struct exec x;
static int bsize;	/* amount of memory being bounced in */
static int iom_begin;
static int iom_added;
static char *addr;

/*
 * funzip_process is called with each buffer read by funzip()
 * or fload().  It deals with the a.out header, then handles text
 * and data.  Once everything has been loaded, we return a negative
 * value to indicate that we are done (hopefully successfully).
 */
int
funzip_process(buf, nbytes)
	char *buf;
	int nbytes;
{
	static char progress_string[] = "/-\\|";
	static int progress = 0;
	static char *eaddr;
	static int textdone;
	int i;
	int ret = nbytes;

	while (nbytes > 0) {
		if (loadstate == -2)
			return (ret);
		if (loadstate < 0)
			return (-1);

		printf("%c\b", progress_string[progress++]);
		if (progress_string[progress] == 0)
			progress = 0;

		/*
		 * First read the exec header
		 */
		if (loadstate < sizeof(struct exec)) {
			i = sizeof(struct exec) - loadstate;
			if (i > nbytes)
				i = nbytes;
			bcopy(buf, ((char *)&x) + loadstate, i);
			buf += i;
			loadstate += i;
			nbytes -= i;
			if (loadstate == sizeof(struct exec)) {
				if (x.a_magic != 0407 && x.a_magic != 0413 &&					    x.a_magic != 0410) {
					printf("Bad format\n");
					loadstate = -1;
					return (-1);
				}
				x.a_entry &= 0x00ffffff;
				if (x.a_entry == 0)
					iom_begin = IOM_BEGIN;
				if (x.a_text + x.a_data >= MAXBOOT) {
					printf("File too large (%dK limit)\n",
					    MAXBOOT / 1024);
					loadstate = -1;
					return (-1);
					
				}
				bsize = 0;
				textdone = 0;
				addr = (char *)x.a_entry;
				if (x.a_text > XLOW_SIZE)
					eaddr = (char *)IOM_END + x.a_text -
					    XLOW_SIZE;
				else
					eaddr = (char *)x.a_entry + x.a_text;
				printf("%d", x.a_text);
			}
			continue;
		}
		/*
		 * "Seek" to CLBYTES on 0413 mode files.
		 */
		if (x.a_magic != 0413 && loadstate == sizeof(struct exec))
			loadstate = CLBYTES;

		if (loadstate < CLBYTES) {
			i = CLBYTES - loadstate;
			if (i > nbytes)
				i = nbytes;
			buf += i;
			loadstate += i;
			nbytes -= i;
			continue;
		}

		if (loadstate == CLBYTES && x.a_entry < LOWMEM)
			bsize = LOWMEM - x.a_entry;

		if (loadstate < CLBYTES + bsize) {
			i = CLBYTES + bsize - loadstate;
			if (i > nbytes)
				i = nbytes;
			bcopy(buf, &lowmem[x.a_entry + loadstate - CLBYTES], i);
			buf += i;
			addr += i;
			loadstate += i;
			nbytes -= i;
			continue;
		}
		if (loadstate == CLBYTES + bsize)
			loadstate = CLBYTES + LOWMEM;

		/*
		 * We are loading either text or data.
		 * addr points to where the next byte should go.
		 * eaddr points to just after the last byte.
		 * We must skip over the I/O hole if it is between
		 * addr and eaddr.
		 */
		if (addr < eaddr) {
			i = eaddr - addr;
			if (i > nbytes)
				i = nbytes;
			if (addr < (char *)iom_begin &&
			    addr + i > (char *)iom_begin)
				i = (char *)iom_begin - addr;
			bcopy(buf, addr, i);
			addr += i;
			buf += i;
			loadstate += i;
			nbytes -= i;
			if (addr == eaddr && textdone == 0) {
				/*
				 * Done with text.  Set up for data
				 */
				textdone = 1;
				printf("+%d", x.a_data);
				if (addr >= (char *)iom_begin)
					iom_added = 1;
					
				if (x.a_magic == 0413 || x.a_magic == 0410)
					while ((int)addr & CLOFSET)
						*addr++ = 0;
				if (addr == (char *)iom_begin) {
					addr = (char *)IOM_END;
					iom_added = 1;
				}
				eaddr = addr + x.a_data;
				if (eaddr > (char *)iom_begin && !iom_added) {
					eaddr += IOM_END - iom_begin;
					iom_added = 1;
				}
			} else if (addr == eaddr && textdone == 1) {
				loadstate = -2;	/* We are done loading! */
			} else if (addr == (char *)iom_begin)
				addr = (char *)IOM_END;
			continue;
		}
		nbytes = 0;
	}
	return (ret);
}

/*
 * Load a file into memory in preparation to run it.
 * The entry location in the a.out header is used as the starting
 * location for the program in physical memory.  If loading a program
 * below LOWMEM, load the portion below LOWMEM in a temporary area
 * until done reading, as the BIOS disk driver depends on the contents
 * of low memory.
 */
boot_file(io, o)
	int io;
	struct boot_options *o;
{
	int i;
	char c;

	bsize = 0;
	loadstate = 0;
	iom_added = 0;
	iom_begin = o->o_basemem;

	/*
	 * Read a.out header, text and data via compressed or
	 * uncompressed stream with callbacks to funzip_process.
	 */
	if (compressed) {
		/*
		 * funzip reads the file sequentially,
		 * calling funzip_process with each chunk.
		 */
		funzip(io);
		if (loadstate != -2)
			goto shread;
	} else {
		fload(io);
		if (loadstate != -2)
			goto shread;
	}
	printf("+%d", x.a_bss);
	/*
	 * Kludge: if program including bss is too large, don't clear bss;
	 * currently only kernels are that large, and the kernel clears bss.
	 */
	if (x.a_text + x.a_data + x.a_bss + x.a_entry >= MAXBOOT)
		x.a_bss = MAXBOOT - (x.a_text + x.a_data + x.a_entry);
	if (addr < (char *)iom_begin && addr + x.a_bss > (char *)iom_begin) {
		DBZERO(addr, iom_begin - (int)addr);
		DBZERO((char *)IOM_END, x.a_bss - (iom_begin - (int)addr));
	} else
		DBZERO(addr, x.a_bss);

	printf(" start 0x%x\n", x.a_entry);

	close(io);

	/*
	 * If we booted from a floppy device, we ask about changing floppies
	 * (so you can mount an alternate root disk) before continuing. This
	 * is set with -changedisk in _PATH_BOOTDEFAULT.
	 */
	if (o->o_changedisk && B_TYPE(bios2realdev(opendev)) == FD_MAJORDEV) {
		printf("File loaded -- change diskettes if necessary, then hit return");
		while (getchar() != '\n')
			;
	}
	o->o_loaded = 1;
	o->o_entry = x.a_entry;
	if (o->o_rootdevset)
		addrealdev(o->o_rootdev);
	else
		addrealdev(bios2realdev(opendev));

	return (1);
shread:
	printf("Short read\n");
	return (0);
}

void
start_prog(o)
	struct boot_options *o;
{
	int i;

	if (o->o_loaded == 0) {
		printf("no program loaded\n");
		return;
	}

	/*
	 * Just prior to calling the program, load up the low memory
	 * We use copy_lowmem so it can mark the BIOS code as
	 * unusable if we happen to clobber it.
	 */
	if (o->o_entry < LOWMEM)
		copy_lowmem(&lowmem[o->o_entry], (char *)o->o_entry, 
			LOWMEM - o->o_entry);

	i = (*((int (*)()) o->o_entry))(o->o_howto, opendev, cyloffset,
	    bootparams);

	if (i)
		printf("exit %d\n", i); 
	o->o_loaded = 0;
}

void
makebootline()
{
	static char x[] = "0123456789abcdef";
	int param[4];
	char linebuf[sizeof line];
	register int i;
	register char *cp = linebuf;
	register char *cp2;
	int type = B_TYPE(bootdev);

	if (type >= ndevs || devsw[type] == NULL) {
		printf("bootdev 0x%x?\n", bootdev);
		return;
	}
	param[0] = B_ADAPTOR(bootdev);
	param[1] = B_CONTROLLER(bootdev);
	param[2] = B_UNIT(bootdev);
	param[3] = B_PARTITION(bootdev);

	for (cp2 = devsw[type]->dv_name; *cp2; *cp++ = *cp2++)
		;

	*cp++ = '(';

	for (i = 0; i < sizeof(param) / sizeof(param[0]); ++i) {
		if (i > 0)
			*cp++ = ',';
		if (param[i] >= 0xa) {
			*cp++ = '0';
			*cp++ = 'x';
		}
		if (param[i] >= 0x10) {
			*cp++ = x[param[i] >> 4];
			param[i] &= 0xf;
		}
		*cp++ = x[param[i]];
	}

	*cp++ = ')';

	for (cp2 = default_boot; *cp++ = *cp2++; )
		;

	for (cp = line, cp2 = linebuf; *cp++ = *cp2++; )
		;
}
