/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: pci.c,v 2.7 1995/12/12 22:26:57 karels Exp $
 */

/*
 * PCI support routines
 */

#include <sys/param.h>
#include <i386/pci/pci.h>

#include <sys/device.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>

#ifndef PCI_DEFAULTBURST 
#define PCI_DEFAULTBURST 5	/* Default max burst is 32LW */
#endif
int	pci_log2_burstsize = PCI_DEFAULTBURST;

static	int pci_mechanism;
int	pci_probe();
void	pci_id __P((pci_devaddr_t *pa));
int	(*pci_probefn)() = pci_probe;
extern	int autodebug;

extern	int pci_scan_buses;
extern	int pci_scan_functions;
int	pci_scan_agents;

int
pci_probe()
{
	pci_devaddr_t pa;
	int found_one = 0, switched = 0;
	int tmp;

	/*
	 * Determine whether PCI is present by attempting to determine
	 * which pci configuration mechanism the machine uses.  Check
	 * for mechanism 2 first as the test for mechanism 1 will
	 * also pay off on mechanism 2 systems.
	 */
	outb(PCI_M2_CSE, 0);
	outb(PCI_M2_FORW, 0);
	if (inb(PCI_M2_CSE) == 0 && inb(PCI_M2_FORW) == 0)
		pci_mechanism = 2;
	else {
		outl(PCI_M1_CAR, PCI_M1_CM);
		if (inl(PCI_M1_CAR) != PCI_M1_CM) {
			aprint_debug("pci config registers not found\n");
			return (0);
		}
		pci_mechanism = 1;
		outl(PCI_M1_CAR, 0);
	}

retry:
	pci_scan_agents = pci_mechanism == 1 ? PCI_M1_AGENTS : PCI_M2_AGENTS;
	bzero(&pa, sizeof (pa));

	for (tmp = 0; tmp < pci_scan_buses; tmp++) {
	    pa.d_bus = tmp;
	    for (pa.d_agent = 0; pa.d_agent < pci_scan_agents; pa.d_agent++) {
		for (pa.d_function = 0; pa.d_function < pci_scan_functions;
		    pa.d_function++) {
			if (pci_inw(&pa, PCI_VENDOR_ID) == 0xffff)
				continue;
			if (found_one == 0) {
				found_one = 1;
				aprint_normal(
				  "pci0 at root: configuration mechanism %d\n",
				    pci_mechanism);
				pci_id_hdr();
			}
			pci_id(&pa);
		}
	    }
	}
	if (found_one == 0) {
		/*
		 * If we went all the way through the pci address
		 * space and didn't see any devices assume that
		 * the wrong configuration mechanism was selected
		 * and try with the other. Some/All Compaq systems
		 * have this problem.
		 */
		if (switched == 0) {
			switched = 1;
			pci_mechanism = pci_mechanism == 1 ? 2 : 1;
			aprint_debug("switching pci configuration mechanisms\n");
			goto retry;
		}
		/*
		 * Normally we should at least see the ISA/EISA
		 * bridge chips even if no boards are plugged
		 * into the motherboard.
		 */
		aprint_debug("no pci devices found\n");
		return (0);
	}

	if (!elcr_present) {
		/* 
		 * Figure out if there is a 82374EB.
		 * Select 82374EB conf register and write id to register.
		 */
		/* XXX we still need a header file for 82374EB registers */
		outb(0x22, 0x2);
		outb(0x23, 0xf);
		if (inb(0x23) == 0xf)		/* did we select 82374 */
			elcr_present = 1;	/* if so, we have elcr regs */
	}
	return (1);
}

/*
 * Enable pci configuration space io and select the address
 * passed in by paddr.  For configuration mechanism 2 only
 * part of the address selection can be done here.  The agent and
 * offset fields are deferred until the actual ins and outs are
 * done to fetch or store the data.
 */
static void
pci_enable(pci_devaddr_t *paddr, u_char offset)
{

	if (pci_mechanism == 2) {
		outb(PCI_M2_CSE, PCI_M2_CM | paddr->d_function << PCI_M2_FN_S);
		outb(PCI_M2_FORW, paddr->d_bus);
	} else
		outl(PCI_M1_CAR,
		    PCI_M1_CM |
		    (paddr->d_bus << PCI_M1_BUS_S) |
		    (paddr->d_agent << PCI_M1_AGENT_S) |
		    (paddr->d_function << PCI_M1_FN_S) |
		    (offset & PCI_M1_OFFSET_M));
}

/*
 * Disable pci configuration space io. This eliminates theoretical
 * problems with other devices using the same port address as
 * are used to talk to the pci configuration space.
 */
static void
pci_disable()
{

	if (pci_mechanism == 2) {
		outb(PCI_M2_CSE, 0);
		outb(PCI_M2_FORW, 0);
	} else
		outl(PCI_M1_CAR, 0);
}

/*
 * Build the pci mechanism 2 port address needed to fetch or store
 * data.  This does the portion of the address selection deferred by 
 * pci_enable.
 */
#define PCI_MODE2ADDR(paddr, offset) \
	(PCI_M2_CBP | ((paddr)->d_agent << PCI_M2_AGENT_S) | (offset))

/*
 * The following 6 routines are exported for actually reading
 * or writing pci configuration space. 
 */
u_char
pci_inb(pci_devaddr_t *paddr, u_char offset)
{
	u_long result;

	pci_enable(paddr, offset);

	if (pci_mechanism == 2) 
		result = inb(PCI_MODE2ADDR(paddr, offset));
	else {
		result = inl(PCI_M1_CDR);
		result >>= 8 * (offset & 0x3);
		result &= 0xff;
	}
	pci_disable();
	return (result);
}

u_short
pci_inw(pci_devaddr_t *paddr, u_char offset)
{
	u_long result;

	pci_enable(paddr, offset);
	if (pci_mechanism == 2) 
		result = inw(PCI_MODE2ADDR(paddr, offset));
	else {
		result = inl(PCI_M1_CDR);
		result >>= 8 * (offset & 0x3);
		result &= 0xffff;
	}
	pci_disable();
	return (result);
}

u_long
pci_inl(pci_devaddr_t *paddr, u_char offset)
{
	u_long result;

	pci_enable(paddr, offset);
	if (pci_mechanism == 2) 
		result = inl(PCI_MODE2ADDR(paddr, offset));
	else
		result = inl(PCI_M1_CDR);
	pci_disable();
	return (result);
}

void
pci_outb(pci_devaddr_t *paddr, u_char offset, u_char value)
{

	pci_enable(paddr, offset);
	if (pci_mechanism == 2) 
		outb(PCI_MODE2ADDR(paddr, offset), value);
	else {
		u_int tmp;
		tmp = inl(PCI_M1_CDR);
		tmp &= ~(0xff << (8 * (offset & 0x3)));
		tmp |= value << (8 * (offset & 0x3));
		outl(PCI_M1_CDR, tmp);
	}
	pci_disable();
}

void
pci_outw(pci_devaddr_t *paddr, u_char offset, u_short value)
{

	pci_enable(paddr, offset);
	if (pci_mechanism == 2) 
		outw(PCI_MODE2ADDR(paddr, offset), value);
	else {
		u_int tmp;

		tmp = inl(PCI_M1_CDR);
		tmp &= ~(0xffff << (8 * (offset & 0x3)));
		tmp |= value << (8 * (offset & 0x3));
		outl(PCI_M1_CDR, tmp);
	}
	pci_disable();
}

void
pci_outl(pci_devaddr_t *paddr, u_char offset, u_long value)
{

	pci_enable(paddr, offset);
	if (pci_mechanism == 2) 
		outl(PCI_MODE2ADDR(paddr, offset), value);
	else
		outl(PCI_M1_CDR, value);
	pci_disable();
}

/* print header for pci_id output */
pci_id_hdr()
{

	aprint_debug("PCI devices:\n");
	aprint_debug("bus agent func vendor device line pin\n");
}

/* 
 * If a device is present at the supplied address print out
 * the address along with the vendor id, device id, interrupt
 * line and interrupt pin.
 */
void
pci_id(pci_devaddr_t *pa)
{

	aprint_debug("%3x %4x %4x %6x %6x %4x %3x\n",
	    pa->d_bus, pa->d_agent, pa->d_function,
    	    pci_inw(pa, PCI_VENDOR_ID),
	    pci_inw(pa, PCI_DEVICE_ID),
	    pci_inb(pa, PCI_I_LINE),
	    pci_inb(pa, PCI_I_PIN));
}

/*
 * Record "claims" to devices as we find them so that each device
 * is found only once.  For now, use a fixed-size array to record
 * these, although that limits the maximum number of devices
 * that can be located.   We should change this to use a linked list. 
 */
#ifndef PCI_CLAIMSIZE
#define PCI_CLAIMSIZE 20
#endif
static	int pci_claimed[PCI_CLAIMSIZE];

/*
 * This routine is called by the individual device probe routines. 
 * It will scan through the pci configuration address space calling
 * the supplied match routine with all device addresses which have
 * not previously been claimed. If the match routine returns true
 * scanning will cease at that point and that address will be
 * marked as claimed.
 */
pci_devaddr_t *
pci_scan(int(*matcher)(pci_devaddr_t*))
{
	static pci_devaddr_t pa;
	int claim_id = 0;
	int claim_ptr = 0;
	int tmp;

	if (PCI_PRESENT() == 0)
		return (NULL);

	bzero(&pa, sizeof (pa));
	for (tmp = 0; tmp < pci_scan_buses; tmp++) {
	    pa.d_bus = tmp;
	    for (pa.d_agent = 0; pa.d_agent < pci_scan_agents; pa.d_agent++) {
		for (pa.d_function = 0; pa.d_function < pci_scan_functions;
		    pa.d_function++) {
			if (pci_claimed[claim_ptr] == ++claim_id) {
				claim_ptr++;
				continue;
			}
			if (matcher(&pa))
				break;
		}
		if (pa.d_function != pci_scan_functions)
			break;
	    }
	    if (pa.d_function != pci_scan_functions)
		    break;
	}
	if (pa.d_function == pci_scan_functions)
		return (NULL);		/* we didn't find a match */

	if (claim_ptr >= PCI_CLAIMSIZE) {
		printf("pci0: too many devices\n");
		return (NULL);
	}
	pci_claimed[claim_ptr] = claim_id;

	if (elcr_present) {
		/*
		 * At this point we know that we have edge/level control
		 * registers. Fetch the interrupt level/line from the
		 * device and set that line up for level triggering. 
		 */
		if ((tmp = pci_inb(&pa, PCI_I_LINE)) != 0) {
			if (tmp == 2)	/* irq 2 means 9 */
			    tmp = 9;
			if (tmp < 8)
			    outb(IO_ELCR1, inb(IO_ELCR1) | (1 << tmp));
			else
			    outb(IO_ELCR2, inb(IO_ELCR2) | (1 << (tmp - 8)));
		}
	}
	return (&pa);
}
