/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc.  All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: ncr.c,v 2.9 1995/12/12 21:15:01 cp Exp $
 */

/*
 * Driver for NCR 53C810 SCSI host bus adapter
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>

#include <machine/cpu.h>

#include <i386/pci/pci.h>

#include "ncrreg.h"
#include "ncr_code.h"		/* actual NCR 53c810 firmware */

/*
 * information for running an operation
 */
typedef struct ncr_op {
	struct ncr_op	*op_next;	/* used to put on various queues */

	struct buf	*op_bp;
	u_char		 op_lun;
	u_char		 op_icmd;	/* operation is for an immediate cmd */
	u_char		 op_done;	/* icmd is done */
	u_char		 op_target;
	u_char		 op_cdblen;
	u_char		 op_read;
	u_char		 op_status;	/* scsi status */	
	u_char		 op_msgout[6];
	struct scsi_cdb  op_cdb;	/* real scsi cdb from caller */
	scintr_fn	 op_bintr;	/* base interrupt handler */
	struct device	*op_bintrdev;	/* base device */
	volatile u_char	 op_shared[NCRR_eot];
} ncr_op_t;

/*
 * Change the size of the slot modify firmware.
 * Selector knows size of ncr_tslot.
 */
typedef struct ncr_tslot {		/* target slot */
	u_char		 ts_flags;
	u_char		 ts_filler[3];
	vm_offset_t	 ts_runop;	/* address of runop code */
	ncr_op_t	*ts_op;
	u_char		 ts_targetrate;
	u_char		 ts_scntl3;
	u_char		 ts_sxfer;
	u_char		 ts_syncrate;
} volatile ncr_tslot_t;

typedef struct ncr_softc {
	struct hba_softc	sc_hba;
#define sc_flags sc_hba.hba_dev.dv_flags
	struct isadev		sc_isa;
	struct intrhand		sc_ih;
	ncr_op_t	*sc_op_go;	/* op currently being started */
	ncr_io_t	*sc_io;
	vm_offset_t	 sc_pio;	/* physical address of device */
	ncr_op_t	*sc_op_free;	/* free ops */
	u_char		 sc_nodisconnect;
	u_char		 sc_nosync;
	u_char		 sc_id;		/* adapter's SCSI ID */
	ncr_tslot_t	 sc_tslots[8];	/* number of targets */
	u_int		 sc_chipclk;
	volatile u_char	 sc_shared[NCRS_eot];
	u_char		 sc_scf;
	u_char		 sc_ccf;
	u_char		 sc_probetime;
	u_char		 sc_probehung;
	u_char		 sc_watchon;
	u_char		 sc_watchinterval;
	u_int		 sc_ioout;	/* number of outstanding io ops */
	u_int		 sc_icnt;	/* count of interrupts */
	u_int		 sc_icmd;	/* icmd is being run */
} ncr_softc_t;

static int ncrprobe __P((struct device *parent, struct cfdata *cf, void *aux));
static void ncrattach __P((struct device *parent, struct device *self,
	    void *aux));
static int ncricmd __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t data, int len, int rw));
static int ncrdump __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t addr, int len));
static void ncrstart __P((struct device *self, struct sq *sq, struct buf *bp,
	    scdgo_fn dgo, struct device *dev));
static int ncrgo __P((struct device *self, int targ, scintr_fn intr,
	    struct device *dev, struct buf *bp, int pad));
static void ncrrel __P((struct device *self));
static void ncrhbareset __P((struct hba_softc *hba, int resetunits));

struct cfdriver ncrcd =
    { 0, "ncr", ncrprobe, ncrattach, DV_DULL, sizeof (ncr_softc_t) };

static struct hbadriver ncrhbadriver =
    { ncricmd, ncrdump, ncrstart, ncrgo, ncrrel, ncrhbareset };

/*
 * shifts to pick up values from flags field
 */
#define NCR_F_NOSYNC_S		24	/* units with sync forced off */
#define NCR_F_NODISCONNECT_S	16	/* units with disconnect forced off */
#define NCR_F_IID_S		4	/* initiator id */
#define NCR_F_CLOCK_S	    	0
#define NCR_F_NOPARITY		0x100
#define NCR_F_SLOWSCSI		0x200
#define NCR_F_NOTE		0x400	/* no tolerant enable */

#define NCR_F_CLOCK_M		0xf
#define NCR_F_IID_M		0xf

#define NCR_F_CLOCK_BITS(x)	(((x) >> NCR_F_CLOCK_S) & NCR_F_CLOCK_M)
#define NCR_F_IID_BITS(x)	(((x) >> NCR_F_IID_S) & NCR_F_IID_M)

u_short ncr_clocktable[] = {5000, 0, 6670, 4000, 3750, 3333, 2500, 2000,
			    1667, 0, 0, 0, 0, 0, 0, 0 };

#define NCR_VENDOR_ID		0x1000
#define NCR_DEVICE_ID_810	0x1
#define NCR_DEVICE_ID_820	0x2
#define NCR_DEVICE_ID_825	0x3
#define NCR_DEVICE_ID_815	0x4
#define NCR_DEVICE_ID_FIRST	0x1
#define NCR_DEVICE_ID_LAST	0x4

#define PA(addr) pmap_extract(kernel_pmap, (vm_offset_t)(addr))
#define R2VWA(offset) ((u_int*)(&op->op_shared[(offset)]))
#define S2VWA(offset) ((u_int*)(&sc->sc_shared[(offset)]))

#define NCR_SLOWWATCH	hz
#define NCR_FASTWATCH	1

/*
 * internal use forward references
 */
static int ncrintr __P((void *sc0));

static ncr_op_t *get_op __P((ncr_softc_t *sc));
static void start_op __P((register ncr_softc_t *sc, ncr_op_t *op));
static void done_op __P((ncr_softc_t *sc, ncr_op_t *op));
static void free_op __P((ncr_softc_t *sc, ncr_op_t *op));

static void ncrreset_hw __P((ncr_softc_t *sc));

static int
ncrmatch(pci_devaddr_t *pa)
{
	u_short vendor;
	u_short dev;
	u_char line;
	extern int autodebug;

	vendor = pci_inw(pa, PCI_VENDOR_ID);
	if (vendor != NCR_VENDOR_ID)
		return (0);

	dev = pci_inw(pa, PCI_DEVICE_ID);
	if (dev < NCR_DEVICE_ID_FIRST || dev > NCR_DEVICE_ID_LAST)
		return (0);

	line = pci_inb(pa, PCI_I_LINE);
	if (line == 0) {	/* this device is not enabled in proms */
		if (autodebug)
		    printf("NCR SCSI HBA rejected line/interrupt level = 0\n");
		return (0);

	}

	return (1);
}

void ncrforceintr(void *PA)
{
	pci_devaddr_t *pa = PA;

	/*
	 * start dma at odd byte to force illegal instruction interrupt
	 */
	pci_outb(pa, DIEN | MEM2CONFIG, NCR_IID);
	pci_outl(pa, DSP | MEM2CONFIG, 1);
	delay(50);
	pci_outb(pa, DIEN | MEM2CONFIG, 0);
	(void) pci_inb(pa, DSTAT | MEM2CONFIG);
}
/*
 * standard probe routine
 */
static int
ncrprobe(struct device *parent, struct cfdata *cf, void *aux)
{
	struct isa_attach_args *ia = aux;
	int i;
	u_short command;
	pci_devaddr_t *pa;

	pa = pci_scan(ncrmatch);
	if (pa == NULL)
		return (0);

	command = pci_inw(pa, PCI_COMMAND);
	command &= ~1;				/* turn off port access */
	command |= 2;				/* turn on memory access */
	pci_outl(pa, PCI_COMMAND, command );

	if (ia->ia_irq == IRQUNK)
		ia->ia_irq = isa_discoverintr(ncrforceintr, (void *) pa);
	if (ffs(ia->ia_irq) - 1 == 0)
		ia->ia_irq = IRQNONE;
	else
		ia->ia_irq |= IRQSHARE;
	if (ia->ia_maddr)
		pci_outl(pa, PCI_MEMORY_BA, (u_long)ia->ia_maddr);
	ia->ia_maddr = (caddr_t)pci_inl(pa, PCI_MEMORY_BA);
	ia->ia_msize = 256;
	return (1);
}

void
ncrwatch(sc0)
	void *sc0;
{
	struct ncr_softc *sc;
	int i, op, s;
	u_char istat;

	sc = sc0;
	sc->sc_watchon = 0;

	if (sc->sc_ioout == 0)
		return;

	s = splbio();

	istat = sc->sc_io->io_istat;
	if ((istat & (NCR_INTF | NCR_IDMA | NCR_ISIP)) == 0) {
		splx(s);
		sc->sc_watchon = 1;
		timeout(ncrwatch, sc, sc->sc_watchinterval);
		return;
	}
	i = sc->sc_icnt;
	splx(s);
	delay(1);
	s = splbio();
	if (sc->sc_icnt == i) {
		if (sc->sc_watchinterval != NCR_FASTWATCH)
			printf("%s: missing interrupt(s)\n",
			    sc->sc_hba.hba_dev.dv_xname);
		ncrintr(sc);
		sc->sc_watchinterval = NCR_FASTWATCH;
	}
	splx(s);
	if (sc->sc_ioout == 0 || sc->sc_watchon)
		return;
	sc->sc_watchon = 1;
	timeout(ncrwatch, sc, sc->sc_watchinterval);
}

#define SETSRTE(offset, id, sxfer, scntl3)  \
	((ncr_srte_t*)(&op->op_shared[(offset)]))->sr_id = (id); \
	((ncr_srte_t*)(&op->op_shared[(offset)]))->sr_sxfer = (sxfer); \
	((ncr_srte_t*)(&op->op_shared[(offset)]))->sr_scntl3 = (scntl3)
	
#define SETBMTE(offset, cv, av) \
	((ncr_bmte_t*)(&op->op_shared[(offset)]))->bm_count = (cv); \
	((ncr_bmte_t*)(&op->op_shared[(offset)]))->bm_addr = \
		pmap_extract(kernel_pmap, (vm_offset_t)(av))

static ncr_op_t *
create_op(ncr_softc_t *sc)
{
	ncr_op_t *op;

	op = malloc(sizeof *op, M_DEVBUF, M_WAITOK);
	bzero(op, sizeof *op);

	SETBMTE(NCRR_t_status, 1, &op->op_status);
	op->op_msgout[1] = MSG_EXT_MESSAGE;
	op->op_msgout[2] = 3;	/* length of message  */
	op->op_msgout[3] = XMSG_SDTR;	/* synchronous data transfer request */
	if (sc->sc_flags & NCR_F_SLOWSCSI)
		op->op_msgout[4] = 200/4;	/* 200 ns 5 MB per second */
	else
		op->op_msgout[4] = 100/4;	/* 100 ns 10 MB per second */
	op->op_msgout[5] = 8;		/* maximum sync offset */


	bcopy(NCRR_code, (void*)op->op_shared, sizeof (NCRR_code));
	ncr_patchit(sc, NCRR_patch, (u_int*)op->op_shared);

	*R2VWA(NCRR_physaddr) = PA(op->op_shared);
	*R2VWA(NCRR_selector) = PA(S2VWA(NCRS_newwork));
	*R2VWA(NCRR_findtarget) = PA(S2VWA(NCRS_findtarget));
	*R2VWA(NCRR_idemsg) = MSG_INIT_DETECT_ERROR;
	return (op);
}

void *mapphys __P((vm_offset_t, int));

/*
 * standard attach routine.
 */
static void
ncrattach(struct device *parent, struct device *self, void *aux)
{
	struct isa_attach_args *ia = aux;
	register ncr_softc_t *sc = (ncr_softc_t *)self;
	int i;
	ncr_op_t *op;
	ncr_io_t *io;
	struct timeval t1, t2;

	aprint_naive(": NCR SCSI");
	if (NCR_F_IID_BITS(sc->sc_flags) != 0)
		sc->sc_id = NCR_F_IID_BITS(sc->sc_flags) & 0x7;
	else
		sc->sc_id = 7;

	sc->sc_nodisconnect = sc->sc_flags >> NCR_F_NODISCONNECT_S;
	sc->sc_nosync = sc->sc_flags >> NCR_F_NOSYNC_S;
	sc->sc_io = mapphys((vm_offset_t)ia->ia_maddr, ia->ia_msize);
	sc->sc_pio = (vm_offset_t)ia->ia_maddr;
	if (ia->ia_irq == IRQNONE)
		printf(": warning, running without interrupts");

	ncrreset_hw(sc);

	io = sc->sc_io;

	sc->sc_chipclk = ncr_clocktable[NCR_F_CLOCK_BITS(sc->sc_flags)];

	aprint_verbose("\n%s: initiator id %d", 
		sc->sc_hba.hba_dev.dv_xname, sc->sc_id); 

	if (sc->sc_chipclk == 0) {
		sc->sc_chipclk = 6670;
		sc->sc_nosync = 0xff;	/* no sync if we don't know clock */
		aprint_verbose(", chip clock unknown");
	} else
		aprint_verbose(", chip clock %d.%dMhz",
			sc->sc_chipclk/100, sc->sc_chipclk %100);


	if (sc->sc_chipclk <= 2500)
		sc->sc_ccf = 1;
	else if (((sc->sc_chipclk * 2) / 3)  <= 2500)
		sc->sc_ccf = 2;
	else if ((sc->sc_chipclk / 2)  <= 2500)
		sc->sc_ccf = 3;
	else 
		sc->sc_ccf = 4;

	if (sc->sc_flags & NCR_F_NOPARITY)
		aprint_verbose(", parity disabled");
	else
		aprint_verbose(", parity enabled");
	if (sc->sc_flags & NCR_F_NOTE)
		aprint_verbose(", TolerANT disabled");
	else
		aprint_verbose(", TolerANT enabled");
	printf("\n");

	printf("Delaying for SCSI bus reset and device self tests");
	DELAY(5000000); 	/* 5 seconds */
	printf("\n");

	sc->sc_hba.hba_driver = &ncrhbadriver;
	/*
	 * Link into isa and set interrupt handler.
	 */
	isa_establish(&sc->sc_isa, &sc->sc_hba.hba_dev);
	if (ia->ia_irq != IRQNONE) {
		sc->sc_ih.ih_fun = ncrintr;
		sc->sc_ih.ih_arg = sc;
		intr_establish(ia->ia_irq, &sc->sc_ih, DV_DISK);
	} else
		sc->sc_watchinterval = NCR_FASTWATCH;

	sc->sc_tslots[7].ts_flags = NCRS_slot_stopper;
	bcopy(NCRS_code, (void*)sc->sc_shared, sizeof (NCRS_code));
	ncr_patchit(sc, NCRS_patch, (u_int*)sc->sc_shared);


	for (i = 0; i < 8; i++)
		S2VWA(NCRS_t0addr)[i] = PA(&sc->sc_tslots[i]);

	*S2VWA(NCRS_physaddr) = PA(&sc->sc_shared);

	io->io_dsp = PA(S2VWA(NCRS_wait4work)); /* start firmware */

	op = create_op(sc);
	free_op(sc, op);
	sc->sc_probetime = 1;
	for (i = 0; i < 8; i++) {
		ncr_tslot_t *ts;
		int hr;
		int tr;

		if (i == sc->sc_id)
			continue;
		if (scsi_test_unit_ready(&sc->sc_hba, i, 0) < 0)
			continue;
		sc->sc_probehung = 0;
		SCSI_FOUNDTARGET(&sc->sc_hba, i);
		if (sc->sc_probehung & !(sc->sc_nosync & (1 << i))) {
			ncrhbareset(&sc->sc_hba, 0);
			printf("%s: all targets forced async\n",
			    sc->sc_hba.hba_dev.dv_xname);
			sc->sc_nosync = 0xff;
			sc->sc_probehung = 0;
			SCSI_FOUNDTARGET(&sc->sc_hba, i);
		}
		if (sc->sc_probehung)
			ncrhbareset(&sc->sc_hba, 0);
		op = create_op(sc);
		free_op(sc, op); 	/* place on free list */
		ts = &sc->sc_tslots[i];
		aprint_verbose("%s:",
		    sc->sc_hba.hba_targets[i]->t_dev.dv_xname);
		if (sc->sc_nodisconnect & (1 << i))
			aprint_verbose(" disconnect disabled,");
		else
			aprint_verbose(" disconnect enabled,");
		if ((ts->ts_sxfer & NCR_MO_M) == 0) {
			aprint_verbose(" asynchronous\n");
			continue;
		}

		tr = ts->ts_targetrate;
		hr = ts->ts_syncrate;
		aprint_verbose(" synchronous out %d.%dMB/s, in %d.%dMB/s",
			hr / 10, hr % 10,
			tr / 10, tr % 10);
		aprint_verbose(", max offset %d\n", ts->ts_sxfer & NCR_MO_M);
	}
	sc->sc_probetime = 0;
	sc->sc_watchinterval = NCR_SLOWWATCH;
}

/*
 * standard icmd routine
 */
static int
ncricmd(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t data,
       int len, int rw)
{
	int status;
	int s;
	ncr_op_t *op;
	ncr_softc_t *sc = (ncr_softc_t *)hba;
	struct buf bufhdr;
	int x;

	if (sc->sc_probetime) {
		if (sc->sc_probehung) 
			return (-1);
		x = 0;
	}

	s = splbio();
	op = get_op(sc);
	bufhdr.b_un.b_addr = data;
	bufhdr.b_iocount = bufhdr.b_bcount = len;
	bufhdr.b_flags = rw;
	bufhdr.b_chain = NULL;
	op->op_bp = &bufhdr;
	op->op_done = 0;
	if (rw & B_READ)
		op->op_read = 1;
	else
		op->op_read = 0;

	op->op_cdblen = SCSICMDLEN(cdb->cdb_bytes[0]);
	bcopy(cdb, &op->op_cdb, op->op_cdblen);
	op->op_icmd = 1;
	op->op_target = targ;
	op->op_lun = cdb->cdb_bytes[1] >> 5;
	sc->sc_icmd = 1;
	start_op(sc, op);
	while (!op->op_done) {
		if (sc->sc_probetime) {
			if (x++ > 3000000) {
				printf("%s: SCSI bus hung.",
				    sc->sc_hba.hba_dev.dv_xname);
				printf(" Chip clock set wrong in flags?\n");
				op->op_status = 0x80;
				sc->sc_probehung = 1;
				break;
			}
		}
		ncrintr(sc);
	}
	status = op->op_status;
	free_op(sc, op);
	sc->sc_icmd = 0;
	splx(s);
	if (status & 0x80)
		return (-1);		/* minus one for timeout */
	return (status);
}


/*
 * standard dump routine. This routine has to
 * map the physical address into a logical address
 * before it can run the io.
 */
extern char *dumpbufp;

static int
ncrdump(struct hba_softc *hba, int targ, struct scsi_cdb *cdb, caddr_t addr,
       int len)
{
	int i;

	for (i = 0; i < len; i += ctob(1)) 
		pmap_enter(kernel_pmap, (vm_offset_t)(dumpbufp + i),
		    (vm_offset_t) addr + i, VM_PROT_READ, TRUE);
	return (ncricmd(hba, targ, cdb, dumpbufp, len, B_WRITE));
}

/*
 * standard start routine.
 * we can always start the io.
 */
static void
ncrstart(struct device *self, struct sq *sq, struct buf *bp, scdgo_fn dgo,
        struct device *dev)
{
	ncr_softc_t *sc = (ncr_softc_t *)self;

	if (bp) {
		if (sc->sc_op_go)
			panic("ncr start with go set");
		sc->sc_op_go = get_op(sc);
		(*dgo)(dev, &sc->sc_op_go->op_cdb);
		return;
	}
	(*dgo)(dev, (struct scsi_cdb *)NULL);
}

/* 
 * standard go routine. Fill in the op
 * and run the io
 */
static int
ncrgo(struct device *self, int targ, scintr_fn intr, struct device *dev,
     struct buf *bp, int pad)
{
	struct ncr_softc *sc = (struct ncr_softc *)self;
	ncr_op_t *op;

	op = sc->sc_op_go;
	sc->sc_op_go = NULL;
	op->op_bp = bp;
	if (bp->b_flags & B_READ)
		op->op_read = 1;
	else
		op->op_read = 0;
	op->op_target = targ;
	op->op_bintr = intr;
	op->op_bintrdev = dev;
	op->op_done = 0;
	op->op_icmd = 0;
	op->op_cdblen = SCSICMDLEN(op->op_cdb.cdb_bytes[0]);
	op->op_lun = op->op_cdb.cdb_bytes[1] >> 5;
	start_op(sc, op);
	return (0);
}

/*
 *	Free up op when ncrrel instead of start called
 *	If a queue exists for multiple luns start next op
 */
static void
ncrrel(struct device *self)
{
	ncr_softc_t *sc = (ncr_softc_t *)self;
	struct sq *sq;

	if (sc->sc_op_go) {
		free_op(sc, sc->sc_op_go);
		sc->sc_op_go = NULL;
	}
	if ((sq = sc->sc_hba.hba_head) != NULL) {
		sc->sc_op_go = get_op(sc);
		(*sq->sq_dgo)(sq->sq_dev, &sc->sc_op_go->op_cdb);
	}
}

/*
 * standard reset code.
 */
static void
ncrhbareset(struct hba_softc *hba, int resetunits)
{
	ncr_softc_t *sc = (ncr_softc_t *)hba;
	ncr_tslot_t *ts;
	int i;

	ncrreset_hw(sc);
	printf("%s: delaying 5 seconds after bus reset\n",
	    sc->sc_hba.hba_dev.dv_xname);
	DELAY(5000000); 	/* 5 seconds */

	for (i = 0; i < 8; i++) {
		ts = &sc->sc_tslots[i];
		if (ts->ts_flags & NCRS_slot_valid)
		    free_op(sc, ts->ts_op);
		ts->ts_flags &= ~(NCRS_slot_valid | NCRS_slot_done);
		ts->ts_runop = 0;
		ts->ts_op = NULL;
		ts->ts_sxfer = 0;	/* no longer sync */
	}
	sc->sc_io->io_dsp = PA(S2VWA(NCRS_wait4work)); /* start firmware */
	if (resetunits)
		scsi_reset_units(hba);
}

ncrintr_isip(ncr_softc_t *sc, ncr_op_t *op)
{
	u_char sist0, sist1, junk;
	int dbc;
	int inpipe;
	ncr_io_t *io = sc->sc_io;

	sist0 = io->io_sist0;
	sist1 = io->io_sist1;
	delay(1);
	junk = io->io_dstat;

	if (sist0 & NCR_PAR) {
		*R2VWA(NCRR_ideflag) = 1;
		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_phaselock]);
		return;
	}

	if (op == NULL) {
		printf("%s: ", sc->sc_hba.hba_dev.dv_xname);
		ncr_dumpreg(sc);
		panic("ncr scsi interrupt with no device active\n");
	}

	/*
	 * phase mismatch 
	 * for now we just handle input and don't check
	 * where is is happening
	 */
	if (sist0 & NCR_PMM) {
		int whichlist;
		ncr_bmte_t *sgl;    /* pointer to actual scatter gather list */
		ncr_bmte_t *sgc;    /* pointer to current item */
		ncr_bmte_t *sgp;    /* partial item created */

		if (io->io_dsp != PA(R2VWA(NCRR_ostop)) &&
		    io->io_dsp != PA(R2VWA(NCRR_istop))) {
			io->io_stest3 |= NCR_CSF;
			io->io_ctest3 = NCR_CLF;
			if (io->io_dsp == PA(R2VWA(NCRR_msg_ostop)))
				sc->sc_nosync |=  1 << op->op_target;
			else
				printf("%s: unexpected phase change\n",
				    sc->sc_hba.hba_dev.dv_xname);
			io->io_dsp = (vm_offset_t)
			    PA(&op->op_shared[NCRR_phaselock]);
			return;
		}

		whichlist = op->op_read ? NCRR_inlist : NCRR_outlist;

		sgl = (ncr_bmte_t*) (op->op_shared +
		    (*R2VWA(whichlist) - PA(op->op_shared)));
		sgp = (ncr_bmte_t*) R2VWA(NCRR_t_partial);
		sgc = (ncr_bmte_t*) R2VWA(NCRR_t_data);

		*sgp = *sgc;
		dbc = io->io_dbc;
		if (!op->op_read) {
			inpipe = io->io_dfifo;
			inpipe -= dbc;
			inpipe &= 0x7f;
			if (io->io_sstat0 & NCR_ORF)
				inpipe++;
			if (io->io_sstat0 & NCR_OLF)
				inpipe++;
			dbc += inpipe;
			io->io_stest3 |= NCR_CSF;
			io->io_ctest3 = NCR_CLF;
		}
		sgp->bm_addr += sgp->bm_count - dbc;
		sgp->bm_count = dbc;
		*R2VWA(NCRR_partial) = 0xff;	

		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_phaselock]);

		return (1);
	}

	/*
	 * really should do more checking here
	 */
	op->op_status = 0x80;
	done_op(sc, op);
	io->io_dsp = PA(S2VWA(NCRS_wait4work)); /* start firmware */
	return (1);
}

static int
ncrintr_done(ncr_softc_t *sc)
{
	int i;
	ncr_tslot_t *ts;

	for (i = 0; i < 8; i++) {
		ts = &sc->sc_tslots[i];
		if (ts->ts_flags & NCRR_slot_done) {
			if (!sc->sc_icmd)
				break;
			if (ts->ts_op->op_icmd)
				break;
		}
	}
	if (i == 8)
		return (0);
	done_op(sc, ts->ts_op);
	return (1);

}

static int
ncrintr(void *sc0)
{
	ncr_softc_t *sc = sc0;
	register ncr_io_t *io = sc->sc_io;
	u_char istat = io->io_istat;
	u_char dstat;
	vm_offset_t dsa;
	ncr_op_t *op;
	int target;
	ncr_tslot_t *ts;

	if ((istat & (NCR_INTF | NCR_IDMA | NCR_ISIP)) == 0)
		return (0);

	sc->sc_icnt++;

	if (istat & NCR_INTF) {
		if (!sc->sc_icmd || sc->sc_probetime)
			io->io_istat = NCR_INTF | (istat & NCR_ISIGP);
		while (ncrintr_done(sc))
			continue;
	}

	if ((istat & (NCR_IDMA | NCR_ISIP)) == 0)
		return (1);

	dsa = io->io_dsa;
	for (target = 0; target < 8; target++) {
		ts = &sc->sc_tslots[target];
		if (ts->ts_runop == dsa)
			break;
	}
	if (target < 8)
		op = ts->ts_op;
	else
		op = NULL;

	if (istat & NCR_ISIP) {
		ncrintr_isip(sc0, op);
		return;
	}

	if (istat & NCR_IDMA == 0)
		return;

	printf("%s: ", sc->sc_hba.hba_dev.dv_xname);
	dstat = io->io_dstat;
	if (dstat & NCR_MDPE) {
		ncr_dumpreg(sc);
		panic("ncr pci bus parity error");
	}
	if (dstat & NCR_BF) {
		ncr_dumpreg(sc);
		panic("ncr pci bus fault");
	}
	if (!(dstat & NCR_SIR)) {
		ncr_dumpreg(sc);
		printf("dstat = %x\n", dstat);
		panic("ncr dma interrupt not claimed");
	}
	switch (io->io_dsps) {
	case NCRR_i_i2long:
		printf("target input transfer too long\n");
		*R2VWA(NCRR_ideflag) = 1;
		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_itoss]);
		break;
	case NCRR_i_o2long:
		printf("target output transfer too long\n"); 
		*R2VWA(NCRR_ideflag) = 1;
		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_otoss]);
		break;
	case NCRR_i_phase:
		ncr_dumpreg(sc);
		panic("ncr no phase found\n");
	case NCRR_i_emsgtossed:
		printf("unknown extended message 0x%x discarded\n",
		    *R2VWA(NCRR_rcvmsg));
		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_phaselock]);
		break;
	case NCRR_i_msgtossed:
		printf("unknown message 0x%x discarded\n",
		    *R2VWA(NCRR_rcvmsg));
		io->io_dsp = (vm_offset_t) PA(&op->op_shared[NCRR_phaselock]);
		break;
	case NCRS_i_ssid_invalid:
		if (sc->sc_probetime) {
			sc->sc_nodisconnect |= 1 << target;
			io->io_scratcha = target;
			io->io_dsp = PA(S2VWA(NCRS_si_restart)); 
			break;
		}
		ncr_dumpreg();
		panic("ncr: old single initiator reselect\n");
		break;
	case NCRS_i_bogusreselect:
		ncr_dumpreg();
		panic("ncr: reselect from non active target\n");
	default:
		ncr_dumpreg(sc);
		panic("unknown firmware interrupt\n");
	}
	return (1);
}

/*
 * return op to free list. 
 * could be macro. But usefull place to
 * put in debugging code
 */
static void
free_op(ncr_softc_t *sc, ncr_op_t *op)
{

	op->op_next = sc->sc_op_free;
	sc->sc_op_free  = op;
}

/*
 * get op off of free list. There should always
 * be one available. If not either this driver or
 * one of the target drivers has a bug.
 */
static ncr_op_t *
get_op(ncr_softc_t *sc)
{
	ncr_op_t *op;

	if (sc->sc_op_free == NULL)
		panic("ncr: get_op no op");
	op = sc->sc_op_free;
	sc->sc_op_free = op->op_next;
	return (op);
}


/*
 * This code originally lifted from ab_subr.c
 *
 * If the transfer uses a chain (bp->b_chain != NULL), we assume that no
 * <addr,len> pair ever crosses a page boundary.  In any case, at most two
 * pages can be partial (one at the start and one at the end).
 *
 * (Lots of panics here, from sheer raging paranoia.)
 */
void
ncr_sgmap(ncr_op_t *op)
{
	int len, i, n, o, pages;
	vm_offset_t v;
	ncr_bmte_t *sg = (ncr_bmte_t*)(&op->op_shared[NCRR_sg]);
	struct buf *bp = op->op_bp;

	/*
	 * ###: We used to xfer 0 bytes at addr 0 for len < 0, but I
	 * think this is an error.  If not, our caller will have to
	 * check len < 0 and change that to len = 0 anyway.
	 */
	len = bp->b_iocount;
	if (len < 0)
		panic("ncr_sgmap");
	if (len == 0) {
		sg->bm_stop = 0xff;
		return;
	}
	v = (vm_offset_t)bp->b_un.b_addr;
	o = v & PGOFSET;
	pages = i386_btop(i386_round_page(len + o));

	/*
	 * Transfer may also be chained, in which case we must step
	 * to the next buffer each time bp->b_bcount runs out (note
	 * that len is the head buffer's bp->b_iocount).
	 */
	if (pages >= NCRR_nsg)	/* need an extra for stopper */
		panic("ncr_sgmap pages");

	if (pages == 1) {
		sg->bm_addr = pmap_extract(kernel_pmap, v);
		sg->bm_count = bp->b_bcount;
		sg->bm_stop = 0;
		(++sg)->bm_stop = 0xff;
		return;
	}

	/* First page may start at some offset, and hence be short. */
	n = bp->b_bcount;
	i = NBPG - o;
	sg->bm_addr = pmap_extract(kernel_pmap, v);
	sg->bm_count = i;
	sg->bm_stop = 0;
	sg++;
	v += i;
	n -= i;

	/* Pages 1 through (pages-1), if pages > 2, are full size. */
	for (i = 2; i < pages; i++) {
		if (n <= 0) {
			if (n < 0 || bp->b_chain == NULL)
				panic("ncr_sgmap mid chain");
			bp = bp->b_chain;
			n = bp->b_bcount;
			v = (vm_offset_t)bp->b_un.b_addr;
			if (v & PGOFSET)
				panic("ncr_sgmap mid addr");
		}
		sg->bm_addr = pmap_extract(kernel_pmap, v);
		sg->bm_count = NBPG;
		sg->bm_stop = 0;
		sg++;
		v += NBPG;
		n -= NBPG;
	}

	/* Last page is n remaining bytes. */
	if (n <= 0) {
		if (n < 0 || bp->b_chain == NULL)
			panic("ncr_sgmap last chain");
		bp = bp->b_chain;
		n = bp->b_bcount;
		v = (vm_offset_t)bp->b_un.b_addr;
		if (v & PGOFSET)
			panic("ncr_sgmap last addr");
	}
	if (n > NBPG)
		panic("ncr_sgmap lastpg %d", n);
	/*
	 * ab_subr looked wrong. It seems like it will plant one extra
	 * address with a length of zero 
	 */
	if (n != 0) {
		sg->bm_addr = pmap_extract(kernel_pmap, v);
		sg->bm_count = n;
		sg->bm_stop = 0;
		sg++;
	}
	sg->bm_stop = 0xff;
}

ncr_patchit(ncr_softc_t *sc, u_int *patch, u_int *code)
{
	int i;

	for (i = 0; patch[i] != 0; i += 2)
		if (patch[i] == NCR_PATCH_TABLEREL)
			code[patch[i + 1]] =
			    PA(code) + code[patch[i + 1]]; 
		else
			code[patch[i + 1]] =
			    sc->sc_pio + code[patch[i + 1]]; 
}

static void
start_op(register ncr_softc_t *sc, ncr_op_t *op)
{
	ncr_op_t *op1;
	ncr_io_t *io = sc->sc_io;
	int i;
	ncr_tslot_t *ts;

	ts = &sc->sc_tslots[op->op_target];
	ts->ts_op = op;
	ts->ts_runop = PA(&op->op_shared);

	SETBMTE(NCRR_t_cmd, op->op_cdblen, &op->op_cdb);
	SETSRTE(NCRR_t_select, op->op_target, ts->ts_sxfer, ts->ts_scntl3);
	*R2VWA(NCRR_syncmsgin) = 0;
	ncr_sgmap(op);
	/*
	 * we could do more here. Like point the direction we are not
	 * going in to a instant stop
	 */
	*R2VWA(NCRR_inlist) = PA(&op->op_shared[NCRR_sg]);
	*R2VWA(NCRR_outlist) = PA(&op->op_shared[NCRR_sg]);

	op->op_msgout[0] = op->op_lun | MSG_IDENTIFY;
	if (!(sc->sc_nodisconnect & (1 << op->op_target)))
		op->op_msgout[0] |= MSG_IDENTIFY_DR;

	if (op->op_cdb.cdb_bytes[0] != CMD_TEST_UNIT_READY ||
	    op->op_lun != 0 || (sc->sc_nosync & (1 << op->op_target))) {
		SETBMTE(NCRR_t_id_msg, 1, op->op_msgout);
	} else {
		*R2VWA(NCRR_syncmsgin) = 0;
		SETBMTE(NCRR_t_id_msg, 6, op->op_msgout);
	}

	*R2VWA(NCRR_slotaddr) = PA(ts);
	*R2VWA(NCRR_partial) = 0;	
	ts->ts_flags |= NCRS_slot_valid | NCRS_slot_newwork;
	io->io_istat = NCR_ISIGP;	/* break reselect */
	sc->sc_ioout++;
	if (!sc->sc_probetime && !sc->sc_watchon) {
		sc->sc_watchon = 1;
		timeout(ncrwatch, sc, sc->sc_watchinterval);
	}
}

static void
setsyncclocks(ncr_softc_t *sc, int desired, ncr_tslot_t *ts)
{
	int bg_rate = 0;
	int bg_divisor;
	int bg_syncclock;
	int bg_syncrate;
	int i, j;

	for (i = 1; i < 5; i++) {
		int syncclock;

		switch (i) {
		case  1:
			syncclock = sc->sc_chipclk;
			break;
		case  2:
			syncclock = (sc->sc_chipclk * 2) / 3;
			break;
		case  3:
			syncclock = sc->sc_chipclk / 2;
			break;
		case  4:
			syncclock = sc->sc_chipclk / 3;
			break;
		}
		if (syncclock > 5000)
			continue;		/* to fast for chip */
		j = howmany(syncclock, desired);
		if (j > 11)
			continue;
		if (j < 4)
			continue;
		if ((syncclock / j) > bg_rate) {
			bg_rate = syncclock / j;
			bg_syncclock = i;
			bg_divisor = j - 4;
		}
		if (bg_rate == 0)
			panic("%s: unable to determine sync clocks\n",
			    sc->sc_hba.hba_dev.dv_xname);
	}
	ts->ts_scntl3 = bg_syncclock << NCR_SCF_S | sc->sc_ccf;
	ts->ts_sxfer = (bg_divisor << NCR_TP_S);
	ts->ts_syncrate = bg_rate / 10;
}
		
static void
done_op(ncr_softc_t *sc, ncr_op_t *op)
{
	int status;
	ncr_bmte_t *sg;
	ncr_bmte_t *sgp;	/* partial at end */
	int resid = 0;
	int whichlist;
	int offset;

	ncr_tslot_t *ts;
	ts = &sc->sc_tslots[op->op_target];
	ts->ts_flags &= ~(NCRS_slot_valid | NCRS_slot_done);
	ts->ts_runop = 0;
	ts->ts_op = 0;
	/*
	 * first see if we we tried to negotiate sync
	 */
	if (op->op_cdb.cdb_bytes[0] == CMD_TEST_UNIT_READY &&
	    op->op_lun == 0 && !(sc->sc_nosync & (1 << op->op_target)) &&
	    (offset = (*R2VWA(NCRR_syncmsgin)) >> 8) != 0) {
		int i;

		i  = ((*R2VWA(NCRR_syncmsgin)) & 0xff) * 4; /* has period */
		if (i == 0)
			i++;
		i = 100000 / i;			/* rate MB * 100 */
		ts->ts_targetrate = i / 10;	/* just so we can print */
		setsyncclocks(sc, i, ts);
		ts->ts_sxfer |= offset;
	}

	sc->sc_ioout--;

	/*
	 * icmd wants op back
	 * it holds its own pointer to it
	 */
	if (op->op_icmd) {
		op->op_done = 1;
		return;
	}
	status = op->op_status;
	if (status & 0x80)
		status = -1;

	if (op->op_read)
		whichlist = NCRR_inlist;
	else
		whichlist = NCRR_outlist;


	sg = (ncr_bmte_t*)
		(op->op_shared + (*R2VWA(whichlist) - PA(op->op_shared)));

	if (*R2VWA(NCRR_partial) == 0xff) {
		sgp = (ncr_bmte_t*) R2VWA(NCRR_t_partial);
		resid = (sgp)->bm_count;
		sg++;
	}
	while (sg->bm_stop == 0)
		resid += (sg++)->bm_count;
	(*op->op_bintr)(op->op_bintrdev, status, resid);
	free_op(sc, op);
}

/*
 * reset the hardware including doing a bus reset. If we allow
 * multiple masters then the bus reset should be conditional.
 */
static void
ncrreset_hw(ncr_softc_t *sc)
{
	ncr_io_t *io = sc->sc_io;

	io->io_istat = NCR_ISRST;
	delay(100);
	io->io_istat = 0;
	io->io_scntl1 = NCR_ARST;	/* reset the bus */
	delay(300000);
	io->io_scntl1 = 0;
	io->io_istat = NCR_ISRST;
	delay(100);
	io->io_istat = 0;

	io->io_scntl0 = NCR_ARB1 | NCR_ARB0;
	if (!(sc->sc_flags & NCR_F_NOPARITY))
		io->io_scntl0 |=  NCR_EPC | NCR_AAP;
	io->io_scid = NCR_RRE | NCR_SRE |  sc->sc_id;
	io->io_dmode = 2 << NCR_BL_S;	/* burst length of 4 */
	io->io_dien = NCR_MDPE | NCR_BF | NCR_SIR | NCR_IID;
	io->io_dcntl = NCR_IRQM | NCR_COM;
	io->io_sien0 = NCR_PMM | NCR_SGE | NCR_UDC | NCR_RST;
	io->io_sien1 = NCR_STO;	/* should allow rest of bits */
	io->io_respid0 = 1 << sc->sc_id;
	if (sc->sc_flags & NCR_F_NOTE)
		io->io_stest3 = 0;
	else
		io->io_stest3 = NCR_TE;
	io->io_stime0 = 0xc;	/* 200 milli seconds about */
}

static int onceonly = 1;

ncr_dumpreg(ncr_softc_t *sc) 
{
	ncr_io_t *io = sc->sc_io;

	if (onceonly == 0)
		return;
	onceonly--;

	printf("SCNTL0 %x, SCNTL1 %x, SCNTL2 %x, SCNTL3 %x,\n",
	    io->io_scntl0, io->io_scntl1, io->io_scntl2, io->io_scntl3); 
	printf("SCID %x, SXFER %x, SDID %x, GPREG %x,\n",
	    io->io_scid, io->io_sxfer, io->io_sdid, io->io_gpreg); 

	printf("SFBR %x, SOCL %x, SSID %x, SBCL %x,\n",
	    io->io_sfbr, io->io_socl, io->io_ssid, io->io_sbcl); 
	printf("DSTAT %x, SSTAT0 %x, SSTAT1 %x, SSTAT2 %x,\n",
	    io->io_dstat, io->io_sstat0, io->io_sstat1, io->io_sstat2); 
	printf("DSA %x\n", io->io_dsa);
	printf("ISTAT %x\n", io->io_istat);
	printf("CTEST0 %x, CTEST1 %x, CTEST2 %x, CTEST3 %x,\n",
	    io->io_ctest0, io->io_ctest1, io->io_ctest2, io->io_ctest3); 
	printf("TEMP %x\n", io->io_temp);
	printf("DFIFO %x, CTEST4 %x, CTEST5 %x, CTEST6 %x,\n",
	    io->io_dfifo, io->io_ctest4, io->io_ctest5, io->io_ctest6); 
	printf("DBC %x, DCMD %x,\n",
	    io->io_dbc, io->io_dcmd); 
	printf("DNAD %x\n", io->io_dnad);
	printf("DSP %x\n", io->io_dsp);
	printf("DSPS %x\n", io->io_dsps);
	printf("SCRATCHA %x\n", io->io_scratcha);
	printf("DMODE %x, DIEN %x, DWT %x, DCNTL %x,\n",
	    io->io_dmode, io->io_dien, io->io_dwt, io->io_dcntl); 
	printf("ADDER %x\n", io->io_adder);
	printf("SIEN0 %x, SIEN1 %x, SIST0 %x, SIST1 %x,\n",
	    io->io_sien0, io->io_sien1, io->io_sist0, io->io_sist1); 
	printf("SLPAR %x, SWIDE %x, MACNTL %x, GPCNTL %x,\n",
	    io->io_slpar, io->io_swide, io->io_macntl, io->io_gpcntl); 
	printf("STIME0 %x, STIME1 %x, RESPID0 %x, RESPID1 %x,\n",
	    io->io_stime0, io->io_stime1, io->io_respid0, io->io_respid1); 
	printf("STEST0 %x, STEST1 %x, STEST2 %x, STEST3 %x,\n",
	    io->io_stest0, io->io_stest1, io->io_stest2, io->io_stest3); 
	printf("SIDL %x\n", io->io_sidl);
	printf("SODL %x\n", io->io_sodl);
	printf("SBDL %x\n", io->io_sbdl);
	printf("SCRATCHB %x\n", io->io_scratchb);
}
