/*-
 * Copyright (c) 1992, 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI $Id: wdc.c,v 2.4 1995/12/12 19:51:18 karels Exp $
 */
 
/*
 * WD 1002 (and IDE) style controller driver
 *
 * Primary function of this driver is to share the controller between
 * the device drivers using it (wd and wdpi).
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/syslog.h>
#include <sys/reboot.h>

#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <i386/isa/isa.h>

#include "wdreg.h"
#include "wdvar.h"

#ifdef EWVDEBUG
#define EWV_DEF
#include "ewvdebug.h"
#else
#define TR(desc, parm1, parm2)
#endif

/*
 * Configuration
 */
#define	RESET_WAIT	5000		/* Max controller reset wait (ms) */
#define MAXDRV		2		/* Max units/controller */

/* Config flags */
#define WDC_NOALTCHK	0x0001		/* Don't look for alternate regs
					   at PCMCIA standard location */

/* per-controller state */
typedef struct wdc_softc {
	struct	device wdc_dev;  	/* base device */
	struct	isadev wdc_id;   	/* ISA device */
	struct	intrhand wdc_ih;	/* interrupt vectoring */
	wdc_req_t *wdc_cur;		/* Currently attached request */
	wdc_req_t *wdc_last;		/* Last request on queue */
	int	wdc_irq;		/* IRQ we're on (bitmask) */
	int	wdc_iobase;		/* I/O port base */
	int	wdc_aiobase;		/* "Alternate" register base */
	wdc_callback_t *wdc_reset_head;	/* Controller reset callbacks */
	int	wdc_count;		/* Number of nested attaches */
} wdc_softc_t;

int	wdcprobe __P((struct device *parent, struct cfdata *cf, void *aux));
void	wdc_forceintr __P((struct isa_attach_args *ia));
void	wdcattach __P((struct device *parent, struct device *self, void *aux));
int	wdc_print __P((void *aux, char *name));
int	wdc_reset __P((int base, int abase));
void	wdc_reinit __P((struct device *self));
int	wdcintr __P((wdc_softc_t *sc));
void	wdc_timeout __P((void *arg));
void	wdc_attach_reset __P((struct device *self, wdc_callback_t *cb));
int	wdc_getiobase __P((struct device *self));
int	wdc_getaiobase __P((struct device *self));
void	wdc_request __P((struct device *self, wdc_req_t *req));
void	wdc_release __P((struct device *self));
int	wdc_getctl __P((struct device *self, wdc_req_t *req));
int	wdc_pollintr __P((struct device *self, int timeout));
void	wdc_getctl_go __P((struct device *dev));

struct cfdriver wdccd =
    { NULL, "wdc", wdcprobe, wdcattach, DV_DULL, sizeof(wdc_softc_t) };

extern int hz;

/* -------------------- Interface to config/system --------------------*/

/*
 * Probe for controller
 */
int
wdcprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	int base = ia->ia_iobase;
	int i;
	int flags = getdevconf(cf, NULL, -1);
	int abase = base + wdalt_norm;

	/* Simple checks for register existence */
	outb(base + wd_sdh, WDSD_IBM);		/* Select drive 0 */
	outb(base + wd_error, 0xaa);		/* Shouldn't be writable */
	outb(base + wd_cyl_lo, 0x55);		/* Should be writable */
	if (inb(base + wd_cyl_lo) != 0x55 || inb(base + wd_error) == 0xaa)
		return (0);

	/*
	 * Reset controller, if it comes unbusy its probably there. We also
	 * use this to determine if the alternate registers are on the
	 * floppy controller or contiguous with the standard registers (as
	 * in PCMCIA cards).
	 */
	if (wdc_reset(base, abase)) {
		if ((flags & WDC_NOALTCHK) == 0) {
			/* Not at normal address, try PCMCIA address */
			abase = base + wdalt_pcmcia;
			if (wdc_reset(base, abase))
				return (0);
		}
	}

	/* Discover IRQ if not given */
	ia->ia_aux = (void *)abase;
	if (ia->ia_irq == IRQUNK) {
		ia->ia_irq = isa_discoverintr(wdc_forceintr, (void *) ia);
		wdc_reset(base, abase);
		if (ffs(ia->ia_irq) - 1 == 0)
			return (0);
	}
	ia->ia_iosize = WD_NPORT;
	return (1);
}

/*
 * Force an interrupt.
 */
void
wdc_forceintr(ia)
	struct isa_attach_args *ia;
{
	int abase = (int)ia->ia_aux;

	outb(ia->ia_iobase + wd_sdh, WDSD_IBM);		/* Select drive 0 */
	outb(abase + wda_ctlr, 0);
	outb(ia->ia_iobase + wd_command, WDCC_NOP);
}

/*
 * Attach controller.
 */
void
wdcattach(parent, self, aux)
	struct device *parent;
	struct device *self;
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	wdc_softc_t *sc = (wdc_softc_t *)self;
	int base = ia->ia_iobase;
	static int timer_started = 0;
	wdc_attach_args_t waa;
	int i;
	int abase = (int)ia->ia_aux;

	sc->wdc_iobase = base;
	sc->wdc_aiobase = abase;
	isa_portalloc(abase, WD_NALT);

	sc->wdc_cur = sc->wdc_last = NULL;
	sc->wdc_reset_head = NULL;
	sc->wdc_count = 0;
	sc->wdc_irq = ia->ia_irq;

	wdc_reset(base, abase);

	printf(": disk controller");
	if (abase == base + wdalt_pcmcia)
		aprint_normal(" (pcmcia)");
	printf("\n");

	/* Attempt to attach both units */
	for (i = 0; i < 2; i++) {
		waa.found = 0;
		waa.drive = i;
		config_found(self, &waa, wdc_print);
	}

	/* Start the controller tick if needed */
	if (!timer_started) {
		timer_started++;
		timeout(wdc_timeout, (caddr_t)0, hz);
	}

	/* Connect */
	isa_establish(&sc->wdc_id, &sc->wdc_dev);
	sc->wdc_ih.ih_fun = wdcintr;
	sc->wdc_ih.ih_arg = (void *) sc;
	intr_establish(ia->ia_irq, &sc->wdc_ih, DV_DISK);
}

/*
 * Print message when a unit is found
 */
int
wdc_print(aux, name)
	void *aux;
	char *name;
{
	wdc_attach_args_t *ap = (wdc_attach_args_t *)aux;

	if (!ap->found)
		return (QUIET);
	if (name)
		aprint_normal(" at %s", name);
	aprint_normal(" drive %d", ap->drive);
	return (UNCONF);
}

/*
 * Soft reset controller, synchronous
 */
int
wdc_reset(base, abase)
	int base;
	int abase;
{
	int i;

	/* Try a controller reset */
	outb(abase + wda_ctlr, WDCTL_RESET|WDCTL_IDIS);
	delay(10000);
	outb(abase + wda_ctlr, WDCTL_IDIS);

	/* Wait for init complete and clear interrupt */
	i = RESET_WAIT;
	while (inb(base + wd_status) & WDCS_BUSY) {
		if (!--i)
			return (1);
		DELAY(1000);
	}

	/* Get reset status, clear interrupt */
	if (inb(base + wd_error) != 1)
		return (1);

	/* Re-enable interrupts */
	outb(abase + wda_ctlr, 0);
	DELAY(1000);
	return (0);
}

/*
 * Reset the controller and call back anyone who wanted to know about it.
 * Note: Ownership of the controller is not changed, the callback must
 *	queue if the unit doesn't already own the controller.
 */
void
wdc_reinit(self)
	struct device *self;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;
	wdc_callback_t *cbp;

	/* Hit the controller hardware */
	wdc_reset(sc->wdc_iobase, sc->wdc_aiobase);

	/* Notify all interested parties */
	for (cbp = sc->wdc_reset_head; cbp; cbp = cbp->next)
		cbp->func(cbp->arg);
}

/*
 * Controller interrupt dispatcher, send to current holder of controller
 */
int
wdcintr(sc)
	wdc_softc_t *sc;
{
	wdc_req_t *req = sc->wdc_cur;

	TR("wdcintr", sc, req);
	if (!req) {
		int wdc = sc->wdc_iobase;
		int status = inb(wdc + wd_status);
		TR("wdcintr: UNEXPECTED", status, 0);
		if (status & WDCS_BUSY) {
			printf("wdc%d: Unexpected interrupt, ctlr busy\n",
				sc->wdc_dev.dv_unit);
		} else if (status == (WDCS_READY|WDCS_SEEKCMPLT))
			;	/* Notebook powering down disk */
		else {
			printf("wdc%d: Unexpected interrupt, stat=%b "
			       "err=0x%x\n",
				sc->wdc_dev.dv_unit, status, WDCS_BITS,
				inb(wdc + wd_error));
		}
		return (-1);
	}
	return (req->intr(req->self));
}

/*
 * Timeout handler, called once a second we decrement the timeout counters
 * on active controller requests and call their timeout routine if they
 * hit 0. A requests timer only runs when the request is connected to
 * the controller (feature? perhaps...).
 */
void
wdc_timeout(arg)
	void *arg;
{
	wdc_softc_t *sc;
	wdc_req_t *req;
	int s = splbio();
	int i;

	for (i = 0; i < wdccd.cd_ndevs; i++) {
		if ((sc = wdccd.cd_devs[i]) == NULL)
			continue;
		req = sc->wdc_cur;
		if (req && req->timeout && --req->timeout == 0)
			req->tout(req->self);
	}
	splx(s);
	timeout(wdc_timeout, (caddr_t) NULL, hz);
}

/* ------------------- Interface to controller users ------------------ */

/*
 * Register a reset callback
 */
void
wdc_attach_reset(self, cb)
	struct device *self;
	wdc_callback_t *cb;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;

	cb->next = sc->wdc_reset_head;
	sc->wdc_reset_head = cb;
}

/*
 * Return I/O base address for controller
 */
int
wdc_getiobase(self)
	struct device *self;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;

	return (sc->wdc_iobase);
}

/*
 * Return I/O base address for "alternate" controller registers
 */
int
wdc_getaiobase(self)
	struct device *self;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;

	return (sc->wdc_aiobase);
}

/*
 * Queue for accesss to the controller, calls go routine immediately if 
 * controller is already attached or if it was free. Caution should be
 * used when overlapping requests, this is primarily designed to allow
 * the interrupt routine an option to not release the controller before
 * queueing the next request.
 *
 * Note: its not permissible to queue for the controller twice if you
 *	don't already own it.
 */
void
wdc_request(self, req)
	struct device *self;
	wdc_req_t *req;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;
	wdc_req_t *cur = sc->wdc_cur;
	int s = splbio();

	TR("wdc_request", req, sc->wdc_count);
	
	if (cur) {
		if (cur == req) {
			/* Notify immediately if she already has it */
			sc->wdc_count++;
			TR("wdc_request: recursive GO", req->self, 
				sc->wdc_count);
			req->go(req->self);
		} else {
			if (req->next != WDC_UNALLOC)
				panic("wdc_request: queueing non-free request");
			/* Queue for access */
			req->next = NULL;
			sc->wdc_last->next = req;
			sc->wdc_last = req;
		}
	} else {
		/* Give it to her right away */
		sc->wdc_cur = sc->wdc_last = req;
		req->next = NULL;
		sc->wdc_count++;
		TR("wdc_request: GO", req->self, 0);
		req->go(req->self);
	}
	splx(s);
}

/*
 * Release controller (must be called at splbio())
 */
void
wdc_release(self)
	struct device *self;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;
	wdc_req_t *req = sc->wdc_cur;
	wdc_req_t *nreq;

	TR("wdc_release req/count", req, sc->wdc_count);
	if (!req)
		panic("wdc_release: no controller attached");
	if (--sc->wdc_count)
		return;
	nreq = sc->wdc_cur = req->next;
	req->next = WDC_UNALLOC;
	if (nreq) {
		TR("wdc_release", nreq, nreq->self);
		sc->wdc_count++;
		nreq->go(nreq->self);
	}
}

/*
 * Request access to controller synchronously
 *
 * It is legal to request the controller if it is already owned, in this
 * case return is immediate. Since this defeats the locking provided at
 * the wdc level the caller must take care to coordinate usage with
 * the prior owner of the controller. There is no analog in the queueing
 * code, it is not permissible to queue for the controller when it is
 * already held (if this is needed a different request block should be
 * used).
 *
 * Must be called at splbio().
 *
 * When done with controller caller must wdc_release().
 */
int
wdc_getctl(self, req)
	struct device *self;
	wdc_req_t *req;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;
	wdc_req_t save_req;

	TR("wdc_getctl: self/req",self,req);

	if (sc->wdc_cur == req) {
		sc->wdc_count++;
		TR("wdc_getctl: controller recusively acquired", sc, req);
		return (0);
	}

	/* Request access */
	save_req = *req;
	req->next = WDC_UNALLOC;
	req->intr = NULL;
	req->self = NULL;
	req->tout = NULL;
	req->timeout = 0;
	req->go = wdc_getctl_go;
	wdc_request(self, req);
	while (sc->wdc_cur != req) {
		/* Wait for an interrupt */
		if (wdc_pollintr((struct device *)sc, 5000000)) {
			wdc_req_t *p;

			printf("wdc%d: Sync controller get stalled\n",
				sc->wdc_dev.dv_unit);

			/* remove from queue */
			for (p = sc->wdc_cur; p != NULL; p = p->next) {
				if (p->next == req) {
					p->next = req->next;
					break;
				}
			}
			*req = save_req;
			return (1);
		}

		/* Call the owner to handle it */
		sc->wdc_cur->intr(sc->wdc_cur->self);
		DELAY(1000);		/* Allow interrupt time to clear */
	}
	save_req.next = req->next;
	*req = save_req;
	TR("wdc_getctl: acquired, count",sc->wdc_count,0);
	return (0);
}

/*
 * Poll for an interrupt on the wdc controller.
 *
 * Returns non-zero if the wait timed out.
 */
int
wdc_pollintr(self, timeout)
	struct device *self;
	int timeout;
{
	wdc_softc_t *sc = (wdc_softc_t *)self;

	while (--timeout) {
		if (icu_read_ir() & sc->wdc_irq)
			break;
		DELAY(1);
	}
	return (timeout == 0);
}

/* ------------------- Utility routines ------------------ */
/*
 * Dummy go routine for synchronous controller acquisition
 */
void
wdc_getctl_go(dev)
	struct device *dev;
{

}
