/*-
 * Copyright (c) 1992, 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: isa.c,v 2.12 1995/12/12 22:40:38 ewv Exp $
 */

/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * William Jolitz.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)isa.c	8.1 (Berkeley) 6/11/93
 */

/*
 * code to manage AT bus
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/malloc.h>
#include <sys/uio.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/reboot.h>

#include <vm/vm_param.h>
#include <machine/cpu.h>
#include <machine/segments.h>

#include <i386/isa/dma.h>
#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>
#include <i386/isa/icu.h>

#define	IDTVEC(name)	__CONCAT(X,name)
/* default interrupt vector table */
extern	IDTVEC(intr0), IDTVEC(intr1), IDTVEC(intr2), IDTVEC(intr3),
	IDTVEC(intr4), IDTVEC(intr5), IDTVEC(intr6), IDTVEC(intr7),
	IDTVEC(intr8), IDTVEC(intr9), IDTVEC(intr10), IDTVEC(intr11),
	IDTVEC(intr12), IDTVEC(intr13), IDTVEC(intr14), IDTVEC(intr15);
extern	IDTVEC(adintr0), IDTVEC(adintr1), IDTVEC(adintr2), IDTVEC(adintr3),
	IDTVEC(adintr4), IDTVEC(adintr5), IDTVEC(adintr6), IDTVEC(adintr7),
	IDTVEC(adintr8), IDTVEC(adintr9), IDTVEC(adintr10), IDTVEC(adintr11),
	IDTVEC(adintr12), IDTVEC(adintr13), IDTVEC(adintr14), IDTVEC(adintr15);

int	(*eisa_probefn)();
int	(*pci_probefn)();
int	eisa_present;
int	pci_present;
int	elcr_present;
isa_type isa_bustype = OLDBUS_ISA;		/* backwards compatability */

/*
 * Fill in interrupt table for used by auto intr discovery code
 * during configuration of kernel, setup interrupt control unit
 *
 * All of these call the dummy interrupt vector isa_adintr() (below).
 */
isa_autoirq()
{
	int sel = GSEL(GCODE_SEL, SEL_KPL);

	/* first icu */
	setidt(32, &IDTVEC(adintr0), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(33, &IDTVEC(adintr1), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(34, &IDTVEC(adintr2), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(35, &IDTVEC(adintr3), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(36, &IDTVEC(adintr4), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(37, &IDTVEC(adintr5), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(38, &IDTVEC(adintr6), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(39, &IDTVEC(adintr7), SDT_SYS386IGT, SEL_KPL, sel);

	/* second icu */
	setidt(40, &IDTVEC(adintr8), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(41, &IDTVEC(adintr9), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(42, &IDTVEC(adintr10), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(43, &IDTVEC(adintr11), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(44, &IDTVEC(adintr12), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(45, &IDTVEC(adintr13), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(46, &IDTVEC(adintr14), SDT_SYS386IGT, SEL_KPL, sel);
	setidt(47, &IDTVEC(adintr15), SDT_SYS386IGT, SEL_KPL, sel);

	/* reset coprocessor interrupt latch */
	outb(IO_NPX+1, 0);

	if (eisa_present) {
		/* take care of some EISA obligations */
		outb(IO_ENMI, 0);	/* turn off extended NMI features */
		outb(IO_ELCR1, 0);	/* edge-triggered interrupts (0-7) */
		outb(IO_ELCR2, 0);	/* edge-triggered interrupts (8-15) */
	}

	/* initialize 8259's */
	outb(IO_ICU1, 0x11);	/* ICW1 */
	outb(IO_ICU1+1, 32);	/* ICW2: high bits of interrupt vectors */
	outb(IO_ICU1+1, 0x4);	/* ICW3: cascade interrupt */
#ifdef AUTO_EOI
	/*
	 * Some machines with old 8259s won't boot with automatic EOI.
	 * Note, this must correspond with EOI sequence in INTR macro (icu.h).
	 */
	outb(IO_ICU1+1, 3);	/* ICW4: auto EOI */
	outb(IO_ICU1, 0x6b);	/* set special mask mode, and ISR for read */
#else
	outb(IO_ICU1+1, 1);	/* ICW4 */
	outb(IO_ICU1, 0xb);	/* OCW3: set ISR on IO_ICU1 read */
#endif
	outb(IO_ICU1+1, 0xff);	/* OCW1: mask all interrupts */

	outb(IO_ICU2, 0x11);
	outb(IO_ICU2+1, 40);
	outb(IO_ICU2+1, 2);	/* ICW3: master's IRQ for this slave */
#ifdef AUTO_EOI
	outb(IO_ICU2+1, 3);
	outb(IO_ICU2, 0x6b);
#else
	outb(IO_ICU2+1, 1);
	outb(IO_ICU2, 0xb);
#endif
	outb(IO_ICU2+1, 0xff);

	INTREN(IRQ_SLAVE);
}

matchbyname(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{

	return (strcmp(cf->cf_driver->cd_name, (char *) aux) == 0);
}

#define	NIRQ	16

/*
 * Convert irq index to standard mask representation
 * (note that IRQ2 is the same as IRQ9, and is not (1 << 2)).
 * Another exception: for PCI, we map 0 to IRQNONE.
 */
u_short irq2mask[] = {
	IRQNONE, IRQ1, IRQ2, IRQ3, IRQ4, IRQ5, IRQ6, IRQ7,
	IRQ8, IRQ9, IRQ10, IRQ11, IRQ12, IRQ13, IRQ14, IRQ15
};

static int irqmap = IRQ0|IRQ_SLAVE;	/* clock and slave are reserved */
int irqshare;

/*
 * Order in which free IRQs should be used for dynamic allocation,
 * attempting to avoid IRQs that might be used by motherboard devices
 * (including those configured for PCI devices) that might not be used.
 * We prefer 16-bit IRQs just in case, then IRQs for devices that would
 * probably have been configured already if present.
 */
u_short irqpref[] = {
	/* generally usable if free */
	IRQ11,
	IRQ10,
	IRQ9,	/* aka IRQ2 */

	/* used by common devices, hopefully useable if not configured yet */
	IRQ5,
	IRQ7,
	IRQ4,
	IRQ3,

	/* either hard disk, or sometimes reserved for PCI */
	IRQ15,
	IRQ14,

	/* commonly used by standard/internal devices */
	IRQ6,	/* floppy controller */
	IRQ12,	/* PS/2 mouse */
	IRQ8,	/* RTC */
	IRQ13,	/* FPU */
	IRQ1,	/* kbd (???) */
};

/*
 * Chose a free IRQ given a mask of choices;
 * return a mask containing one free IRQ,
 * or 0 if none if the IRQs in the mask are free.
 */
isa_irqalloc(irqmask)
	int irqmask;
{
	u_short *p;

	irqmask &= ~irqmap;
	for (p = irqpref; p < &irqpref[sizeof(irqpref)/sizeof(irqpref[0])]; p++)
		if (irqmask & *p)
			return (*p);
	/* no useable irqs */
	return (0);
}

/*
 * Get current interrupt status registers from the 8259's (non-destructive).
 * This returns a composite of the two 8259 IR registers in a format that
 * can be masked with the IRQx constants from icu.h.
 *
 * The return value will no longer be correct unless the interrupt(s)
 * of interest are blocked from before the call until the value is used.
 * Never call this under splmem_fast().
 */
int
icu_read_ir()
{
	int ir;

	/*
	 * Block interrupts while switching from IS to IR,
	 * then switch back to IS after reading IR.  The interrupt
	 * entry macros for IRQ 7 and 15 need access to IS,
	 * and assume that they do not need to switch.
	 */
	asm("cli");
	outb(IO_ICU1, ICU_OCW_SEL|ICU_READ_IR);
	outb(IO_ICU2, ICU_OCW_SEL|ICU_READ_IR);
	ir = inb(IO_ICU1) | inb(IO_ICU2) << 8;

	outb(IO_ICU1, ICU_OCW_SEL|ICU_READ_IS);
	outb(IO_ICU2, ICU_OCW_SEL|ICU_READ_IS);
	asm("sti");
	return (ir);
}

/*
 * We keep track of the allocation of low-numbered I/O ports
 * as that is where most of the action is.
 */
#define	ISA_NPORT_CHECK	0x400

#define	FREE8	0		/* NBBY (== 8) consecutive free ports */
#define	BUSY8	0xff		/* NBBY (== 8) consecutive busy ports */
#define	BUSY2	0x03		/* 2 busy ports, then 6 free */
#define	BUSY4	0x0f		/* 4 busy ports, then 4 free */
#define	FREE16	FREE8, FREE8
#define	BUSY16	BUSY8, BUSY8
#define	FREE64	FREE16, FREE16, FREE16, FREE16

/* initialized to include internally-used ports like dma, icu, etc. */
u_char	isa_portmap[ISA_NPORT_CHECK / NBBY] = {
	BUSY8,  FREE8,  FREE16, BUSY2, FREE8, FREE16,		/*   0 -  3f */
	BUSY4,  FREE8,  FREE16,  0x02, FREE8, BUSY2, FREE8,	/*  40 -  7f */
	BUSY16, FREE16, BUSY2,  FREE8, FREE16,			/*  80 -  bf */
	BUSY16, BUSY16, FREE16, FREE16,				/*  c0 -  ff */
	FREE64,			/* 100 - 13f */
	FREE64,			/* 140 - 17f */
	FREE64,			/* 180 - 1bf */
	FREE64,			/* 1c0 - 1ff */
	FREE64,			/* 200 - 23f */
	FREE64,			/* 240 - 27f */
	FREE64,			/* 280 - 2bf */
	FREE64,			/* 2c0 - 2ff */
	FREE64,			/* 300 - 33f */
	FREE64,			/* 340 - 37f */
	FREE64,			/* 380 - 3bf */
	FREE64,			/* 3c0 - 3ff */
};

/*
 * Check whether "size" ports are free beginning at "base".
 * The minimum size is 1, but 0 is interpreted as 1
 * for convenience when the size has not yet been set.
 */
isa_portcheck(base, size)
	int base, size;
{
	register int i;

	if (size == 0)
		size = 1;
	for (i = base; i < ISA_NPORT_CHECK &&  i < base + size; i++)
		if (isset(isa_portmap, i))
			return (0);
	return (1);
}

/*
 * Mark "size" ports as allocated beginning at "base".
 */
void
isa_portalloc(base, size)
	int base, size;
{
	register int i;

	for (i = base; i < ISA_NPORT_CHECK &&  i < base + size; i++)
		setbit(isa_portmap, i);
}

int autodebug = 0;		/* compatibility for older drivers */
int autoprint = 0;		/* output control for autoconf print */
extern struct bootparam *getbootparam();

/* Bus names indexed by ia_bustype locator value */
char *isa_bus_names[] = {
	"unknown%d",		/* Any/Unknown - shouldn't happen */
	"isa%d",	
	"eisa%d",
	"mca%d",
	"pci%d",
	"pcmcia%d"
};

/*
 * Probe for a device on the ISA bus
 */
isa_devprobe(dev, cf, aux)
	struct device *dev;
	struct cfdata *cf;
	void *aux;
{
	struct isa_attach_args iaa, *ia = &iaa;
	struct bootparam *bp;
	struct boot_devspec ds, *dsp;
	int irq, share, oflags;
	int matchlevel;
	int isaprint();
	
	if (sizeof(ds.ds_loc) / sizeof(ds.ds_loc[0]) <= DSLOC_FLAGS)
		panic("boot_devspec too small");
	/*
	 * look through boot params for locator overrides.
	 * Take the last one matching in case user typed
	 * line in more than once. Easier than making boot
	 * only pass it in once.
	 */
	/* XXX should change this to use getdevconf to get devspec and flags */
	ds.ds_validmask = 0;
	for (bp = NULL; bp = getbootparam(B_DEVSPEC, bp); ) {
		dsp = (struct boot_devspec *) B_DATA(bp);
		if (strcmp(cf->cf_driver->cd_name, dsp->ds_driver) != 0)
			continue;
		if (cf->cf_unit != dsp->ds_unit)
			continue;
		ds = *dsp;
	}

#define ASSIGN_ONE(to,from)				\
	if ((ds.ds_validmask & (1 << (from))) == 0)	\
		(to) = cf->cf_loc[(from)];		\
	else						\
		(to) = ds.ds_loc[(from)]

	ASSIGN_ONE(ia->ia_iobase, LOC_IOBASE); 
	ASSIGN_ONE(ia->ia_iosize, LOC_IOSIZE);	/* usually not yet known */
	ASSIGN_ONE(ia->ia_msize, LOC_MSIZE);	/* len of io mem */
	ASSIGN_ONE(ia->ia_drq, LOC_DRQ);
	ASSIGN_ONE(ia->ia_bustype, LOC_BUSTYPE); /* not sure this is worth it */
	if ((ds.ds_validmask & (1 << (LOC_MADDR))) == 0)
		ia->ia_maddr = (caddr_t) cf->cf_loc[LOC_MADDR];
	else
		ia->ia_maddr = (caddr_t) ds.ds_loc[LOC_MADDR];
	ASSIGN_ONE(irq, LOC_IRQ);		/* usually IRQUNK */ 

#undef ASSIGN_ONE

	switch (irq) {
	case -1:
		ia->ia_irq = IRQUNK;		/* unknown */
		break;
	case 0:
		ia->ia_irq = IRQNONE;
		break;
	default:
		ia->ia_irq = irq_indextomask(irq);
		break;
	}

	/*
	 * See whether device is supposedly attached
	 * to a bus that we do not have.
	 */
	switch (ia->ia_bustype) {
	case BUS_EISA:
		if (EISA_PRESENT() == 0)
			return (0);
		break;

	case BUS_PCI:
		if (PCI_PRESENT() == 0)
			return (0);
		break;
	}

	if (autoprint & AC_DEBUG) {
		aprint_debug("\nprobe %s%d",
		    cf->cf_driver->cd_name, cf->cf_unit);
		/* XXX set parent bus type for isaprint */
		switch (ia->ia_bustype) {
		case BUS_EISA:
			aprint_debug(" at eisa?");
			break;
		case BUS_PCI:
			aprint_debug(" at pci?");
			break;
		default:
			break;
		}
		isaprint(ia, NULL);
		if (autoprint & AC_ASK) {
			aprint_debug(" [yn]? ");
			if (getchar() == 'n') {
				aprint_debug("\n");
				return (0);
			}
		}
		aprint_debug(": ");
	}

	/*
	 * If the port address is -1, do not probe the device.
	 * Used to disable a device from the boot devspec.
	 */
	if (ia->ia_iobase == (u_short)-1) {
		aprint_debug("port is -1, skipped\n");
		return (0);
	}

	/*
	 * Check that at least the base port is available.
	 * The size is usually 0 at this point.  The driver may
	 * make a complete check if desirable, including any
	 * discontiguous ports.
	 */
	if (ia->ia_iobase && isa_portcheck(ia->ia_iobase, ia->ia_iosize) == 0) {
		aprint_debug("base port in use\n");
	 	return (0);
	}

	oflags = cf->cf_flags;
	if (ds.ds_validmask & (1 << DSLOC_FLAGS))
		cf->cf_flags = ds.ds_loc[DSLOC_FLAGS];

	/* 
	 * driver dependent match is responsible for filling in the proper
	 * ia_irq, ia_drq, ia_maddr, ia_iosize, etc.
	 */
	if (!(matchlevel = (*cf->cf_driver->cd_match)(dev, cf, (void *)ia))) {
		aprint_debug("probe failed\n");
		cf->cf_flags = oflags;
		return (0);
	}

	/*
	 * Check for port overlap again, as we should now
	 * have the correct size.
	 */
	if (ia->ia_iobase && isa_portcheck(ia->ia_iobase, ia->ia_iosize) == 0) {
		printf("Warning: probe of %s%d may have modified ports 0x%x - 0x%x, already in use\n",
		    cf->cf_driver->cd_name, cf->cf_unit,
		    ia->ia_iobase + 1, ia->ia_iobase + ia->ia_iosize - 1);
		cf->cf_flags = oflags;
		return (0);
	}
	if (ia->ia_bustype == BUS_ANY)
		ia->ia_bustype = BUS_ISA;

	/*
	 * irq conflict map -- warn if more than one device uses
	 * an irq unless all of them set IRQSHARE; sharing may
	 * or may not work otherwise.
	 */
	share = ia->ia_irq & IRQSHARE;
	irq = ia->ia_irq &~ IRQSHARE;
	if (irqmap & irq) {
		if ((irq & irqshare) == 0 || share == 0)
			printf("WARNING: conflict at irq %d\n",
			    ffs(ia->ia_irq) - 1);
	} else if (share)
		irqshare |= irq;
	irqmap |= irq;

	if (ia->ia_drq == (u_short)-1)
		ia->ia_drq = DRQNONE;

	isa_portalloc(ia->ia_iobase, ia->ia_iosize);
	aprint_debug("attached\n");

	/* 
	 * XXX
	 * Set parent bus name based on the result from probe (usually what
	 * was in the original cfdata record). We do this trickery since 
	 * the other bustypes (pci, eisa) are not actually different
	 * parent devices, everything is still actually a child of isa0.
	 * They may become full-fledged busses in the future, but currently
	 * we need to intermix them to control the order of device probes.
	 */
	if ((unsigned int)ia->ia_bustype <= BUS_PCMCIA)
		sprintf(dev->dv_xname, isa_bus_names[ia->ia_bustype],
		    dev->dv_unit);

	config_attach(dev, cf, ia, isaprint);

	/* XXX restore isa0 name */
	sprintf(dev->dv_xname, "isa%d", dev->dv_unit);
	cf->cf_flags = oflags;
	return (matchlevel);
}

/*
 * Determine the type of bus present.
 * Currently, we check for eisa now and defer everything else
 * until later.  Note: this is called early, from init386,
 * so that we know whether eisa is present when isa_autoirq is called.
 * This should be fixed.
 */
isa_probe()
{

	if (!bcmp(ISA_HOLE_VADDR(EISA_ID_OFFSET), EISA_ID, EISA_ID_LEN)) {
		eisa_present = 1;
		elcr_present = 1;
		isa_bustype = OLDBUS_EISA;	/* backwards compatability */
	}
	return (1);
}

/*
 * Attach the isa bus.
 *
 * Our main job is to attach the ISA bus and iterate down the list 
 * of `isa devices' (children of that node).
 */
static void
isa_attach(parent, dev, aux)
	struct device *parent, *dev;
	void *aux;
{
	extern int do_page;
	int old_page = do_page;

	printf("\n");

	if (autoprint & (AC_PAGE | AC_DEBUG))	/* for pci_probe */
		do_page = 1;
	if (eisa_probefn)
		eisa_present = (*eisa_probefn)();
	if (pci_probefn)
		pci_present = (*pci_probefn)();

	/* page output if debugging, but not asking per device */
	do_page = (autoprint & AC_PAGE);
	config_search(isa_devprobe, dev, aux);
	isa_normalirq();
	do_page = old_page;
}

struct cfdriver isacd =
    { NULL, "isa", matchbyname, isa_attach, DV_DULL, sizeof(struct isa_softc) };

isaprint(aux, name)
	void *aux;
	char *name;
{
	register struct isa_attach_args *ia = aux;
	int irq;

	if (name)
		aprint_normal(" at %s", name);
	if (ia->ia_iobase)
		aprint_normal(" iobase 0x%x", ia->ia_iobase);
	if (irq = (ia->ia_irq &~ IRQSHARE))
		if (irq == IRQUNK)
			aprint_normal(" irq ?");
		else
			aprint_normal(" irq %d", ffs(irq) - 1);
	if (ia->ia_drq != DRQNONE)
		if (ia->ia_drq == (u_short)-1)
			aprint_normal(" drq ?");
		else
			aprint_normal(" drq %d", ia->ia_drq);
	if (ia->ia_maddr) {
		prf_t mprint;

		if (ia->ia_bustype == BUS_PCI)
			mprint = aprint_verbose;
		else
			mprint = aprint_normal;
		(*mprint)(" maddr 0x%x-0x%x", ia->ia_maddr,
		    ia->ia_maddr + ia->ia_msize - 1);
	}

	return (UNCONF);
}

/*
 * Each attached device calls isa_establish after it initializes
 * its isadev portion.  This links the device into the isa chain.
 */
void
isa_establish(id, dev)
        register struct isadev *id;
        register struct device *dev;
{
        register struct isa_softc *sc = (struct isa_softc *)dev->dv_parent;

        id->id_dev = dev;
        id->id_bchain = sc->sc_isadev;
        sc->sc_isadev = id;
}


int mask0, mask1, mask2, mask3, mask4, mask5, mask6, mask7, 
    mask8, mask9, mask10, mask11, mask12, mask13, mask14, mask15;

int *masktbl[NIRQ] = {
	&mask0, &mask1, &mask2, &mask3, &mask4, &mask5, &mask6, &mask7, 
	&mask8, &mask9, &mask10, &mask11, &mask12, &mask13, &mask14, &mask15
};

struct intrhand *intrhand[NIRQ] = {
	NULL,		/* intr 0: Clock */
	NULL,		/* intr 1: Keyboard */
	NULL,		/* intr 2 */
	NULL,		/* intr 3 */
	NULL,		/* intr 4 */
	NULL,		/* intr 5 */
	NULL,		/* intr 6 */
	NULL,		/* intr 7 */
	NULL,		/* intr 8 */
	NULL,		/* intr 9 */
	NULL,		/* intr 10 */
	NULL,		/* intr 11 */
	NULL,		/* intr 12 */
	NULL,		/* intr 13: coproc */
	NULL,		/* intr 14: harddrive */
	NULL,		/* intr 15 */
};

int	isa_straycount[16];	/* unclaimed interrupts on all irq's */
int	isa_defaultstray[2];	/* "default interrupts" on irq 7/15 */

/*
 * Attach an interrupt handler to the vector chain for the given level.
 * Each driver's attach routine must call this to register its interrupt.
 * Currently this can be called only during autoconfiguration (before
 * isa_normalirq() runs) with interrupts disabled.
 */
void
intr_establish(intr, ih, devtype)
	int intr;
	struct intrhand *ih;
	enum devclass devtype;
{
	register struct intrhand **p, *q;
	int index = ffs(intr) - 1;

	for (p = &intrhand[index]; (q = *p) != NULL; p = &q->ih_next)
		continue;
	*p = ih;
	ih->ih_next = NULL;
	switch (devtype) {
		case DV_DISK:
		case DV_TAPE:
			INTRMASK(biomask, intr);
			*masktbl[index] |= intr;
			break;
		case DV_TTY:
			INTRMASK(ttymask, intr);
			*masktbl[index] |= intr;
			break;
		case DV_NET:
			INTRMASK(impmask, intr);
			*masktbl[index] |= intr;
			break;
		case DV_CLK:
			*masktbl[index] |= highmask;
			break;
		default:	/* DV_COPROC, DV_DULL */
			*masktbl[index] |= intr;
			break;
	}
	irqmap |= intr;
	INTREN(intr);
}

/*
 * Normal interrupt vector (Torek-style) which acts as a switch to 
 * the driver.
 */
void
isa_intrswitch(frame)
	struct intrframe frame;
{
	register struct intrhand *ih;
	int expected = 0;

	for (ih = intrhand[frame.if_vec]; ih; ih = ih->ih_next) {
		/* 
		 * Call the appropriate driver's interrupt handler.
		 * For now at least, can't tell if interrupts
		 * will be 1-for-1 with requests, so call all handlers,
		 * ignoring return values.
		 */
		/*
		 * awkward code below avoids compiler strangeness with
		 * ?: conditional
	  	 */
		if (ih->ih_arg) {
			if ((*ih->ih_fun)(ih->ih_arg)) {
				expected++;
				ih->ih_count++;
			}
		} else {
			if ((*ih->ih_fun)(&frame)) {
				expected++;
				ih->ih_count++;
			}
		}
	}
	if (expected == 0) {
		isa_straycount[frame.if_vec]++;
		/* Ignore stray clock and intr 7 (problem on some machines) */
		if (frame.if_vec != 0 && frame.if_vec != 7)
			log(LOG_ERR, "stray interrupt on ISA irq %d\n",
			    frame.if_vec);
	}
}

#if 0	/* currently done in-line */
/*
 * Set specified irq to be level triggered if possible.
 * Note that irq is an index (0-15), not an IRQX mask.
 */
isa_leveltrig(irq)
	int irq;
{

	if (!elcr_present)
		return (0);
	if (irq == 2)		/* irq 2 means 9 */
		irq = 9;
	if (irq < 8)
		outb(IO_ELCR1, inb(IO_ELCR1) | (1 << irq));
	else
		outb(IO_ELCR2, inb(IO_ELCR2) | (1 << (irq - 8)));
	return (1);
}
#endif

volatile int autointr;
static int icount;

/*
 * This is the "dummy" interrupt vector used during autoconfig
 * to catch interrupts generated by prodding the cards.
 */ 
isa_adintr(frame)
	struct intrframe frame;
{

	/* ignore clk & kbd */
	if (frame.if_vec != 0 && frame.if_vec != 1 && autointr == 0)
		autointr = frame.if_vec;
	if (icount++ > 50)
		frame.if_ppl = protomask;
}

/*
 * Used by each driver to "prod" the card and get it to generate
 * an interrupt, and then return which interrupt it got.
 *
 * Driver supplies prod routine (as force argument) and arg to
 * that routine.
 */  
isa_discoverintr(force, arg)
	void (*force)();
	void *arg;
{
	volatile int *aip = &autointr;
	int i;

	aprint_debug("check intr: ");
	icount = 0;
	splx(0);			/* all interrupts enabled */
	DELAY(10000);
	*aip = 0;

	/* 
 	 * call driver-specific cattle prod 
	 */
	(*force)(arg);

	/* 
	 * wait for card to interrupt 
	 */
	for (i = 0; (i < 100) && (!*aip); i++)
		DELAY(10000);

	(void) spl0();		/* only hard clock enabled now */
	if (*aip)
		aprint_debug("irq %d ", *aip);
	else
		aprint_debug("none ");
	return (1 << *aip);
}

isa_got_intr()
{

	return (autointr ? (1 << autointr) : 0);
}

int tty_net_devs = 0;	/* any tty-class devices that may be used for net? */
int orig_impmask;

/*
 * Set up "real" interrupt vectors [we're ready to run!]
 */
isa_normalirq()
{
	int sel = GSEL(GCODE_SEL, SEL_KPL), i;
	int mask;
	extern int safepri;

	/*
	 * Make sure we catch all the interrupt mask group for
	 * each interrupt 
	 */
	for (i = 0; i < NIRQ; i++) {
		mask = *masktbl[i];
		if (mask & biomask)
			mask |= biomask;
		if (mask & impmask)
			mask |= impmask;
		/*
		 * block bio, net devices during tty interrupts
		 * to minimize latency for ttys; this give ttys
		 * priority over all but clock.
		 */
		if (mask & ttymask)
			mask |= ttymask | biomask | impmask;
		*masktbl[i] = mask | nonemask | IRQSOFT;
	}

	/*
	 * Block unassigned interrupts at all times.
	 */
	biomask |= nonemask | IRQSOFT;
	ttymask |= nonemask | IRQSOFT;
	impmask |= nonemask | IRQSOFT;
	protomask = nonemask | IRQSOFT;

	/*
	 * If we're using networks over tty devices (SLIP, PPP, parallel-port
	 * Ethernet) then any tty intr is potentially a net intr.
	 * Note: adding ttymask to impmask after the individual interrupts'
	 * masks were set (above) means that network device interrupts
	 * will not block tty device interrupts, but that splimp()
	 * will block both.
	 */
	orig_impmask = impmask;
	if (tty_net_devs)
		impmask |= ttymask;

	safepri = nonemask;		/* allow NFS sync after crash */

	/* first icu */
#if 0	/* clock is done in enablertclock */
	setidt(32, &IDTVEC(intr0),  SDT_SYS386IGT, SEL_KPL, sel);
#endif
	setidt(33, &IDTVEC(intr1),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(34, &IDTVEC(intr2),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(35, &IDTVEC(intr3),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(36, &IDTVEC(intr4),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(37, &IDTVEC(intr5),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(38, &IDTVEC(intr6),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(39, &IDTVEC(intr7),  SDT_SYS386IGT, SEL_KPL, sel);

	/* second icu */
	setidt(40, &IDTVEC(intr8),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(41, &IDTVEC(intr9),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(42, &IDTVEC(intr10),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(43, &IDTVEC(intr11),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(44, &IDTVEC(intr12),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(45, &IDTVEC(intr13),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(46, &IDTVEC(intr14),  SDT_SYS386IGT, SEL_KPL, sel);
	setidt(47, &IDTVEC(intr15),  SDT_SYS386IGT, SEL_KPL, sel);
}

tty_net_devices(change)
	int change;
{

	tty_net_devs += change;
	if (tty_net_devs)
		impmask |= ttymask;
	else
		impmask = orig_impmask;
}

struct at_dma_state at_dma_state[8];

/*
 * The routine at_setup_dmachan should be called once by each driver
 * that will be using DMA, generally from the attach routine,
 * or at least from the top half.
 * NOTE: the memory in the bounce buffer must be physically contiguous,
 * and must not span a 64K boundary (128K for channels 5-7);
 * thus this is currently unsafe for maxsz > NBPG.
 *
 * Drivers using the DMA_RAW flag to at_dma need not call at_setup_dmachan().
 */
void
at_setup_dmachan(chan, maxsz)
	int chan;
	int maxsz;
{
	register struct at_dma_state *atd = &at_dma_state[chan];

	if (atd->bounce == NULL) {
		atd->bounce = (caddr_t) malloc(maxsz, M_DEVBUF, M_WAITOK);
		atd->bphys = kvtop(atd->bounce);
		atd->maxsz = maxsz;
	}
}

/*
 * Start DMA on channel "chan" to copy "nbytes" bytes to (read)
 * or from (write) kernel virtual or physical memory address "addr".
 * The address is kernel virtual unless the ATDMA_RAW flag is set.
 *
 * Flags:
 *	ATDMA_READ	- start a read to memory (else write to device)
 *	ATDMA_RAW	- address is physical and contiguous, don't check or
 *			  attempt to use bounce buffers
 *	ATDMA_AUTO	- run in auto restart mode (ATDMA_RAW must also be set
 *			  to use this)
 *
 * The previous version used the first parameter as "isread",
 * 1 for read and 0 for write.
 */
void
at_dma(flags, addr, nbytes, chan)
	int flags;
	caddr_t addr;
	int nbytes;
	int chan;
{
	register struct at_dma_state *atd = &at_dma_state[chan];
	int phys;
	int s, needcopy;

	/*
	 * If the transfer spans one or more page boundaries, the physical
	 * pages may well not be contiguous.  Could check, but that means
	 * one or two extra calls to pmap_extract(); just copy if
	 * we span pages for now.
	 */
	if (flags & ATDMA_RAW)
		phys = (int)addr;
	else {
		if (trunc_page(addr) != trunc_page(addr + nbytes - 1))
			needcopy = 1;
		else {
			phys = kvtop(addr);
			if (phys + nbytes - 1 > ISA_MAXADDR)
				needcopy = 1;
			else
				needcopy = 0;
		}

		atd->addr = NULL;
		if (needcopy) {
			if (nbytes > atd->maxsz)
				panic("at_dma size");
			phys = atd->bphys;

			/*
			 * copy to bounce buffer now on write,
			 * save address for copy after DMA on read
			 */
			if (flags & ATDMA_READ) {
				atd->addr = addr;
				atd->nbytes = nbytes;
			} else
				bcopy(addr, atd->bounce, nbytes);
		}
	}
	/*
	 * Start DMA
	 */
	if (chan > 3) {
		/*
		 * Channels 5-7 use 16-bit transfers, with size in words.
		 * (Channel 4 is used internally to cascade 8237A's.)
		 */
		chan -= 4;
		nbytes = (nbytes >> 1) - 1;
		s = splhigh();
		outb(DMA2_CLEARFF, 0);
		if (flags & ATDMA_READ)
			outb(DMA2_MODE, DMA_CNUM(chan) | DMA_READ |
			    (flags & DMA_AUTOINIT));
		else
			outb(DMA2_MODE, DMA_CNUM(chan) | DMA_WRITE |
			    (flags & DMA_AUTOINIT));

		/*
		 * Address bit 0 is ignored;
		 * address bits 1-16 go into the 8237A address register;
		 * address bits 17-23 go into the page register bits 1-7.
		 */
		outb(DMA2_ADDR(chan), phys >> 1);
		outb(DMA2_ADDR(chan), phys >> 9);
		outb(DMA2_PAGEREG(chan), phys >> 16);
		outb(DMA2_COUNT(chan), nbytes);
		outb(DMA2_COUNT(chan), nbytes >> 8);
		outb(DMA2_MASK, DMA_ECHAN(chan));
	} else {
		nbytes--;
		s = splhigh();
		outb(DMA1_CLEARFF, 0);
		if (flags & ATDMA_READ)
			outb(DMA1_MODE, DMA_CNUM(chan) | DMA_READ |
			    (flags & DMA_AUTOINIT));
		else
			outb(DMA1_MODE, DMA_CNUM(chan) | DMA_WRITE |
			    (flags & DMA_AUTOINIT));

		outb(DMA1_ADDR(chan), phys);
		outb(DMA1_ADDR(chan), phys >> 8);
		outb(DMA1_PAGEREG(chan), phys >> 16);
		outb(DMA1_COUNT(chan), nbytes);
		outb(DMA1_COUNT(chan), nbytes >> 8);
		outb(DMA1_MASK, DMA_ECHAN(chan));
	}	
	splx(s);
}

/*
 * Abort a DMA operation.
 * Could dig out the count and return actual/resid count,
 * but we don't need that now.
 */
void
at_dma_abort(chan)
	int chan;
{

	if (chan > 3)
		outb(DMA2_MASK, DMA_DCHAN(chan - 4));
	else
		outb(DMA1_MASK, DMA_DCHAN(chan));
}

/*
 * Set cascade mode on a DMA channel
 */
void
at_dma_cascade(chan)
	int chan;
{

	if (chan > 3) {
		chan -= 4;
		outb(DMA2_MODE, DMA_CNUM(chan) | DMA_CASCADE);
		outb(DMA2_MASK, DMA_ECHAN(chan));
	} else {
		outb(DMA1_MODE, DMA_CNUM(chan) | DMA_CASCADE);
		outb(DMA1_MASK, DMA_ECHAN(chan));
	}
}

#if 0
#define ISA_PORT_A	0x92
#define PORT_A_WATCHDOG	(1 << 4)
#endif

#define ISA_PORT_B	0x61
#define PORT_B_PARITY	(1 << 7)
#define PORT_B_IOCHAN	(1 << 6)
#define	PORT_B_BITS	"\20\010parity_error\007channel_check"

#define ENMI_WATCHDOG	(1 << 7)
#define ENMI_BUSTIMEOUT	(1 << 6)
#define ENMI_IOSTATUS	(1 << 5)
#define	ENMI_BITS	"\20\010watchdog\007bus_timeout\006nmi_pend"

extern	int nmi_panic;

/*
 * Handle a NMI, possibly a machine check.
 * Return true to simulate fatal trap, false to continue.
 */
isa_nmi(cd)
	int cd;
{
	int isa_port_b = inb(ISA_PORT_B);

	printf("\nNMI: port B(61)=%b", isa_port_b, PORT_B_BITS);
	if (EISA_PRESENT())
		printf(", ENMI(461)=%b", inb(IO_ENMI), ENMI_BITS);
	printf("\n");
	if (nmi_panic) {
		if (isa_port_b & PORT_B_PARITY)
			panic("RAM parity error");
		if (isa_port_b & PORT_B_IOCHAN)
			panic("I/O channel check hardware error");
	}
	return (0);
}
