/*
 * Copyright (c) 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: if_tr.c,v 2.6 1995/12/13 03:58:22 ewv Exp $
 */

/*
 * IBM TRA 4/16 (Natl Semi TROPIC chipset)
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/kernel.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/if_types.h>
#include <net/route.h>
#include <net/if_token.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <machine/cpu.h>
#include <machine/stdarg.h>

#include <net/bpf.h>
#include <net/bpfdesc.h>

#include "if_trreg.h"

/* Configuration */
#define	RETRY_CAP	256	/* Longest open retry wait (sec) */
#define RESET_DELAY	70000	/* Time to hold reset (usec) */
#define INIT_WAIT	3000	/* How long to wait for init (ms) */
#define TX_BSIZE	536	/* Size of a transmit buffer */
#define TX_DATA		(TX_BSIZE - sizeof(tr_fpb_t))
#define TX_NUMBUFS	9	/* Number of TX buffers */
#define RX_MINBUFS	4	/* Minimum number of RX buffers */
#define RX_BSIZE	1032	/* Receive buffer size */
#define DONE_MAXQ	60	/* Max number of done queue entries */
#define OPEN_WDOG	30	/* Open deadman timeout */

/*
 * Software status per adapter.
 */
typedef struct tr_softc {
	struct	device tr_dev;		/* base device (must be first) */
	struct	isadev tr_id;		/* ISA device */
	struct	intrhand tr_ih;		/* interrupt vectoring */	
	struct	arpcom tr_ac;		/* Ethernet common part */
	struct	atshutdown tr_ats;	/* Shutdown callback */

	/* Config */
	int	tr_iobase;		/* IO area base */
	caddr_t	tr_maddr;		/* Physical (ISA) address of shrd mem */
	caddr_t	tr_vaddr;		/* Kernel virtual addr of shared mem */
	tr_mio_t *tr_mvaddr;		/* Memory mapped IO area base (KV) */

	/* State */
	u_char	tr_flags;		/* Misc state */
	int	tr_retry;		/* secs before next open retry */
	u_char	tr_features;		/* Feature bits (init status) */

	/* Shared ram areas */
	u_short	tr_srb;			/* system request buffer offset */
	tr_fptca_t *tr_fpta;		/* fast path transmit area */
	tr_rxarb_t *tr_arb;		/* adapter request buffer */
	tr_asb_t *tr_asb;		/* adapter status buffer */

	/* Transmit */
	int	tr_txavail;		/* TX buffers on freelist */
	u_char	tr_txcorr;		/* Transmit correlator */

	/* Receive done queue */
	int	tr_dqlen;		/* length of done q/busy flag */
	u_short tr_doneq[DONE_MAXQ];	/* queue for saving asb requests */
	int	tr_dqh;			/* head and tail pointers in done Q */
	int	tr_dqt;
} tr_softc_t;

#define	tr_if	tr_ac.ac_if		/* ifnet struct */
#define	tr_addr	tr_ac.ac_enaddr		/* hardware (i.e. ethernet) address */

/* tr_softc_t.tr_flags */
#define TR_OPENING		0x01	/* open in progress */

/* Prototypes */
int trprobe __P((struct device *parent, struct cfdata *cf, void *aux));
void tr_forceintr __P((void *aux));
void trattach __P((struct device *parent, struct device *self, void *aux));
tr_mio_t *tr_getmio __P((int base));
int tr_reset __P((tr_softc_t *sc));
int trinit __P((int unit));
void tr_open __P((tr_softc_t *sc));
void tr_open_done __P((tr_softc_t *sc));
int tr_open_err __P((tr_softc_t *sc, u_char ret_code, u_short open_errcode));
void tr_shutdown __P((void *arg));
int trstart __P((struct ifnet *ifp));
void tr_txint __P((tr_softc_t *sc));
void tr_rxint __P((tr_softc_t *sc));
void tr_rxdone __P((tr_softc_t *sc, u_short bufr));
void tr_asbfree __P((tr_softc_t *sc));
void tr_post_rxdone __P((tr_softc_t *sc, u_short bufr));
int trint __P((tr_softc_t *sc));
int trwatchdog __P((int unit));
void tr_close __P((tr_softc_t *sc));
void tr_ringstat __P((tr_softc_t *sc));
int trioctl __P((struct ifnet *ifp, int cmd, caddr_t data));
void tr_halt __P((tr_softc_t *sc));
void tr_down __P((tr_softc_t *sc));
void trprf __P((tr_softc_t *sc, char *fmt, ...));

struct cfdriver trcd = {
	NULL, "tr", trprobe, trattach, DV_IFNET, sizeof(tr_softc_t)
};

/*
 * Open error messages
 *
 * Phase
 */
static char *phaserr[] = {
	"(unknown phase 0)",
	"lobe media test",
	"physical insertion",
	"address verification",
	"roll call poll",
	"request parameters",
	"(unknown phase 6)",
	"(unknown phase 7)"
};

/*
 * Error
 */
static char *openerr[] = {
	"(unknown 0x0)",
	"function failure",
	"signal loss",
	"wire fault",
	"ring speed mismatch",
	"timeout",
	"ring failure",
	"ring beaconing",
	"duplicate node address",
	"parameter request",
	"remove received",
	"(unknown 0xb)",
	"(unknown 0xc)",
	"(unknown 0xd)",
	"(unknown 0xe)",
	"(unknown 0xf)",
};
	
/*
 * Number of TX buffers needed in the pool to start a packet transmit. This
 * is: ((MTU+TX_BSIZE-1)/TX_BSIZE)+1  - or -
 *	Number of buffers needed for a maximum sized packet plus one for the
 *	end of the free queue.
 */
#define TX_NEEDED	5	/* # of buffers needed to start a packet */

/*
 * Produce a pointer to a fast path transmit buffer given the shared ram
 * offset of its next_bufr field in network order.
 */
extern inline tr_fpb_t *
tr_gfpb(sc, offset)
	tr_softc_t *sc;
	u_short offset;
{

	return ((tr_fpb_t *)(sc->tr_vaddr + ntohs(offset) - NEXTOFF));
}

/*
 * Return kernel virtual pointer given shared RAM offset
 */
extern inline void *
tr_mptr(sc, offset)
	tr_softc_t *sc;
	u_short offset;
{

	return ((void *)(sc->tr_vaddr + offset));
}

/* ---------------------- Initialization -------------------- */

/*
 * See if interface exists
 */
int
trprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	int pio = ia->ia_iobase;
	tr_mio_t *mio;
	int id;
	u_long mio_paddr;
	int i;

	/*
	 * Check ID nibbles
	 */
	mio = tr_getmio(pio);
	id = 0;
	for (i = 0; i < 8; i += 2) {
		id <<= 4;
		id |= mio->chnid[i] & 0xf;
	}
	if (id != TR_ISA_ID)
		return (0);

	ia->ia_msize = 
	    8192 << ((mio->rw.rrr_odd & TR_RRR_WMASK) >> TR_RRR_SHFT);
	ia->ia_iosize = TR_IOSIZE;
	if (ia->ia_irq == IRQUNK) {
		ia->ia_irq = isa_discoverintr(tr_forceintr, aux);
		outb(pio + TR_RESET, 1);
		delay(RESET_DELAY);
		outb(pio + TR_RUN, 1);
		if (ffs(ia->ia_irq) - 1 == 0)
			return(0);
	}
	return (1);
}

/*
 * Force an interrupt by doing an illegal access to shared RAM
 */
void
tr_forceintr(aux)
	void *aux;
{
	struct isa_attach_args *ia = aux;
	int pio = ia->ia_iobase;
	tr_mio_t *mio;
	int *sram;
	int i;

	outb(pio + TR_RESET, 1);
	delay(RESET_DELAY);
	outb(pio + TR_RUN, 1);

	mio = tr_getmio(pio);
	mio->or.isrp_even = INT_ENABLE | CHCK_IRQ;
	for (i = 0; i < INIT_WAIT; i++) {
		if (mio->rw.isrp_odd & SRB_RESP)
			break;
		delay (1000);
	}
	mio->and.isrp_odd = ~SRB_RESP;
	mio->rw.rrr_even = (u_int)(ia->ia_maddr) >> 12;

	sram = (int *)ISA_HOLE_VADDR(ia->ia_maddr);
	*sram = 0;
}

/*
 * Attach interface to system
 */
void
trattach(parent, self, aux)
	struct device *parent;
	struct device *self;
	void *aux;
{
	tr_softc_t *sc = (tr_softc_t *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	struct ifnet *ifp = &sc->tr_if;
	int pio = ia->ia_iobase;
	tr_mio_t *mio;
	tr_init_srb_t *srb;
	u_short address_addr;

	/* Print warm fuzzy */
	aprint_naive(": IBM Token Ring");

	sc->tr_iobase = pio;
	sc->tr_mvaddr = mio = tr_getmio(pio);

	sc->tr_vaddr = ISA_HOLE_VADDR(ia->ia_maddr);
	sc->tr_maddr = ia->ia_maddr;
	sc->tr_retry = 1;

	ifp->if_name = trcd.cd_name;
	ifp->if_unit = sc->tr_dev.dv_unit;

	if (tr_reset(sc) != 0)
		return;

	ifp->if_init = trinit;
	ifp->if_ioctl = trioctl;
	ifp->if_start = trstart;
	ifp->if_watchdog = trwatchdog;
	ifp->if_flags = IFF_BROADCAST;
	token_attach(ifp);

	bpfattach(&ifp->if_bpf, ifp, DLT_IEEE802, sizeof(struct token_header));

	isa_establish(&sc->tr_id, &sc->tr_dev);
	sc->tr_ih.ih_fun = trint;
	sc->tr_ih.ih_arg = (void *)sc;
	intr_establish(ia->ia_irq, &sc->tr_ih, DV_NET);

	/* Schedule shutdown callback */
	sc->tr_ats.func = tr_shutdown;
	sc->tr_ats.arg = (void *)sc;
	atshutdown(&sc->tr_ats, ATSH_ADD);

	/* print h/w info */
	if (sc->tr_features & RING_SPEED_16)
		aprint_normal(": 16 mbit");
	else
		aprint_normal(": 4 mbit"); 
	printf("\n");
	aprint_normal("tr%d: address %s\n", sc->tr_if.if_unit,
	    ether_sprintf(sc->tr_addr));
}

/*
 * Extract base address of mio area
 */
tr_mio_t *
tr_getmio(base)
	int base;
{
	tr_mio_t *mio;
	u_long mio_paddr;

	mio_paddr = inb(base + TR_SETUP);
	mio_paddr = ((mio_paddr & ~3) << 11) | 0x80000L;
	mio = (tr_mio_t *) ISA_HOLE_VADDR(mio_paddr);
	return (mio);
}

/*
 * Reset adapter hardware
 */
int
tr_reset(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	int pio = sc->tr_iobase;
	u_short bup_code;
	int i;
	char *p;
	tr_init_srb_t *srb;

	/* Adapter hard reset */
	outb(pio + TR_RESET, 1);
	delay (RESET_DELAY);
	outb(pio + TR_RUN, 1);

	/* Enable interrupts */
	mio->or.isrp_even = INT_ENABLE | CHCK_IRQ;

	/* Clear cached pointers and state */
	sc->tr_srb = 0;
	sc->tr_asb = 0;
	sc->tr_arb = 0;
	sc->tr_flags = 0;
	sc->tr_dqh = sc->tr_dqt = sc->tr_dqlen = 0;

	/* Wait for adapter to initialize */
	for (i = 0; i < INIT_WAIT; i++) {
		if (mio->rw.isrp_odd & SRB_RESP)
			break;
		delay (1000);
	}

	/* disable interrupts */
	mio->and.isrp_even = ~INT_ENABLE;

	if ((mio->rw.isrp_odd & SRB_RESP) == 0) {
		printf("\n");
		trprf(sc, "tr_reset: adapter not responding\n");
		return (-1);
	}

	/* clear response bit */
	mio->and.isrp_odd = ~SRB_RESP;

	/* Enable shared RAM at configured address */
	mio->rw.rrr_even = (u_int)(sc->tr_maddr) >> 12;

	/* Remember where the srb is */
	sc->tr_srb = ntohs(mio->rw.wrb);

	/* Check initialization return codes */
	srb = (tr_init_srb_t *)tr_mptr(sc,sc->tr_srb);
	bup_code = ntohs(srb->bup_code);

	if (srb->cmd != INIT_COMPLETE || bup_code != 0) {
		printf("\n");
		trprf(sc,"tr_reset: init failed: resp=0x%x, bup=0x%x\n",
		    srb->cmd, bup_code);
		return (-1);
	}

	sc->tr_features = srb->status;
	if ((sc->tr_features & FPT_AVAIL) == 0) {
		printf("\n");
		trprf(sc, "tr_reset: unsupported card revision "
		    "(no fast path)\n");
		return (-1);
	}

	/* For certain 64K cards must initialize parity */
	if (mio->totram == TR_RAM64_B && 
	    (mio->rw.rrr_odd&TR_RRR_WMASK) == TR_RRR_64K) {
		u_char *p = tr_mptr(sc, 0xfe00);

		for (i = 0; i <= 0x200; i++)
			p[i] = 0;
	}

	/* Extract MAC address */
	p = tr_mptr(sc, ntohs(srb->encoded_addr));
	for (i = 0; i < ISO88025_ADDR_LEN; i++)
		sc->tr_addr[i] = p[i];

	return (0);
}

/*
 *  Initialize interface and start open sequence
 */
int
trinit(unit)
	int unit;
{
	struct tr_softc *sc = trcd.cd_devs[unit];
	struct ifnet *ifp = &sc->tr_if;

	if (ifp->if_addrlist == NULL)
		return (0);			/* no address */

	if (ifp->if_flags & IFF_RUNNING)
		return (0);

	if (tr_reset(sc)) {
		tr_halt(sc);
		tr_down(sc);
		return (0);
	}
	tr_open(sc);
	return (0);
}

/*
 * Open the adapter and make it ready for tx/rx. This entails setting up
 * fast path transmit then issueing the open command.
 */
void
tr_open(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_cfpr_t *fpsrb = tr_mptr(sc, sc->tr_srb);
	tr_open_srb_t *opsrb;
	int i;

	sc->tr_flags |= TR_OPENING;

	/* Set up and issue the fast path config command */
	fpsrb->cmd = DIR_CFG_FAST_PATH_RAM;
	fpsrb->parm1 = htons((16 + (TX_BSIZE * TX_NUMBUFS) + 7) / 8);
	fpsrb->parm2 = htons(TX_BSIZE);
	mio->or.isra_odd = SRB_CMD;

	/* Wait for completion or timeout */
	for (i = 0; i < INIT_WAIT; i++) {
		if (mio->rw.isrp_odd & SRB_RESP)
			break;
		delay (1000);
	}

	if ((mio->rw.isrp_odd & SRB_RESP) == 0) {
		trprf(sc, "tr_open: fast path config timeout\n");
		tr_halt(sc);
		tr_down(sc);
		return;
	}
	mio->and.isrp_odd = ~SRB_RESP;

	/* Check return code */
	if (fpsrb->rc != 0) {
		trprf(sc, "tr_open: fast path config error: %d\n", fpsrb->rc);
		tr_halt(sc);
		tr_down(sc);
		return;
	}

	/* Save address of fast path area and new srb */
	sc->tr_fpta = tr_mptr(sc, ntohs(fpsrb->parm1));
	sc->tr_srb = ntohs(fpsrb->parm2);

	/* Enable interrupts */
	mio->or.isrp_even = INT_ENABLE | CHCK_IRQ;

	/* Set up an open command */
	opsrb = (tr_open_srb_t *)tr_mptr(sc, sc->tr_srb);

	bzero(opsrb, sizeof(*opsrb));

	opsrb->cmd = DIR_OPEN_ADAPTER;
	opsrb->num_rcv_buf = htons(RX_MINBUFS);
	opsrb->rcv_buf_len = htons(RX_BSIZE);
	opsrb->dhb_length = htons(TX_BSIZE);

	/* interrupt the adapter */
	mio->or.isra_odd = SRB_CMD;

	/* arm timeout */
	sc->tr_if.if_timer = OPEN_WDOG;
}

/* 
 * Called when the adapter open has completed. Cache interesting addresses
 * in the softc and possibly initiate retries.
 */
void
tr_open_done(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_open_resp_t *oresp = tr_mptr(sc, sc->tr_srb);
	struct ifnet *ifp = &sc->tr_if;

	/* clear the interrupt and watchdog timer */
	mio->and.isrp_odd = ~SRB_RESP;
	ifp->if_timer = 0;
	sc->tr_flags &= ~TR_OPENING;

	if (oresp->rc != 0) {
		/* Print error msg and determine if fatal */
		int die = tr_open_err(sc, oresp->rc, ntohs(oresp->err));

		if (die == 0 && (ifp->if_flags & IFF_UP) != 0) {
			/*
			 * If non-fatal and interface is up schedule an 
			 * open retry. Use a binary backoff to the cap 
			 * (256 secs).
			 */
			int bkoff = sc->tr_retry<<1;

			if (bkoff <= RETRY_CAP)
				sc->tr_retry = bkoff;
			trprf(sc, "insertion retry in %d seconds\n",
				sc->tr_retry);
			ifp->if_timer = sc->tr_retry;
			ifp->if_flags &= ~IFF_RUNNING;
		} else {
			/* Fatal error, leave the interface down. */
			tr_halt(sc);
			tr_down(sc);
		}
		return;
	}

	/* Initialize driver state */
	sc->tr_asb = tr_mptr(sc, ntohs(oresp->asb));
	sc->tr_arb = tr_mptr(sc, ntohs(oresp->arb));
	sc->tr_srb = ntohs(oresp->srb);
	sc->tr_txavail = ntohs(sc->tr_fpta->buffer_count);
	sc->tr_txcorr = 0;
	sc->tr_retry = 1;
	if (sc->tr_txavail != TX_NUMBUFS) {
		trprf(sc, "Short on transmit buffers, got %d (FATAL)\n",
			sc->tr_txavail);
		tr_halt(sc);
		tr_down(sc);
	}

	ifp->if_flags |= IFF_RUNNING;
	trprf(sc, "inserted in ring\n");

	if (ifp->if_flags & IFF_UP) {
		/* XXX should do an arprequest here to check for dup addrs */
		trstart(ifp);
	} else
		tr_close(sc);
}

/*
 * Decode an open error and return determination of how fatal it should
 * be (if 1 is returned the error is fatal and no further recovery should
 * be attempted).
 */
int
tr_open_err(tr_softc_t *sc, u_char ret_code, u_short open_errcode)
{
	int rc;

	rc = 1;				/* Errors are fatal by default */
	trprf(sc, "open failure");
	switch(ret_code) {
	case OPEN_INVALID:
		printf(": invalid command code");
		break;

	case OPEN_ALREADY:
		printf(": adapter already open");
		break;

	case OPEN_MISSING:
		printf(": missing paramters");
		break;

	case OPEN_CANCEL: {
		int phase = (open_errcode >> 4) & 0x7;
		int err = open_errcode & 0x0f;

		printf(" %x during %s: %s", open_errcode, phaserr[phase], 
			openerr[err]);
		switch (open_errcode) {
		case 0x11:
		case 0x26:
		case 0x27:
		case 0x32:
		case 0x35:
		case 0x36:
		case 0x37:
		case 0x42:
		case 0x45:
		case 0x46:
		case 0x47:
		case 0x55:
		case 0x56:
		case 0x57:
		case 0x59:
			/* Above errors should be retried */
			rc = 0;
			break;
		}
		break;
	}

	case OPEN_NORX:
		printf(": not enough receive buffers");
		break;

	case OPEN_BADNODE:
		printf(": bad node address");
		break;

	case OPEN_BADRXLEN:
		printf(": bad RX buffer length");
		break;

	case OPEN_BADTXLEN:
		printf(": bad TX buffer length");
		break;

	default:
		printf(": unknown: 0x%x", ret_code);
		break;
	}

	if (rc)
		printf(" (FATAL)\n");
	else
		printf("\n");
	return (rc);
}

/*
 * Called at system shutdown time, unmap shared memory and hold card in
 * reset.
 */
void
tr_shutdown(arg)
	void *arg;
{
	tr_softc_t *sc = (tr_softc_t *)arg;
	tr_mio_t *mio = sc->tr_mvaddr;
	int pio = sc->tr_iobase;

	mio->rw.rrr_even = 0;		/* Turn off shared RAM */
	outb(pio + TR_RESET, 1);	/* Hold adapter in reset */
}

/* ------------------ Transmit processing ---------------------- */

/*
 * Start output from interface queue if possible
 */
int
trstart(ifp)
	struct ifnet *ifp;
{
	int unit = ifp->if_unit;
	tr_softc_t *sc = trcd.cd_devs[ifp->if_unit];
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_fptca_t *fpta = sc->tr_fpta;
	int tot_len;
	struct mbuf *m, *mfirst;
	caddr_t mptr;
	int mlen;
	tr_fpb_t *buf;
	tr_fpb_t *fbuf;
	caddr_t bptr;
	int blen;

	if ((ifp->if_flags & IFF_RUNNING) == 0)
		return 0;

	/*
	 * Don't initiate output unless there is sufficient space
	 * for an MTU sized packet plus a free queue tail placeholder.
	 */
nextpkt:
	if (sc->tr_txavail < TX_NEEDED)
		return (0);

	/* Board is ready, see if anything waiting to go */
	IF_DEQUEUE(&ifp->if_snd, mfirst);
	if (mfirst == NULL)
		return (0);

	/* Keep track of total packet size */
	tot_len = 0;
	/* Set up for first card buffer */
	fbuf = buf = tr_gfpb(sc, fpta->fqh);
	bptr = (caddr_t)buf->data;
	blen = TX_DATA;
	sc->tr_txavail--;

	/* Set up first mbuf in chain */
	m = mfirst;
	mptr = mtod(m, caddr_t);
	mlen = m->m_len;

	for (;;) {
		int this_len = min(mlen, blen);

		bcopy(mptr, bptr, this_len);

		tot_len += this_len;
		blen -= this_len;

		if ((mlen -= this_len) == 0) {
			/* Move to next mbuf */
			m = m->m_next;
			if (m == NULL) {
				buf->buffer_len = htons(TX_BSIZE - blen);
				break;
			}
			mptr = mtod(m, caddr_t);
			mlen = m->m_len;
		} else
			mptr += this_len;

		if (blen == 0) {
			/* Move to next card buffer */
			buf->buffer_len = htons(TX_DATA);
			buf = tr_gfpb(sc, buf->next_bufr);
			bptr = (caddr_t)buf->data;
			blen = TX_DATA;
			sc->tr_txavail--;
		} else
			bptr += this_len;
	}

	/*
	 * Update frame values
	 */
	fbuf->last_bufr = htons(((caddr_t)&buf->next_bufr) - sc->tr_vaddr);
	fbuf->frame_len = htons(tot_len);
	fbuf->stn_id = DIRECT_STATION;
	fbuf->cmd = TRANSMIT_DIR_FRAME;

	/* Only the card uses this */
	fbuf->corr = sc->tr_txcorr++ & 0x7f;

	/*
	 * Send (something) to BPF
	 */
	if (ifp->if_bpf)
		bpf_mtap(ifp->if_bpf, mfirst);
	m_freem(mfirst);

	/*
	 * Update the free queue head and tell the card about it
	 */
	fpta->fqh = htons(1);		/* Avoids byte races w/8bit bus */
	fpta->fqh = buf->next_bufr;
	mio->or.isra_odd = FPT_REQ;

	ifp->if_opackets++;
	ifp->if_flags |= IFF_OACTIVE;

	goto nextpkt;
}

/*
 * Transmit complete interrupt. Process completed transmit buffers and
 * put them back on the free queue.
 */
void
tr_txint(sc)
	tr_softc_t *sc;
{
	tr_fptca_t *fpta = sc->tr_fpta;
	tr_fpb_t *fp;

	for (;;) {
		u_short cqt = fpta->cqt;

		if (cqt != fpta->cqt)
			continue;
		if (fpta->fqt == fpta->cqt) {
			struct ifnet *ifp = &sc->tr_if;

			if (sc->tr_txavail == TX_NUMBUFS)
				ifp->if_flags &= ~IFF_OACTIVE;
			trstart(ifp);
			return;
		}

		/*
		 * Free queue tail is the same as our completion queue head,
		 * it points to the buffer just before the next buffer to
		 * be processed at completion time.
		 */
		fp = tr_gfpb(sc, fpta->fqt);
		fp = tr_gfpb(sc, fp->next_bufr);
		if (fp->rc != 0)
			sc->tr_if.if_oerrors++;
		sc->tr_txavail += 
		    (ntohs(fp->frame_len) + TX_DATA - 1) / TX_DATA;
		fpta->fqt = fp->last_bufr;
	}
}

/* ------------------- Receive processing -------------------- */
 
/* 
 * Receive interrupt. Pad at the start of the packet for better alignment.
 */
void
tr_rxint(sc)
	tr_softc_t *sc;
{
	struct ifnet *ifp = &sc->tr_if;
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_rxarb_t *arb = sc->tr_arb;
	int mac_pad = sizeof(struct token_max_hdr) - arb->lan_hdr_len;
	u_short rxbufr = arb->bufr;
	int pkt_resid;
	caddr_t b_ptr;
	int b_resid;
	tr_rxb_t *rb;
	caddr_t m_ptr;
	int m_resid;
	struct mbuf *mfirst;
	struct mbuf **mlink;
	struct mbuf *m;

	pkt_resid = ntohs(arb->frame_len);

	/* Set up first card buffer */
	rb = tr_mptr(sc, ntohs(rxbufr));
	b_resid = ntohs(rb->buf_len);
	b_ptr = (caddr_t)rb->data;

	/* Set up first mbuf */
	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == 0)
		goto discard;
	m->m_pkthdr.len = pkt_resid;
	m->m_pkthdr.rcvif = ifp;
	m_resid = MHLEN - mac_pad;
	if (pkt_resid+mac_pad > MINCLSIZE) {
		MCLGET(m, M_DONTWAIT);
		if (m->m_flags & M_EXT)
			m_resid = MCLBYTES-mac_pad;
	}
	m->m_len = min(m_resid, pkt_resid);
	m_ptr = m->m_data = mtod(m, caddr_t)+mac_pad;
	mfirst = m;
	mlink = &m->m_next;

	for (;;) {
		int this_len = min(m_resid, b_resid);

		bcopy(b_ptr, m_ptr, this_len);

		if ((pkt_resid -= this_len) == 0) break;

		if ((b_resid -= this_len) == 0) {
			u_short nrb = ntohs(rb->next_buf);
#ifdef DIAGNOSTIC
			if (nrb == 0) {
				trprf(sc, "tr_rxint: next RX buffer is null\n");
				goto mdiscard;
			}
#endif
			rb = (tr_rxb_t *)tr_mptr(sc, nrb-2);
			b_resid = ntohs(rb->buf_len);
#ifdef DIAGNOSTIC
			if (b_resid > pkt_resid) {
				trprf(sc, "tr_rxint: receive buffer too big\n");
				goto mdiscard;
			}
#endif
			b_ptr = (caddr_t)rb->data;
		} else
			b_ptr += this_len;

		if ((m_resid -= this_len) == 0) {
			MGET(m, M_DONTWAIT, MT_DATA);
			if (m == NULL)
				goto mdiscard;
			*mlink = m;
			mlink = &m->m_next;
			m_resid = MHLEN;
			if (pkt_resid > MINCLSIZE) {
				MCLGET(m, M_DONTWAIT);
				if (m->m_flags & M_EXT)
					m_resid = MCLBYTES;
			}
			m->m_len = min(m_resid, pkt_resid);
			m_ptr = mtod(m, caddr_t);
		} else
			m_ptr += this_len;
#ifdef DIAGNOSTIC
		if (m_resid < 0 || b_resid < 0) {
			trprf(sc, "tr_rxint: bad residual: "
				  "m_resid=%d b_resid=%d\n", m_resid, b_resid);
			goto mdiscard;
		}
#endif
	}
		
	/*
	 * Ack the interrupt and send ASB response to free buffer at card.
	 */
	mio->and.isrp_odd = ~ARB_CMD;
	mio->or.isra_odd = ARB_FREE;
	tr_rxdone(sc, rxbufr);

	ifp->if_ipackets++;

	/* Send to BPF if open */
	if (ifp->if_bpf)
		bpf_mtap(ifp->if_bpf, mfirst);

	/* Dispose to upper layer */
	token_input(&sc->tr_if, mfirst);
	return;

mdiscard:
	m_freem(mfirst);
discard:
	/* Ack interrupt (as above) and free RX buffer */
	mio->and.isrp_odd = ~ARB_CMD;
	mio->or.isra_odd = ARB_FREE;
	tr_rxdone(sc, rxbufr);
}

/*
 * Post receive complete ASB if possible, else queue.
 */
void
tr_rxdone(tr_softc_t *sc, u_short bufr)
{
	tr_mio_t *mio = sc->tr_mvaddr;
	u_short *doneq = sc->tr_doneq;
	int dqt = sc->tr_dqt;

	/* Send now if ASB is free */
	if (sc->tr_dqlen++ == 0) {
		tr_post_rxdone(sc, bufr);
		return;
	}

#ifdef DIAGNOSTIC
	if (sc->tr_dqlen > DONE_MAXQ) {
		trprf(sc, "tr_rxdone: done queue overflow\n");
		panic("tr: done queue overflow");
	}
#endif

	/* Put in done queue */
	sc->tr_doneq[dqt] = bufr;
	sc->tr_dqt = (dqt + 1) % DONE_MAXQ;

	/*
	 * Short circuit interrupt process, the ASB might actually be free.
	 */
	if (mio->rw.isrp_odd & ASB_FREE)
		tr_asbfree(sc);
}

/*
 * Run the next ASB on the queue when the ASB comes free.
 */
void
tr_asbfree(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	u_short *tr_doneq = sc->tr_doneq;
	int dqh = sc->tr_dqh;
	u_short bufr;

	/* clear ASB free interrupt */
	mio->and.isrp_odd = ~ASB_FREE;
	if (--sc->tr_dqlen == 0)
		return;

#ifdef DIAGNOSTIC
	if (sc->tr_dqlen < 0) {
		trprf(sc, "done Q length invalid: %d\n", sc->tr_dqlen);
		panic("tr: done Q corrupt");
	}
#endif

	bufr = tr_doneq[dqh];
	sc->tr_dqh = (dqh + 1) % DONE_MAXQ;
	tr_post_rxdone(sc, bufr);
}

/*
 * Post receive complete message
 */
void
tr_post_rxdone(tr_softc_t *sc, u_short bufr)
{
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_asb_t *asb = sc->tr_asb;

	/* Fill in the ASB */
	asb->cmd = RECEIVED_DATA;
	asb->rc = 0;
	asb->stn_id = DIRECT_STATION;
	asb->rbuf = bufr;

	/* Tell the adapter */
	mio->or.isra_odd =  ASB_RESP | ASB_FREE_REQ;
}

/*------------------- Main interrupt handler ----------------------*/

/* 
 * Handle interrupts
 */
int
trint(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	int pio = sc->tr_iobase;
	u_char isrp_odd;
	int valid;

	valid = 0;
	for (;;) {
		/*
		 * Look for meaty interrupts, if none left handle the
		 * wimpy ones.
		 */
		isrp_odd = mio->rw.isrp_odd;
		if (isrp_odd == 0) {
			if (valid == 0) {
				u_char isrp_even = mio->rw.isrp_even;

				if (isrp_even & ACCESS_INT)
					mio->and.isrp_even = ~ACCESS_INT;
				if (isrp_even & ERROR_INT)
					mio->and.isrp_even = ~ERROR_INT;
			}
			break; /* all done handling interrupts */
		}
		valid++;

		if (isrp_odd & ASB_FREE) {
			tr_asbfree(sc);
			continue;
		}

		/* No commands are used that return an SSB */
		if (isrp_odd & SSB_RESP) {
			trprf(sc, "Unexpected SSB complete interrupt\n");
			mio->and.isrp_odd = ~SSB_RESP;
			continue;
		}


		if (isrp_odd & SRB_RESP) {
			u_char srb_resp = *(u_char *)tr_mptr(sc,sc->tr_srb);
			switch (srb_resp) {
			case DIR_OPEN_ADAPTER:
				tr_open_done(sc);
				break;

			default:
				trprf(sc, "unexpected SRB response to cmd %d\n",
					srb_resp);
				mio->and.isrp_odd = ~SRB_RESP;
				break;
			}
			continue;
		}

		if (isrp_odd & ARB_CMD) {
			u_char arb_cmd = *(u_char *)sc->tr_arb;

			switch (arb_cmd) {
			case RECEIVED_DATA:
				tr_rxint(sc);
				break;

			case RING_STATUS_CHANGE:
				tr_ringstat(sc);
				break;

			default:
				trprf(sc, "Unexpected ARB cmd: %d\n", arb_cmd);
				mio->and.isrp_odd = ~ARB_CMD;
				mio->or.isra_odd = ARB_FREE;
				break;
			}
			continue;
		}

		if (isrp_odd & ADAPTER_CHECK) {
			trprf(sc, "Adapter check\n");
			mio->and.isrp_odd = ~ADAPTER_CHECK;
			continue;
		}

		if (isrp_odd & FPT_COMPL) {
			mio->and.isrp_odd = ~FPT_COMPL;
			tr_txint(sc);
			continue;
		}

		trprf(sc, "unexpected interrupt, isrp_odd=0x%x\n", isrp_odd);
		break;
	}

out:
	outb(pio + TR_IEN, 0x1);		/* Re-enable interrupts */
	return (1);
}

/*------------------- Misc/utility routines ------------------ */

/*
 * Watchdog timer expiration, guards against open failures and is used
 * to reopen after being booted from the ring.
 */
int
trwatchdog(unit)
	int unit;
{
	struct tr_softc *sc = trcd.cd_devs[unit];
	struct ifnet *ifp = &sc->tr_if;

	if (sc->tr_flags & TR_OPENING)
		trprf(sc,"open timeout\n");

	tr_halt(sc);

	if ((ifp->if_flags & IFF_UP) == 0)
		return (0);

	/*
	 * Start another open, this may reschedule the watchdog
	 */
	trprf(sc, "retrying ring insertion\n");
	trinit(ifp->if_unit);
	return (0);
}

/*
 * Close the adapter (quite violently)
 */
void
tr_close(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_cfpr_t *srb = tr_mptr(sc, sc->tr_srb);

	if (sc->tr_flags & TR_OPENING)
		return;

	/* Stop dead in its tracks, great when the board hangs */
	tr_halt(sc);

	if (sc->tr_if.if_flags & IFF_UP) {
		trprf(sc, "attempting reinsertion\n");
		trinit(sc->tr_if.if_unit);
	}
}

/* 
 * Report ring status changes and initiate error recovery if needed.
 */
void
tr_ringstat(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	tr_ringstat_t *rstat = (tr_ringstat_t *)sc->tr_arb;
	u_short ring_stat = ntohs(rstat->ring_stat);

	if ((ring_stat & SIGNAL_LOSS) != 0)
		trprf(sc, "Can't detect any signal\n");

	if ((ring_stat & HARD_ERROR) != 0)
		trprf(sc, "Beacon frames are being received or transmitted\n");

	if ((ring_stat & TRANSMIT_BEACON) != 0)
		trprf(sc, "Adapter is transmitting beacon frames\n");

	if ((ring_stat & LOBE_WIRE_FAULT) != 0)
		trprf(sc, "Open or short circuit detected\n");

	if ((ring_stat & AUTO_REMOVAL) != 0)
		trprf(sc, "Adapter hardware error detected\n");

	if ((ring_stat & REMOVE_RECEIVED) != 0)
		trprf(sc, "Remove MAC frame received\n");

	if ((ring_stat & SINGLE_STATION) != 0)
		trprf(sc, "No other sites detected\n");

	if ((ring_stat & RING_RECOVERY) != 0)
		trprf(sc, "Adapter is receiving or transmitting contention "
			"MAC frames\n");

	/* Clear the interrupt and release the ARB */
	mio->and.isrp_odd = ~ARB_CMD;
	mio->or.isra_odd = ARB_FREE;

	/*
	 * If the adapter closed itself close again to make sure (this
	 * will also initiate open retries if the interface is still up).
	 */
	if (ring_stat & (LOBE_WIRE_FAULT | AUTO_REMOVAL)) {
		trprf(sc, "removed from ring\n");
		tr_close(sc);
	} else if (ring_stat & REMOVE_RECEIVED) {
		trprf(sc, "removed from ring by network manager\n");
		tr_halt(sc);
		tr_down(sc);
	}
}

/*
 * Ioctl handler
 */
int
trioctl(ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *)data;
	tr_softc_t *sc = trcd.cd_devs[ifp->if_unit];
	int rc;
	int s = splimp();

	rc = 0;
	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			trinit(ifp->if_unit);
			arp_ifinit((struct arpcom *)ifp, ifa);
			break;
#endif INET
		default:
			trinit(ifp->if_unit);
			break;
		}
		break;

	case SIOCSIFFLAGS:
		if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) == IFF_RUNNING) {
			tr_close(sc);
		} else if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) == IFF_UP) {
			trinit(ifp->if_unit);
		}
		sc->tr_retry = 1;
		break;
	default:
		rc = EINVAL;
	}
	splx(s);
	return (rc);
}

/*
 * Halt adapter
 */
void
tr_halt(sc)
	tr_softc_t *sc;
{
	tr_mio_t *mio = sc->tr_mvaddr;
	int pio = sc->tr_iobase;

	/* Disable shared RAM */
	mio->rw.rrr_even = (u_int)(sc->tr_maddr) >> 12;

	/* Hold reset */
	outb(pio + TR_RESET, 1);

	sc->tr_flags &= ~TR_OPENING;
	sc->tr_if.if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);
	sc->tr_if.if_timer = 0;
	if_qflush(&sc->tr_if.if_snd);
}

/*
 * Down interface on really fatal errors
 */
void
tr_down(sc)
	tr_softc_t *sc;
{

	sc->tr_if.if_flags &= ~IFF_UP;
}

/*
 * Print message with tr header
 */
void
#if __STDC__
trprf(tr_softc_t *sc, char *fmt, ...)
#else
trprf(sc, fmt, va_alist)
	tr_softc_t *sc;
	char *fmt;
	va_dcl;
#endif
{
	va_list ap;

#if __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	printf("%s%d: %r", sc->tr_if.if_name, sc->tr_if.if_unit, fmt, ap);
	va_end(ap);
}
