/*-
 * Copyright (c) 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * Copyright (c) 1995 SDL Communications, Inc.  All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	This driver source code is adapted from RISCom/h2 driver
 *	'if_rh.c' v1.6 Copyright BSDI.
 *
 *	BSDI $Id: if_n2.c,v 2.10 1995/12/13 00:03:43 cp Exp $
 */


/*
 * SDL Communications RISCom/N2 dual sync serial port driver
 *		    Notes 
 *   1. In configuring the kernel, a memory size of 32k (32768)
 *	should be used for boards equipped with 64k DP-RAM.
 *	The driver divides the RAM into 2 halves,
 *	half for port 0 and half for port 1.
 *	An example configuration entry:
 *   device  ntwo0 at isa? port 0x300 iomem 0xe0000 iosiz 32768 flags 0
 *
 *	The flags field is broken down such the the high
 *	16 bits is for channel 1 and the low 16 bits is for channel
 *	0. The high order 4 bits of each channel are used to
 *	specify the use of an internal clock. The high order bit of
 *	these four bits indicates  that the internal clock is to
 *	be supplied to the cable. The next 3 bits are the rate out of a table.
 *	This is only needed for direct connect, never with a CSU/DSU. 
 *	The low order 3 bits of each channel are used as flags.
 *	Bit 0 of the half word (the low order bit) causes DCD to be ignored.
 *	Bit 1 makes the driver silent about errors.
 *	Bit 2 cause the driver to gate the receive clock to the
 *		transmit clock. This is only needed for X.21 interfaces
 *		which don't have crossover cables.
 *	
 */

/*
 * TODO:
 *   alternative HDLC checksum
 */

#undef	RHDEBUG
#undef	RHDEBUG_X


#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/proc.h>

#if INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>		/* XXX for slcompress/if_p2p.h */
#endif

#include <net/if.h>
#include <net/netisr.h>
#include <net/if_types.h>
#include <net/if_p2p.h>
#include <net/if_c_hdlc.h>

#include <i386/isa/icu.h>
#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>

#include "if_n2reg.h"
#include "if_n2ioctl.h"





#define LOBYTE(y)    (y & 0xff)		      /* bits 0-7  of long value */
#define HIBYTE(y)    ((y & 0xff00) >> 8)      /* bits 8-15 of long value */
#define BIBYTE(y)    ((y & 0xff0000) >> 16)   /* bits 16-23 of long value */
#define LOWORD(y)    (y & 0xffff)	      /* bits 0-15 of long value */
#define	 MAXDESC  16	 /* max number of descriptors */
#define Kb  *1024

static int clktab[] =
    { 56000, 9600, 19200, 38400, 64000, 500000, 1000000, 1500000};


struct n2_softc {
	struct	device sc_dev;		/* base device */
#define sc_flags sc_dev.dv_flags

	struct	isadev sc_id;		/* ISA device */
	struct	intrhand sc_ih;		/* interrupt vectoring */
	struct	p2pcom sc_p2pcom;	/* point-to-point common stuff */
#define sc_if	sc_p2pcom.p2p_if	/* network-visible interface */ 
#define sc_unit sc_if.if_unit

	struct n2_softc *sc_nextchan;	/* next channel's n2_softc */
	struct n2_softc *sc_firstchan;	/* first channel's n2_softc */

	u_short sc_addr;		/* base i/o port */
	u_short sc_msci;		/* channel-adjusted MSCI address */
	u_short sc_dma;			/* channel-adjusted DMAC address */
	int sc_rate;			/* baud rate on channel */
	n2_type_t sc_type;		/* type of board */

	struct	hd_cbd *sc_txdesp;	/* TX buffer descriptors */
	struct	hd_cbd *sc_rxdesp;	/* RX buffer descriptor */

	u_char	sc_chan;		/* channel number */
	u_char	sc_tdin;		/* current TX descrip. */
	u_char	sc_tdout;		/* current TX descrip. */
	u_char	sc_tdend;		/* last TX  descrip. */
	u_char	sc_tdlast;
	u_char	sc_rdin;		/* current RX descrip. */
	u_char	sc_rdout;		/* current RX descrip. */
	u_char	sc_rdend;		/* last RX  descrip. */
	u_char	sc_dcd;			/* buffered state of dcd */
	u_char	sc_ignoredcd;	
	u_char	sc_quiet;		/* silent about rx data errors */
	u_char	sc_srcclk;		/* we drive the clock output lines */
	u_char	sc_rx2tx;		/* transmitter uses receive clock */
	u_char	sc_up;			/* configured up */
	u_char	sc_timeron;		/* timer is running */
	u_char	sc_watchdog;		/* watch dog timeout */
	u_char	sc_pcr_open;		/* pcr value for open window */
	u_char	sc_pcr_closed;		/* pcr value for close window */

	int	sc_memsize;		/* the memory window size */
	caddr_t sc_vaddr;		/* virtual memory address */
	u_char	sc_psr;			/* Page Scan Register */
	u_int	sc_memoff;		/* memory offset from DP-RAM base */
	struct	atshutdown sc_ats;	/* shutdown routine */
	int	sc_ihc;			/* hang in interrupt detection */

};

int	n2probe __P((struct device *, struct cfdata *, void *));
void	n2forceintr __P((void *));
void	n2attach __P((struct device *, struct device *, void *));
int	n2ioctl __P((struct ifnet *, int, caddr_t));
int	n2intr __P((struct n2_softc *));
int	n2start __P((struct ifnet *));
int	n2mdmctl __P((struct p2pcom *, int));
void	n2shutdown __P((void *));
void	n2watch __P((void *));

static n2init __P((struct n2_softc *));
static void n2eanble __P((struct n2_softc *));
static void n2disable __P((struct n2_softc *));
static void n2read __P((struct n2_softc *, char *, int));
static void n2rx_intr __P((struct n2_softc *));
static void n2tx_intr __P((struct n2_softc *));
static void n2start_tx __P((struct n2_softc *));
static u_char n2getpsr __P((int));
static int n2testmem __P((struct isa_attach_args *));

extern int hz;

struct cfdriver ntwocd =
	{ NULL, "ntwo", n2probe, n2attach, DV_IFNET, sizeof(struct n2_softc) };


n2sel_page(sc)
	struct n2_softc *sc;
{
	static struct n2_softc *n2mem_sc;

	if (n2mem_sc == sc)
		return;

	if (n2mem_sc != NULL) {
		outb(n2mem_sc->sc_addr + N2_PCR, sc->sc_pcr_closed);
	}
	n2mem_sc = sc;
	outb(sc->sc_addr + N2_PCR, sc->sc_pcr_open);
	outb(sc->sc_addr + N2_PSR, sc->sc_psr);
}


	
/*
 * Probe routine
 */
n2probe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	int i, iobase;
	u_char tval, actval;
	u_char bar;

	iobase = ia->ia_iobase;

	/*
	 * Check configuration parameters
	 */
	if (!N2_IOBASEVALID(iobase)) {
		printf("** ntwo%d: illegal base address %x\n",cf->cf_unit,iobase);
		return (0);
	}
	if (ia->ia_msize < 16 Kb) {
		printf("minimum window size is 16Kb");
		return (0);
	}
	if (ia->ia_msize < 32 Kb)
		ia->ia_msize = 16 Kb;
	else if (ia->ia_msize < 64 Kb)
		ia->ia_msize = 32 Kb;
	else if (ia->ia_msize < 128 Kb)
		ia->ia_msize = 64 Kb;
	else if (ia->ia_msize > 128 Kb) {
		aprint_normal("size adjusted to maximum window size of 128 Kb");  
		ia->ia_msize = 128 Kb;
	}
	/*
	 * Check if card is present
	 */
	tval = 0xaa;
	outb(iobase, tval);
	if ( (actval=inb(iobase)) != tval)  {
		return (0);
	}
	/* Reset SCA */
	outb(iobase+N2_PCR, 0);	   /* reset and close window */ 
	outb(iobase+N2_PSR, 0);	   /* disable DMA */

	if (ia->ia_irq != IRQUNK) {
		i = ffs(ia->ia_irq) - 1;
		if (!N2_IRQVALID(i)) {
			printf("ntwo%d: illegal IRQ %d\n", cf->cf_unit, i);
			return (0);
		}
	}
	/*
	 * Automagic irq detection
	 */
	if (ia->ia_irq == IRQUNK)
		ia->ia_irq = isa_discoverintr(n2forceintr, (void *) ia);

	/* Reset SCA */
	outb(iobase, 0);
	if (ffs(ia->ia_irq) - 1 == 0)  
		return (0);

	bar = (u_int)ia->ia_maddr >> 12;
	outb(iobase + N2_BAR, bar);  /* write to memory base address reg */
	if ((tval = inb(iobase+N2_BAR)) != bar)	 {
		aprint_debug("ntwo%d: BAR wrote %x , read %x \n",
		    cf->cf_unit, bar, tval);
		return (0);
	}

	outb(iobase + N2_PCR, N2_BUS16 | N2_ENWIN | N2_ENVPM); /* enable mem */
	ia->ia_aux = NULL;
	if (!n2testmem(ia)) {
		aprint_verbose(
		    "ntwo%d: failed in 16 bit mode, trying 8 bit mode\n",
		    cf->cf_unit);
		outb(iobase + N2_PCR, N2_BUS8 | N2_ENWIN | N2_ENVPM); 
		if (!n2testmem(ia)) {
			aprint_verbose("ntwo%d: giving up\n",
				cf->cf_unit);
			outb(iobase + N2_PCR, 0); 
			return (0);
		}
		aprint_verbose("ntwo%d: 8 bit mode successful\n", cf->cf_unit);
		ia->ia_aux++;
	}
	outb(iobase + N2_PCR, 0); 
	ia->ia_iosize = N2_NPORT;
	return (1);
}

static int
n2testmem(ia)
	struct isa_attach_args *ia;
{
	u_short *ptr = (ushort *)ISA_HOLE_VADDR(ia->ia_maddr);
	int size = ia->ia_msize;
	int iobase = ia->ia_iobase;
	int ssize;
	int i,j;
	u_char tval;

	ssize = ia->ia_msize/2;
	tval = n2getpsr(ia->ia_msize);

	outb(iobase + N2_PSR, tval);   /* write to Page Scan register */
	for (i = 0; i < ssize; i++)
		ptr[i] = 0;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != 0) {
			aprint_debug("bank: 0, addr: %x wrote: 0, read %x\n",
			    i, j);
			return (0);
		}

	outb(iobase + N2_PSR, tval | 1);
	for (i = 0; i < ssize; i++)
		ptr[i] = 0;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != 0) {
			aprint_debug("bank: 1, addr: %x wrote: 0, read %x\n",
			    i, j);
			return (0);
		}

	outb(iobase + N2_PSR, tval);   /* write to Page Scan register */
	for (i = 0; i < ssize; i++)
		ptr[i] = 0xffff;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != 0xffff) {
			aprint_debug(
			    "bank: 0, addr: %x wrote: 0xffff, read %x\n",
			    i, j);
			return (0);
		}

	outb(iobase + N2_PSR, tval | 1);
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != 0) {
			aprint_debug("wrote bank 0, bank one changed\n");
			return (0);
		}

	for (i = 0; i < ssize; i++)
		ptr[i] = 0xffff;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != 0xffff) {
			aprint_debug(
			    "bank: 1, addr: %x wrote: 0xffff, read %x\n",
			    i, j);
			return (0);
		}

	outb(iobase + N2_PSR, tval);   /* write to Page Scan register */
	for (i = 0; i < ssize; i++)
		ptr[i] = i & 0xffff;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != i & 0xffff) {
			aprint_debug(
			    "bank: 0, addr: %x wrote: %x, read %x\n",
			    i, i & 0xffff, j);
			return (0);
		}

	outb(iobase + N2_PSR, tval | 1);   /* write to Page Scan register */
	for (i = 0; i < ssize; i++)
		ptr[i] = i & 0xffff;
	for (i = 0; i < ssize; i++)
		if ((j = ptr[i]) != i & 0xffff) {
			aprint_debug("bank: 1, addr: %x wrote: %x, read %x\n",
			    i, i & 0xffff, j);
			return (0);
		}

	return (1);
}

static void
n2forceintr(aux)
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;

	/* Enable interrupt on tx buffer empty (it sure is) */
	outb(ia->ia_iobase, N2_RUN );
	routb(ia->ia_iobase, CMD, CMD_TXENB);
	routb(ia->ia_iobase, IE0, ST0_TXRDY | ST0_TXINT);
	routb(ia->ia_iobase, IER0, ISR0_TXRDY0 | ISR0_TXINT0);
}

static u_char
n2getpsr(int memsize)
{
	u_char psr;
	switch (memsize) {
	case 16 Kb:
		psr = N2_WS16K;
		break;
	case 32 Kb:
		psr = N2_WS32K;
		break;
	case 64 Kb:
		psr = N2_WS64K;
		break;
	case 128 Kb:
		psr = N2_WS128K;
		break;
	}
	return (psr);
}

static n2_getflags(sc, flags)
register struct n2_softc *sc;
int flags;
{

	sc->sc_rate = clktab[N2RATE(flags)];
	sc->sc_ignoredcd = N2NODCD(flags);
	sc->sc_quiet = N2QUIET(flags);
	sc->sc_srcclk = N2SRCCLK(flags);
	sc->sc_rx2tx = N2RX2TX(flags);
}

/*
 * Interface exists: make available by filling in network interface
 * records.  System will initialize the interface when it is ready
 * to accept packets.
 */
void
n2attach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct n2_softc *sc = (struct n2_softc *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	struct ifnet *ifp;
	struct n2_softc *csc;
	int i,iobase,memsize,mem16;
	ushort rval,wval,*waddr;
	u_char ctl,bar,val;

	iobase = ia->ia_iobase;
	memsize = ia->ia_msize;

	sc->sc_pcr_closed = N2_RUN | N2_ENVPM;
	if (ia->ia_aux == 0)
		sc->sc_pcr_closed |= N2_BUS16;
	else
		sc->sc_pcr_closed |= N2_BUS8;

	sc->sc_pcr_open = sc->sc_pcr_closed | N2_ENWIN;

	outb(iobase + N2_PCR, sc->sc_pcr_closed);

	DELAY(5);
	routb(iobase, IER0, 0);		/* MSCI disable interrupt */
	routb(iobase, DMER, 0);		/* DMA master enable - DISable */
 
	/*
	 * Load the wait controller registers
	 */
	routb(iobase, WCRL, 0);
	routb(iobase, WCRM, 0);
	routb(iobase, WCRH, 0);
	routb(iobase, PCR,  0);	  /* SCA Priority Control Reg */

	sc->sc_psr = n2getpsr(ia->ia_msize) | N2_DMAEN;
	routb(iobase, DMER, DMER_DME);	/* DMA master enable */

	/*
	 * Initialize interface data
	 */
	for (i = 0, csc = sc; i < N2_NCHANS; i++) {
		if (i) {     /* Allocate memory for port 1 and higher */
			csc = (struct n2_softc *) malloc(
				sizeof(struct n2_softc), M_DEVBUF, M_NOWAIT);
			if (csc == 0) {
				printf("n2%d: no memory for the channel 1 data",
					sc->sc_dev.dv_unit);
				return;
			}
			bzero((caddr_t) csc, sizeof(struct n2_softc));
			sc->sc_nextchan = csc;
		}
		csc->sc_nextchan = 0;
		csc->sc_firstchan = sc; 
		csc->sc_addr = ia->ia_iobase;
		csc->sc_msci = ia->ia_iobase + N2_SCA(MCHAN * i);
		csc->sc_dma = ia->ia_iobase + N2_SCA(CHAN * i);
		csc->sc_memsize = memsize;
		csc->sc_vaddr = ISA_HOLE_VADDR(ia->ia_maddr);
		if (i != 0) {
			aprint_naive(": RISCom/N2");
			n2_getflags(csc, sc->sc_flags >> 16);
			aprint_normal(" board type = ");
			switch (inb(iobase + N2_IDR)) {
			case N2_CSUIDR:
				csc->sc_type = N2CSU;
				aprint_normal("N2CSU");
				break;
			case N2_DDSIDR:
				csc->sc_type = N2DDS;
				aprint_normal("N2DDS");
				break;
			default:
				csc->sc_type = N2;
				aprint_normal("N2");
				break;
			}
			printf("\n");
		} else {
			n2_getflags(csc, sc->sc_flags);
			csc->sc_type = N2;
		}
		csc->sc_chan = i;
		csc->sc_psr = sc->sc_psr | i;
		csc->sc_pcr_open = sc->sc_pcr_open;
		csc->sc_pcr_closed = sc->sc_pcr_closed;

		n2init(csc);

		ifp = &csc->sc_if;
		ifp->if_unit = (sc->sc_dev.dv_unit * N2_NCHANS) + i;
		ifp->if_name = "ntwo";
		ifp->if_mtu = HDLCMTU;
		ifp->if_flags = IFF_POINTOPOINT | IFF_MULTICAST;
		ifp->if_type = IFT_NONE;
		ifp->if_ioctl = n2ioctl;
		ifp->if_output = 0;		/* crash and burn on access */
		ifp->if_start = n2start;
		csc->sc_p2pcom.p2p_mdmctl = n2mdmctl;
		if_attach(ifp);
		p2p_attach(&csc->sc_p2pcom);
	}

	/* Register the interrupt handler and isa device */
	isa_establish(&sc->sc_id, &sc->sc_dev);
	sc->sc_ih.ih_fun = n2intr;
	sc->sc_ih.ih_arg = (void *)sc;
	intr_establish(ia->ia_irq, &sc->sc_ih, DV_NET);

	sc->sc_ats.func = n2shutdown;
	sc->sc_ats.arg = (void *)sc;
	atshutdown(&sc->sc_ats,ATSH_ADD);
}

n2ioctl (ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct n2_softc *sc0 = ntwocd.cd_devs[ifp->if_unit / N2_NCHANS];
	struct n2_softc *sc = (ifp->if_unit % N2_NCHANS)?
					sc0->sc_nextchan : sc0;
	struct proc *p = curproc;
	int error;

	switch (cmd) {
	case N2IOCGTYPE:
		((struct n2ioc_gtype*)data)->gt_type = sc->sc_type;
		return (0);
	case N2IOCSBYTE:
	    {
		n2ioc_io_t *io = (n2ioc_io_t*) data;

		if (error = suser(p->p_ucred, &p->p_acflag))
			return (error);
		outb(sc->sc_addr + io->io_offset, io->io_data);
		return (0);
	    }

	case N2IOCGBYTE:
	    {
		n2ioc_io_t *io = (n2ioc_io_t*) data;

		if (error = suser(p->p_ucred, &p->p_acflag))
			return (error);
		io->io_data = inb(sc->sc_addr + io->io_offset);
		return (0);
	    }

	/*
	 * Set cus data polarity. i_int being set means invert;
	 * This code counts on the fact that the csu channel is
	 * always the second channel on the board
	 */
	case N2IOCSPOLARITY:
	    {
		n2ioc_int_t *i = (n2ioc_int_t*) data;
		int s;

		if (error = suser(p->p_ucred, &p->p_acflag))
			return (error);

		if (i->i_int){
			sc->sc_psr |= N2_CSU_INVERT;
			sc0->sc_psr |= N2_CSU_INVERT;
		} else {
			sc->sc_psr &= ~N2_CSU_INVERT;
			sc0->sc_psr &= ~N2_CSU_INVERT;
		}
		s = spltty();
		n2sel_page(sc0);
		n2sel_page(sc);
		splx(s);
		return (0);
	    }
	case N2IOCSFLAGS:
	    {
		n2ioc_int_t *i = (n2ioc_int_t*) data;

		if (error = suser(p->p_ucred, &p->p_acflag))
			return (error);
		n2_getflags(sc, i->i_int);
		return (0);
	    }
	default:
		return (p2p_ioctl(ifp, cmd, data));
	}
}

void n2shutdown(arg)
void *arg;
{
	struct n2_softc *sc = (struct n2_softc *) arg;

	routb(sc->sc_dma, DIR, 0);	/* disable interrupts */
	routb(sc->sc_dma, DIR+TX, 0);
	outb(sc->sc_addr + N2_PCR, 0);	/* close window */
}

/*
 * Initialize the channel
 */
static n2init(sc)
register struct n2_softc *sc;
{
	int base = sc->sc_addr;
	int dma = sc->sc_dma;
	struct hd_cbd *desp;
	int s,i,chan,n;
	u_int bufoff, despoff, despoff0; 
	int bufcnt;
	int tbufcnt;
	int rbufcnt;

	chan = sc->sc_chan;

	/* Disable DMA interrupts */
	routb(dma, DIR, 0);
	routb(dma, DIR+TX, 0);
	routb(sc->sc_msci, IE1, 0); 

	routb(sc->sc_msci, IE0, 0);

	s = splimp();

	n2sel_page(sc);	 /* select page */
	bzero(sc->sc_vaddr, sc->sc_memsize);
	/*
	 * calculate offset to this channels memory as viewed from
	 * the onboard processor. This is because there the on board
	 * processor has a linear address space while the host processor
	 * is viewing the memory through one of two windows.
	 */
	sc->sc_memoff = (chan == 0) ? 0 : sc->sc_memsize;

	/*
	 * Calculate the memory layout parameters
	 * Create one to one relationship between slots in
	 * ring buffer and actually buffer memory
	 * transmit buffer count has to be limited because interrupt
	 * counter is used for transmit only.
	 */
	bufcnt = sc->sc_memsize / (HDLCMAX + sizeof (struct hd_cbd));
	tbufcnt = bufcnt / 3;		/* 1/3 transmit */
	if (tbufcnt > 15)
		tbufcnt = 15;
	rbufcnt = bufcnt - tbufcnt;	/* 2/3 receive */
	sc->sc_rdend = rbufcnt - 1;
	sc->sc_tdend = tbufcnt - 1;

	sc->sc_rdin = sc->sc_rdout = 0;

	desp = (struct hd_cbd *)  sc->sc_vaddr;
	sc->sc_rxdesp = desp;		/* for host cpu */

	despoff0 = despoff = sc->sc_memoff;
	bufoff = sc->sc_memoff + sizeof(struct hd_cbd) * bufcnt;
	for (i = 1; i <= rbufcnt; i++, desp++) {
		desp->cbd_len = 0;
		desp->cbd_bp1 = BIBYTE(bufoff);
		desp->cbd_bp0 = LOWORD(bufoff);
		bufoff += HDLCMAX;	 /* point to next buffer */
		desp->cbd_stat = 0;	   
		despoff +=  sizeof(struct hd_cbd);  
		desp->cbd_next = (i == rbufcnt) ? despoff0 : despoff;
	} 
	despoff -=  sizeof(struct hd_cbd);  
	routb(dma, EDAL, despoff);
	routb(dma, EDAH, despoff >> 8);
	despoff +=  sizeof(struct hd_cbd);  
	routb(dma, BFLL, LOBYTE(HDLCMAX));
	routb(dma, BFLH, HIBYTE(HDLCMAX));
	routb(dma, CDAL, LOBYTE(despoff0));
	routb(dma, CDAH, HIBYTE(despoff0));
	routb(dma, CPB,	 BIBYTE(despoff0));

	sc->sc_tdin = sc->sc_tdout = sc->sc_tdlast = 0;
	sc->sc_txdesp = desp;		
	despoff0 = despoff;

	for (i = 1; i <= tbufcnt; i++, desp++)	 {
		desp->cbd_len = 0;
		desp->cbd_bp1 = BIBYTE(bufoff);
		desp->cbd_bp0 = LOWORD(bufoff);
		bufoff += HDLCMAX;	 /* point to next buffer */
		desp->cbd_stat = ST2_EOT;      
		despoff += sizeof(struct hd_cbd); 
		desp->cbd_next = (i == tbufcnt) ? despoff0 : despoff;
	} 

	routb(dma, EDAL+TX, 0xf0);		/* an unreachable address */
	routb(dma, EDAH+TX, 0xff);
	routb(dma, CDAL+TX, LOBYTE(despoff0));
	routb(dma, CDAH+TX, HIBYTE(despoff0));
	routb(dma, CPB+TX,  BIBYTE(despoff0));

	splx(s);
	return (1);
}

/*
 * Set transmission parameters and enable the channel
 */
static void
n2enable(sc)
	register struct n2_softc *sc;
{
	register msci = sc->sc_msci;
	register dma  = sc->sc_dma;
	int br, div, clkmode, mcr,ier1,iobase;
	int chan =  sc->sc_unit % N2_NCHANS;

	iobase = sc->sc_addr;

	routb(msci, CMD, CMD_CHRST);

	/* Reset the channel */

	/* Clear DMA channels */
	routb(dma, DCR, DCR_ABORT);	/* software abort - to initial state */
	routb(dma, DCR+TX, DCR_ABORT);	/* software abort - to initial state */
	routb(dma, DSR, 0);		/* DMA disable - to halt state */
	routb(dma, DSR+TX, 0);		/* DMA disable - to halt state */

	n2init(sc);

	/* Set DMA mode to chain mode */
	routb(dma, DMR, DMR_TMOD | DMR_NF ); /* set chain mode */
	routb(dma, DMR+TX, DMR_TMOD | DMR_NF | DMR_CNTE); /* set chain mode */
	routb(dma, DCR, DCR_FCTCLR);		/* clear frame counter */
	routb(dma, DCR+TX, DCR_FCTCLR);		/* clear frame counter */

	/* HDLC, standard CRC */
	routb(msci, MD0, MD0_HDLC | MD0_CRCENB | MD0_CRCINIT | MD0_CRCCCITT);

	/* No address field checks */
	routb(msci, MD1, MD1_NOADR);

	/* NRZ, full duplex */
	routb(msci, MD2, MD2_NRZ | MD2_DUPLEX);

	/* Idle fill pattern -- flag character */
	routb(msci, IDL, 0x7e);

	/* Reset RX to validate new modes */
	routb(msci, CMD, CMD_RXRST);

	/* Set FIFO depths */
	routb(msci, TRC0, 16);	  /* lowat */

	/* Load the baud rate generator registers */
	/* FIXME
	 * The following stuff is not so great
	 * the div is really used as the reload
	 * counter and br is the divisor. The
	 * div &01 isn't really what should be
	 * there.
	 */
	div = RN2_OSC_FREQ / sc->sc_rate;
	if (div == 0)
		div = 1;
	br = 0;

	/*
	 * avoid br=0 -- it yields duty ratio != 0.5
	 */
	if (div > 10 || !(div & 01)) {
		do {
			br++;
			div /= 2;
		} while (div > 127);
	}
	routb(msci, TMC, div);

	mcr = inb(sc->sc_addr+N2_MCR);

	if (sc->sc_srcclk)  {
		routb(msci, RXS, RXS_BRG | br);
		routb(msci, TXS, TXS_BRG | br);
		if (chan)  mcr |= N2_CLKOUT1;
		else	   mcr |= N2_CLKOUT0;
	} else	{
		routb(msci, RXS, RXS_LINE);
		if (sc->sc_rx2tx)
			routb(msci, TXS, TXS_RXC);
		else
			routb(msci, TXS, TXS_LINE);
		if (chan)  mcr &= ~N2_CLKOUT1;
		else	   mcr &= ~N2_CLKOUT0;
	}

	/*  enable interrupts */
	ier1 = rinb(iobase,IER1);
	if (chan)  {
		ier1 |= ISR1_DMIA1R | ISR1_DMIB1R | ISR1_DMIA1T | ISR1_DMIB1T ;
		mcr = (mcr & ~N2_DTR1) | N2_TE1;	/* set DTR to ON */
	}
	else   {
		ier1 |= ISR1_DMIA0R | ISR1_DMIB0R | ISR1_DMIA0T | ISR1_DMIB0T ;
		mcr = (mcr & ~N2_DTR0) | N2_TE0;	    /* set DTR to ON */
	}

	/*  enable interrupts  */
	routb(iobase, IER1, ier1);
	routb(dma, DIR, DSR_EOT | DSR_COF | DSR_BOF | DSR_EOM);
	routb(dma, DIR+TX, DSR_EOT | DSR_COF | DSR_BOF | DSR_EOM);

	/*  set	 DTR & RTS to ON  */
	outb(sc->sc_addr+N2_MCR, mcr);

	/* Enable SCA transmitter and receiver */
	routb(msci, CMD, CMD_TXENB);
	routb(msci, CMD, CMD_RXENB);


	/* Raise RTS and start transmission of the idle pattern */
	routb(msci, CTL, CTL_IDLC | CTL_UNDRC);

	/* Start RX DMA */
	routb(dma, DSR, DSR_DE | DSR_EOT);

	/* Kickstart output */
	sc->sc_if.if_flags &= ~IFF_OACTIVE;
	sc->sc_watchdog = 0;
	(void) n2start(&sc->sc_if);
}

/*
 *   See if anything to transmit
 */
int
n2start(ifp)
struct ifnet *ifp;
{
	struct n2_softc *sc0 = ntwocd.cd_devs[ifp->if_unit / N2_NCHANS];
	struct n2_softc *sc = (ifp->if_unit % N2_NCHANS)?
					sc0->sc_nextchan : sc0;

	struct mbuf *m,*m0;
	struct hd_cbd *desp;
	struct ifqueue *ifq;
	caddr_t bp,buffer;
	int i,len,newtxin;

	sc0->sc_ihc = 0;

	/*
	 * Check if we have an available outgoing frame header 
	 * Should always be true if we don't have dcd
	 */
	newtxin = sc->sc_tdin + 1;
	if (newtxin > sc->sc_tdend)
		newtxin = 0;
	if (newtxin == sc->sc_tdout) {
		return (0);  /* TX Q full */
	}

	for (;;) {
		ifq = &sc->sc_p2pcom.p2p_isnd;
		m = ifq->ifq_head;
		if (m == NULL)	{
			ifq = &sc->sc_if.if_snd;
			m = ifq->ifq_head;
		}
		if (m == NULL) {
			return (0);
		}
		IF_DEQUEUE(ifq, m);
		if (sc->sc_dcd)
			break;
		/* Discard packet if no DCD */
		m_freem(m);
	}

	n2sel_page(sc);

	/*
	 * Fill in the buffer (it can be done at lower priority)
	 */
	desp = &sc->sc_txdesp[sc->sc_tdin];
	bp = sc->sc_vaddr + ((desp->cbd_bp1 << 16) + desp->cbd_bp0) -
		sc->sc_memoff;
	buffer = bp;

	len = 0;
	for (m0 = m; m != 0; m = m->m_next) {
		if (m->m_len == 0)
			continue;
		bcopy(mtod(m, caddr_t), bp, m->m_len);
		bp += m->m_len;
		len += m->m_len;
	}
	m_freem(m0);

	desp->cbd_len = len;	/* frame size */
	desp->cbd_stat = 0x80;	    /* Set EOM bit for frame */
	sc->sc_tdin = newtxin;
	n2start_tx(sc);
	return (0);
}

/*
 * Start output
 */
static void
n2start_tx(sc)
struct n2_softc *sc;
{
	int i,last;
	struct hd_cbd *desp, *lastdesp;

	if ((sc->sc_tdin == sc->sc_tdout) ||
	    (sc->sc_if.if_flags & IFF_OACTIVE))
		return ;

	sc->sc_if.if_flags |= IFF_OACTIVE;  
	sc->sc_watchdog =  5;		/* won't hack real slow links  */

	desp = & sc->sc_txdesp[sc->sc_tdout];
	if (sc->sc_tdin == 0)
		last = sc->sc_tdend;
	else
		last = sc->sc_tdin - 1 ;
	lastdesp = & sc->sc_txdesp[last];
	lastdesp->cbd_stat = 0x81;	/* Set EOM & EOT bits for last frame */
	sc->sc_tdlast = sc->sc_tdin;

	/*
	 * Start output
	 */
	routb(sc->sc_dma, DSR+TX, DSR_DE);
}


/*
 * Interrupt routine
 */
n2intr(sc)
register struct n2_softc *sc;
{
	u_char isr1;
	int rval = 0;

	while ((isr1 = rinb(sc->sc_addr, ISR1)) != 0) {
		if (sc->sc_ihc++ > 200) {
			n2enable(sc);
			n2enable(sc->sc_nextchan);
			sc->sc_ihc = 0;
		}
		rval = 1;

		if (isr1 & (ISR1_DMIA0R | ISR1_DMIB0R))
			n2rx_intr(sc);
		if (isr1 & (ISR1_DMIA0T | ISR1_DMIB0T))
			n2tx_intr(sc);
		if (isr1 & (ISR1_DMIA1R | ISR1_DMIB1R))
			n2rx_intr(sc->sc_nextchan);
		if (isr1 & (ISR1_DMIA1T | ISR1_DMIB1T))
			n2tx_intr(sc->sc_nextchan);

	}
	return rval;
}
static void
n2rx_intr(sc)
	register struct n2_softc *sc;
{

	caddr_t rxptr;
	struct hd_cbd *desp;
	u_char st;
	u_int len, i, eda;
	u_char *sp;

	n2sel_page(sc);

	/*
	 * If we find a full buffer without  eom we just toss it
	 * When the finally buffer shows up it should always have
	 * bad crc. If it has good crc then someone is sending us
	 * out of spec stuff. Hopefully the next layer up will detect
	 * this.
	 */
	desp = &sc->sc_rxdesp[sc->sc_rdout];
	routb(sc->sc_dma, DSR, DSR_DEWD | DSR_EOM);
	while ((len = desp->cbd_len) != 0) {
	    if ((st = desp->cbd_stat) & ST2_EOM) {

		i = desp->cbd_bp0 + (desp->cbd_bp1 << 16) - sc->sc_memoff;
		if (len > HDLCMAX || i > sc->sc_memsize ||
		    i + len > sc->sc_memsize) {
			log(LOG_ERR, "ntwo%d: onboard memory inconsistent\n",
			    sc->sc_unit);
			n2enable(sc);		/* reset board */
			return;
		}

		/* Analyse frame status */
		if (st & (ST2_SHRT|ST2_ABT|ST2_FRME|ST2_OVRN|ST2_CRCE)) {
		    if (!sc->sc_quiet) {
			/*
			 * put into english as this is useful to end user
			 */
			char b[80];
#define BUF &b[strlen(b)]

			b[0] = 0;
			sc->sc_if.if_ierrors++;
			if (st & ST2_SHRT)
				sprintf(BUF, "-Short Frame");
			if (st & ST2_ABT)
				sprintf(BUF, "-Abort");
			if (st & ST2_FRME)
				sprintf(BUF, "-Residue Bit");
			if (st & ST2_OVRN)
				sprintf(BUF, "-Overrun");
			if (st & ST2_CRCE)
				sprintf(BUF, "-CRC");
			log(LOG_ERR, "ntwo%d: Receive packet status %x%s\n",
			    sc->sc_unit, st, b);
#undef BUF
		    }
		} else { 
			n2read(sc, sc->sc_vaddr + i, len);
			sc->sc_if.if_ipackets++;
		}
	    }
	    desp->cbd_len = 0;
	    desp->cbd_stat = 0;
	    eda = (caddr_t)(desp) - sc->sc_vaddr;
	    eda += sc->sc_memoff;
	    routb(sc->sc_dma, EDAL, eda);
	    routb(sc->sc_dma, EDAH, eda >> 8);
	    if (++sc->sc_rdout > sc->sc_rdend)
		    sc->sc_rdout = 0;
	    desp = &sc->sc_rxdesp[sc->sc_rdout];
	}
	st = rinb(sc->sc_dma, DSR);
	st &= DSR_EOT | DSR_BOF | DSR_COF;
	if (st) {
		/*
		 * If we got here we think the dma engine has
		 * stopped. Don't bother to complain about
		 * overrun errors. Restart DMA engine
		 */
		sc->sc_if.if_ierrors++;
		if (st & (DSR_EOT | DSR_COF))
			log(LOG_ERR, "ntwo%d: RXdmaerr %x\n",
			    sc->sc_unit, st);
		routb(sc->sc_dma, DSR, st | DSR_DE );
	}
}

static void
n2tx_intr(sc)
	register struct n2_softc *sc;
{

	u_char st, count;

	n2sel_page(sc);

	count = rinb(sc->sc_dma, FCT+TX);		

	/*  Analyze DMA status */
	st = rinb(sc->sc_dma, DSR+TX);

	while ( (st & DSR_EOM) && count--)  {
		routb(sc->sc_dma, DSR+TX, DSR_EOM | DSR_DEWD); /*write EOM bit*/
		sc->sc_if.if_opackets++;
	}

	if (st & (DSR_BOF | DSR_COF)) {
		sc->sc_if.if_oerrors++;
		log(LOG_ERR, "ntwo%d: TXdmaerr %x\n",
		    sc->sc_unit, st);
	}

	st &= DSR_EOT | DSR_BOF | DSR_COF;
	if (st)
		routb(sc->sc_dma, DSR+TX, st | DSR_DEWD);

	if (st & DSR_EOT)  {
		sc->sc_tdout = sc->sc_tdlast;
		/* Restart transmission */
		sc->sc_if.if_flags &= ~IFF_OACTIVE;
		sc->sc_watchdog = 0;
		n2start_tx(sc);
		(void) n2start(&sc->sc_if);
	}
}

/*
 * Modem control
 * flag = 0 -- disable line; flag = 1 -- enable line
 */
int
n2mdmctl(pp, flag)
	struct p2pcom *pp;
	int flag;
{
	struct n2_softc *sc0 = ntwocd.cd_devs[pp->p2p_if.if_unit / N2_NCHANS];
	struct n2_softc *sc = (pp->p2p_if.if_unit % N2_NCHANS) ?
				sc0->sc_nextchan : sc0;
	u_char br;
	int s;

	s = splimp();
	if (flag) {
		sc->sc_up = 1;
		n2enable(sc);
		if (!sc->sc_timeron) {
			sc->sc_timeron = 1;
			timeout(n2watch, sc, hz);
		}

		if (sc->sc_ignoredcd)  {
			sc->sc_dcd =  1;
			if (sc->sc_p2pcom.p2p_modem)
				(*sc->sc_p2pcom.p2p_modem)(pp, flag);
			splx(s);
			return;
		}

		/* hardware uses negative logic */
		if (!(rinb(sc->sc_msci, ST3) & ST3_DCD)) {
			sc->sc_dcd = 1;
			if (sc->sc_p2pcom.p2p_modem)
				(*sc->sc_p2pcom.p2p_modem)(pp, flag);
		} else
			sc->sc_dcd = 0;
		splx(s);
		return;
	}

	sc->sc_up = 0;
	/* disable interrupts */
	routb(sc->sc_dma, DIR, 0);
	routb(sc->sc_dma, DIR+TX, 0);

	/* Halt dma in progress */
	routb(sc->sc_dma, DSR, 0);
	routb(sc->sc_dma, DSR+TX, 0);
	routb(sc->sc_dma, DCR, DCR_ABORT);
	routb(sc->sc_dma, DCR+TX, DCR_ABORT);

	/* disable transmitter and receiver */
	routb(sc->sc_msci, CMD, CMD_RXDIS);
	routb(sc->sc_msci, CMD, CMD_TXDIS);
	routb(sc->sc_msci, CTL, CTL_RTS);	/* drop RTS */
	br = inb(sc->sc_addr+N2_MCR);	/* drop DTR and disable transmitter */
	if (sc->sc_chan)  {
		br &= ~N2_TE1;
		br |= N2_DTR1;
	} else {
		br &= ~N2_TE0;
		br |= N2_DTR0;
	}
	outb(sc->sc_addr+N2_MCR, br);
	splx(s);
	if (sc->sc_p2pcom.p2p_modem)
		(*sc->sc_p2pcom.p2p_modem)(pp, flag);
}

/*
 * Pass the accepted IP packet to upper levels
 */
static void
n2read(sc, cp, totlen)
	register struct n2_softc *sc;
	register caddr_t cp;
	int totlen;
{
	if (totlen == 0)
		return;
	
	if ((*sc->sc_p2pcom.p2p_hdrinput)(&sc->sc_p2pcom, (u_char *)cp,
	    totlen) >= 0)
		(*sc->sc_p2pcom.p2p_input)(&sc->sc_p2pcom, 0);
}

void
n2watch(sc0)
	void *sc0;
{
	struct n2_softc *sc;
	int s;

	sc = sc0;
	if (!sc->sc_up) {
		sc->sc_timeron = 0;
		return;
	}
	timeout(n2watch, sc, hz); 
	s = splimp();

	if (sc->sc_watchdog) {
		sc->sc_watchdog--;
		if (!sc->sc_watchdog) {
			log(LOG_ERR, "ntwo%d: output hung, resetting\n",
			    sc->sc_unit);
			n2enable(sc);
		}
	}

	if (sc->sc_ignoredcd) {
		splx(s);
		return;
	}

	/* hardware neg logic */
	if (!(rinb(sc->sc_msci, ST3) & ST3_DCD)) {
		if (sc->sc_dcd) {
			splx(s);
			return;
		}

		log(LOG_NOTICE, "ntwo%d: DCD raised\n", sc->sc_unit);
		sc->sc_dcd = 1;
		if (sc->sc_p2pcom.p2p_modem)
			(*sc->sc_p2pcom.p2p_modem)(&sc->sc_p2pcom, 1);
	} else {
		if (!sc->sc_dcd) {
			splx(s);
			return;
		}

		log(LOG_NOTICE, "ntwo%d: DCD dropped\n", sc->sc_unit);
		sc->sc_dcd = 0;
		n2enable(sc);	/* toss packets on board */
		if (sc->sc_p2pcom.p2p_modem)
			(*sc->sc_p2pcom.p2p_modem)(&sc->sc_p2pcom, 0);
	}
	splx(s);
	return;
}
