/*
 * Copyright (c) 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: smc83C825.h,v 2.1 1995/09/22 21:41:08 ewv Exp $
 */

/*
 * SMC 83C825 token ring controller
 *
 * Note: Below definitions assume Intel address/data mode (SCP_IAF | SCP_IDF).
 */

/* Address pointers are 4 byte quantities but do not always align as such */
typedef u_char sptr_t[4];

/*
 * System configuration pointer block (SCP)
 *	Pointed to by ISCP
 */
typedef struct smtr_scp {
	u_short	ctl;			/* Misc settings */
	sptr_t	scbp;			/* System control block */
	sptr_t	cbp;			/* Command block chain start */
	sptr_t	sbp;			/* Intr status block */
	u_char	sbsiz;			/* Size of intr blk - 2 bytes */
	u_char	pad1;
} scp_t;

/* Bits for scp_t.ctl */
#define SCP_BSTMASK	0x001f
#define SCP_MWC		0x2000
#define SCP_IDF		0x4000
#define SCP_IAF		0x8000

/* Macros to extract values from ISB words (u_short's) */
#define ISB_TYPE(x)	((x) & 0xff)	/* Interrupt type in ISB */
#define ISB_SUBTYPE(x)	((x) >> 8)	/* Subtype field in ISB */

/* ISB interrupt types */
#define SMIT_MAC3	0x01		/* MAC type 3 interrupt */
#define SMIT_MACCNT	0x02		/* MAC error counter interrupt */
#define SMIT_MAC2	0x04		/* MAC type 2 interrupt */
#define SMIT_TX		0x05		/* Transmit frame interrupt */
#define SMIT_ETX	0x06		/* End of TX queue reached */
#define SMIT_RRI	0x07		/* Non-MAC receive resource intr */
#define SMIT_MRRI	0x08		/* MAC receive resource intr */
#define SMIT_RX		0x09		/* Non-MAC frame receive interrupt */
#define SMIT_MRX	0x0a		/* MAC frame receive interrupt */
#define SMIT_FIFO	0x0b		/* TRC FIFO error */
#define SMIT_ACS	0x0c		/* Action command interrupt */
#define SMIT_MAC1	0x0d		/* MAC type 1 interrupt */
#define SMIT_TRCI	0x0e		/* TRC init complete */

/* Subtype bits for SMIT_ACS */
#define SMACS_CI	0x4000		/* CI detected in CB */
#define SMACS_CE	0x8000		/* CE detected in CB */

/*
 * System control block (SCB)
 */
typedef struct smtr_scb {
	u_short	ctl;			/* Control/command word */
	u_char	iack;			/* IACK code */
	u_char	pad0;
	u_short	resctl;			/* resume control bits */
	u_short imctl;			/* interrupt mask control */
	u_short	imstate;		/* current interrupt mask bits */
} scb_t;

/* Bits for smtr_scb.ctl */
#define SMCTL_CMDMASK	0x000f		/* Command mask */
#define SMCTL_NOP	0x0000		/* NOP command */
#define SMCTL_REMOVE	0x0001		/* Remove from ring */
#define SMCTL_SUSCB	0x0002		/* Suspend CB chain processing */
#define SMCTL_SETIM	0x0003		/* Set interrupt mask */
#define SMCTL_CLRIM	0x0004		/* Clear interrupt mask */
#define SMCTL_IV	0x2000		/* IACK code valid */
#define SMCTL_RV	0x4000		/* Resume control word valid */
#define SMCTL_SV	0x8000		/* SCB valid */

/* Bits for smtr_scb.resctl */
#define SMRES_CB	0x0001		/* action command block (CB) */
#define SMRES_RXMAC_FCB	0x0010		/* RX_MAC_FCB chain */
#define SMRES_RXMAC_BDB	0x0020		/* RX_MAC_BDB chain */
#define SMRES_RXNM_FCB	0x0040		/* RX_NONMAC_FCB chain */
#define SMRES_RXNM_BDB	0x0080		/* RX_NONMAC_BDB chain */
#define SMRES_TX0_FCB	0x0100		/* TX access class 0 FCB chain */
#define SMRES_TX1_FCB	0x0200		/* TX access class 1 FCB chain */
#define SMRES_TX2_FCB	0x0400		/* TX access class 2 FCB chain */
#define SMRES_TX3_FCB	0x0800		/* TX access class 3 FCB chain */
#define SMRES_TX4_FCB	0x1000		/* TX access class 4 FCB chain */
#define SMRES_TX5_FCB	0x2000		/* TX access class 5 FCB chain */
#define SMRES_TX6_FCB	0x4000		/* TX access class 6 FCB chain */
#define SMRES_TX7_FCB	0x8000		/* TX access class 7 FCB chain */

/* Bits for smtr_scb.imctl (interrupt enable/disable bits) */
#define SMIM_MAC3	0x0002		/* Mac type 3 (MAC_ARC_INDICATE) */
#define SMIM_COUNT	0x0004		/* Mac error counters */
#define SMIM_MAC2	0x0010		/* Mac type 2 (TRC_STATUS_CHANGE_IND) */
#define SMIM_TX		0x0020		/* Transmit frame */
#define SMIM_TXQ	0x0040		/* End of transmit Q */
#define SMIM_NMRSRC	0x0080		/* Non-mac resource interrupt */
#define SMIM_MACRSRC	0x0100		/* Mac resource interrupt */
#define SMIM_NMRX	0x0200		/* Non-mac frame received */
#define SMIM_MACRX	0x0400		/* Mac frame received */
#define SMIM_FIFO	0x0800		/* TRC FIFO status */
#define SMIM_ACS	0x1000		/* Action command status */
#define SMIM_MAC1	0x2000		/* Mac type 1 (self removed) */
#define SMIM_TRCI	0x4000		/* TRC init sequence */

/*
 * Action command block (CB)
 */
typedef struct smtr_cb {
	u_short	stat;			/* Command status */
	u_short	ctl;			/* Control bits */
	sptr_t	ncbp;			/* Pointer to next CB */
	u_char	pcmd;			/* P and command */
	u_char	parm;			/* command specific parameter */
	u_char	wcnt;			/* word count */
	u_char	iap;			/* Internal address pointer */
	sptr_t	eap;			/* External address pointer */
} cb_t;

/* Bits for smtr_cb.stat */
#define	SMST_DONE	0x8000		/* Command done */
#define SMST_MASK	0x00ff		/* Return code mask */

/* Bits for smtr_cb.ctl */
#define SMCBC_CI	0x2000		/* Command interrupt */
#define SMCBC_CE	0x8000		/* End of CB chain */

/* CB command codes (P and command) */
#define SMCB_NOP	0x00		/* NOP */
#define SMCB_TEST	0x01		/* Test */
#define SMCB_TEST2	0x81		/* Test */
#define SMCB_INSERT	0x02		/* Insert in ring */
#define SMCB_ABORT	0x82		/* Abort */
#define SMCB_REMOVE	0x83		/* Remove from ring */
#define SMCB_WVAL	0x04		/* Write value */
#define SMCB_RVAL	0x05		/* Read value */
#define SMCB_INIT_MAC	0x06		/* Initialize */
#define SMCB_INIT_TXRX	0x86		/* Initialize */
#define SMCB_RTRC	0x07		/* Read TRC status */

/* SMCB_TEST subcommand codes */
#define SMTST_MACREG	0x00		/* TRC MAC registers test */
#define SMTST_TRCILOOP	0x01		/* TRC internal loopback */
#define SMTST_TCTILOOP	0x02		/* TRC/TRI loopback */
#define SMTST_TRCROM	0x03		/* TRC rom test (P=0) */
#define SMTST_TRCROM	0x03		/* Hist interface regs test (P=1) */
#define SMTST_LOBE	0x04		/* Lobe media test */
#define SMTST_ANALOG	0x05		/* Analog test */

/* smtr_cb.cmdst and abort codes for SMCB_INAB */
#define SMCST_TX0	0x01		/* TX class 0 FCB active */
#define SMCST_TX1	0x02		/* TX class 0 FCB active */
#define SMCST_TX2	0x04		/* TX class 0 FCB active */
#define SMCST_TX3	0x08		/* TX class 0 FCB active */
#define SMCST_TX4	0x10		/* TX class 0 FCB active */
#define SMCST_TX5	0x20		/* TX class 0 FCB active */
#define SMCST_TX6	0x40		/* TX class 0 FCB active */
#define SMCST_TX7	0x80		/* TX class 0 FCB active */

/* SMCB_RVAL/WVAL internal addresses (number in parens is length in words) */
#define SMIA_MACSTATE	0x11		/* (1) MAC state */
#define SMIA_SAL	0x28		/* (3) SA of last AMP or SMP */
#define SMIA_PDROP	0x3b		/* (2) Physical drop number */
#define SMIA_UNA	0x3e		/* (3) Upstream neighbor address */
#define SMIA_PRIID	0x4b		/* (9) Product instance ID */
#define SMIA_CLRCNT	0x7a		/* (6) error counters, read/clear */
#define SMIA_CNT	0x80		/* (6) error counters, read */
#define SMIA_CONF	0xa0		/* (29) config block */

/* Values for SMIA_MACSTATE */
#define SMMS_BYPASS	0x00		/* Bypass */
#define SMMS_INSERT	0x01		/* Inserted */
#define SMMS_INIT	0x02		/* Initialize */
#define SMMS_TXCLTK	0x03		/* TX_CL_TK */
#define SMMS_STANDBY	0x04		/* Standby */
#define SMMS_TXBEACON	0x05		/* TX beacon */
#define SMMS_ACTIVE	0x06		/* Active */
#define SMMS_TXPURGE	0x07		/* TX purge */

/* layout of SMIA_CONF and initialization block for SMCB_INIT (MAC, P==0?) */
typedef struct smia_conf {
	u_short	conf0;			/* Config reg 0 */
	u_short conf1;			/* Config reg 1 */
	u_short tblock[13];		/* Timer setup block */
	u_short pad0;
	u_short imac[3];		/* Individual MAC address */
	u_short iga[3];			/* Individual group MAC address */
	u_short fa[3];			/* Functional address */
	u_char bsga[2];			/* Bit significant group address */
	u_short srn;			/* Source ring or bridge number */
	u_short trn;			/* Target ring number */
	/* following fields are only used in SMCB_INIT */
	u_char pdrop[4];		/* Physical drop number (SMCB_INIT) */
	u_char priid[18];		/* Product instance ID */
} mac_conf_t;

/* Bits for smia_conf.conf0 */
#define	SMIAC0_FTRIG_MASK	0x000f	/* FIFO DMA trigger point */
#define SMIAC0_MRCLK		0x0010	/* MAC clock divide by 4 */
#define SMIAC0_THDREN		0x0020	/* Must be set to 1 */
#define SMIAC0_ETREN		0x0100	/* Enable early token release */
#define SMIAC0_RXOWNB		0x0200	/* Receive own frames */
#define SMIAC0_RXATMAC		0x0400	/* Rx attention MAC frames */
#define SMIAC0_PROMIS		0x0800	/* Promiscuous mode */
#define SMIAC0_USETPT		0x1000	/* Always set to 1 */
#define SMIAC0_SAVBAD		0x2000	/* Save bad frames */
#define SMIAC0_ONEQUE		0x4000	/* Use single receive Q */
#define SMIAC0_DISSMV		0x8000	/* Disable automatic self remove */

/* Bits for smia_conf.conf1 */
#define SMIAC1_TIFCNT_MASK	0x000f	/* Transmit interframe spacing */
#define SMIAC1_RXALMC		0x0010	/* Receive ALL multicast frames */
#define SMIAC1_RXALNSR		0x0020	/* Receive all non-src routing frames */
#define SMIAC1_SROUTE		0x0040	/* Enable source routing */
#define SMIAC1_SGLRT		0x0080	/* Single route bridge */

/* Timer offsets in smia_conf.tblock */
#define	SMIAC_PRESCALE	0
#define SMIAC_TPT	1
#define SMIAC_TQP	2
#define SMIAC_TNT	3
#define SMIAC_TBT	4
#define SMIAC_TSM	5
#define SMIAC_TAM	6
#define SMIAC_TBR	7
#define SMIAC_TER	8
#define SMIAC_TGT	9
#define SMIAC_THT	10
#define SMIAC_TRR	11
#define SMIAC_TVX	12

/* external parameter block for SMCB_INIT (INIT_TX_RX, P==1??) */
typedef struct sm_txrx_init {
	sptr_t	tx0_fcb;		/* TX access class 0 FCB */
	sptr_t	tx1_fcb;		/* TX access class 1 FCB */
	sptr_t	tx2_fcb;		/* TX access class 2 FCB */
	sptr_t	tx3_fcb;		/* TX access class 3 FCB */
	sptr_t	tx4_fcb;		/* TX access class 4 FCB */
	sptr_t	tx5_fcb;		/* TX access class 5 FCB */
	sptr_t	tx6_fcb;		/* TX access class 6 FCB */
	sptr_t	tx7_fcb;		/* TX access class 7 FCB */
	sptr_t	rxmac_fcb;		/* RX MAC FCB */
	sptr_t	rxmac_bdb;		/* RX MAC BDB */
	sptr_t	rxnm_fcb;		/* RX non-MAC FCB */
	sptr_t	rxnm_bdb;		/* RX non-MAC BDB */
} txrx_init_t;

/* SMCB_RTRC status block */
typedef struct sm_trcstat {
	u_char	err_cnt[12];		/* Error counters */
	u_short	ringstat;		/* Ring status */
	u_short	btype;			/* Beacon type */
	u_short active_err;		/* Active error code */
	u_char	lastsa[6];		/* SA of last AMP or SMP */
	u_char	una[6];			/* Upstream neighbor address */
	u_short	ucode_version;		/* TRC microcode version */ 
	u_short status;			/* TRC status indicate */
	u_short	pad0;
} trcstat_t;

/* Error counter offsets for sm_trcstat.errcnt */
#define	SMERR_INTERNAL	0
#define SMERR_LINE	1
#define SMERR_AC	2
#define SMERR_BURST	3
#define SMERR_ADTRANS	5
#define SMERR_RCONGEST	6
#define SMERR_LOSTFR	7
#define SMERR_FREQ	8
#define SMERR_FRCOPY	9
#define SMERR_TOKEN	11

/* Bits for sm_trcstat.ringstat */
#define SMRS_RINGREC	0x0020		/* Ring recovery in progress */
#define SMRS_ONLY	0x0040		/* Only station on ring */
#define SMRS_CNTOVFL	0x0080		/* Mac error counter overflow */
#define SMRS_REMOVE	0x0100		/* Remove frame received */
#define SMRS_AUTORMERR	0x0400		/* Auto removal error */
#define SMRS_LOBEFLT	0x0800		/* Lobe wire fault */
#define SMRS_TXBEACON	0x1000		/* Transmitting beacon */
#define SMRS_SOFTERR	0x2000		/* TRC sent REPORT_ERROR MAC frame */
#define SMRS_HARDERR	0x4000		/* TRC is sending/receiving beacons */
#define SMRS_SIGLOSS	0x8000		/* Loss of ring signal */

/* Bits for sm_trcstat.status */
#define	SMTS_MACCNT	0x0001		/* MAC error counter changed */
#define SMTS_RINGSTAT	0x0002		/* Ring status changed */
#define SMTS_UNA	0x0004		/* UNA changed */
#define SMTS_RRQINIT	0x0008		/* Request host to send MAC init */

/*
 * Frame control block (FCB)
 */
typedef struct smtr_fcb {
	u_short	frst;			/* Frame status */
	u_short ctl;			/* Control bits */
	sptr_t	nfcbp;			/* Next FCB pointer */
	sptr_t	bdbp;			/* Associated data descriptor */
	u_short	flen;			/* Frame length (bytes) */
	sptr_t	aux;			/* s/w use (TRC ignores) */
} fcb_t;

/* Bits for smtr_fcb.frst / TX packets */
#define SMTXST_CR2	0x0004		/* CR2 from rcvd FC */
#define SMTXST_AR2	0x0008		/* AR2 from rcvd FC */
#define SMTXST_CR1	0x0040		/* CR1 from rcvd FC */
#define SMTXST_AR1	0x0080		/* AR1 from rcvd FC */
#define SMTXST_ED	0x0100		/* E from rcvd ED */
#define SMTXST_FD	0x8000		/* Frame transmitted */

/* Bits for smtr_fcb.frst / RX packets */
#define SMRXST_ERR	0x0001		/* Any error (OR of bits 1-6) */
#define SMRXST_FCS	0x0002		/* FCS error */
#define SMRXST_ALIGN	0x0004		/* Alignment error */
#define SMRXST_NOISE	0x0008		/* Noise error */
#define SMRXST_SIZE	0x0020		/* <18 or >64K bytes error */
#define SMRXST_EBIT	0x0080		/* E bit of ED field set */
#define SMRXST_MMASK	0x0300		/* Match mask */
#define SMRXST_INDIV	0x0000		/* 	Individual match */
#define SMRXST_GROUP	0x0100		/*	Group match */
#define SMRXST_FUNC	0x0200		/* 	Functional match */
#define SMRXST_BCAST	0x0300		/*	Broadcast match */
#define SMRXST_DMATCH	0x0400		/* Destination addr matched */
#define SMRXST_SROUTE	0x0800		/* Source routing frame */
#define SMRXST_SELF	0x1000		/* Src addr == Individual address */
#define SMRXST_BOTHC	0x2000		/* Both C bits set */
#define SMRXST_BOTHA	0x4000		/* Both A bits set */
#define SMRXST_FD	0x8000		/* Frame received */

/* Bits for smtr_fcb.ctl */
#define SMFCT_NTC	0x0001		/* Do not transmit CRC */
#define SMFCT_TFS	0x0002		/* Post returned FS (TX only) */
#define SMFCT_FI	0x2000		/* FCB interrupt */
#define SMFCT_FW	0x4000		/* FCB warning interrupt */
#define SMFCT_FE	0x8000		/* End of FCB chain */

/*
 * Buffer descriptor block (BDB)
 */
typedef struct smtr_bdb {
	u_short	flags;			/* Buffer flags */
	sptr_t	nbdbp;			/* Next BDB in chain */
	sptr_t	buf;			/* Data buffer pointer */
	u_short	bflen;			/* Data buffer length (bytes) */
} bdb_t;

/* Bits for smtr_bdb.flags */
#define SMBFL_BW	0x4000		/* generate RX BDB warning interrupt */
#define SMBFL_BE	0x8000		/* End of BDB chain */
