/*-
 * Copyright (c) 1993, 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: example.c,v 2.2 1995/09/01 15:43:22 torek Exp $
 */

/*
 * Example device driver for "xx" device, illustrating autoconfiguration
 * data structures and subroutines for ISA-bus devices.
 * This driver uses a range of I/O ports, an interrupt,
 * and device memory in the 640 K to 1 MB (0xA0000 to 0x100000) I/O space.
 */
#include <sys/param.h>			/* ALWAYS included */
#include <sys/systm.h>			/* general kernel functions */
#include <sys/device.h>			/* generic device definitions */
#include <i386/isa/isa.h>		/* ISA bus parameters */
#include <i386/isa/isavar.h>		/* ISA-specific data structures */
#include <i386/isa/icu.h>		/* interrupt definitions */
#include <i386/isa/dma.h>		/* DMA definitions */
#include "xxreg.h"			/* xx register definitions */
#include "xxvar.h"			/* xx data structure definitions */
#include "xxioctl.h"			/* xx-specific ioctl definitions */

struct xx_type {
	char	*xx_name;
	/* parameters describing each type of device supported */
} [XX_NTYPES] = {
	/* definitions for each type of device supported */
	{ "model X" },
	{ "model Y" },
};

struct xx_softc {
	struct  device sc_dev;		/* base device, must be first */
	struct  isadev sc_id;		/* ISA device */
	struct  intrhand sc_ih;		/* interrupt vectoring */

	int	sc_base;		/* I/O port base */
	int	sc_membase;		/* kernel address of device memory */
	int	sc_memsize;		/* size of device memory */
	int	sc_dmachan;		/* DMA channel */
	struct	xx_type *sc_type;	/* type-specific device parameters */

	struct	evcnt sc_intr;		/* display no. of interrupts */
	struct	evcnt sc_resync;	/* and number of resynch attempts */

	int	sc_open;		/* device is open */
	struct	selinfo sc_wsel;	/* Selecting process for write */
	/* additional private storage per device located */
};

int	xxprobe __P((struct device *, struct cfdata *, void *));
void	xxforceintr __P((void *));
void	xxattach __P((struct device *, struct device *, void *));
void	xxstart __P((struct xx_softc *, struct uio *));
int	xxintr __P((void *));

struct cfdriver xxcd =
	{ NULL, "xx", xxprobe, xxattach, DV_DULL, sizeof(struct xx_softc) };

struct	xx_type *xx_check_type __P((struct isa_attach_args *));
 
/* /dev interface functions */
int	xxopen __P((dev_t, int, int, struct proc *));
int	xxclose __P((dev_t, int, int, struct proc *));
/*	xxread __P((dev_t, struct uio *, int)) -- read() prohibited */
int	xxwrite __P((dev_t, struct uio *, int));
int	xxselect __P((dev_t, int, struct proc *));
/*	xxmmap __P((dev_t, int, int)) -- mmap() prohibited */
int	xxioctl __P((dev_t, int, caddr_t, int, struct proc *));

struct devsw xxsw = {
	&xxcd,
	xxopen, xxclose, noread, xxwrite, xxioctl, xxselect, nommap,
	nostrat, nodump, nopsize, 0,
	nostop
};

/*
 * Probe the hardware to see if it is present
 */
xxprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;

	/*
	 * Check whether device registers appear
	 * to be present at this address.
	 */
	if (!xx_test_registers(ia->ia_iobase))
		return (0); 		/* device not present here */

	/*
	 * Check/test shared memory in 640K to 1MB "hole".  ia_maddr is
	 * a physical address, ISA_HOLE_VADDR converts to kernel virtual.
	 */
	if (!xx_memory_ok(ISA_HOLE_VADDR(ia->ia_maddr)))
		return (0); 		/* memory not present here */

	/*
	 * If we support multiple device subtypes, etc., we can pass
	 * this information to xxforceintr and/or xxattach using ia->ia_aux.
	 * Here we pass a pointer to the structure describing the subtype.
	 */
	ia->ia_aux = xx_check_type(ia);

	if (ia->ia_irq == IRQUNK) {
		ia->ia_irq = isa_discoverintr(xxforceintr, aux);
		/* disable device interrupts here */
		if (ffs(ia->ia_irq) - 1 == 0)
			return (0);	/* no interrupt */
	}

	ia->ia_iosize = XX_NPORT;	/* reserve this many ports */
	return (1);			/* device appears to be present here */
}

/*
 * force device to interrupt for autoconfiguration
 */ 
void
xxforceintr(aux)
	void *aux;
{
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;

	xx_intr_enable(ia->ia_iobase, (struct xx_type *) ia->ia_aux);
	/* count interrupt events */
	/*
	 * The device should now have interrupted.  If it has not,
	 * initiate some activity to force an interrupt.
	 */
	if (!isa_got_intr())
		xx_poke_harder();
}
 
/*
 * Interface exists: initialize softc structure
 * and attach to bus, interrupts, etc.
 */
void
xxattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct xx_softc *sc = (struct xx_softc *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;

	/* record device location for all future accesses */
	sc->sc_base = ia->ia_iobase;
	sc->sc_membase = ISA_HOLE_VADDR(ia->ia_maddr);
	sc->sc_memsize = ia->ia_msize;
	sc->sc_dmachan = ia->ia_drq;
	sc->sc_type = (struct xx_type *) ia->ia_aux;

	printf(": %s\n", sc->sc_type->xx_name);

	/* attach to isa bus */
	isa_establish(&sc->sc_id, &sc->sc_dev);

	/* attach interrupt handler */
	sc->sc_ih.ih_fun = xxintr;
	sc->sc_ih.ih_arg = (void *)sc;
	intr_establish(ia->ia_irq, &sc->sc_ih, DV_xxx);	/* fill in device type */

	/* attach event counters */
	evcnt_attach(&sc->sc_intr, &sc->sc_dev, "intr");
	evcnt_attach(&sc->sc_resync, &sc->sc_dev, "resync");

	/* allocate and initialize DMA channel, given maximum I/O size */
	at_setup_dmachan(sc->sc_dmachan, XX_MAXIOSIZE);
}
 
int
xxopen(dev, flag, fmt, p)
	dev_t dev;
	int flag, fmt;
	struct proc *p;
{
	int unit = minor(dev);
	struct xx_softc *sc;

	/* Validate unit number */
	if (unit >= xxcd.cd_ndevs || (sc = xxcd.cd_devs[unit]) == NULL)
		return (ENXIO);

	if (sc->sc_open == 0) {
		if (xxinit(sc))		/* initialize device */
			sc->sc_open = 1;
		else
			return (EIO);
	}
	return (0);			/* success */
}

int
xxclose(dev, flag, fmt, p)
	dev_t dev;
	int flag, fmt;
	struct proc *p;
{
	int s;
	struct xx_softc *sc = xxcd.cd_devs[minor(dev)];

	/* Mark as not open */
	sc->sc_open = 0;
	xxshutdown(sc);			/* turn off device */

	return(0);
}

int
xxwrite(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	int n, s, error;
	struct xx_softc *sc = xxcd.cd_devs[minor(dev)];

	/* Loop while more data remaining */
	while (uio->uio_resid != 0) {
		while (device busy) {
			error = tsleep(sc, PZERO | PCATCH, "xxwrit", 0);
			if (error != 0)
				return (error);
		}
		xxstart(sc, uio);
	}
	return (0);
}

void
xxstart(sc, uio)
	struct xx_softc *sc;
	struct uio *uio;
{

	/* compute or allocate kernel address */
	at_dma(uio->uio_rw == UIO_READ, kaddr, count, sc->sc_dmachan);
	/* initiate I/O */
	if (error)
		at_dma_abort(sc->sc_dmachan);
}

/*
 * Device interrupt handler
 */
int
xxintr(sc0)
	void *sc0;
{
	struct xx_softc *sc = sc0;
	register int base = sc->sc_base;

	/* check device status, etc. */
	at_dma_terminate(sc->sc_dmachan);
	/* notify process selecting for write */
	selwakeup(&sc->sc_wsel);
	/* count interrupt events */
	sc->sc_intr.ev_count++;
	return (1);	/* interrupt was expected */
}

/*
 * Select function that supports only select for write
 */
int
xxselect(dev, rw, p)
	dev_t dev;
	int rw;
	struct proc *p;
{
	int s, ret;
	struct xx_softc *sc = xxcd.cd_devs[minor(dev)];

	s = splXXX();		/* block xx-class interrupts */
	switch (rw) {
	case FREAD:
		/* Silly to select for input on output-only device */
		ret = 1;		/* go ahead and try to read! */
		break;
	case FWRITE:
		/* Return true if queue almost empty */
		if (sc->sc_outq.c_cc < LOWAT)
			ret = 1;
		else {
			ret = 0;
			selrecord(p, &sc->sc_wsel);
		}
		break;
	case 0:		/* exceptional condition */
		ret = 0;
		break;
	}
	splx(s);
	return (ret);
}

int
xxioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	struct xx_softc *sc = xxcd.cd_devs[minor(dev)];

	switch (cmd) {
	default:
		return (ENOTTY);
	}
	return (0);
}
