/*-
 * Copyright (c) 1994, 1995 Matt Thomas (matt@lkg.dec.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *	BSDI $Id: decether.c,v 2.3 1995/11/03 17:20:40 ewv Exp $
 *
 * Id: decether.c,v 1.5 1995/07/13 15:14:59 thomas Exp
 *
 * Log: decether.c,v
 * Revision 1.5  1995/07/13  15:14:59  thomas
 * mcbits is int now
 *
 * Revision 1.4  1995/05/24  22:06:20  thomas
 * mcmask is now the number of bits
 *
 * Revision 1.3  1995/04/24  23:41:08  thomas
 * FreeBSD flips the type in if_ethersubr.c
 *
 * Revision 1.2  1995/04/24  19:03:43  thomas
 * Fix EISA DE422 support.  Correct Copyright message.
 * Remove unused fields.
 *
 * Revision 1.1  1995/04/23  22:07:13  thomas
 * Initial revision
 *
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/malloc.h>

#if defined(__FreeBSD__)
#include <machine/cpufunc.h>
#elif defined(__bsdi__)
#include <machine/inline.h>
#endif

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/route.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <vm/vm.h>

#include "bpfilter.h"
#if NBPFILTER > 0
#include <net/bpf.h>
#endif

#include <i386/isa/decether.h>

void
decether_input(
    struct arpcom *ac,
    caddr_t seg1,
    size_t total_len,
    size_t len1,
    caddr_t seg2)
{
    struct ether_header eh;
    struct mbuf *m;

    if (total_len - sizeof(eh) > ETHERMTU
	    || total_len - sizeof(eh) < ETHERMIN) {
	ac->ac_if.if_ierrors++;
	return;
    }
    eh = *(struct ether_header *) seg1;
#if defined(__bsdi__)
    eh.ether_type = ntohs(eh.ether_type);
#endif

#if NBPFILTER > 0
    if (seg2 == NULL) {
	if (ac->ac_if.if_bpf != NULL)
	    bpf_tap(ac->ac_if.if_bpf, seg1, total_len);
#endif
	/*
	 * If this is single cast but not to us
	 * drop it!
	 */
	if ((eh.ether_dhost[0] & 1) == 0
		&& !DECETHER_ADDREQUAL(eh.ether_dhost, ac->ac_enaddr))
	    return;
#if NBPFILTER > 0
    }
#endif

    MGETHDR(m, M_DONTWAIT, MT_DATA);
    if (m == NULL) {
	ac->ac_if.if_ierrors++;
	return;
    }
    if (total_len + 2 > MHLEN) {
	MCLGET(m, M_DONTWAIT);
	if ((m->m_flags & M_EXT) == 0) {
	    m_free(m);
	    ac->ac_if.if_ierrors++;
	    return;
	}
    }
    m->m_data += 2;
    bcopy(seg1, mtod(m, caddr_t), len1);
    if (seg2 != NULL)
	bcopy(seg2, mtod(m, caddr_t) + len1, total_len - len1);
#if NBPFILTER > 0
    if (ac->ac_if.if_bpf != NULL && seg2 != NULL)
	bpf_tap(ac->ac_if.if_bpf, mtod(m, caddr_t), total_len);
    /*
     * If this is single cast but not to us
     * drop it!
     */
    if ((eh.ether_dhost[0] & 1) == 0
	   && !DECETHER_ADDREQUAL(eh.ether_dhost, ac->ac_enaddr)) {
	m_freem(m);
	return;
    }
#endif
    m->m_data += sizeof(eh);
    m->m_pkthdr.len = m->m_len = total_len - sizeof(eh);
    m->m_pkthdr.rcvif = &ac->ac_if;
    ether_input(&ac->ac_if, &eh, m);
}

int
decether_ifioctl(
    struct ifnet *ifp,
    int cmd,
    caddr_t data)
{
    struct arpcom *ac = (struct arpcom *) ifp;
    int s, error = 0;

    s = splimp();

    switch (cmd) {
	case SIOCSIFADDR: {
	    struct ifaddr *ifa = (struct ifaddr *)data;

	    ifp->if_flags |= IFF_UP;
	    switch(ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET: {
#ifndef __bsdi__
		    ac->ac_ipaddr = IA_SIN(ifa)->sin_addr;
#endif
		    (*ifp->if_init)(ifp->if_unit);
#if defined(__FreeBSD__) || defined(__bsdi__)
		    arp_ifinit(ac, ifa);
#endif
#ifndef __bsdi__
		    arpwhohas(ac, &IA_SIN(ifa)->sin_addr);
#endif
		    break;
		}
#endif /* INET */

#ifdef NS
		/* This magic copied from if_is.c; I don't use XNS,
		 * so I have no way of telling if this actually
		 * works or not.
		 */
		case AF_NS: {
		    struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);
		    if (ns_nullhost(*ina)) {
			ina->x_host = *(union ns_host *)(ac->ac_enaddr);
		    } else {
			ifp->if_flags &= ~IFF_RUNNING;
			bcopy((caddr_t)ina->x_host.c_host,
			      (caddr_t)ac->ac_enaddr,
			      sizeof(ac->ac_enaddr));
		    }

		    (*ifp->if_init)(ifp->if_unit);
		    break;
		}
#endif /* NS */

		default: {
		    (*ifp->if_init)(ifp->if_unit);
		    break;
		}
	    }
	    break;
	}

	case SIOCSIFFLAGS: {
	    (*ifp->if_init)(ifp->if_unit);
	    break;
	}

#if defined(MULTICAST) || defined(__FreeBSD__)
	case SIOCADDMULTI:
	case SIOCDELMULTI: {
	    /*
	     * Update multicast listeners
	     */
	    if (cmd == SIOCADDMULTI)
		error = ether_addmulti((struct ifreq *)data, ac);
	    else
		error = ether_delmulti((struct ifreq *)data, ac);

	    if (error == ENETRESET) {
		/* reset multicast filtering */
		(*ifp->if_init)(ifp->if_unit);
		error = 0;
	    }
	    break;
	}

#endif /* MULTICAST */

	default: {
	    error = EINVAL;
	}
    }

    splx(s);
    return error;
}

/*
 *  This is the standard method of reading the DEC Address ROMS.
 *  I don't understand it but it does work.
 */
int
decether_read_macaddr(
    unsigned char *hwaddr,
    int ioreg,
    int skippat)
{
    int cksum, rom_cksum;
    unsigned char addrbuf[6];
    
    if (!skippat) {
	int idx, idx2, found, octet;
	static u_char testpat[] = { 0xFF, 0, 0x55, 0xAA, 0xFF, 0, 0x55, 0xAA };
	idx2 = found = 0;
    
	for (idx = 0; idx < 32; idx++) {
	    octet = inb(ioreg);
	    
	    if (octet == testpat[idx2]) {
		if (++idx2 == sizeof(testpat)) {
		    ++found;
		    break;
		}
	    } else {
		idx2 = 0;
	    }
	}
	
	if (!found)
	    return -1;
    }

    if (hwaddr == NULL)
	hwaddr = addrbuf;

    cksum = 0;
    hwaddr[0] = inb(ioreg);
    hwaddr[1] = inb(ioreg);

    /* hardware adddress can't be multicast */
    if (hwaddr[0] & 1)
	return -1;

    cksum = *(u_short *) &hwaddr[0];

    hwaddr[2] = inb(ioreg);
    hwaddr[3] = inb(ioreg);
    cksum *= 2;
    if (cksum > 65535) cksum -= 65535;
    cksum += *(u_short *) &hwaddr[2];
    if (cksum > 65535) cksum -= 65535;

    hwaddr[4] = inb(ioreg);
    hwaddr[5] = inb(ioreg);
    cksum *= 2;
    if (cksum > 65535) cksum -= 65535;
    cksum += *(u_short *) &hwaddr[4];
    if (cksum >= 65535) cksum -= 65535;

    /* 00-00-00 is an illegal OUI */
    if (hwaddr[0] == 0 && hwaddr[1] == 0 && hwaddr[2] == 0)
	return -1;

    rom_cksum = inb(ioreg);
    rom_cksum |= inb(ioreg) << 8;
	
    if (cksum != rom_cksum)
	return -1;
    return 0;
}

static void
decether_multicast_op(
    unsigned short *mctbl,
    int mcbits,
    const u_char *mca,
    int enable)
{
    u_int idx, bit, data, crc = 0xFFFFFFFFUL;

#ifdef __alpha
    for (data = *(__unaligned u_long *) mca, bit = 0; bit < 48; bit++, data >>= 
1)
        crc = (crc >> 1) ^ (((crc ^ data) & 1) ? DECETHER_CRC32_POLY : 0);
#else
    for (idx = 0; idx < 6; idx++)
        for (data = *mca++, bit = 0; bit < 8; bit++, data >>= 1)
            crc = (crc >> 1) ^ (((crc ^ data) & 1) ? DECETHER_CRC32_POLY : 0);
#endif
    /*
     * The following two lines convert the N bit index into a longword index
     * and a longword mask.  
     */
    if (mcbits < 0) {
	mcbits = -mcbits;
	crc >>= (32 - mcbits);
    }
    crc &= (1 << mcbits) - 1;
    bit = 1 << (crc & 0x0F);
    idx = crc >> 4;

    /*
     * Set or clear hash filter bit in our table.
     */
    if (enable) {
	mctbl[idx] |= bit;		/* Set Bit */
    } else {
	mctbl[idx] &= ~bit;		/* Clear Bit */
    }
}

void
decether_multicast_filter(
    struct arpcom *ac,
    const int mcbits,
    unsigned short *mctbl)
{
#if defined(MULTICAST) || defined(__FreeBSD__)
    struct ether_multistep step;
    struct ether_multi *enm;
#endif
#ifdef ISO
    extern char all_es_snpa[];
#endif

    bzero(mctbl, mcbits / 8);

    if (ac->ac_if.if_flags & IFF_ALLMULTI)
	return;
    decether_multicast_op(mctbl, mcbits, etherbroadcastaddr, TRUE);
#ifdef ISO
    decether_multicast_op(mctbl, mcbits, all_es_snpa, TRUE);
#endif

#if defined(MULTICAST) || defined(__FreeBSD__)
    ETHER_FIRST_MULTI(step, ac, enm);
    while (enm != NULL) {
	if (!DECETHER_ADDREQUAL(enm->enm_addrlo, enm->enm_addrhi)) {
	    ac->ac_if.if_flags |= IFF_ALLMULTI;
	    return;
	}
	decether_multicast_op(mctbl, mcbits, enm->enm_addrlo, TRUE);
	ETHER_NEXT_MULTI(step, enm);
    }
#endif
}

