/*--------------------------------------------------------------*/
/* Cyclom-Y Device Driver for BSDI BSD/OS 2.x.			*/
/*								*/
/* Copyright (C) 1995 Cyclades Corporation. All rights reserved.*/
/*								*/
/* This code was modified by Marcio Saito from original code	*/
/* originally developed by Brian Litzinger.			*/
/*								*/
/* V_2.0.0 07/12/95	Marcio Saito	First Cyclades version	*/
/*					after we got the code	*/
/*					from Mediacity.		*/
/*--------------------------------------------------------------*/

#define	VERSION	"V_2.0.0 (07/12/95)"

/* #define CYDEBUG 1 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/kernel.h>
#include <sys/syslog.h>
#include <sys/device.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <machine/cpu.h>

#include "cyreg.h"
#include "ic/cl-cd1400.h"

#define POLLSLICE 100 /*ms*/


#if 0
#ifdef CYDEBUG
int CYDEBUG = CYDEBUG;


cydoutb(a, v)
	int a, v;
{

	dprintf(("outb(%x,%x)\n", a, v));
	outb(a, v);
}

cydinb(a)
	int a;
{
	int v = inb(a);

	dprintf(("inb(%x)->%x\n", a, v));
	return (v);
}

#define outb		cydoutb
#define inb		cydinb

#else	/* !CYDEBUG */

#define dprintf(x)	/* nothing */
#endif
#endif

#define dprintf(x)	/* */
#define UNIT(dev)	dv_unit(dev)
#define LINE(dev)	dv_subunit(dev)

#define MAX_CHAN 16

#define	CYIOC_NFL	0x4381
#define	CYIOC_RFL	0x4382

struct cy_softc {
	struct	device cy_dev;		/* base device */
	struct	isadev cy_id;		/* ISA device */
	struct  intrhand cy_ih;		/* interrupt vectoring */
	struct	ttydevice_tmp cy_ttydev; /* tty stuff */
	struct	tty cy_tty[MAX_CHAN];   /* Per-channel tty structures */
	short	cy_softdtr;		/* software copy of DTR */
	short	cy_txint;		/* TX interrupt is in progress */
	short   cy_inintr;		/* Interrupt in progess */
	short   cy_pollactive;          /* Polling active */
	caddr_t	cy_addr[MAX_CHAN];	/* base i/o address */
	char	cy_init[MAX_CHAN];	/* line has been inited since reset */
	char	cy_cmd[MAX_CHAN];	/* command bytes per channels */
	char	cy_pendesc[MAX_CHAN];   /* pending escapes */
	u_int	cy_orun[MAX_CHAN];	/* overruns */
        unsigned char cy_srer[MAX_CHAN];
        caddr_t cy_base;
        int     cy_NbrCD1400s;
        int     cy_NbrPorts;
};

/*
 * cymctl commands
 */
enum cymctl_cmds { GET, SET, BIS, BIC };

int cyprobe __P((struct device *, struct cfdata *, void *));
void cyattach __P((struct device *, struct device *, void *));
int cyopen __P((dev_t, int, int, struct proc *));
int cyclose __P((dev_t, int, int, struct proc *));
int cyread __P((dev_t, struct uio *, int));
int cywrite __P((dev_t, struct uio *, int));
int cyintr __P((struct cy_softc *));
void cystart __P((struct tty *));
int cyioctl __P((dev_t, int, caddr_t, int, struct proc *));
void cychancmd __P((caddr_t, int));
int cyparam __P((struct tty *, struct termios *));
int cyselect __P((dev_t, int, struct proc *));
void cystop __P((struct tty *, int));
static void cycd1400init __P((int));
static void cyforceintr __P((void *));
static int cymctl __P((dev_t, enum cymctl_cmds, int));
static int cyspeed __P((long, int *));
static void cypoll();

struct cfdriver cycd = {
	NULL, "cy", cyprobe, cyattach, DV_TTY, sizeof(struct cy_softc)
};

struct devsw cysw = {
	&cycd,
	cyopen, cyclose, cyread, cywrite, cyioctl, cyselect, nommap,
	nostrat, nodump, nopsize, 0,
	cystop
};

static int cy_rflow[MAX_CHAN] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

/*
 * Probe routine
 */
cyprobe(parent, cf, aux)
    struct device *parent;
    struct cfdata *cf;
    void *aux;
{
    register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
    int i,j;
    caddr_t b;
    int flags;
    int ports;
    int NbrCD1400s;
    unsigned char version;


    flags = cf->cf_flags;

    NbrCD1400s = (flags&0x07)+1;
    ports = NbrCD1400s*4;

    b = ISA_HOLE_VADDR(ia->ia_maddr);

    rinb(b,cy_RESET_16);
    delay(500);

    for (i=0; i<NbrCD1400s; i++) {
        caddr_t base = b + i * CD1400_MEMSIZE;

        /* wait for chip to become ready for new command */
        for (j = 0; j < 500; j += 50) {
            delay(50);  /* wait 50 us */
            if (!rinb(base,CD1400_CCR)) {
                break;
            }
        }

        /* clear the GFRCR register */
        routb(base,CD1400_GFRCR,0);

        /* issue a reset command */
        routb(base,CD1400_CCR,CCR_HARDRESET);

        /* wait for the CD1400 to initialise itself */
        for (j = 0; j < 10000; j += 50) {
            delay(50);  /* wait 50 us */

            /* retrieve firmware version */
            version = rinb(base,CD1400_GFRCR);
            if (version)
                break;
        }
        /* anything in the 40-4f range is fine */
        if ((version & 0xf0) != 0x40) {
            printf("Cyclom-Y %s: Warning: Cyclom-Y not found.\n",VERSION);
            return 0;
        }
    }
    if (ia->ia_irq == IRQUNK) {
            printf("Cyclom-Y %s: Warning: No intr from Cyclom-Y.\n",VERSION);
	    return (0);
    }
    printf("Cyclom-Y %s: Cyclom-Y board ready.\n",VERSION);
    return (1);
}

/*
 * Attach routine
 */
void 
cyattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	register struct cy_softc *sc = (struct cy_softc *) self;
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	register caddr_t base;
	caddr_t b;
        int flags;
        int i,j,k;


        flags = sc->cy_dev.dv_flags;
        sc->cy_NbrCD1400s = (flags&0x07)+1;
        sc->cy_NbrPorts   = sc->cy_NbrCD1400s * 4;
        printf(" ports %d",sc->cy_NbrPorts);


	sc->cy_base = b = ISA_HOLE_VADDR(ia->ia_maddr);

	for (i=0,k=0;i<sc->cy_NbrCD1400s;i++) {
	    base = b + i * CD1400_MEMSIZE;
	    routb(base, CD1400_PPR, CD1400_CLOCK_25_1MS);
	    for (j=0;j<CD1400_NO_OF_CHANNELS;j++,k++) {
		sc->cy_addr[k] = base;
		/* cy_channel_init */
		routb(base, CD1400_CAR, j&0x03);
		cychancmd(base,CCR_SOFTRESET);
		routb(base, CD1400_LIVR, 0);
	    }
	}

	routb(b,cy_CLEAR_INTR,0);	/* Clear interrupts */

	/* Initialize interrupt/id structures */
	isa_establish(&sc->cy_id, &sc->cy_dev);
	sc->cy_ih.ih_fun = cyintr;
	sc->cy_ih.ih_arg = (void *) sc;
	intr_establish(ia->ia_irq, &sc->cy_ih, DV_TTY);

	strcpy(sc->cy_ttydev.tty_name, cycd.cd_name);
	sc->cy_ttydev.tty_unit = sc->cy_dev.dv_unit;
	sc->cy_ttydev.tty_base = sc->cy_dev.dv_unit * 16;
	sc->cy_ttydev.tty_count = sc->cy_NbrPorts;
	sc->cy_ttydev.tty_ttys = sc->cy_tty;
	tty_attach(&sc->cy_ttydev);
	sc->cy_inintr = 0;
	sc->cy_pollactive = 1;
#ifdef cy_STATS
        sc->cy_statclk = 1000;	/* 1000ms = 1 sec */
#endif
	timeout(cypoll,(caddr_t)sc,POLLSLICE);
	printf("\n");
}

/*
 * Open line
 */
cyopen(dev, flag, mode, p)
	dev_t dev;
	int flag;
	int mode;
	struct proc *p;
{
	register struct tty *tp;
	int s;
	int error;
	register caddr_t base;
	int unit, chan;
	struct cy_softc *sc;


	unit = UNIT(dev);
	if (unit >= cycd.cd_ndevs || (sc = cycd.cd_devs[unit]) == 0)
		return (ENXIO);
	chan = LINE(dev);
	if (chan >= sc->cy_NbrPorts)
		return (ENXIO);
	tp = &sc->cy_tty[chan];
	base = sc->cy_addr[chan];
	dprintf(("cyopen: base %lx, chan %d\n",base,chan));
	tp->t_oproc = cystart;
	tp->t_param = cyparam;
	tp->t_dev = dev;
	if ((tp->t_state & TS_ISOPEN) == 0) {
		tp->t_state |= TS_WOPEN;
		if (tp->t_ispeed == 0)
			tp->t_termios = deftermios;
		else
			ttychars(tp);
		cyparam(tp, &tp->t_termios);
		ttsetwater(tp);
	} else if ((tp->t_state & TS_XCLUDE) && p->p_ucred->cr_uid != 0)
		return (EBUSY);

	error = 0;
	s = spltty();
	(void) cymctl(dev, SET, TIOCM_DTR|TIOCM_RTS);

	routb(base, CD1400_CAR, chan&0x03);
	if (rinb(base, CD1400_MSVR2) & MSVR2_CD) {
		dprintf(("CD PRESENT\n"));
		tp->t_state |= TS_CARR_ON;
	}
	if (!(flag & O_NONBLOCK)) {
		while (!(tp->t_cflag & CLOCAL) && !(tp->t_state & TS_CARR_ON)) {
			tp->t_state |= TS_WOPEN;
			error = ttysleep(tp, (caddr_t)&tp->t_rawq,
			    TTIPRI | PCATCH, ttopen, 0);
			if (error) {
				/*
				 * Disable line and drop DTR.
				 * Note, this is wrong if another open might
				 * be in progress.
				 */
#if 1
				/*cychancmd(base, CCR_TXDIS | CCR_RXDIS);*/
				sc->cy_init[chan] = 0;
				(void) cymctl(dev, SET, 0);
#endif
				break;
			}
		}
	}
	splx(s);
	if (error == 0)
		error = (*linesw[tp->t_line].l_open)(dev, tp);
#ifdef cy_STATS
	sc->stats_ints_xmit[chan] = 0;
	sc->stats_ints_recv[chan] = 0;
	sc->stats_ints_mdm [chan] = 0;
#endif
	return (error);
}

/*
 * Close line
 */
cyclose(dev, flag, mode, p)
	dev_t dev;
	int flag;
	int mode;
	struct proc *p;
{
	struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];
	int chan = LINE(dev);
	register struct tty *tp = &sc->cy_tty[chan];
	register caddr_t base = sc->cy_addr[chan];
	int s;

	dprintf(("cyclose:\n"));

	(*linesw[tp->t_line].l_close)(tp, flag);

	/* Disable line */
	s = spltty();
	routb(base, CD1400_CAR, chan&0x03);
	cychancmd(base, CCR_TXDIS | CCR_RXDIS);
	sc->cy_init[chan] = 0;
	routb(base, CD1400_SRER, sc->cy_srer[chan]=0);
	splx(s);

	/* Hang up */
	if ((tp->t_cflag & HUPCL) || (tp->t_state & TS_WOPEN) ||
	    (tp->t_state & TS_ISOPEN) == 0)
	    (void) cymctl(dev, SET, 0);

	ttyclose(tp);
	if (sc->cy_orun[chan]) {
		log(LOG_ERR, "cy%d line %d: %d overruns\n", UNIT(dev), chan,
		    sc->cy_orun[chan]);
		sc->cy_orun[chan] = 0;
	}
#ifdef cy_STATS
	{
		int i;
		printf("cy%d:%d\n\t", UNIT(dev), LINE(dev));
		for (i = 0; i < 3; i++)
			printf("I%d=%x ", i, int_cnt[minor(dev)][i]);
		for (i = 0; i < 2; i++)
			printf("L%d=%x ", i, loops[UNIT(dev)][i]);
		printf("\n");
	}
#endif
	return (0);
}

/*
 * Read from line
 */
cyread(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];
	struct tty *tp = &sc->cy_tty[LINE(dev)];

	dprintf(("cyread:\n"));
	return ((*linesw[tp->t_line].l_read)(tp, uio, flag));
}

/*
 * Write to line
 */
cywrite(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];
	struct tty *tp = &sc->cy_tty[LINE(dev)];

	dprintf(("cywrite:\n"));
	return ((*linesw[tp->t_line].l_write)(tp, uio, flag));
}

cyselect(dev, flag, p)
	dev_t dev;
	int flag;
	struct proc *p;
{
	struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];

	return (ttyselect(&sc->cy_tty[LINE(dev)], flag, p));
}

static void
cypoll(sc)
    struct cy_softc *sc;
{
    register struct tty *tp = &sc->cy_tty[1];
    register caddr_t base = sc->cy_addr[1];
    register int chan;
    int s;

    s = spltty();
#if 0
    printf("cypoll: cy_inintr=%d ",sc->cy_inintr);
    printf("%s ", tp->t_state&TS_BUSY?"TS_BUSY":"!TS_BUSY");
    printf("t_outq=%d ",tp->t_outq.c_cc);
    printf("SRER[1] = %x ",sc->cy_srer[1]);
    printf("SVRR=%x ",rinb(base,CD1400_SVRR)&0xff);
    printf("\n");
#endif
    if (sc->cy_inintr==0) {
	for (chan=0;chan<sc->cy_NbrPorts;chan++) {
	    tp   = &sc->cy_tty[chan];

	    if (tp->t_state&TS_BUSY /*&& (sc->cy_srer[chan]&SRER_TXD)==0*/) {
		base = sc->cy_addr[chan];
		dprintf(("prodding %d\n",chan));
		routb(base, CD1400_CAR, chan&0x03);
		routb(base, CD1400_SRER, sc->cy_srer[chan]=SRER_MDM|SRER_RXD|SRER_TXD);
	    }
	}
    }
#ifdef cy_STATS
    sc->cy_statclk -= POLLSLICE;
    if (sc->cy_statclk<=0) {

       sc->cy_statclk = 1000;	/* 1000ms = 1 sec */
    }
#endif
    splx(s);
    if (sc->cy_pollactive)
	timeout(cypoll,(caddr_t)sc,POLLSLICE);
}

/*
 * Interrupt routine
 */
cyintr(sc)
	register struct cy_softc *sc;
{
	register struct tty *tp;
	register b, c;
	register unsigned cnt;
	register chan;
	register caddr_t base;
	int cd;
	int domore;

    sc->cy_inintr = 1;
    dprintf(("cyintr:\n"));
  more:
    domore = 0;
    for (cd=0;cd<sc->cy_NbrCD1400s;cd++) {
	base = sc->cy_addr[cd*4]; 	/* WARNING */

	dprintf(("*I%d*\n", sc->cy_dev.dv_unit));
	if (base == 0) {
		printf("cy%d: bogus interrupt\n", sc->cy_dev.dv_unit);
		sc->cy_inintr = 0;
		return (1);
	}

	/* Read Board Status Register */
	dprintf(("reading from %lx\n",(unsigned long)base));
	while (b=rinb(base,CD1400_SVRR)) {
	    domore = 1;
	    /*
	     * Need to add some code to allow return if this card is hogging
	     */

	    /* Receiver interrupt */
	    if (b & CD1400_SVRR_RX) {
		unsigned char save_rir = rinb(base,CD1400_RIR);
		int           chan     = cd*CD1400_NO_OF_CHANNELS + (save_rir&0x03);
		unsigned char save_car = rinb(base,CD1400_CAR);
		routb(base,CD1400_CAR,save_rir);

		tp = &sc->cy_tty[chan];
#ifdef cy_STATS
		sc->stats_ints_recv[chan]++;
#endif

		switch(rinb(base,CD1400_RIVR)&0x07) {

		    case 3:
			/* Get the count of received characters */
			cnt = rinb(base, CD1400_RDCR);

			/* If the line wasn't opened, throw data into bit bucket */
			if ((tp->t_state & TS_ISOPEN) == 0) {
			    while (cnt--)
				(void) rinb(base, CD1400_RDSR);
			    goto rxout;
			}

			while (cnt--) {
			    c = ((rinb(base, CD1400_RDSR))&0xff);
			    (*linesw[tp->t_line].l_rint)(c, tp);
			}
		    break;

		    case 7:
			(void) rinb(base,CD1400_RDSR); /* Get status */
			(void) rinb(base,CD1400_RDSR); /* Get bad data */
		    break;
		}

	      rxout:
		routb(base,CD1400_RIR,save_rir&0x3f);
		routb(base,CD1400_CAR,save_car);
	    }



	    /* TX interrupt? */
	    if (b & CD1400_SVRR_TX) {
		unsigned char save_tir = rinb(base,CD1400_TIR);
		int           chan     = cd*CD1400_NO_OF_CHANNELS + (save_tir&0x03);
		unsigned char save_car = rinb(base,CD1400_CAR);
		routb(base,CD1400_CAR,save_tir);

		tp = &sc->cy_tty[chan];
#ifdef cy_STATS
		sc->stats_ints_xmit[chan]++;
#endif

		/* (Re-)start transmit */
		if (tp->t_state & TS_FLUSH) {
		    tp->t_state &= ~(TS_BUSY|TS_FLUSH);
		    /* Disable TX interrupts */
		    routb(base, CD1400_SRER, sc->cy_srer[chan]=SRER_MDM|SRER_RXD);
		} else {
		    tp->t_state &= ~TS_BUSY;
		    sc->cy_txint = 1;
		    (*linesw[tp->t_line].l_start)(tp);
		    sc->cy_txint = 0;
		    /* If nothing to send, disable TX interrupts */
		    if ((tp->t_state&TS_BUSY) == 0)	
			routb(base, CD1400_SRER, sc->cy_srer[chan]=SRER_MDM|SRER_RXD);

		}
		routb(base, CD1400_TIR,(save_tir & 0x3f));
		routb(base, CD1400_CAR,save_car);

	    }
	    /* goto out */

	    /* Modem Ctl interrupt? */
	    if (b & CD1400_SVRR_MDM) {
		unsigned char save_mir = rinb(base,CD1400_MIR);
		int           chan = cd * CD1400_NO_OF_CHANNELS + (save_mir & 0x3);
		unsigned char save_car = rinb(base,CD1400_CAR);
		routb(base,CD1400_CAR,save_mir);

		tp = &sc->cy_tty[chan];
#ifdef cy_STATS
		sc->stats_ints_mdm[chan]++;
#endif
		
		if ((rinb(base, CD1400_MISR) & MISR_CDCHG)) {
			/* Get the value of CD */
			if (rinb(base,CD1400_MSVR2) & MSVR2_CD)
				(void) (*linesw[tp->t_line].l_modem)(tp, 1);	
			else if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
			    (void) cymctl(tp->t_dev, SET, 0);

		   /* WARNING: May need to clear change bits here */
		}
		routb(base,CD1400_MIR,save_mir&0x3f);
		routb(base,CD1400_CAR,save_car);

	    }
	}
    }
    if (domore) goto more;
    sc->cy_inintr = 0;
    routb(sc->cy_base, cy_CLEAR_INTR, 0);
    return (1);
}

/*
 * Start transmission
 */
void
cystart(tp)
	register struct tty *tp;
{
	register caddr_t base;
	register c, count;
	int s, chan;
	register struct cy_softc *sc;

	dprintf(("cystart:\n"));
	/*
	 * Check if there is work to do and we're able to do more.
	 */
	s = spltty();
	if (tp->t_state & TS_BUSY ||
	    (tp->t_state & (TS_TIMEOUT|TS_TTSTOP) &&
	    (tp->t_state & (TS_XON_PEND|TS_XOFF_PEND)) == 0)) {
		dprintf(("leaving early\n"));
		goto out;
	}

	if (tp->t_outq.c_cc <= tp->t_lowat)
		ttyowake(tp);

	sc = cycd.cd_devs[UNIT(tp->t_dev)];
	chan =  LINE(tp->t_dev);
	base = sc->cy_addr[chan];

	/*
	 * If not in interrupt context, TDR is not available.
	 * Simply enable TX interrupt if there is output to be done.
	 */
	if (sc->cy_txint == 0) {
		if (tp->t_outq.c_cc ||
		    tp->t_state & (TS_XON_PEND|TS_XOFF_PEND) ||
		    sc->cy_cmd[chan] || sc->cy_pendesc[chan]) {
			tp->t_state |= TS_BUSY;
			routb(base, CD1400_CAR, chan&0x03);
		        routb(base, CD1400_SRER, sc->cy_srer[chan]=SRER_MDM|SRER_RXD|SRER_TXD);
		}
		dprintf(("leaving with TXD ints enabled\n"));
		goto out;
	}

	/*
	 * Process pending commands
	 */
	count = CD1400_MAX_FIFO_SIZE;
	if (c = sc->cy_cmd[chan]) {
		sc->cy_cmd[chan] = 0;
		routb(base, CD1400_TDR, CD1400_C_ESC);
		routb(base, CD1400_TDR, c);
		count -= 2;
	}
	if (sc->cy_pendesc[chan]) {
		sc->cy_pendesc[chan] = 0;
		routb(base, CD1400_TDR, CD1400_C_ESC);
		count--;
	}
	

	if (tp->t_state & (TS_XON_PEND|TS_XOFF_PEND)) {
		if (tp->t_state & TS_XON_PEND) {
			routb(base, CD1400_TDR, tp->t_cc[VSTART]);
			tp->t_state &= ~TS_XON_PEND;
		} else {
			routb(base, CD1400_TDR, tp->t_cc[VSTOP]);
			tp->t_state &= ~TS_XOFF_PEND;
		}
		if (tp->t_state & (TS_TIMEOUT|TS_TTSTOP))
			count = 0;
		else
			count--;
	}

	/*
	 * Run regular output queue
	 */
	while (tp->t_outq.c_cc && count--) {
		c = getc(&tp->t_outq);
		if (c == CD1400_C_ESC) {
			if (count == 0)		/* oops */
				sc->cy_pendesc[chan]++;
			else {
				routb(base, CD1400_TDR, CD1400_C_ESC);
				count--;
			}
		}
	        dprintf(("txd: %c (%x)\n",(c<0x20||c>0x7e)?' ':c,c));
		routb(base, CD1400_TDR, c);
	}
        if (count<CD1400_MAX_FIFO_SIZE /*tp->t_outq.c_cc*/)
	    tp->t_state |= TS_BUSY;

out:
	splx(s);
}

/*
 * Ioctl routine
 */
cyioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
        struct proc *p;
{
	register struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];
	register struct tty *tp = &sc->cy_tty[LINE(dev)];
	register int error;
	int s;
 
	dprintf(("cyioctl:\n"));

	if (cmd == CYIOC_RFL) {
		cy_rflow[UNIT(dev)] = 1;
		return(0);
	}
	if (cmd == CYIOC_NFL) {
		cy_rflow[UNIT(dev)] = 0;
		return(0);
	}

	error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, data, flag, p);
	if (error >= 0)
		return (error);
	error = ttioctl(tp, cmd, data, flag, p);
	if (error >= 0)
		return (error);

	s = spltty();
	switch (cmd) {
	case TIOCSBRK:
		/* Start sending BREAK */
		sc->cy_cmd[LINE(tp->t_dev)] = CD1400_C_SBRK;
		cystart(tp);
		break;

	case TIOCCBRK:
		/* Stop sending BREAK */
		sc->cy_cmd[LINE(tp->t_dev)] = CD1400_C_EBRK;
		cystart(tp);
		break;

	case TIOCSDTR:			/* set DTR */
		(void) cymctl(dev, BIS, TIOCM_DTR);
		break;

	case TIOCCDTR:			/* clear DTR */
		(void) cymctl(dev, BIC, TIOCM_DTR);
		break;

	case TIOCMSET:
		(void) cymctl(dev, SET, * (int *) data);
		break;

	case TIOCMBIS:
		(void) cymctl(dev, BIS, * (int *) data);
		break;

	case TIOCMBIC:
		(void) cymctl(dev, BIC, * (int *) data);
		break;

	case TIOCMGET:
		* (int *) data = cymctl(dev, GET, 0);
		break;

	default:
		splx(s);
		return (ENOTTY);
	}
	splx(s);
	return (0);
}

int cy_fifothresh_lo = 8;	/* FIFO depth, half of FIFO, < 38.4k */
int cy_fifothresh_hi = 4;	/* FIFO depth, >= 38.4k */

int cy_doenable = 0;	/* should not be needed, defeats optimization if set */
/*
 * Set parameters and enable the line
 */
cyparam(tp, t)
	register struct tty *tp;
	register struct termios *t;
{
	int s, chan;
	register struct cy_softc *sc;
	register caddr_t base;
	register c;
	int iprescaler, oprescaler;
	int ispeed, ospeed;
	int cflag = t->c_cflag;
	int iflag = t->c_iflag;

	dprintf(("cyparam:\n"));
	/* short-circuit the common case where there is no hardware change */

	if (tp->t_cflag == t->c_cflag && tp->t_state&TS_ISOPEN &&
	    tp->t_iflag == t->c_iflag &&
	    tp->t_ispeed == t->c_ispeed && tp->t_ospeed == t->c_ospeed)
	    	return (0);

	if ((tp->t_cflag & CLOCAL) == 0 && t->c_cflag & CLOCAL)
		wakeup((caddr_t) &tp->t_rawq);

	tp->t_ispeed = t->c_ispeed;
	tp->t_ospeed = t->c_ospeed;
	tp->t_cflag = t->c_cflag;
	tp->t_iflag = t->c_iflag;

	/* Select line */
	sc = cycd.cd_devs[UNIT(tp->t_dev)];
	chan = LINE(tp->t_dev);
	base = sc->cy_addr[chan];
	s = spltty();
	routb(base, CD1400_CAR, chan&0x03);

	/* ospeed == 0 is for HANGUP */
	if (tp->t_ospeed == 0) {
		(void) cymctl(tp->t_dev, SET, 0);
		cychancmd(base, CCR_TXDIS | CCR_RXDIS);
		sc->cy_init[chan] = 0;
		splx(s);
		return (0);
	}

        if ((ospeed = cyspeed(t->c_ospeed, &oprescaler)) < 0) {
            splx(s);
            return(EINVAL);
        }
        routb(base,CD1400_TBPR,ospeed);
        routb(base,CD1400_TCOR,oprescaler);

        if ((ispeed = cyspeed(t->c_ispeed, &iprescaler)) < 0) {
            splx(s);
            return(EINVAL);
        }
        routb(base,CD1400_RBPR,ispeed);
        routb(base,CD1400_RCOR,iprescaler);
	
	/* Load COR1 */
	switch (tp->t_cflag & CSIZE) {
	case CS5:
		c = COR1_5BITS;
		break;
	case CS6:
		c = COR1_6BITS;
		break;
	case CS7:
		c = COR1_7BITS;
		break;
	case CS8:
		c = COR1_8BITS;
		break;
	}
#if 0
	printf("%s ",tp->t_cflag & CSTOPB ? "CSTOPB":"!CSTOPB");
	printf("%s ",tp->t_cflag & PARENB ? "PARENB":"!PARENB");
	printf("%s ",tp->t_cflag & PARODD ? "PARODD":"!PARODD");
	printf("%s ",tp->t_cflag & PARODD ? "PARODD":"!PARODD");
#endif
	if (tp->t_cflag & CSTOPB)
		c |= COR1_2SB;
	if (tp->t_cflag & PARENB) {
		c |= COR1_NORMPAR;
		if (tp->t_cflag & PARODD)
			c |= COR1_ODDP;
#if 0
		if ((tp->t_iflag & INPCK) == 0)
			c |= COR1_IGNORE;
#endif
	} else
		c |= COR1_IGNORE;
	routb(base, CD1400_COR1, c);
	dprintf(("cor1=%x\n",c));

	/* Load COR2 */
	c = COR2_ETC;
	if (tp->t_cflag & CCTS_OFLOW)
		c |= COR2_CTSAE;
#ifdef wrong
	/*
	 * COR2_RTSAO enables traditional RTS (high when there is something
	 * to transmit), not RTR (high when ready to receive).
	 */
	if (tp->t_cflag & CRTS_IFLOW)
		c |= COR2_RTSAO;
#endif

	/* there should be some logic to enable on-chip Xon/Xoff flow ctl */
	routb(base, CD1400_COR2, c);
	dprintf(("cor2=%x\n",c));

	/* Load COR3 */
	if (tp->t_ispeed >= 38400)
		routb(base, CD1400_COR3, cy_fifothresh_hi);	/* FIFO depth */
	else
		routb(base, CD1400_COR3, cy_fifothresh_lo);	/* FIFO depth */
	/* set the Receive Timeout Period to 20ms */
	routb(base, CD1400_RTPR, 20);

	/* Inform CD1400 engine about new values in COR registers */
	cychancmd(base, CCR_CORCHG1 | CCR_CORCHG2 | CCR_CORCHG3);
	DELAY(500);
	
        /* Load COR4 */
        c = 0;
        if (iflag&IGNCR)
	    c |= COR4_IGNCR;
	if (iflag&IGNBRK)
	    c |= COR4_IGNBRK;
	if (!(iflag&BRKINT))
	    c |= COR4_NBRKINT;
	if (iflag&IGNPAR)
	    c |= COR4_PFODISC;
	else {
	    if (iflag&PARMRK)
		c |= COR4_PFOMARK;
	    else
		c |= COR4_PFONULL;
	}
	routb(base,CD1400_COR4,c);
	dprintf(("cor4=%x\n",c));

        /* Load COR5 */
        c = 0;
        if (iflag&ISTRIP)
	    c |= COR5_ISTRIP;
	if (t->c_iflag&IEXTEN)
	    c |= COR5_IEXTEN;
	routb(base,CD1400_COR5,c);
	dprintf(("cor5=%x\n",c));


        c = 0;
	if (tp->t_cflag & CRTS_IFLOW) {
	    c |= 8;
	}
	if (cy_rflow[chan] != 0) {
		routb(base, CD1400_MCOR1, MCOR1_CDZD|c);
	} else {
		routb(base, CD1400_MCOR1, MCOR1_CDZD);
	}

	dprintf(("cyparam:1\n"));
	if (sc->cy_init[chan] == 0 || cy_doenable) {
		sc->cy_init[chan] = 1;
		/* Load modem control parameters */
		routb(base, CD1400_MCOR2, MCOR2_CDOD);

		/* Finally enable transmitter and receiver */
		cychancmd(base, CCR_TXEN | CCR_RXEN);
		routb(base, CD1400_SRER, sc->cy_srer[chan]=SRER_MDM|SRER_RXD); /* WARNING */
		dprintf(("cyparam:2\n"));
	}
	splx(s);
	dprintf(("cyparam:d\n"));
	return (0);
}

/*
 * Write a command to the Channel Command Register,
 * making sure it is not busy before writing the command.
 * The channel must already have been selected.
 */
void
cychancmd(base, cmd)
	caddr_t base;
	int cmd;
{
	int i;

	for (i = 0; i < 100; i++) {
		if (rinb(base, CD1400_CCR) == 0)
			goto ready;
		DELAY(100);
	}
	printf("cy: ccr not ready\n");
ready:
	routb(base, CD1400_CCR, cmd);
}

/*
 * Stop output on a line
 */
/*ARGSUSED*/
void
cystop(tp, flag)
	register struct tty *tp;
	int flag;
{
	int s;

	s = spltty();
	if (tp->t_state & TS_BUSY) {
		if ((tp->t_state & TS_TTSTOP) == 0)
			tp->t_state |= TS_FLUSH;
	}
	splx(s);
}

/*
 * Modem control routine.
 */
static int
cymctl(dev, cmd, bits)
	dev_t dev;
	enum cymctl_cmds cmd;
	int bits;
{
	register struct cy_softc *sc = cycd.cd_devs[UNIT(dev)];
	register line = LINE(dev);
	register caddr_t base = sc->cy_addr[line];
	register msvr;

	dprintf(("cymctl%x: cmd=%d bits=%x base=%lx\n",
	         minor(dev), cmd, bits, (unsigned long)base));

	routb(base, CD1400_CAR, line&0x03);

	switch (cmd) {
	case GET:
		msvr = rinb(base, CD1400_MSVR2);
		bits = TIOCM_LE;
		if (msvr & MSVR2_DTR)
			if(cy_rflow[UNIT(dev)] != 0) {
			    bits |= TIOCM_RTS;
			} else {
			    bits |= TIOCM_DTR;
			}
		if (msvr & MSVR2_CTS)
			bits |= TIOCM_CTS;
		if (msvr & MSVR2_DSR)
			bits |= TIOCM_DSR;
		if (msvr & MSVR2_CD)
			bits |= TIOCM_CAR;
		if (msvr & MSVR2_RI)
			bits |= TIOCM_RI;
		msvr = rinb(base, CD1400_MSVR1);
		if (msvr & MSVR1_RTS)
			if(cy_rflow[UNIT(dev)] != 0) {
			    bits |= TIOCM_DTR;
			} else {
			    bits |= TIOCM_RTS;
			}
		return (bits);

	case SET:
		if (bits&TIOCM_RTS) {
		    if (cy_rflow[UNIT(dev)] != 0) {
			routb(base, CD1400_MSVR2,MSVR2_DTR);
		    } else {
			routb(base, CD1400_MSVR1,MSVR1_RTS);
		    }
		} else {
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR2,0x00); /* lower DTR */
		    } else {
		        routb(base, CD1400_MSVR1,0x00); /* lower RTS */
		    }
		}
		if (bits&TIOCM_DTR) {
		    sc->cy_softdtr |= 1 << line;
		    if (cy_rflow[UNIT(dev)] != 0) {
			routb(base, CD1400_MSVR1,MSVR1_RTS);
		    } else {
			routb(base, CD1400_MSVR2,MSVR2_DTR);
		    }
		} else {
		    sc->cy_softdtr &= ~(1 << line);
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR1,0x00); /* lower RTS */
		    } else {
		        routb(base, CD1400_MSVR2,0x00); /* lower DTR */
		    }
		}
		break;

	case BIS:
		if (bits & TIOCM_RTS) {
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR1,MSVR2_DTR);
		    } else {
		        routb(base, CD1400_MSVR2,MSVR1_RTS);
		    }
		}
		if (bits & TIOCM_DTR) {
		    sc->cy_softdtr |= 1 << line;
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR1,MSVR1_RTS);
		    } else {
		        routb(base, CD1400_MSVR2,MSVR2_DTR);
		    }
		}
		 
		break;

	case BIC:
		if (bits & TIOCM_RTS) {
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR2,0x00); /* lower DTR */
		    } else {
		        routb(base, CD1400_MSVR1,0x00); /* lower RTS */
		    }
		}
		if (bits & TIOCM_DTR) {
		    sc->cy_softdtr &= ~(1 << line);
		    if (cy_rflow[UNIT(dev)] != 0) {
		        routb(base, CD1400_MSVR1,0x00); /* lower RTS */
		    } else {
		        routb(base, CD1400_MSVR2,0x00); /* lower DTR */
		    }
		}
		break;
	}

	/* Enable/disable receiver on open/close */
	if (cmd == SET) {
		routb(base, CD1400_CAR, line&0x03);
		if (bits == 0) {
		    cychancmd(base, CCR_RXDIS);
		} else {
		    cychancmd(base, CCR_RXEN);
		}
	}
	return (0);
}

static int
cyspeed(speed, prescaler_io)
    long speed;
    int *prescaler_io;
{
    int         actual;
    int         error;
    int         divider;
    int         prescaler;
    int         prescaler_unit;

    if (speed == 0)
        return 0;

    if (speed < 0 || speed > 150000)
        return -1;

    /* determine which prescaler to use */
    for (prescaler_unit = 4, prescaler = 2048; prescaler_unit;
            prescaler_unit--, prescaler >>= 2) {
        if (CYCLOM_CLOCK/prescaler/speed > 63)
            break;
    }

    divider = (CYCLOM_CLOCK/prescaler*2/speed + 1)/2;   /* round off */
    if (divider > 255)
        divider = 255;
    actual = CYCLOM_CLOCK/prescaler/divider;
    error = ((actual-speed)*2000/speed +1)/2;   /* percentage */

    /* 3.0% max error tolerance */
    if (error < -30 || error > 30)
        return -1;

#if 0
    printf("speed = %ld\n",speed);
    printf("prescaler = %d (%d)\n", prescaler, prescaler_unit);
    printf("divider = %d (%x)\n", divider, divider);
    printf("actual = %d\n", actual);
    printf("error = %d\n", error);
#endif

    *prescaler_io = prescaler_unit;
    return divider;
} /* end of cyspeed() */
