/*-
 * Copyright (c) 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: bha.c,v 2.3 1995/11/13 16:17:04 cp Exp $
 */

/*
 * Buslogic (AHA-compatible hardware) host adapter driver.
 *
 * This variant make use of the Buslogic 32-bit mode, avoiding
 * DMA bounce buffers.
 *
 * TODO:
 *	Compute CCB's when buffers are queued
 *	Add support for multiple outstanding commands on the same target
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>
#include <i386/isa/ahareg.h>
#include <i386/isa/abvar.h>

/*
 * Having more than one or two mboxes is useful mainly for debugging.
 * These sizes are calculated to fit inside 2 KB.
 */
#define	NMBOX	8
#define	NSCCB	12
#define	MAXSG	17

/*
 * See abvar.h for common variables.  Note, this MUST begin with the
 * ab_softc.
 */
struct bha_softc {
	struct	ab_softc sc_ab;
#define	sc_hba		sc_ab.ab_hba

	/* in and out mailbox info */
#define	mbobase(sc)	((struct mbox32 *)(sc)->sc_ab.ab_mbobase)
#define	mbolim(sc)	((struct mbox32 *)(sc)->sc_ab.ab_mbolim)
#define	mbibase(sc)	((struct mbox32 *)(sc)->sc_ab.ab_mbibase)
#define	mbilim(sc)	((struct mbox32 *)(sc)->sc_ab.ab_mbilim)
	struct	mbox32 *sc_outbox;	/* next available out mbox */
	struct	mbox32 *sc_inbox;	/* next expected in mbox */

	/* command control block information */
#define	sccbbase(sc)	((struct soft_ccb *)(sc)->sc_ab.ab_sccbv)
#define	sccblim(sc)	((struct soft_ccb *)(sc)->sc_ab.ab_sccblim)
	struct	soft_ccb *sc_nextsccb;	/* find free SCCBs starting here */
	struct	soft_ccb *sc_gosccb;	/* send sccb from bhastart to bhago */
};

/*
 * Macros for incrementing circular pointers.
 */
#define	INC_CIR(v, base, lim) { \
	if (++(v) >= (lim)) \
		(v) = (base); \
}
#define	INC_MBI(mb, sc)	INC_CIR(mb, mbibase(sc), mbilim(sc))
#define	INC_MBO(mb, sc)	INC_CIR(mb, mbobase(sc), mbolim(sc))
#define	INC_SCCB(s, sc)	INC_CIR(s, sccbbase(sc), sccblim(sc))

/*
 * Software CCBs for 32-bit BHA-only code.
 */
struct soft_ccb {
	struct	ccb32 sccb_ccb;		/* hardware CCB (MUST BE FIRST) */
	scintr_fn sccb_intr;		/* base interrupt handler */
	struct	device *sccb_intrdev;	/* link back to associated unit */
	time_t	sccb_stamp;		/* timestamp (for watchdog) */
	struct	sg32 sccb_sg[MAXSG];	/* scatter/gather map */
};

/*
 * Given the physical address of a CCB (in, e.g., a Buslogic mbox), convert
 * to the corresponding virtual address for that CCB.
 */
#define	BHA_CCB_TO_SCCB(sc, p) \
	((struct soft_ccb *)((int)(sc)->sc_ab.ab_sccbv + \
	    ((int)(p) - (int)(sc)->sc_ab.ab_sccb_phys)))

void	bhaattach __P((struct device *parent, struct device *dev, void *args));
static	int bhaccb __P((struct bha_softc *sc, struct soft_ccb *sccb,
	    int synch, int mbcmd));
int	bhadump __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t buf, int len));
int	bhago __P((struct device *self, int targ, scintr_fn intr,
	    struct device *dev, struct buf *bp, int pad));
void	bhahbareset __P((struct hba_softc *hba, int resetunits));
int	bhaicmd __P((struct hba_softc *hba, int targ, struct scsi_cdb *cdb,
	    caddr_t buf, int len, int rw));
static	struct mbox32 *bhainbox __P((struct bha_softc *sc, struct mbox32 *mb));
int	bhaintr __P((void *sc0));
int	bhaprobe __P((struct device *parent, struct cfdata *cf, void *aux));
void	bhastart __P((struct device *self, struct sq *sq, struct buf *bp,
	    scdgo_fn dgo, struct device *dev));
void	bharel __P((struct device *self));
static	void bhareset __P((struct bha_softc *sc, int hard));
void	bhawatch __P((void *sc0));
static	struct soft_ccb *sccballoc __P((struct bha_softc *sc,
	    struct scsi_cdb *cdb));
static	void sccbinit __P((struct bha_softc *sc, struct soft_ccb *sccb,
	    int targ, scintr_fn intr, struct device *dev, struct buf *bp));

struct cfdriver bhacd =
    { 0, "bha", bhaprobe, bhaattach, DV_DULL, sizeof(struct bha_softc) };
static struct hbadriver bhahbadriver =
    { bhaicmd, bhadump, bhastart, bhago, bharel, bhahbareset };

/*
 * Look for an empty outbox.
 * If we don't find one, return NULL.
 */
static __inline struct mbox32 *
scanmbo(sc, start)
	register struct bha_softc *sc;
	struct mbox32 *start;
{
	struct mbox32 *mbo;
	int i;

	for (mbo = start, i = 0; i < sc->sc_ab.ab_mbox_count; ++i) {
		if (mbo->mb_status == MBOX_O_FREE)
			return (mbo);
		INC_MBO(mbo, sc);
	}
	return (NULL);
}

/*
 * Look for a filled inbox.
 * If we don't find one, return start.
 */
static struct mbox32 *
scanmbi(sc, start)
	register struct bha_softc *sc;
	struct mbox32 *start;
{
	struct mbox32 *mbi;
	int i;

	for (mbi = start, i = 0; i < sc->sc_ab.ab_mbox_count; ++i) {
		if (mbi->mb_status != MBOX_I_FREE)
			return (mbi);
		INC_MBI(mbi, sc);
	}
	return (start);
}

/*
 * Probe for a Buslogic (only).  If we find it we will use it in 32-bit
 * mode (hence we cannot talk to an Adaptec).
 */
int
bhaprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{

	return (ab_probe((struct isa_attach_args *)aux, 1));
}

/*
 * Make sure that we don't get stuck.
 * We check for stale mailboxes.
 */
u_long bha_missed_mbif;
u_long bha_aborted_command;
u_long bha_dobusreset = AB_DOBUSRESET;
u_long bha_wedgetime = AB_WEDGETIME;

void
bhawatch(sc0)
	void *sc0;
{
	struct bha_softc *sc;
	struct soft_ccb *sccb;
	struct mbox32 *mb;
	int i, op, s;
	time_t now;
	int fasttoolong = 0;
	int slowtoolong = 0;

	sc = sc0;
	sc->sc_ab.ab_watchon = 0;
	if (sc->sc_ab.ab_ioout == 0)
		return;

	/* did we miss an MBIF interrupt? */
	mb = sc->sc_inbox;
	while ((mb = scanmbi(sc, mb))->mb_status != MBOX_I_FREE) {
		s = splbio();	/* avoid races */
		if (mb->mb_status != MBOX_I_FREE) {
			++bha_missed_mbif;
			if (!sc->sc_ab.ab_modifyintdone)
				ab_modifyint(&sc->sc_ab);
			mb = bhainbox(sc, mb);
		}
		splx(s);
		if (sc->sc_ab.ab_ioout == 0)
			return;
	}

	if (!sc->sc_ab.ab_watchon) {
		sc->sc_ab.ab_watchon = 1;
		timeout(bhawatch, sc, sc->sc_ab.ab_watchtime);
	}

	if (--sc->sc_ab.ab_wedgecounter != 0)
		return;
	sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;

	/*
	 * Scan sccb list, look for stale entries.
	 * If a read or write entry hasn't been serviced in a reasonable
	 * amount of time, complain (should abort the transaction?).
	 */
	now = time.tv_sec;
	sccb = sc->sc_nextsccb;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
		op = sccb->sccb_ccb.ccb_cdb6.cdb_cmd;
		if (sccb->sccb_ccb.ccb_opcode != CCB_FREE &&
		    sccb->sccb_stamp < (now - bha_wedgetime)) {
		    	s = splbio();
		    	if (sccb->sccb_ccb.ccb_opcode != CCB_FREE &&
			    sccb->sccb_stamp < (now - bha_wedgetime)) {
			    	if (op == CMD_READ10 || op == CMD_WRITE10 ||
				     op == CMD_READ || op == CMD_WRITE) {
					fasttoolong = 1;
					printf("%s: operation time exceeds %d seconds\n",
					    sc->sc_hba.hba_dev.dv_xname,
					    now - sccb->sccb_stamp);
				} else
					slowtoolong = 1;
				if (bha_dobusreset)
					break;
			}
			splx(s);
		}
		INC_SCCB(sccb, sc);
	}
	if (fasttoolong && !slowtoolong && bha_dobusreset) {
		scintr_fn save_intr[NSCCB];	
		struct	device *save_intrdev[NSCCB];

		s = splbio();
		sccb = sc->sc_nextsccb;
		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			if (sccb->sccb_ccb.ccb_opcode != CCB_FREE) {
				save_intr[i] = sccb->sccb_intr;
				save_intrdev[i] = sccb->sccb_intrdev;
			} else 
				save_intr[i] = NULL;
			INC_SCCB(sccb, sc);
		}

		printf("%s: attempting bus reset",
		    sc->sc_hba.hba_dev.dv_xname);

		sc->sc_ab.ab_ioout = 0;
		bzero(sc->sc_ab.ab_mboxv,
		    2 * NMBOX * sizeof(struct mbox32) + 
		    NSCCB * sizeof(struct soft_ccb));

		bhareset(sc, AB_HARDRESET);
		bzero(sc->sc_ab.ab_mboxv,
		    2 * NMBOX * sizeof(struct mbox32) + 
		    NSCCB * sizeof(struct soft_ccb));

		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			sccb->sccb_ccb.ccb_opcode = CCB_FREE;
			INC_SCCB(sccb, sc);
		}

		DELAY(5000000);
		printf("\n");
		for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
			struct sq *sq;

			if (save_intr[i] == NULL)
				continue;

			sc->sc_ab.ab_hba.hba_intr = save_intr[i];
			sc->sc_ab.ab_hba.hba_intrdev = save_intrdev[i];

			(*sc->sc_hba.hba_intr)(sc->sc_hba.hba_intrdev,
			    HBAINTR_BUSRESET, 0);

			if ((sq = sc->sc_hba.hba_head) != NULL) {
				sc->sc_hba.hba_head = sq->sq_forw;
				sc->sc_gosccb =
				    sccballoc(sc, (struct scsi_cdb *)NULL);
				(*sq->sq_dgo)(sq->sq_dev, (struct scsi_cdb *)
				    &sc->sc_gosccb->sccb_ccb.ccb_cdb);
			}
		}
		sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;
		splx(s);
	}
}

void
bhaattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct bha_softc *sc = (struct bha_softc *)self;
	struct soft_ccb *sccb;
	int i;

	/*
	 * See ahaattach().
	 */
	sc->sc_ab.ab_ih.ih_fun = bhaintr;
	ab_attach(&sc->sc_ab, (struct isa_attach_args *)aux, 1,
	    NMBOX, sizeof(struct mbox32),
	    NSCCB, sizeof(struct soft_ccb));
	sc->sc_outbox = mbobase(sc);
	sc->sc_inbox = mbibase(sc);

	sccb = sc->sc_ab.ab_sccbv;
	sc->sc_nextsccb = sccb;
	sc->sc_gosccb = NULL;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++sccb, ++i)
		sccb->sccb_ccb.ccb_opcode = CCB_FREE;

	sc->sc_hba.hba_driver = &bhahbadriver;
	bhareset(sc, AB_SOFTRESET);

	ab_attach2(&sc->sc_ab, NULL);

}

static void
bhareset(sc, hard)
	struct bha_softc *sc;
	int hard;
{
	u_char out[5];

	out[0] = sc->sc_ab.ab_mbox_count;
	*(int *)&out[1] = sc->sc_ab.ab_mbox_phys;
	ab_reset(sc->sc_ab.ab_port, BHA_MBOX_INIT_32, out, hard);
	if (hard == AB_HARDRESET)
		scsi_reset_units(&sc->sc_hba);
}

void
bhahbareset(hba, resetunits)
	struct hba_softc *hba;
	int resetunits;
{
	struct bha_softc *sc = (struct bha_softc *)hba;

	if (resetunits)
		bhareset(sc, AB_HARDRESET);
	else
		bhareset(sc, AB_SOFTRESET);

}

/*
 * Allocate and initialize a software CCB.
 * Must be called at splbio().
 * XXX we wouldn't need splbio() if we didn't call this from bhaintr()!
 */
static struct soft_ccb *
sccballoc(sc, cdb)
	struct bha_softc *sc;
	struct scsi_cdb *cdb;
{
	struct soft_ccb *sccb, *n;
	int i;

	sccb = sc->sc_nextsccb;
	for (i = 0; i < sc->sc_ab.ab_sccb_count; ++i) {
		if (sccb->sccb_ccb.ccb_opcode == CCB_FREE)
			break;
		INC_SCCB(sccb, sc);
	}
	if (sccb->sccb_ccb.ccb_opcode != CCB_FREE)
		panic("sccballoc");
	sccb->sccb_stamp = time.tv_sec;
	sccb->sccb_ccb.ccb_opcode = CCB_CMD_SG_RDL;
	n = sccb;
	INC_SCCB(n, sc);
	sc->sc_nextsccb = n;

	if (cdb)
		bcopy(cdb, sccb->sccb_ccb.ccb_cdbbytes,
		    SCSICMDLEN(cdb->cdb_bytes[0]));

	return (sccb);
}

/*
 * Initialize (most of) an SCCB.
 * We assume that the CDB has already been set up.
 */
static void
sccbinit(sc, sccb, targ, intr, dev, bp)
	struct bha_softc *sc;
	struct soft_ccb *sccb;
	int targ;
	scintr_fn intr;
	struct device *dev;
	struct buf *bp;
{
	struct ccb32 *ccb;
	int len, sgpages;
	u_int phys;

	/* XXX do we need direction bits if we always check residual count? */
	ccb = &sccb->sccb_ccb;
	ccb->ccb_control = 0;
	ccb->ccb_targ = targ;
	ccb->ccb_lun = ccb->ccb_cdbbytes[1] >> 5;
	ccb->ccb_cdblen = SCSICMDLEN(ccb->ccb_cdbbytes[0]);

	/* Build scatter/gather map, if needed. */
	sgpages = ab_sgmap(bp, &sccb->sccb_sg[0], MAXSG, &phys);
	if (sgpages != 0) {
		/* Point the CCB at the scatter/gather map. */
		phys = ((int)&sccb->sccb_sg[0] - (int)sc->sc_ab.ab_sccbv) +
		    sc->sc_ab.ab_sccb_phys;
		len = sgpages * sizeof(struct sg32);
		ccb->ccb_opcode = CCB_CMD_SG_RDL;
	} else {
		/* Point the CCB directly at the transfer. */
		ccb->ccb_opcode = CCB_CMD_RDL;
		len = bp->b_iocount;
	}
	ccb->ccb_data = phys;
	ccb->ccb_datalen = len;

	/* XXX when we permit multiple commands per target, must change this */
	ccb->ccb_rqslen = 1;	/* disable automatic request sense */
	ccb->ccb_rsv1 = 0;	/* XXX necessary? */
	ccb->ccb_rsv2 = 0;	/* XXX necessary? */
	ccb->ccb_rsv3 = 0;	/* XXX necessary? */

	sccb->sccb_intr = intr;
	sccb->sccb_intrdev = dev;
}

/*
 * Fire off a CCB.
 * If synch is set, poll for completion.
 * Must be called at splbio().
 */
static int
bhaccb(sc, sccb, synch, mbcmd)
	struct bha_softc *sc;
	struct soft_ccb *sccb;
	int synch;
	int mbcmd;
{
	struct ccb32 *ccb;
	u_long ccb_phys;
	struct mbox32 *mb, *n;
	int aha, status;

	if ((mb = scanmbo(sc, sc->sc_outbox)) == NULL)
		/* if we run out of mailboxes here, it's big trouble */
		panic("bhaccb scanmbo");
	n = mb;
	INC_MBO(n, sc);
	sc->sc_outbox = n;
	ccb = &sccb->sccb_ccb;
	ccb_phys = sc->sc_ab.ab_sccb_phys +
	    ((u_long)ccb - (u_long)sc->sc_ab.ab_sccbv);
	mb->mb_ccb = ccb_phys;
	mb->mb_cmd = mbcmd;

	/* start the adapter's outbox scan */
	aha = sc->sc_ab.ab_port;
	sc->sc_ab.ab_istat = inb(AHA_INTR(aha)); /* XXX should do more */
	outb(AHA_STAT(aha), AHA_C_IRST);
	while (inb(AHA_STAT(aha)) & AHA_S_CDF)
		continue;
	outb(AHA_DATA(aha), AHA_START_SCSI_CMD);

	if (!synch) {
		sc->sc_ab.ab_ioout++;
		if (!sc->sc_ab.ab_watchon) {
			sc->sc_ab.ab_watchon = 1;
			sc->sc_ab.ab_wedgecounter = sc->sc_ab.ab_wedgetime;
			timeout(bhawatch, sc, sc->sc_ab.ab_watchtime);
		}
		return (0);
	}

	/* wait for notification */
	while ((inb(AHA_INTR(aha)) & AHA_I_ANY) == 0)
		continue;
	sc->sc_ab.ab_istat = inb(AHA_INTR(aha));
	outb(AHA_STAT(aha), AHA_C_IRST);
	if ((sc->sc_ab.ab_istat & AHA_I_MBIF) == 0)
		panic("bhaccb missing mbif");

	/* scan for the inbox */
	/* XXX what do we do if the command hangs? */
	mb = sc->sc_inbox;
	for (;;) {
		mb = scanmbi(sc, mb);
		if (mb->mb_status == MBOX_I_FREE)
			continue;
		if (mb->mb_ccb == ccb_phys)
			break;
		INC_MBI(mb, sc);
	}
	sc->sc_ab.ab_resid = ccb->ccb_datalen;
	if ((sc->sc_ab.ab_stat = ccb->ccb_hastat) == CCB_H_NORMAL)
		status = ccb->ccb_tarstat;
	else if (ccb->ccb_cdbbytes[0] == CMD_TEST_UNIT_READY)
		/* probing, don't make noise */
		status = -1;
	else {
		ab_error(&sc->sc_ab, ccb->ccb_hastat);
		status = -1;
	}
	mb->mb_status = MBOX_I_FREE;
	if (sc->sc_gosccb == sccb)
		sc->sc_gosccb = 0;
	ccb->ccb_opcode = CCB_FREE;

	return (status);
}


int
bhaicmd(hba, targ, cdb, buf, len, rw)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len, rw;
{
	struct bha_softc *sc = (struct bha_softc *)hba;
	struct soft_ccb *sccb;
	int s, error;
	struct buf bufhdr;

	s = splbio();
	sccb = sccballoc(sc, cdb);
	splx(s);
	bufhdr.b_un.b_addr = buf;
	bufhdr.b_iocount = bufhdr.b_bcount = len;
	bufhdr.b_flags = rw;
	bufhdr.b_chain = NULL;
	sccbinit(sc, sccb, targ, NULL, NULL, &bufhdr);
	s = splbio();
	error = bhaccb(sc, sccb, 1, MBOX_O_START);
	splx(s);
	return (error);
}


int
bhadump(hba, targ, cdb, buf, len)
	struct hba_softc *hba;
	int targ;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len;
{
	struct bha_softc *sc = (struct bha_softc *)hba;
	struct soft_ccb *sccb;
	struct ccb32 *ccb;
	u_long from, to, lastfrom;

	/*
	 * Several assumptions:
	 * +	The upper-level dump code always calls us on aligned
	 *	2^n chunks which never cross 64 KB physical memory boundaries
	 * +	We're running in virtual mode with interrupts blocked
	 */

	sccb = sccballoc(sc, cdb);
	ccb = &sccb->sccb_ccb;

	ccb->ccb_opcode = CCB_CMD_RDL;
	ccb->ccb_control = 0;
	ccb->ccb_targ = targ;
	ccb->ccb_lun = ccb->ccb_cdbbytes[1] >> 5;
	ccb->ccb_cdblen = SCSICMDLEN(ccb->ccb_cdbbytes[0]);
	ccb->ccb_rqslen = 1;	/* disable automatic request sense */

	ccb->ccb_data = (u_long)buf;
	ccb->ccb_datalen = len;

	sccb->sccb_intr = NULL;
	sccb->sccb_intrdev = 0;

	return (bhaccb(sc, sccb, 1, MBOX_O_START));
}

/*
 * Start a transfer.
 *
 * Since the Buslogic handles transactions in parallel (with disconnect /
 * reconnect), we don't have to queue requests for the host adapter.
 *
 * This code is NOT re-entrant: (*dgo)() must call bhago() without calling
 * bhastart() first (because of the sc_gosccb kluge).
 */
void
bhastart(self, sq, bp, dgo, dev)
	struct device *self;
	struct sq *sq;
	struct buf *bp;
	scdgo_fn dgo;
	struct device *dev;
{
	struct bha_softc *sc = (struct bha_softc *)self;

	if (bp) {
		/* asynch transaction */
		sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
		(*dgo)(dev,
		    (struct scsi_cdb *)&sc->sc_gosccb->sccb_ccb.ccb_cdb);
		return;
	}
	/* let bhaicmd() allocate its own sccb */
	(*dgo)(dev, (struct scsi_cdb *)NULL);
}

/*
 * Get the host adapter going on a command.
 *
 * XXX should we allocate the DMA channel here, rather than dedicate one?
 * XXX must be called at splbio() since interrupts can call it
 * XXX but it'd be much better for the work to get done at low ipl!
 */
int
bhago(self, targ, intr, dev, bp, pad)
	struct device *self;
	int targ;
	scintr_fn intr;
	struct device *dev;
	struct buf *bp;
	int pad;
{
	struct bha_softc *sc = (struct bha_softc *)self;

	sccbinit(sc, sc->sc_gosccb, targ, intr, dev, bp);
	return (bhaccb(sc, sc->sc_gosccb, 0, MBOX_O_START));
}

/*
 * Handle a filled in-mailbox; return the next inbox.
 * Leaves sc_inbox pointing at the next inbox as a side effect.
 */
static struct mbox32 *
bhainbox(sc, mbi)
	struct bha_softc *sc;
	struct mbox32 *mbi;
{
	struct soft_ccb *sccb = BHA_CCB_TO_SCCB(sc, mbi->mb_ccb);
	struct ccb32 *ccb = &sccb->sccb_ccb;
	struct sq *sq;
	int status = -1;

	sc->sc_ab.ab_hba.hba_intr = sccb->sccb_intr;
	sc->sc_ab.ab_hba.hba_intrdev = sccb->sccb_intrdev;
	sc->sc_ab.ab_resid = ccb->ccb_datalen;
	sc->sc_ab.ab_targ = ccb->ccb_targ;
	sc->sc_ab.ab_lun = ccb->ccb_lun;
	sc->sc_ab.ab_stat = ccb->ccb_hastat;
	sc->sc_ab.ab_ioout--;
	sc->sc_ab.ab_iototal++;

	switch (mbi->mb_status) {

	case MBOX_I_ERROR:
		if (ccb->ccb_hastat == CCB_H_NORMAL) {
			status = ccb->ccb_tarstat;
			if ((status & STS_MASK) != STS_GOOD)
				break;
		}
		/* print vaguely informative messages */
		ab_prmbox(&sc->sc_ab, mbi->mb_status,
		    ccb->ccb_opcode, ccb->ccb_hastat);
		break;

	case MBOX_I_COMPLETED:
		status = ccb->ccb_tarstat;
		break;

	case MBOX_I_ABORTED:
		break;

	case MBOX_I_ABORT_FAILED:
		if (ccb->ccb_opcode == CCB_FREE) {
			mbi->mb_status = MBOX_I_FREE;
			INC_MBI(mbi, sc);
			return (sc->sc_inbox = mbi);
		}
		/* FALLTHROUGH */

	default:
		ab_prmbox(&sc->sc_ab, mbi->mb_status,
		    ccb->ccb_opcode, ccb->ccb_hastat);
		break;
	}

	/* free up resources */
	mbi->mb_status = MBOX_I_FREE;
	if (sc->sc_gosccb == sccb)
		sc->sc_gosccb = NULL;
	ccb->ccb_opcode = CCB_FREE;

	if (sc->sc_hba.hba_intr && sc->sc_hba.hba_intrdev) {
		/*
		 * For non-immediate commands,
		 * pass status back to higher levels and
		 * start the next transfer.
		 */
		(*sc->sc_hba.hba_intr)(sc->sc_hba.hba_intrdev, status,
		    sc->sc_ab.ab_resid);
		if ((sq = sc->sc_hba.hba_head) != NULL) {
			sc->sc_hba.hba_head = sq->sq_forw;
			sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
			(*sq->sq_dgo)(sq->sq_dev, (struct scsi_cdb *)
			    &sc->sc_gosccb->sccb_ccb.ccb_cdb);
		}
	}

	/* increment to the next mailbox */
	INC_MBI(mbi, sc);
	return (sc->sc_inbox = mbi);
}

int
bhaintr(sc0)
	void *sc0;
{
	register struct bha_softc *sc = sc0;
	register struct mbox32 *mbi;

	if (!ab_intr(&sc->sc_ab))
		return (0);
	mbi = sc->sc_inbox;
	if ((sc->sc_ab.ab_istat & AHA_I_MBIF) && mbi->mb_status == MBOX_I_FREE)
		mbi = scanmbi(sc, mbi);
	while (mbi->mb_status != MBOX_I_FREE)
		mbi = bhainbox(sc, mbi);
	return (1);
}

void
bharel(self)
	struct device *self;
{
	struct bha_softc *sc = (struct bha_softc *)self;
	struct soft_ccb *sccb;
	struct sq *sq;

	if ((sccb = sc->sc_gosccb) != NULL) {
		if (sc->sc_gosccb == sccb)
			sc->sc_gosccb = NULL;
		sccb->sccb_ccb.ccb_opcode = CCB_FREE;
	}
	if ((sq = sc->sc_hba.hba_head) != NULL) {
		sc->sc_gosccb = sccballoc(sc, (struct scsi_cdb *)NULL);
		(*sq->sq_dgo)(sq->sq_dev,
		    (struct scsi_cdb *)&sc->sc_gosccb->sccb_ccb.ccb_cdb);
	}
}
