/*-
 * Copyright (c) 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: ab_subr.c,v 2.9 1995/12/12 20:25:38 karels Exp $
 */

/*
 * Common subroutines for aha.c and bha.c.
 */

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/dma.h>
#include <i386/isa/icu.h>
#include <i386/isa/ahareg.h>
#include <i386/isa/abvar.h>
#include <i386/eisa/eisa.h>

#ifndef DRQNONE
#define	DRQNONE	0	/* probably should be -1, and should be elsewhere */
#endif

#define AB_FASTWATCH 1

#ifndef AB_SLOWWATCH		/* allow override from config */
#define AB_SLOWWATCH (hz/10)
#endif

static const char cmd_out_size[] =
    { 0, 4, 0, 0, 0, 1, 4, 1, 1, 1, 0, 0, 2, 1, 0, 0,	/* 00-0f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 3, 0, 1,	/* 10-1f */
      0, 2, 35,3, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0,	/* 20-2f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 30-3f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 40-4f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 50-5f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 60-6f */
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 70-7f */
      0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0	/* 80-8f */
      };

/*
 * Send a command to the host adapter and recover the output immediately.
 */
int
ab_fastcmd(port, cmd, out, in)
	int port, cmd;
	u_char *out, *in;
{
	int cnt, s;

	s = splbio();
	while ((inb(AHA_STAT(port)) & AHA_S_IDLE) == 0)
		continue;

	if (inb(AHA_INTR(port)) & AHA_I_ANY)
		outb(AHA_STAT(port), AHA_C_IRST);

	/* send command and data bytes */
	for (cnt = cmd_out_size[cmd]; cnt >= 0; --cnt) {
		outb(AHA_DATA(port), cmd >= 0 ? cmd : *out++);
		cmd = -1;
		while (inb(AHA_STAT(port)) & AHA_S_CDF)
			continue;
		if (inb(AHA_INTR(port)) & AHA_I_HACC)
			break;
	}

	/* receive bytes */
	while ((inb(AHA_INTR(port)) & AHA_I_HACC) == 0) {
		if (inb(AHA_STAT(port)) & AHA_S_DF)
			*in++ = inb(AHA_DATA(port));
	}

	/* look for an error */
	if (inb(AHA_STAT(port)) & AHA_S_INVDCMD) {
		outb(AHA_STAT(port), AHA_C_IRST);
		splx(s);
		return (EIO);
	}

	outb(AHA_STAT(port), AHA_C_IRST);

	splx(s);
	return (0);
}

/*
 * Probe an AHA or Buslogic board.
 */
int
ab_probe(args, require_32bit)
	struct isa_attach_args *args;
	int require_32bit;
{
	int port, i, is32;
	u_char in[20], out[4];

	static int reset_port;
	char *s;
#define fail(m)	{ s = (m); goto failure; }

	/* check for diagnostic failure and reserved bits */
	port = args->ia_iobase;
	if (inb(AHA_STAT(port)) & (AHA_S_DIAGF|AHA_S_RSV02))
		return (0);
	if (inb(AHA_INTR(port)) & (AHA_I_RSV40|AHA_I_RSV20|AHA_I_RSV10))
		return (0);

	/*
	 * Do hard reset, then wait up to 5 seconds for self-test to
	 * complete. Note this pause is usually very fast.
	 * The 5 second pause is in ab_attach2, and is needed because
	 * of the reset we are doing here.
	 */
	if (reset_port != port) {
		DELAY(100000);
		outb(AHA_STAT(port), AHA_C_HRST);
		for (i = 0;
		     (inb(AHA_STAT(port)) & (AHA_S_STST|AHA_S_DIAGF)) ==
		     AHA_S_STST; ++i) {
			if (i >= 50)
				return (0);
			DELAY(100000);
		}
		reset_port = port;
	}
	if ((inb(AHA_STAT(port)) &
	     (AHA_S_STST|AHA_S_DIAGF|AHA_S_INIT|AHA_S_IDLE)) !=
	    (AHA_S_INIT|AHA_S_IDLE))
		fail("selftest failed");

	/* check for Buslogic */
	is32 = 0;
	if (!ab_fastcmd(port, AHA_INQUIRY, (u_char *)0, in) &&
	    (in[0] == 'A' && in[1] == 'A')) {

		/*
		 * Check for BT-542B, which does not do 32-bit addressing.
		 * Return code of A means its a PC/AT bus adapter.
		 */
		out[0] = 4;
		if (!ab_fastcmd(port, BHA_INQUIRE_EXTENDED, out, in) &&
		    (in[0] != 'A'))
			is32 = 1;
	}
	if (require_32bit && !is32)
		fail("not 32 bit");

	if (ab_fastcmd(port, AHA_CONFIG_DATA, (u_char *)0, in))
		fail("config_data failed");

	if (is32)
		args->ia_drq = DRQNONE;
	else {
		switch (in[0]) {
		case 0x80:	args->ia_drq = 7; break;
		case 0x40:	args->ia_drq = 6; break;
		case 0x20:	args->ia_drq = 5; break;
		case 0x01:	args->ia_drq = 0; break;
		default:	fail("unknown drq");
		}
	}
	switch (in[1]) {
	case 0x40:	args->ia_irq = IRQ15; break;
	case 0x20:	args->ia_irq = IRQ14; break;
	case 0x08:	args->ia_irq = IRQ12; break;
	case 0x04:	args->ia_irq = IRQ11; break;
	case 0x02:	args->ia_irq = IRQ10; break;
	case 0x01:	args->ia_irq = IRQ9; break;
	default:	fail("unknown irq");
	}
	args->ia_maddr = 0;
	args->ia_msize = 0;
	args->ia_iosize = AHA_NPORT;

	return (1);

failure:
	aprint_debug("%s: ", s);
	return (0);
#undef fail
}

/*
 * Floppy DMA buffers are so small that 11 microseconds of delay
 * (the default) can cause a data late.
 *
 * XXX: the names below should be changed, but may be embedded in
 *	various gdb scripts now.
 */
u_char aha_bus_on_time = 8;
u_char aha_bus_off_time = 4;

static void
set_burst_mode(port)
	int port;
{

	ab_fastcmd(port, AHA_BUS_ON_TIME, &aha_bus_on_time, (u_char *)0);
	ab_fastcmd(port, AHA_BUS_OFF_TIME, &aha_bus_off_time, (u_char *)0);
}

/*
 * Common code between Adaptec and Buslogic attach (part 1 of 2).
 */
void
ab_attach(ab, args, is32, nmbox, mboxsize, nsccb, sccbsize)
	struct ab_softc *ab;
	struct isa_attach_args *args;
	int is32, nmbox;
	size_t mboxsize;
	int nsccb;
	size_t sccbsize;
{
	struct soft_ccb *sccb;
	int port = args->ia_iobase;
	caddr_t va;
	u_int pa;
	size_t bytes;
	int can32, smallfloppyfifo;
	char *vendor, *pref, *id;
	u_char in[20], in2[20], out[20];

	ab->ab_watchtime = AB_SLOWWATCH;
	ab->ab_wedgetime = (AB_WEDGETIME * hz) / AB_SLOWWATCH;
	/*
	 * Decode adapter type & print vendor and ID. If the device
	 * can do 32-bit addressing, so we can warn if the user
	 * should configure a `bha'.
	 */
	ab->ab_port = port;
	ab_fastcmd(port, AHA_INQUIRY, (u_char *)0, in);
	smallfloppyfifo = 1;
	ab->ab_sync_base = 200;
	vendor = "Adaptec";
	pref = " AHA-";
	switch (in[0]) {
	case 0:
		id = "1540/16 (unsupported)";
		break;
	case '0':
		id = "1540 (unsupported)";
		break;
	case 'A':
		if (in[1] != 'A') {
			id = "1540B/1542B";
			break;
		}
		vendor = "Buslogic";
		pref = " ";
		ab->ab_isbuslogic=1;
		if (args->ia_drq != DRQNONE) 
			id = "BT-542B";
		else
			id = "EISA/PCI/VLB";
		break;
	case 'B':
		id = "1640 (unsupported)";
		break;
	case 'C':
		id = "1740A/1742A";
		smallfloppyfifo = 0;
		break;
	case 'D':
		id = "1540C/1542C";
		smallfloppyfifo = 0;
		ab->ab_sync_base = 100;
		break;
	case 'E':
		id = "1540CF/1542CF";
		smallfloppyfifo = 0;
		ab->ab_sync_base = 100;
		break;
	case 'F':
		id = "1540CP/1542CP";
		smallfloppyfifo = 0;
		ab->ab_sync_base = 100;
		break;
	default:
		aprint_normal(": unknown type %c%c Adaptec card", in[0], in[1]);
		vendor = NULL;
		break;
	}
	aprint_naive(": %s SCSI", vendor);
	if (vendor != NULL)
		aprint_normal(": %s%s%s", vendor, pref, id);
	aprint_verbose(" rev %c%c", in[2], in[3]);
	if (args->ia_drq == DRQNONE)
		ab->ab_is32capable = 1;
	if (is32)
		aprint_verbose(" (32-bit)");
	else if (args->ia_drq == DRQNONE)
		printf(
	"\n%s: 32-bit capable -- add bha driver for better performance",
		    ab->ab_hba.hba_dev.dv_xname);
	printf("\n");

	/* pick up initiator id */
	ab_fastcmd(port, AHA_CONFIG_DATA, (u_char *)0, in);
	ab->ab_id = in[2];

	/*
	 * Program the mailbox lock on the 1542C and late-model 1542Bs.
	 */
	if (ab_fastcmd(port, AHA_GET_LOCK, (u_char *)0, in) == 0) {
#ifdef DIAGNOSTIC
		if (in[0] & 0x8)
		    aprint_verbose("%s: extended BIOS translation enabled\n",
			ab->ab_hba.hba_dev.dv_xname);
#endif
		if (in[1]) {
			out[0] = 0;
			out[1] = in[1];
			if (ab_fastcmd(port, AHA_SET_LOCK, out, (u_char *)0))
				panic("ab_attach clear mbox lock");
#ifdef DIAGNOSTIC
			aprint_verbose("%s: unlocked\n",
			    ab->ab_hba.hba_dev.dv_xname);
#endif
		}
	}

	/*
	 * Link into isa and set interrupt handler (our caller
	 * has already set ab->ab_ih.ih_fun).
	 */
	isa_establish(&ab->ab_isa, &ab->ab_hba.hba_dev);
	ab->ab_ih.ih_arg = ab;
	intr_establish(args->ia_irq, &ab->ab_ih, DV_DISK);
	ab->ab_irq = args->ia_irq;

	/*
	 * Allocate space for mailboxes and SCCBs (thence to be
	 * initialized by our caller).  These must be contiguous
	 * because they are accessed with physical addresses from
	 * the card's bus master DMA.
	 */
	bytes = 2 * nmbox * mboxsize + nsccb * sccbsize;
	if (bytes > NBPG)
		panic("ab_attach sizes");
	va = malloc(bytes, M_DEVBUF, M_WAITOK);
	bzero(va, bytes);
	pa = pmap_extract(kernel_pmap, (vm_offset_t)va);
	ab->ab_mboxv = va;
	ab->ab_mbox_phys = pa;
	ab->ab_mbox_count = nmbox;
	ab->ab_mbolim = va + 1 * nmbox * mboxsize;
	ab->ab_mbilim = va + 2 * nmbox * mboxsize;

	/* Sccb immediately follows mboxes. */
	va += 2 * nmbox * mboxsize;
	pa += 2 * nmbox * mboxsize;
	ab->ab_sccbv = va;
	ab->ab_sccb_phys = pa;
	ab->ab_sccb_count = nsccb;
	ab->ab_sccblim = va + nsccb * sccbsize;

	/* negotiate bus master DMA for our DMA channel */
	/* XXX this is only needed if BIOS is disabled */
	if (args->ia_drq != DRQNONE)
		at_dma_cascade(args->ia_drq);

	if (smallfloppyfifo)
		set_burst_mode(port);
}

/*
 * Finish up the attach, after sccb's have been initialized and the
 * host bus adapter has been reset.  If the given function pointer is
 * not NULL, call it for each target.
 */
void
ab_attach2(ab, func)
	register struct ab_softc *ab;
	int (*func) __P((struct ab_softc *, int));
{
	int targ;
	u_char tgmap[8];
#ifdef DIAGNOSTIC
	int first, port, syncoff;
	u_char in[20], out[20];
#endif

	printf("Delaying for SCSI bus reset and device self tests");
	DELAY(5000000);
	printf("\n");

#ifdef DIAGNOSTIC
	/*
	 * Print out several host adapter parameters
	 * that are primarily of interest in debugging.
	 */
	first = 1;
	port = ab->ab_port;
	out[0] = 17;
	ab_fastcmd(port, AHA_SETUP_DATA, out, in);
	aprint_verbose("%s:", ab->ab_hba.hba_dev.dv_xname);
	if (in[0] & 1) {
		aprint_verbose(" synch negotiation");
		first = 0;
	}
	if (in[0] & 2) {
		aprint_verbose("%s parity", first ? "" : ",");
		first = 0;
	}
	if (!ab->ab_is32capable) {
		aprint_verbose("%s bus transfer ", first ? "" : ",");
		/* XXX custom values for standard numbers are guesses */
		/* XXX values of 99 and ff from observation on 1542CF */
		switch (in[1]) {
		case 0xff:		aprint_verbose("3.3 Mb/s"); break;
		case 0: case 0xbb:	aprint_verbose("5 Mb/s");   break;
		case 1:	case 0xa2: case 0x99:
					aprint_verbose("6.7 Mb/s"); break;
		case 2:	case 0x91:	aprint_verbose("8 Mb/s");   break;
		case 3:	case 0x80:	aprint_verbose("10 Mb/s");  break;
		case 4: case 0xaa:	aprint_verbose("5.7 Mb/s"); break;
		default:
			aprint_verbose("0x%x custom", in[1]); break;
			break;
		}
		aprint_verbose(", bus on/off %d/%d us", in[2], in[3]);
	}
	aprint_verbose("\n");
#endif

	ab_fastcmd(ab->ab_port, AHA_INSTALLED_DEVS, (u_char *)0, tgmap);
	for (targ = 0; targ < 8; ++targ) {
		if (tgmap[targ] == 0)
			continue;
		if (func != NULL) {
			if (func(ab, targ))
				continue;
		}
		SCSI_FOUNDTARGET(&ab->ab_hba, targ);
#ifdef DIAGNOSTIC
		/*
		 * Ideally this would appear earlier,
		 * but synch negotiation won't occur without
		 * data transfer and it's simple to let
		 * target attach routines do this for us.
		 */
		out[0] = 17;
		ab_fastcmd(port, AHA_SETUP_DATA, out, in);
		syncoff = in[targ + 8] & 0xf;
		if (!(syncoff && !(ab->ab_isbuslogic && !ab->ab_is32capable)) &&
		    !(!ab->ab_isbuslogic) && (in[16] & (1 << targ)))
			continue;
		aprint_verbose("%s:",
		    ab->ab_hba.hba_targets[targ]->t_dev.dv_xname);
		if (syncoff && !(ab->ab_isbuslogic && !ab->ab_is32capable)) {
			/*
			 * Calculate synchronous xfer rate from period
			 * and base rate.
			 */
			static const char *const rates[] = {
				"??", "??", "10", "6.67", "5", "4", "3.33",
				"2.86", "2.50", "2.22", "2", "1.81",
			};
			int i, syncper;

			syncper = (in[targ + 8] >> 4) & 7;
			i = syncper + ab->ab_sync_base;
			if (ab->ab_is32capable)
				i /= 100;
			else
				i /= 50;
			if (i >= sizeof rates / sizeof *rates)
				i = 0;
			aprint_verbose(
		" synch transfer period %d ns, offset %d (%s MB/s)",
			    syncper * 50 + ab->ab_sync_base, syncoff,
			    rates[i]);
		}
		if ((!ab->ab_isbuslogic) && (in[16] & (1 << targ))) {
			if (syncoff)
				aprint_verbose(",");
			aprint_verbose(" disconnect off");
		}
		aprint_verbose("\n");
#endif
	}

	/* XXX if there's only one target, disable disconnection option? */
	/* XXX how about dynamically attached units? */
}

/*
 * Send the given reset command.
 */
void
ab_reset(port, cmd, out, hard)
	int port, cmd, hard;
	u_char *out;
{

	if (hard == AB_HARDRESET)
		outb(AHA_STAT(port), AHA_C_HRST);	/* reset on the bus */
	else
		outb(AHA_STAT(port), AHA_C_SRST);	/* just controller */
	DELAY(100000);	/* for the 1542C */
	while ((inb(AHA_STAT(port)) & (AHA_S_INIT|AHA_S_IDLE)) !=
	    (AHA_S_INIT|AHA_S_IDLE))
		continue;
	if (ab_fastcmd(port, cmd, out, (u_char *)0))
		panic("ab_reset");

	/* XXX do we need to explicitly disable MBOA interrupts? */
}

/*
 * Given a transfer described by a `struct buf', figure out what kind
 * of transfer is needed (direct or scatter/gather).  If scatter/gather
 * is required, set up the given s/g map and return the number of entries;
 * otherwise store the physical address for the transfer's page(s) and
 * return 0.
 *
 * If the transfer uses a chain (bp->b_chain != NULL), we assume that no
 * <addr,len> pair ever crosses a page boundary.  In any case, at most two
 * pages can be partial (one at the start and one at the end).
 *
 * (Lots of panics here, from sheer raging paranoia.)
 */
int
ab_sgmap(bp, sg, nsg, physp)
	struct buf *bp;
	struct sg32 *sg;
	int nsg;
	u_int *physp;
{
	int len, i, n, o, pages;
	vm_offset_t v;

	/*
	 * ###: We used to xfer 0 bytes at addr 0 for len < 0, but I
	 * think this is an error.  If not, our caller will have to
	 * check len < 0 and change that to len = 0 anyway.
	 */
	len = bp->b_iocount;
	if (len < 0)
		panic("ab_sgmap");
	if (len == 0) {
		*physp = 0;		/* physadr = 0 */
		return (0);		/* no scatter/gather */
	}
	v = (vm_offset_t)bp->b_un.b_addr;
	o = v & PGOFSET;
	pages = i386_btop(i386_round_page(len + o));
	if (pages == 1) {
		/* Transfer lies entirely within a page. */
		if (bp->b_chain)
			panic("ab_sgmap chain 1page");
		*physp = pmap_extract(kernel_pmap, v);
		return (0);		/* no scatter/gather */
	}

	/*
	 * Transfer too big (at least 2 pages) -- we'll need a
	 * scatter/gather map.  (Actually, if the underlying pages
	 * are contiguous, we could be OK, but this would get
	 * complicated, especially when using bounce buffers in
	 * aha.c, so just use the map.)
	 *
	 * Transfer may also be chained, in which case we must step
	 * to the next buffer each time bp->b_bcount runs out (note
	 * that len is the head buffer's bp->b_iocount).
	 */
	if (pages > nsg)
		panic("ab_sgmap pages");

	/* First page may start at some offset, and hence be short. */
	n = bp->b_bcount;
	i = NBPG - o;
	sg->sg_addr = pmap_extract(kernel_pmap, v);
	sg->sg_len = i;
	sg++;
	v += i;
	n -= i;

	/* Pages 1 through (pages-1), if pages > 2, are full size. */
	for (i = 2; i < pages; i++) {
		if (n <= 0) {
			if (n < 0 || bp->b_chain == NULL)
				panic("ab_sgmap mid chain");
			bp = bp->b_chain;
			n = bp->b_bcount;
			v = (vm_offset_t)bp->b_un.b_addr;
			if (v & PGOFSET)
				panic("ab_sgmap mid addr");
		}
		sg->sg_addr = pmap_extract(kernel_pmap, v);
		sg->sg_len = NBPG;
		sg++;
		v += NBPG;
		n -= NBPG;
	}

	/* Last page is n remaining bytes. */
	if (n <= 0) {
		if (n < 0 || bp->b_chain == NULL)
			panic("ab_sgmap last chain");
		bp = bp->b_chain;
		n = bp->b_bcount;
		v = (vm_offset_t)bp->b_un.b_addr;
		if (v & PGOFSET)
			panic("ab_sgmap last addr");
	}
	if (n > NBPG)
		panic("ab_sgmap lastpg %d", n);
	sg->sg_addr = pmap_extract(kernel_pmap, v);
	sg->sg_len = n;

	return (pages);
}

#define	AB_MAXHASTAT	CCB_H_INVPARM

const char * const ab_hastat[AB_MAXHASTAT + 1] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0,
	"target selection timeout",
	"data overrun or underrun",
	"target unexpectedly freed the SCSI bus",
	"target bus phase sequence error",
	0,
	"invalid ccb opcode",
	"linked ccb doesn't have same LUN",
	"invalid direction in target mode",
	"duplicate ccb received in target mode",
	"invalid ccb or segment list parameter",
};

void
ab_error(ab, hastat)
	struct ab_softc *ab;
	int hastat;
{

	if ((unsigned)hastat <= AB_MAXHASTAT && ab_hastat[hastat])
		printf("%s: %s (target %d, lun %d)\n",
		    ab->ab_hba.hba_dev.dv_xname, ab_hastat[hastat],
		    ab->ab_targ, ab->ab_lun);
	else
		printf("%s: internal error 0x%x (target %d, lun %d)\n",
		    ab->ab_hba.hba_dev.dv_xname, hastat,
		    ab->ab_targ, ab->ab_lun);
}

/*
 * Print messages for various unusual mbox status cases.
 * NOTE, THIS ONLY PRINTS; callers should check for abnormality first.
 */
void
ab_prmbox(ab, mbstat, ccbop, hastat)
	struct ab_softc *ab;
	int mbstat, ccbop, hastat;
{

	switch (mbstat) {

	case MBOX_I_ERROR:
		ab_error(ab, hastat);
		break;

	case MBOX_I_COMPLETED:	/* callers should handle this themselves */
		panic("ab_prmbox COMPLETED");

	case MBOX_I_ABORTED:
		panic("ab_prmbox ABORTED");

	case MBOX_I_ABORT_FAILED:
		printf("%s: command abort failed (target %d, lun %d)\n",
		    ab->ab_hba.hba_dev.dv_xname, ab->ab_targ, ab->ab_lun);
		break;

	default:
		printf("%s: bad mbox status 0x%x\n",
		    ab->ab_hba.hba_dev.dv_xname, mbstat);
		panic("ab_prmbox");
	}
}

/*
 * Common code for ahaintr and bhaintr.
 */
int
ab_intr(ab)
	struct ab_softc *ab;
{
	int port, istat;

	port = ab->ab_port;
	ab->ab_istat = istat = inb(AHA_INTR(port));
	outb(AHA_STAT(port), AHA_C_IRST);

	if ((istat & AHA_I_ANY) == 0)
		return (0);

	if (istat & AHA_I_HACC) {
		if (inb(AHA_STAT(port)) & AHA_S_INVDCMD) {
			printf("%s: invalid host adapter command\n",
			    ab->ab_hba.hba_dev.dv_xname);
			panic("ab_intr HACC");
		}
		printf("%s: interrupt for immediate command\n",
		    ab->ab_hba.hba_dev.dv_xname);
	}

	if (istat & AHA_I_SCRD)
		printf("%s: SCSI bus reset\n", ab->ab_hba.hba_dev.dv_xname);

	if (istat & AHA_I_MBOA)
		printf("%s: mailbox out available\n",
		    ab->ab_hba.hba_dev.dv_xname);

	return (1);
}

void
ab_modifyint(ab)
	struct ab_softc *ab;
{

	if (ab->ab_setlevel) {
		ab->ab_modifyintdone = 1;
		if (ab->ab_iototal < 3) {
			printf("%s: warning, interrupts not working, %s\n",
			    ab->ab_hba.hba_dev.dv_xname, "using polling");
			ab->ab_watchtime = AB_FASTWATCH;
			ab->ab_wedgetime = (AB_WEDGETIME * hz) / AB_FASTWATCH;
		}
		return;
	}

	ab->ab_setlevel = 1;
	if (ab->ab_iototal > 1) {
		ab->ab_modifyintdone = 1;
		return;
	}

	/* figure out if we have IO_ELCR1/2 */
	if (!elcr_present) {
		/*
		 * select 82374EB config register
		 * and write id to register
		 */
		outb(0x22, 0x2);
		outb(0x23, 0xf);
		if (inb(0x23) != 0xf)	/* did we select 82374 */
			return;
		elcr_present = 1;
	}

	/*
	 * we believe we have one. Now go ahead and change
	 * this interrupt to level triggering.
	 */
	outb(IO_ELCR1, inb(IO_ELCR1) | ab->ab_irq);
	outb(IO_ELCR2, inb(IO_ELCR2) | (ab->ab_irq >> 8));
}
