/*-
 * Copyright (c) 1992, 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *      BSDI $Id: machdep.c,v 2.13 1995/12/13 03:13:04 karels Exp $
 */

/*-
 * Copyright (c) 1982, 1987, 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * William Jolitz.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)machdep.c	8.2 (Berkeley) 1/12/94
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/signalvar.h>
#include <sys/kernel.h>
#include <sys/map.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/buf.h>
#include <sys/reboot.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/file.h>
#include <sys/callout.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/msgbuf.h>
#include <sys/exec.h>
#include <sys/ioctl.h>
#include <sys/tty.h>
#include <sys/sysctl.h>
#include <sys/sysinfo.h>	/* XXX? */

#include <net/netisr.h>

#include <vm/vm.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>

#include <machine/cpu.h>
#include <machine/reg.h>
#include <machine/psl.h>
#include <i386/i386/specialreg.h>
#include <i386/i386/fp_reg.h>
#include <i386/i386/cons.h>
#include <i386/isa/isa.h>
#include <i386/isa/timerreg.h>
#include <i386/isa/rtc.h>
#include <i386/isa/ic/i8042.h>

extern vm_offset_t avail_end;

struct proc *npxproc;
void set_led __P((int val));

/*
 * Declare this as initialized data so we can patch it.
 */
int	nswbuf = 0;

int	msgbufmapped;		/* set when safe to use msgbuf */

/*
 * Machine-dependent startup code
 */
int	boothowto = 0;
long	dumplo;
int	physmem, maxmem;
extern	int basemem;
extern	int bootdev;

extern	int cpu;			/* cpu type */
extern	int maxbufmem;
extern	int nmbclusters, maxmbclusters;

cpu_startup(firstaddr)
	int firstaddr;
{
	register unsigned i;
	register caddr_t v;
	vm_offset_t minaddr, maxaddr;
	vm_size_t size;

	/*
	 * Initialize error message buffer (at end of core).
	 */
	/* avail_end was pre-decremented in pmap_bootstrap to compensate */
	for (i = 0; i < btoc(sizeof(struct msgbuf)); i++)
		pmap_enter(kernel_pmap, (vm_offset_t) msgbufp + i * NBPG,
		    avail_end + i * NBPG, VM_PROT_ALL, TRUE);
	msgbufmapped = 1;

#if defined(KGDB) && defined(KGDB_PORT)
	{
	extern int kgdb_debug_init;

	if (kgdb_debug_init) {
		kgdb_connect(1);
		kgdb_debug_init = 0;	/* already done */
	}
	}
#endif
#ifdef KDB
	kdb_init();			/* startup kernel debugger */
#endif
	/*
	 * Good {morning,afternoon,evening,night}.
	 */
	aprint_normal(version);
	identifycpu();
	aprint_normal("real mem = %d\n", ctob(maxmem));

	/*
	 * Allocate space for system data structures.
	 * The first available real memory address is in "firstaddr".
	 * The first available kernel virtual address is in "v".
	 * As pages of kernel virtual memory are allocated, "v" is incremented.
	 * As pages of memory are allocated and cleared,
	 * "firstaddr" is incremented.
	 */

	/*
	 * Make two passes.  The first pass calculates how much memory is
	 * needed and allocates it.  The second pass assigns virtual
	 * addresses to the various data structures.
	 */
	firstaddr = 0;
again:
	v = (caddr_t)firstaddr;

#define	valloc(name, type, num) \
	    (name) = (type *)v; v = (caddr_t)((name)+(num))
#define	valloclim(name, type, num, lim) \
	    (name) = (type *)v; v = (caddr_t)((lim) = ((name)+(num)))
	valloc(callout, struct callout, ncallout);
	valloc(swapmap, struct map, nswapmap = maxproc * 10);
	if (nswbuf == 0) {
		/*
		 * Allocate enough swap buffers so that 5% of memory
		 * could be paged at once, with an average transfer size
		 * of MAXBSIZE.  Force the number to be even for historical
		 * reasons.
		 */
		nswbuf = (physmem / (20 * MAXBSIZE / NBPG)) &~ 1;
		if (nswbuf > 256)
			nswbuf = 256;		/* sanity */
		else if (nswbuf < 50)
			nswbuf = 50;		/* more sanity */
	}
	valloc(swbuf, struct buf, nswbuf);

	/*
	 * End of first pass, size has been calculated so allocate memory
	 */
	if (firstaddr == 0) {
		size = (vm_size_t) v;
		firstaddr = (int)kmem_alloc(kernel_map, round_page(size));
		if (firstaddr == 0)
			panic("startup: no room for tables");
		goto again;
	}
	/*
	 * End of second pass, addresses have been assigned
	 */
	if ((vm_size_t)(v - firstaddr) != size)
		panic("startup: table size inconsistency");
#if 0	/* we don't use these currently */
	/*
	 * Allocate a submap for exec arguments.  This map effectively
	 * limits the number of processes exec'ing at any time.
	 */
	exec_map = kmem_suballoc(kernel_map, &minaddr, &maxaddr,
				 16*NCARGS, TRUE);
	/*
	 * Allocate a submap for physio
	 */
	phys_map = kmem_suballoc(kernel_map, &minaddr, &maxaddr,
				 VM_PHYS_SIZE, TRUE);
#endif

	/*
	 * Finally, allocate mbuf pool.  Since mclrefcnt is an off-size
	 * we use the more space efficient malloc in place of kmem_alloc.
	 */
	mclrefcnt = (char *)malloc(maxmbclusters + CLBYTES / MCLBYTES,
	    M_MBUF, M_NOWAIT);
	bzero(mclrefcnt, maxmbclusters + CLBYTES / MCLBYTES);
	mb_map = kmem_suballoc(kernel_map, (vm_offset_t *)&mbutl, &maxaddr,
	    maxmbclusters * MCLBYTES, FALSE);
	/*
	 * Initialize callouts
	 */
	callfree = callout;
	for (i = 1; i < ncallout; i++)
		callout[i-1].c_next = &callout[i];
	callout[i-1].c_next = NULL;

	aprint_verbose("avail mem = %d\n", ptoa(cnt.v_free_count));
	aprint_verbose("buffer cache = %d\n", maxbufmem);

	/*
	 * Set up CPU-specific registers, cache, etc.
	 */
	initcpu();

	/*
	 * Set up buffers, so they can be used to read disk labels.
	 */
	bufinit();

	/* Start real time and statistics clocks. */
	initclocks();

	/*
	 * Configure the system.
	 */
	aprint_naive("Checking devices and configuration...\n");
	configure();
}

#ifdef PGINPROF
/*
 * Return the difference (in microseconds)
 * between the  current time and a previous
 * time as represented  by the arguments.
 * If there is a pending clock interrupt
 * which has not been serviced due to high
 * ipl, return error code.
 */
/*ARGSUSED*/
vmtime(otime, olbolt, oicr)
	register int otime, olbolt, oicr;
{

	return (((time.tv_sec-otime)*60 + lbolt-olbolt)*16667);
}
#endif

struct sigframe {
	int	sf_signum;
	int	sf_code;
	struct	sigcontext *sf_scp;
	sig_t	sf_handler;
	int	sf_eax;	
	int	sf_edx;	
	int	sf_ecx;	
	struct	save87 sf_fpu;
	struct	sigcontext sf_sc;
} ;

#ifndef NO_KSTACK
extern int kstack[];
#endif

/*
 * Send an interrupt to process.
 *
 * Stack is set up to allow sigcode stored
 * in u. to call routine, followed by kcall
 * to sigreturn routine below.  After sigreturn
 * resets the signal mask, the stack, and the
 * frame pointer, it returns to the user
 * specified pc, psl.
 */
void
sendsig(catcher, sig, mask, code)
	sig_t catcher;
	int sig, mask;
	unsigned code;
{
	register struct proc *p = curproc;
	register int *regs;
	register struct sigframe f, *fp;
	struct sigacts *psp = p->p_sigacts;
	int oonstack, frmtrap;
	int frmvm86, userframesize;
#ifdef COPY_SIGCODE
	extern int szsigcode;
#endif

	regs = p->p_md.md_regs;
        oonstack = psp->ps_sigstk.ss_flags & SA_ONSTACK;
	frmtrap = curpcb->pcb_flags & FM_TRAP;
	if (frmtrap && (regs[tEFLAGS] & PSL_VM)) {
		frmvm86 = 1;
		userframesize = sizeof(struct sigframe) +
		    sizeof(struct trapframe_vm86);
	} else {
		frmvm86 = 0;
		userframesize = sizeof(struct sigframe);
        }
	/*
	 * Allocate and validate space for the signal handler
	 * context. Note that if the stack is in data space, the
	 * call to grow() is a nop, and the useracc() check
	 * will fail if the process has not already allocated
	 * the space with a `brk'.
	 */
	if ((psp->ps_flags & SAS_ALTSTACK) &&
	    (psp->ps_sigstk.ss_flags & SA_ONSTACK) == 0 &&
	    (psp->ps_sigonstack & sigmask(sig))) {
		fp = (struct sigframe *)(psp->ps_sigstk.ss_base +
		    psp->ps_sigstk.ss_size - userframesize);
		psp->ps_sigstk.ss_flags |= SA_ONSTACK;
	} else if (frmvm86) {		/* XXX why check? */
		fatalsig(p, SIGILL);
		return;
	} else {
		if (frmtrap)
			fp = (struct sigframe *)(regs[tESP] -
			    sizeof(struct sigframe));
		else
			fp = (struct sigframe *)(regs[sESP] -
			    sizeof(struct sigframe));
	}

	if ((unsigned)fp <= USRSTACK - ctob(p->p_vmspace->vm_ssize)) 
 		(void)grow(p, (unsigned)fp);

 	if (!useracc((caddr_t)fp, userframesize, B_WRITE)) {
  		/*
  		 * Process has trashed its stack; give it an illegal
		 * instruction to halt it in its tracks.
		 */
		fatalsig(p, SIGILL);
		return;
	}

        /*
         * Copy trap frame on top of the user stack.
         */
	if (frmvm86)
		copyout(regs, (void *)((unsigned)fp + sizeof(struct sigframe)),
		    sizeof(struct trapframe_vm86));

	/* 
	 * Build the argument list for the signal handler.
	 */
	f.sf_signum = sig;
	f.sf_code = code;
	f.sf_scp = &fp->sf_sc;
	f.sf_handler = catcher;

	/* save scratch registers */
	if (frmtrap) {
		f.sf_eax = regs[tEAX];
		f.sf_edx = regs[tEDX];
		f.sf_ecx = regs[tECX];
	} else {
		f.sf_eax = regs[sEAX];
		f.sf_edx = regs[sEDX];
		f.sf_ecx = regs[sECX];
	}

	/*
	 * Save floating point state on the user stack.
	 * We also restore the default control word.
	 * XXX This saves users who got nailed in the middle
	 * of floating-to-int operations, but may screw users
	 * who intelligently manipulate the control word...
	 */
	if ((rcr0() & CR0_EM) == 0 && npxproc) {
		/* CR0_EM is off only if we have hardware floating point */
		asm volatile ("fnsave %0" : "=m" (curpcb->pcb_savefpu));
		npxinit(curproc, FPC_DEFAULT);
	}
	f.sf_fpu = curpcb->pcb_savefpu;
	curpcb->pcb_savefpu.sv_env.en_cw = FPC_DEFAULT;
	curpcb->pcb_savefpu.sv_env.en_sw = 0;
	curpcb->pcb_savefpu.sv_env.en_tw = -1;

	/*
	 * Build the signal context to be used by sigreturn.
	 */
	f.sf_sc.sc_onstack = oonstack;
	f.sf_sc.sc_mask = mask;
	if (frmtrap) {
		f.sf_sc.sc_sp = regs[tESP];
		f.sf_sc.sc_fp = regs[tEBP];
		f.sf_sc.sc_pc = regs[tEIP];
		f.sf_sc.sc_ps = regs[tEFLAGS] & ~PSL_T;
		regs[tESP] = (int)fp;
		regs[tEIP] = (int) PS_STRINGS -
		    roundup(szsigcode, sizeof (double));
		if (frmvm86) {
			extern int _ucodesel, _udatasel;

			regs[tEFLAGS] = PSL_USERSET;
			regs[tCS] = _ucodesel;
			regs[tDS] = _udatasel;
			regs[tES] = _udatasel;
			regs[tSS] = _udatasel;
		}
	} else {
		f.sf_sc.sc_sp = regs[sESP];
		f.sf_sc.sc_fp = regs[sEBP];
		f.sf_sc.sc_pc = regs[sEIP];
		f.sf_sc.sc_ps = regs[sEFLAGS] & ~PSL_T;
		regs[sESP] = (int)fp;
		regs[sEIP] = (int) PS_STRINGS -
		    roundup(szsigcode, sizeof (double));
	}

	copyout(&f, fp, sizeof f);
}

/*
 * System call to cleanup state after a signal
 * has been taken.  Reset signal mask and
 * stack state from context left by sendsig (above).
 * Return to previous pc and psl as specified by
 * context left by sendsig. Check carefully to
 * make sure that the user has not modified the
 * psl to gain improper priviledges or to cause
 * a machine fault.
 *
 * Ugly hack: we don't bother to use copyin() to
 * recover the sigframe because we won't take a
 * write fault on the stack (avoiding 386 COW problems).
 */
struct sigreturn_args {
	struct sigcontext *sigcntxp;
};
/* ARGSUSED */
sigreturn(p, uap, retval)
	struct proc *p;
	struct sigreturn_args *uap;
	int *retval;
{
	register struct sigcontext *scp;
	register struct sigframe *fp;
	register int *regs = p->p_md.md_regs;

	fp = (struct sigframe *) regs[sESP];
	if (useracc((caddr_t)fp, sizeof (*fp), 0) == 0)
		return (EINVAL);
	if (fp->sf_sc.sc_ps & PSL_VM && useracc((caddr_t)fp + sizeof (*fp),
	    sizeof(struct trapframe_vm86), 0) == 0)
		return (EINVAL);

	if (fp->sf_sc.sc_ps & PSL_VM) {
		/*
		 * If the signal context claims we are going back to v86
		 * mode, make sure the flags register that the processor
		 * will see agrees.  Otherwise, we'll get a general 
		 * protection trap when trying to load trash into the
		 * cs at the iret in swicth_to_vm86.
		 */
		struct trapframe_vm86 *tfvm86;
		u_int *flagsp;

		tfvm86 = (struct trapframe_vm86 *)((caddr_t)fp + sizeof (*fp));
		flagsp = (u_int *)&tfvm86->tf_eflags86;

		if ((*flagsp & PSL_VM) == 0)
			return (EINVAL);

		*flagsp = (*flagsp & ~PSL_USERCLR) | PSL_USERSET | PSL_VM;
		if ((p->p_md.md_flags & MDP_IOPL) == 0)
			*flagsp &= ~PSL_IOPL;
	}

	/* restore scratch registers */
	regs[sEAX] = fp->sf_eax;
	regs[sEDX] = fp->sf_edx;
	regs[sECX] = fp->sf_ecx;

	/*
	 * Restore floating point state.
	 */
	curpcb->pcb_savefpu = fp->sf_fpu;
	if ((rcr0() & CR0_EM) == 0 && npxproc)
		/* XXX does this cause an exception if one was pending? */
		asm volatile ("frstor %0" : : "m" (curpcb->pcb_savefpu));

	scp = fp->sf_scp;
	if (useracc((caddr_t)scp, sizeof (*scp), 0) == 0)
		return (EINVAL);

	if (scp->sc_onstack & 01)
		p->p_sigacts->ps_sigstk.ss_flags |= SA_ONSTACK;
	else
		p->p_sigacts->ps_sigstk.ss_flags &= ~SA_ONSTACK;
	p->p_sigmask = scp->sc_mask &~ sigcantmask;
	if (fp->sf_sc.sc_ps & PSL_VM) {
		if (p->p_md.md_connarea == 0)
		  	p->p_md.md_connarea = (void *) fp->sf_eax;
		switch_to_vm86((caddr_t)fp + sizeof (*fp),
#ifndef NO_KSTACK
		       (unsigned int) kstack + UPAGES * NBPG -
#else
		       (unsigned int) p->p_addr + UPAGES * NBPG -
#endif
		       sizeof(struct trapframe_vm86),
		       sizeof(struct trapframe_vm86));
		/* NOTREACHED */
	}

	regs[sEBP] = scp->sc_fp;
	regs[sESP] = scp->sc_sp;
	regs[sEIP] = scp->sc_pc;
	regs[sEFLAGS] = (fp->sf_sc.sc_ps & ~PSL_USERCLR) | PSL_USERSET;
	if ((p->p_md.md_flags & MDP_IOPL) == 0)
		regs[sEFLAGS] &= ~PSL_IOPL;
	return (EJUSTRETURN);
}

int	waittime = -1;

boot(howto)
	int howto;
{
	int devtype;
	extern long numbufcache;
	extern struct proc *prevproc;
	static struct proc *saveproc;
	static int atshutdowndone;

	/* take a snap shot before clobbering any registers */
	saveproc = curproc;
	if (curproc == NULL) {
		curproc = prevproc;
#if 0		/* should be unnecessary */
		curpcb = &curproc->p_addr->u_pcb;
#endif
	}
	savectx(curproc->p_addr, 0);

	if ((howto&RB_NOSYNC) == 0 && waittime < 0) {
		waittime = 0;
		(void) spl0();
		bootsync();

		/*
		 * If we've been adjusting the clock, the todr
		 * will be out of synch; adjust it now.
		 */
		resettodr();
	}
	splhigh();
	devtype = major(rootdev);
	if (howto & RB_DUMP) {
		curproc = saveproc;
		dumpsys();
	}
	/* now do other shutdown activity, including devices */
	if (atshutdowndone++ == 0)
		doatshutdown();
	if (howto&RB_HALT) {
		printf("\nSystem is halted; %s\n\n",
		    "hit reset, turn power off, or press return to reboot");
		while (cngetc() != '\n')
			;
	}
#ifdef lint
	printf("%d\n", devtype);
#endif
	reset_cpu();
	for (;;)
		;
	/*NOTREACHED*/
}

#define	DUMP_CHUNK	(128 * 1024)

int	dumpmag = 0x8fca0101;	/* magic number for savecore */
int	dumpsize = 0;		/* also for savecore */
/*
 * Doadump comes here after turning off memory management and
 * getting on the dump stack, either when called above, or by
 * the auto-restart code.
 */
dumpsys()
{
	caddr_t addr;
	int resid;
	int this_len;
	daddr_t bn;
	int rc;
	int blink = 1;

	if (dumpdev == NODEV) {
		/* pause about 15 sec. so panic message can be read */
		DELAY(15*1000000);
		return;
	}
	if (dumpsize == 0)
		dumpconf();
	printf("\ndumping to dev %d,%d,%d, offset %d\n",
	    major(dumpdev), dv_unit(dumpdev), dv_subunit(dumpdev), dumplo);
	printf("dump ");

	resid = ctob(dumpsize);
	addr = (caddr_t)0;
	bn = dumplo;
	rc = -1;
	while (resid > 0) {
		/* Blink the keyboard LED's in case we are in X */
		blink = (blink > 2) ? 1 : blink << 1;
		set_led(blink);

		/* Let user know how far to go */
		if (((int)addr & 0xfffff) == 0)
			printf("%d ", resid >> 20);	/* MB remaining */

		if ((this_len = resid) > DUMP_CHUNK)
			this_len = DUMP_CHUNK;
		if (addr < (caddr_t)basemem &&
		    addr + this_len > (caddr_t)basemem)
			this_len = basemem - (int)addr;
		if (addr == (caddr_t)basemem)
			this_len = IOM_END - (int)basemem;
		else {
			rc = devsw[major(dumpdev)]->d_dump(dumpdev,
			    bn, addr, this_len);
			if (rc != 0)
				break;
		}
		resid -= this_len;
		addr += this_len;
		bn += this_len >> DEV_BSHIFT;
	}

	if (rc)
		set_led(0);	/* Turn all off if dump fails */

	switch (rc) {
	case 0:
		printf("succeeded\n");
		set_led(7);		/* Turn all on when dump OK */
		/* pause about 5 sec. more so panic message can be read */
		DELAY(5*1000000);
		return;
		
	case ENXIO:
		printf("device bad\n");
		break;

	case EFAULT:
		printf("device not ready\n");
		break;

	case EINVAL:
		printf("area improper\n");
		break;

	case EIO:
		printf("i/o error\n");
		break;

	default:
		printf("unknown error\n");
		break;
	}
	printf("\n\n");
	/* pause about 15 sec. so panic message can be read */
	DELAY(15*1000000);
}

/*
 * Set the keyboard LED's (very simple, goal is to run even if 
 * KB is disconnected)
 */
static void
set_led(val)
	int val;
{
	int i = 10000;	/* Wait up to 10ms */

	while ((inb(KBSTAT) & (KBS_ORDY|KBS_IFULL)) != KBS_ORDY && i-- > 0)
		delay(1);
	inb(KBDATA);			/* Clear 'output' buffer */
	outb(KBDATA, KCMD_SETLED);
	while ((inb(KBSTAT) & (KBS_ORDY|KBS_IFULL)) != KBS_ORDY && i-- > 0)
		delay(1);
	inb(KBDATA);
	outb(KBDATA, val);
}

#ifdef HZ
/*
 * microtime function for non-default values of HZ;
 * locore.s contains a more accurate function for the 
 * default value.
 */
microtime(tvp)
	register struct timeval *tvp;
{
	register int u, s;
	register volatile struct timeval *p = &time;
	static struct timeval lasttime;

	do {
		s = p->tv_sec;
		u = p->tv_usec;
	} while (u != p->tv_usec || s != p->tv_sec);
	if (u == lasttime.tv_usec && s == lasttime.tv_sec && ++u >= 1000000) {
		u -= 1000000;
		s++;
	}
	lasttime.tv_sec = s;
	lasttime.tv_usec = u;
	tvp->tv_sec = s;
	tvp->tv_usec = u;
}
#endif

initcpu()
{
}

struct biosgeom *biosgeomp;
int biosgeomlen;

/*
 * machine dependent system variables.
 */
cpu_sysctl(name, namelen, oldp, oldlenp, newp, newlen, p)
	int *name;
	u_int namelen;
	void *oldp;
	size_t *oldlenp;
	void *newp;
	size_t newlen;
	struct proc *p;
{
	extern int cntlaltdel, cntlaltdelcore, nmi_panic;

	/* all sysctl names at this level except BIOS are terminal */
	if (namelen != 1 && name[0] != CPU_BIOS)
		return (ENOTDIR);		/* overloaded */

	switch (name[0]) {
	case CPU_CONSDEV:
		return (sysctl_rdstruct(oldp, oldlenp, newp, &cn_tty->t_dev,
		    sizeof(cn_tty->t_dev)));
	case CPU_BIOS:
		/* all sysctl names at this level are terminal */
		if (namelen != 2)
			return (ENOTDIR);		/* overloaded */
		switch (name[1]) {
		case BIOS_DISKGEOM:
			if (biosgeomp == NULL)
				return (ESRCH);		/* needs translation */
			return (sysctl_rdstruct(oldp, oldlenp, newp, biosgeomp,
			    biosgeomlen));
		default:
			return (EOPNOTSUPP);
		}
	case CPU_CNTLALTDEL:
		return (sysctl_int(oldp, oldlenp, newp, newlen,
		    &cntlaltdel));
	case CPU_CNTLALTDELCORE:
		return (sysctl_int(oldp, oldlenp, newp, newlen,
		    &cntlaltdelcore));
	case CPU_PARITYERRORPANIC:
		return (sysctl_int(oldp, oldlenp, newp, newlen, &nmi_panic));
	default:
		return (EOPNOTSUPP);
	}
	/* NOTREACHED */
}

/*

/*
 * Initialize 386 and configure to run kernel
 */

/*
 * Initialize segments & interrupt table
 */

union descriptor gdt[NGDT];

/* interrupt descriptor table */
struct gate_descriptor idt[32+16];

/* local descriptor table */
union descriptor ldt[NLDT];

struct	i386tss	tss, panic_tss;
char emergency_stack[1024];

extern  struct user *proc0paddr;

/* software prototypes -- in more palatable form */
struct soft_segment_descriptor gdt_segs[] = {
	/* Null Descriptor */
{	0x0,			/* segment base address  */
	0x0,			/* length - all address space */
	0,			/* segment type */
	0,			/* segment descriptor priority level */
	0,			/* segment descriptor present */
	0,0,
	0,			/* default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Code Descriptor for kernel */
{	0x0,			/* segment base address  */
	0xfffff,		/* length - all address space */
	SDT_MEMERA,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	1,			/* default 32 vs 16 bit size */
	1  			/* limit granularity (byte/page units)*/ },
	/* Data Descriptor for kernel */
{	0x0,			/* segment base address  */
	0xfffff,		/* length - all address space */
	SDT_MEMRWA,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	1,			/* default 32 vs 16 bit size */
	1  			/* limit granularity (byte/page units)*/ },
	/* LDT Descriptor */
{	(int) ldt,			/* segment base address  */
	sizeof(ldt)-1,		/* length - all address space */
	SDT_SYSLDT,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	0,			/* unused - default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Null Descriptor - Placeholder */
{	0x0,			/* segment base address  */
	0x0,			/* length - all address space */
	0,			/* segment type */
	0,			/* segment descriptor priority level */
	0,			/* segment descriptor present */
	0,0,
	0,			/* default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Panic Tss Descriptor */
{	(int) &panic_tss,		/* segment base address  */
	sizeof(tss)-1,		/* length - all address space */
	SDT_SYS386TSS,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	0,			/* unused - default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Stack Descriptor for kernel */
{	0x0,			/* segment base address  */
	((unsigned) (USRSTACK+NBPG) >> PGSHIFT)-1,/* set red zone in kstack */
	SDT_MEMRWDA,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	1,			/* default 32 vs 16 bit size */
	1  			/* limit granularity (byte/page units)*/ },
	/* Proc 0 Tss Descriptor */
{	(int) &tss,			/* segment base address  */
	sizeof(tss)-1,		/* length - all address space */
	SDT_SYS386TSS,		/* segment type */
	0,			/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	0,			/* unused - default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ }};

struct soft_segment_descriptor ldt_segs[] = {
	/* Null Descriptor - overwritten by call gate */
{	0x0,			/* segment base address  */
	0x0,			/* length - all address space */
	0,			/* segment type */
	0,			/* segment descriptor priority level */
	0,			/* segment descriptor present */
	0,0,
	0,			/* default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Null Descriptor - overwritten by call gate */
{	0x0,			/* segment base address  */
	0x0,			/* length - all address space */
	0,			/* segment type */
	0,			/* segment descriptor priority level */
	0,			/* segment descriptor present */
	0,0,
	0,			/* default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Null Descriptor - overwritten by call gate */
{	0x0,			/* segment base address  */
	0x0,			/* length - all address space */
	0,			/* segment type */
	0,			/* segment descriptor priority level */
	0,			/* segment descriptor present */
	0,0,
	0,			/* default 32 vs 16 bit size */
	0  			/* limit granularity (byte/page units)*/ },
	/* Code Descriptor for user */
{	0x0,			/* segment base address  */
	btoc(VM_MAXUSER_ADDRESS)-1,/* length - all of user space */
	SDT_MEMERA,		/* segment type */
	SEL_UPL,		/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	1,			/* default 32 vs 16 bit size */
	1  			/* limit granularity (byte/page units)*/ },
	/* Data Descriptor for user */
{	0x0,			/* segment base address  */
	btoc(VM_MAXUSER_ADDRESS)-1,/* length - all of user space */
	SDT_MEMRWA,		/* segment type */
	SEL_UPL,		/* segment descriptor priority level */
	1,			/* segment descriptor present */
	0,0,
	1,			/* default 32 vs 16 bit size */
	1  			/* limit granularity (byte/page units)*/ } };

/* table descriptors - used to load tables by microp */
struct region_descriptor r_gdt = {
	sizeof(gdt)-1,(char *)gdt
};

struct region_descriptor r_idt = {
	sizeof(idt)-1,(char *)idt
};

setidt(idx, func, typ, dpl, sel)
	char *func;
{
	struct gate_descriptor *ip = idt + idx;

	ip->gd_looffset = (int)func;
	ip->gd_selector = sel;
	ip->gd_stkcpy = 0;
	ip->gd_xx = 0;
	ip->gd_type = typ;
	ip->gd_dpl = dpl;
	ip->gd_p = 1;
	ip->gd_hioffset = ((int)func)>>16 ;
}

#define	IDTVEC(name)	__CONCAT(X, name)
extern	IDTVEC(div), IDTVEC(dbg), IDTVEC(nmi), IDTVEC(bpt), IDTVEC(ofl),
	IDTVEC(bnd), IDTVEC(ill), IDTVEC(dna), IDTVEC(fpusegm),
	IDTVEC(tss), IDTVEC(missing), IDTVEC(stk), IDTVEC(prot),
	IDTVEC(page), IDTVEC(rsvd), IDTVEC(fpu), IDTVEC(align),
	IDTVEC(rsvd1), IDTVEC(rsvd2), IDTVEC(rsvd3), IDTVEC(rsvd4),
	IDTVEC(rsvd5), IDTVEC(rsvd6), IDTVEC(rsvd7), IDTVEC(rsvd8),
	IDTVEC(rsvd9), IDTVEC(rsvd10), IDTVEC(rsvd11), IDTVEC(rsvd12),
	IDTVEC(rsvd13), IDTVEC(rsvd14), IDTVEC(rsvd14), IDTVEC(syscall);

int lcr0(), lcr3(), rcr0(), rcr2();
int _udatasel, _ucodesel, _gsel_tss;

double_fault_handler()
{

	printf("double fault: eip=%x esp=%x\n", tss.tss_eip, tss.tss_esp);
	panic("double fault");
}

extern struct bootparamhdr *bootparamp;
struct biosinfo *biosinfo;

checkbootparams()
{
	struct bootparamhdr *bhp = bootparamp;
	struct bootparam *bp;
	extern int autodebug, kernspace;

	if (bhp && bhp->b_magic == BOOT_MAGIC) {
		for (bp = B_FIRSTPARAM(bhp); bp; bp = B_NEXTPARAM(bhp, bp))
		    	switch (bp->b_type) {
			case B_BIOSGEOM:
				biosgeomp = (struct biosgeom *) B_DATA(bp);
				biosgeomlen = bp->b_len -
				    sizeof(struct bootparam);
				break;
			case B_BIOSINFO:
				biosinfo = (struct biosinfo *) B_DATA(bp);
				break;
#ifdef BSDGEOM
			case B_BSDGEOM:
				setbsdgeom((struct bsdgeom *) B_DATA(bp));
				break;
#endif
			case B_AUTODEBUG:
				autoprint = *(int *) B_DATA(bp);
				/* backward compatibility: */
				autodebug = autoprint & (AC_DEBUG | AC_ASK);
				break;
			case B_REALBOOTDEV:
				bootdev = *(int *) B_DATA(bp);
				break;
			case B_KERNSPACE:
				kernspace = btoc(*(int *) B_DATA(bp));
				break;
			case B_CONSOLE: {
				struct bootcons *bc;
				extern int comconsole, com_cnaddr, com_cnrate;

				bc = (struct bootcons *) B_DATA(bp);
				switch (bc->type) {
				case BOOTCONS_KBDISP:
					comconsole = -1;
					break;
				case BOOTCONS_COM:
					comconsole = bc->unit;
					com_cnaddr = bc->val0;
					com_cnrate = bc->val1;
					break;
				}
				break;
				}
			default:
				break;
		}
	}
}

/*
 * Fetch a boot parameter of the specified type.
 * If bp is null, fetch the first; otherwise, fetch the next one
 * after the one specified.
 */
struct bootparam *
getbootparam(type, bp)
	int type;
	struct bootparam *bp;
{
	struct bootparamhdr *bhp = bootparamp;

	if (bhp && bhp->b_magic == BOOT_MAGIC) {
		if (bp)
			bp = B_NEXTPARAM(bhp, bp);
		else
			bp = B_FIRSTPARAM(bhp);
		for (; bp; bp = B_NEXTPARAM(bhp, bp))
		    	if (bp->b_type == type)
				return (bp);
	}
	return (NULL);
}

/*
 * Fetch boot time configuration information for a device identified by 
 * the cfdata structure and/or unit. If a record is not found for the
 * named device, use a wildcard record if found.
 *
 * The return value is the flags field to use for the requested device,
 * if no boot time overrides are found the flags field from the cfdata
 * structure is returned. If 'dsp' is non-NULL a pointer to the found
 * boot_devspec record is filled into it (or NULL if none was found).
 *
 * When called from probe the unit argument should be -1. When called at 
 * attach time the unit number should be the actual unit number (since 
 * the number found in the cf_unit field may be invalid).
 */
int
getdevconf(cf, dspp, unit)
	struct cfdata *cf;
	struct boot_devspec **dspp;
	int unit;
{
	struct bootparam *parm;
	struct boot_devspec *dsp;
	struct boot_devspec *wild;
	int flags;

	if (unit == -1)
		unit = cf->cf_unit;
	wild = NULL;
	parm = NULL;
	while ((parm = getbootparam(B_DEVSPEC, parm)) != NULL) {
		dsp = (struct boot_devspec *)B_DATA(parm);
		if (strcmp(dsp->ds_driver, cf->cf_driver->cd_name) != 0)
			continue;
		if (dsp->ds_unit == B_WILDDEV) {
			wild = dsp;
			continue;
		}
		if (dsp->ds_unit != unit)
			continue;
		if ((dsp->ds_validmask & (1 << DSLOC_FLAGS)) == 0)
			continue;
		break;
	}
	if (parm == NULL)
		dsp = wild;
	if (dsp != NULL)
		flags = dsp->ds_loc[DSLOC_FLAGS];
	else
		flags = cf->cf_flags;
	if (dspp != NULL)
		*dspp = dsp;
	return (flags);
}

init386(first)
{
	extern ssdtosd(), lgdt(), lidt(), lldt(), etext; 
	int x, i;
	int sel;
	struct gate_descriptor *gdp;
#ifndef COPY_SIGCODE
	extern int sigcode,szsigcode;
#endif

	proc0.p_addr = proc0paddr;

	/* make gdt memory segments */
	gdt_segs[GCODE_SEL].ssd_limit = btoc((int) &etext + NBPG);
	for (x=0; x < NGDT; x++) ssdtosd(gdt_segs+x, gdt+x);
	/* Note. eventually want private ldts per process */
	for (x=0; x < 5; x++) ssdtosd(ldt_segs+x, ldt+x);

	/* exceptions */
	sel = GSEL(GCODE_SEL, SEL_KPL);
	setidt(0, &IDTVEC(div),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(1, &IDTVEC(dbg),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(2, &IDTVEC(nmi),  SDT_SYS386TGT, SEL_KPL, sel);
 	setidt(3, &IDTVEC(bpt),  SDT_SYS386TGT, SEL_UPL, sel);
	setidt(4, &IDTVEC(ofl),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(5, &IDTVEC(bnd),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(6, &IDTVEC(ill),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(7, &IDTVEC(dna),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(9, &IDTVEC(fpusegm),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(10, &IDTVEC(tss), SDT_SYS386TGT, SEL_KPL, sel);
	setidt(11, &IDTVEC(missing),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(12, &IDTVEC(stk), SDT_SYS386TGT, SEL_KPL, sel);
	setidt(13, &IDTVEC(prot),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(14, &IDTVEC(page),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(15, &IDTVEC(rsvd),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(16, &IDTVEC(fpu),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(17, &IDTVEC(align),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(18, &IDTVEC(rsvd1),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(19, &IDTVEC(rsvd2),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(20, &IDTVEC(rsvd3),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(21, &IDTVEC(rsvd4),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(22, &IDTVEC(rsvd5),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(23, &IDTVEC(rsvd6),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(24, &IDTVEC(rsvd7),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(25, &IDTVEC(rsvd8),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(26, &IDTVEC(rsvd9),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(27, &IDTVEC(rsvd10),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(28, &IDTVEC(rsvd11),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(29, &IDTVEC(rsvd12),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(30, &IDTVEC(rsvd13),  SDT_SYS386TGT, SEL_KPL, sel);
	setidt(31, &IDTVEC(rsvd14),  SDT_SYS386TGT, SEL_KPL, sel);

	sel = GSEL(GPANIC_SEL, SEL_KPL);
	setidt(8, 0,  SDT_SYSTASKGT, SEL_KPL, sel);	/* double fault */

#include	"isa.h"
#if	NISA >0
	(void) isa_probe();
	/*
	 * Set up ICUs in autoconfig mode.
	 * Note that this does not yet enable interrupts,
	 * as we are not yet prepared for them.
	 */
	isa_autoirq();	
#endif

	lgdt(gdt, sizeof(gdt)-1);
	lidt(idt, sizeof(idt)-1);
	lldt(GSEL(GLDT_SEL, SEL_KPL));

	/*
	 * Initialize the console before we print anything out.
	 */
	cninit();

	physmem = maxmem;
	if (maxmem > btoc(IOM_END))
		physmem -= btoc(IOM_END - basemem);

	vm_set_page_size();
	/* call pmap initialization to make new kernel address space */
	pmap_bootstrap(first, 0);
	/* now running on new page tables, configured,and u/iom is accessible */

	/* make a initial tss so microp can get interrupt stack on syscall! */
	tss.tss_esp = tss.tss_esp0 = tss.tss_esp1 = tss.tss_esp2 =
		(int) &emergency_stack[sizeof emergency_stack - 24];
	tss.tss_ss = tss.tss_ss0 = tss.tss_ss1 = tss.tss_ss2 =
		GSEL(GSTACK_SEL, SEL_KPL);
	tss.tss_cr3 = IdlePTD;
	tss.tss_eip = (int) double_fault_handler;
	/* interrupts off for panic_tss; eflags not used in main tss */
	tss.tss_eflags = PSL_MBO;
	/* most user registers remain 0 */
	tss.tss_ds = tss.tss_es = tss.tss_fs = tss.tss_gs =
		GSEL(GDATA_SEL, SEL_KPL);
	tss.tss_cs = GSEL(GCODE_SEL, SEL_KPL);
	tss.tss_ldt = GSEL(GLDT_SEL, SEL_KPL);
	panic_tss = tss;
#ifndef NO_KSTACK
	tss.tss_esp0 = (int) kstack + UPAGES*NBPG;	/* XXX */
#endif

	/*
	 * By default, deny user processes access to I/O ports.
	 * However, for the moment, allow access to the VGA ports
	 * (ports 0x3B0 thru 0x3DF) for backward compatibility.
	 */
	tss.tss_ioopt |= ((u_int)&tss.tss_iomap - (u_int)&tss.tss_link) << 16;
	for (i = 0; i < USER_IOPORTS / NBBY; i++)
		if (i < 0x3b0 / NBBY || i > 0x3e0 / NBBY)
			tss.tss_iomap[i] = 0xff; /* disallow these ports */
		else
			tss.tss_iomap[i] = 0x0;  /* allow these ports */

	_gsel_tss = GSEL(GPROC0_SEL, SEL_KPL);
	ltr(_gsel_tss);

	/* make a call gate with which to reenter kernel */
	gdp = &ldt[L43BSDCALLS_SEL].gd;

	x = (int) &IDTVEC(syscall);
	gdp->gd_looffset = x;
	gdp->gd_selector = GSEL(GCODE_SEL,SEL_KPL);
	gdp->gd_stkcpy = 0;
	gdp->gd_type = SDT_SYS386CGT;
	gdp->gd_dpl = SEL_UPL;
	gdp->gd_p = 1;
	gdp->gd_hioffset = x >> 16;

	/* copy down to LDT 0 (LDEFCALLS_SEL), the default */
	ldt[LDEFCALLS_SEL] = ldt[L43BSDCALLS_SEL];

	/* transfer to user mode */

	_ucodesel = LSEL(LUCODE_SEL, SEL_UPL);
	_udatasel = LSEL(LUDATA_SEL, SEL_UPL);

	/* setup proc 0's pcb */
	proc0.p_addr->u_pcb.pcb_flags = 0;
	proc0.p_addr->u_pcb.pcb_ptd = IdlePTD;
	curpcb = &proc0.p_addr->u_pcb;		/* so we can take traps */

	/* save a copy of the default syscall descriptor in the pcb */
	curpcb->pcb_ldtdefcall = ldt[LDEFCALLS_SEL];

	/*
	 * Enable features that aren't available on the 386.
	 * We turn on the write protect bit so that we can prevent
	 * modification of kernel text and also do COW more efficiently,
	 * and we turn on the numeric error bit so that we don't need
	 * an external interrupt to report an FP error.
	 * Disable coprocessor unless/until we find one.
	 */
	i = rcr0() | CR0_EM;	/* start emulating */
	if (cpu > CPU_386)
		i |= CR0_WP;
	lcr0(i);

	/*
	 * Now that we're prepared to take interrupts, enable them.
	 * Note that all IRQs are masked, so the only interrupts
	 * which can arrive now are 'spurious' interrupts on IRQ7 & IRQ15.
	 */
	asm volatile ("sti");
}

/*
 * Main calls consinit to locate the console,
 * but we have already done the work in init386 (calling cninit).
 */
consinit()
{

}

extern struct pte	*CMAP1, *CMAP2;
extern caddr_t		CADDR1, CADDR2;
/*
 * zero out physical memory
 * specified in relocation units (NBPG bytes)
 */
clearseg(n)
	u_int n;
{

	if (*(int *)CMAP2)
		panic("clearseg");
	*(int *)CMAP2 = PG_V | PG_KW | ctob(n);
	/* we assume that a non-valid PTE cannot be in the TLB */
	bzero(CADDR2, NBPG);
	*(int *)CMAP2 = 0;
	tbis(CADDR2);
}

#ifdef unused
/*
 * copy a page of physical memory
 * specified in relocation units (NBPG bytes)
 */
copyseg(frm, n)
	u_int frm, n;
{

	if (*(int *)CMAP2)
		panic("copyseg");
	*(int *)CMAP2 = PG_V | PG_KW | ctob(n);
	/* we assume that a non-valid PTE cannot be in the TLB */
	bcopy((void *)frm, (void *)CADDR2, NBPG);
	*(int *)CMAP2 = 0;
	tbis(CADDR2);
}
#endif

/*
 * copy a page of physical memory
 * specified in relocation units (NBPG bytes)
 */
physcopyseg(frm, to)
	u_int frm, to;
{

	if (*(int *)CMAP1 || *(int *)CMAP2)
		panic("physcopyseg");
	*(int *)CMAP1 = PG_V | PG_KW | ctob(frm);
	*(int *)CMAP2 = PG_V | PG_KW | ctob(to);
	/* we assume that a non-valid PTE cannot be in the TLB */
	bcopy(CADDR1, CADDR2, NBPG);
	*(int *)CMAP1 = 0;
	*(int *)CMAP2 = 0;
	tbis(CADDR1);
	tbis(CADDR2);
}

/*
 * insert an element into a queue 
 */
#undef insque
_insque(element, head)
	register struct prochd *element, *head;
{
	element->ph_link = head->ph_link;
	head->ph_link = (struct proc *)element;
	element->ph_rlink = (struct proc *)head;
	((struct prochd *)(element->ph_link))->ph_rlink=(struct proc *)element;
}

/*
 * remove an element from a queue
 */
#undef remque
_remque(element)
	register struct prochd *element;
{
	((struct prochd *)(element->ph_link))->ph_rlink = element->ph_rlink;
	((struct prochd *)(element->ph_rlink))->ph_link = element->ph_link;
	element->ph_rlink = (struct proc *)0;
}

vmunaccess() {}

/*
 * Machine chip & clock speed configuration
 * For now, only distinguish 386, 486, Pentium (no SX).
 * The cliplevel values are mostly computed as averages
 * of the higher values for that speed and the lower values
 * for the next speed, but are fudged or interpolated
 * in a few cases.
 */
char	s386[] = "80386";
char	s486[] = "80486";
char	s586[] = "Pentium";
char	s686[] = "Pentium Pro";
char	cpu_model[sizeof("Pentium Pro")];
char	machine[] = MACHINE;

struct cpuident cpuid[] = {
	CPU_386,	45, 		s386,	8,	/* 8 MHz  */
	CPU_386,	65,		s386, 	16,	/* 16 MHz */
	CPU_386,	92,		s386, 	20,	/* 20 MHz */
	CPU_386,	137,		s386, 	25,	/* 25 MHz */
	CPU_386,	190,		s386,	33,	/* 33 MHz */
	CPU_386,	240,		s386,	40,	/* 40 MHz */
	CPU_386,	INT_MAX,	s386, 	0,

	CPU_486,	300,		s486,	8,	/* 8 MHz */
	CPU_486,	458,		s486,	25,	/* 25 MHz */
	CPU_486,	583,		s486,	33,	/* 33 MHz */
	CPU_486,	705,		s486, 	40,	/* 40 MHz DX2 */
	CPU_486,	868,		s486, 	50,	/* 50 MHz */
	CPU_486,	1089,		s486, 	66,	/* 66 MHz DX2 */
	/* the following is high, in part because lots of 66's hit 1060 */
	CPU_486,	1300,		s486, 	75,	/* 75 MHz DX4 */
	CPU_486,	1700,		s486, 	100,	/* 100MHz DX4 */
	CPU_486,	INT_MAX,	s486, 	0,

	CPU_586,	1400,		s586, 	8,	/* 8 MHz */
	CPU_586,	1615,		s586, 	60,	/* 60 MHz */
	CPU_586,	1974,		s586, 	66,	/* 66 MHz */
	CPU_586,	2400,		s586, 	90,	/* 90 MHz */
	CPU_586,	2700,		s586, 	100,	/* 100 MHz */
	CPU_586,	3240,		s586, 	120,	/* 120 MHz */
	/* the following are sheer guesses */
	CPU_586,	3590,		s586, 	133,	/* 133 MHz */
	CPU_586,	4050,		s586, 	150,	/* 150 MHz */
	CPU_586,	INT_MAX,	s586, 	0,	/* unknown */

	/* the following are sheer guesses */
	CPU_686,	1400,		s686, 	8,	/* 8 MHz */
	CPU_686,	3000,		s686, 	133,	/* 133 MHz */
	CPU_686,	INT_MAX,	s686, 	0,	/* unknown */
	-1,		-1, 		0,	0
};

identifycpu()
{
	register struct cpuident *cp;
	extern int cpu_id;		/* from cpuid instruction */
	extern int delay_mult;

	delay_calibrate();

	for (cp = cpuid; cp->cputype >= 0; cp++) {
		if (cpu == cp->cputype && delay_mult <= cp->cliplevel) {
			printf("cpu = %s ", cp->ident);
			if (cp->mhz)
				aprint_normal("(about %d MHz) ", cp->mhz);
#if 0
			else
				aprint_normal("(unknown speed) ");
#endif
			if (cpu_id)
				aprint_verbose("model %d, stepping %d",
				    (cpu_id & 0xf0) >> 4, cpu_id & 0xf);
			printf("\n");
			break;
		}
	}
	if (cp->cputype < 0)
		printf("Unidentified CPU type (%d) and speed\n", cpu);
/* #ifdef DEBUG */
	aprint_verbose("delay multiplier %d\n", delay_mult);
/* #endif */

	/*
	 * Set up cpu_model string, now that we know.
	 */
	strncpy(cpu_model, cp->ident, sizeof(cpu_model));
}

int delay_mult = 640;			/* in case used before calibration */
int delay_usec = 1000000;		/* max delay per loop */

delay(usec)
	int usec;
{
	int thistime;
	volatile int delay_count;

	while (usec > 0) {
		thistime = min(usec, delay_usec);

		delay_count = (thistime * delay_mult + 64) / 128;
		while (delay_count-- > 0)
			;

		usec -= thistime;
	}
}

/*
 * The counter chip counts down from about 12000 to 0 every 10 milliseconds.
 * Wrap around is funny, so we use the counter by waiting until it has
 * a high value, then doing a short delay, then reading the counter to
 * get a delta.  To find the right multipler for the DELAY loop, we start
 * at the minimum, and try every value until a request one millisecond delay
 * actually takes >= 1 millisecond.  Here are some measurements:
 *
 *  Machine              Multiplier
 *  TPE 50 MHz 486          537
 *  Bell Tech 16 MHz 386     42
 *
 * It takes about 1 second of real time to find the multiplier on the TPE,
 * and less than a second on the Bell Tech.  In 100 trials, the TPE got
 * 537 or 536 in every trial, and the Bell Tech got 42 in every trial.
 *
 * The granularity is about 100/multiplier, so it is about 2us for a
 * slow machine, and about 200ns for a fast one.
 *
 * The accuracy of the resulting delays is about 1% (assuming the user
 * doesn't change the TURBO switch!)
 */
#define	MIN_MULT	32		/* smallest believable multiplier */
#define	MAX_MULT	10000
#define	DEF_MULT	1000		/* default, if we can't figure */

delay_calibrate()
{
	int counts_per_millisec;
	int start, end;
	int s;
	char *fail = NULL;

	initrtclock();		/* sets mode 2 */

	s = splhigh();
	/* the following assumes we are using mode 2, not mode 3 */
	counts_per_millisec = (TIMER_FREQ + 500) / 1000;

	for (delay_mult = 1; delay_mult < MAX_MULT; delay_mult++) {
		/*
		 * Make sure that product of delay_mult and delay_usec
		 * in delay() will not overflow.
		 */
		if (delay_mult > (INT_MAX - 64) / delay_usec)
			delay_usec /= 10;
		while ((start = readrtclock()) < 8000 && start != -1)
			;
		delay(1000);
		end = readrtclock();

		if (start == -1 || end == -1) {
			fail = "impossible values from timer 0";
			break;
		}
		if (end > start || (start - end) > counts_per_millisec)
			break;
	}
	splx(s);

	if (fail == NULL) {
		if (delay_mult == MAX_MULT)
			fail = "multiplier hit maximum";
		if (delay_mult < MIN_MULT) {
			printf("delay multiplier %d!\n", delay_mult);
			fail = "unbelievably low multiplier";
		}
	}
	if (fail)
		printf("Warning: delay calibration: %s (timer not working?)\n",
		    fail);
	if (fail || delay_mult < MIN_MULT)
		delay_mult = DEF_MULT;
}

/*
 * Given a starting I/O port number and a count, allow user processes
 * to access those ports directly.  Currently applies to all user processes,
 * eventually should apply only to the calling process.
 */
mapioport(base, cnt, p)
	int base, cnt;
	struct proc *p;
{
	register int i;

	if (i = suser(p->p_ucred, &p->p_acflag))
		return (i);
	for (i = base; i < base + cnt && i < USER_IOPORTS; i++)
		tss.tss_iomap[i / NBBY] &= ~(1 << (i % NBBY));
	return (0);
}

/*
 * Given a starting I/O port number and a count, deny user processes
 * direct access to those ports.  Currently applies to all user processes,
 * eventually should apply only to the calling process.
 */
unmapioport(base, cnt, p)
	int base, cnt;
	struct proc *p;
{
	register int i;

	if (i = suser(p->p_ucred, &p->p_acflag))
		return (i);
	for (i = base; i < base + cnt && i < USER_IOPORTS; i++)
		tss.tss_iomap[i / NBBY] |= 1 << (i % NBBY);
	return (0);
}

/*
 * Enable IOPL privilege for the specified process,
 * assumed to be the current process.  This allows the process
 * to access all I/O ports, disable/enable interrupts, and generally
 * raise havoc.
 */
enable_IOPL(p)
	struct proc *p;
{
	int i;

	if (i = suser(p->p_ucred, &p->p_acflag))
		return (i);
	p->p_md.md_flags |= MDP_IOPL;
	p->p_md.md_regs[sEFLAGS] |= PSL_IOPL;
	return (0);
}

/*
 * Disable IOPL privilege for the specified process,
 * assumed to be the current process.  This requires no special
 * privilege.
 */
disable_IOPL(p)
	struct proc *p;
{

	p->p_md.md_flags &= ~MDP_IOPL;
	p->p_md.md_regs[sEFLAGS] |= PSL_IOPL;
	return (0);
}
