/*-
 * Copyright (c) 1991, 1992, 1993, 1994 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: exec_machdep.c,v 2.1 1995/02/03 07:26:04 polk Exp $
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/exec.h>
#include <sys/user.h>
#include <sys/vnode.h>

#include <vm/vm.h>

#include <machine/cpu.h>
#include <machine/psl.h>
#include <machine/segments.h>

#include <sys/text.h>

#ifdef COFF

#include <sys/malloc.h>
#include <sys/mman.h>
#include <vm/vm_kern.h>
#include <machine/coff_compat.h>

static int exec_aout_shlib __P((struct text *));
static int exec_coff_binary __P((struct text *));
static int load_emulator __P((struct text *));
static struct exec_load **load_file __P((off_t, struct vnode *, vm_offset_t,
    vm_size_t, vm_prot_t, struct exec_load **));

#endif

/*
 * Given the initial page of an executable file,
 * find out what format it uses and convert the header
 * into a series of load commands.
 * This function may call machine-independent code
 * to handle formats common to all architectures
 * (e.g. interpreted scripts, 0413 binaries, etc.).
 */
int
analyze_exec_header(xp)
	struct text *xp;
{
	int (*handler) __P((struct text *));
	int magic;
	int error = 0;

	/*
	 * We loop here because the specific exec_*() function
	 * may not complete the analysis with just one pass.
	 * A script requires at least two passes (to start an interpreter);
	 * conceivably shared library loads may need multiple passes too.
	 */
	do {
		if (xp->x_size < sizeof (unsigned long))
			return (ENOEXEC);

		/*
		 * On the 386, the magic number currently lies
		 * in the first two bytes.
		 */
		magic = *(unsigned short *) xp->x_header;

		switch (magic) {

		/* interpreters (note byte order dependency) */

		case '#' | '!' << 8:
			handler = exec_interpreter;
			break;

		/* BSD native formats */

		case QMAGIC:
#ifdef COFF
			if (xp->x_flags & X_LIBRARY) {
				handler = exec_aout_shlib;
				break;
			}
#endif
			handler = exec_compact_demand_load_binary;
			break;

		case ZMAGIC:
#ifdef COFF
			if (xp->x_flags & X_LIBRARY) {
				handler = exec_aout_shlib;
				break;
			}
#endif
			handler = exec_demand_load_binary;
			break;

		case NMAGIC:
			handler = exec_shared_binary;
			break;

		case OMAGIC:
			handler = exec_unshared_binary;
			break;

		/* foreign formats */

#ifdef COFF
		case COFF_MAGIC:
			handler = exec_coff_binary;
			xp->x_flags |= X_COFF;
			break;
#endif

		default:
			return (ENOEXEC);
		}
	} while ((error = (*handler)(xp)) == EAGAIN);

#ifdef COFF
	if (!error && (xp->x_flags & X_LIBRARY) == 0 &&
	    (xp->x_flags & X_COFF ||
	    xp->x_proc->p_md.md_flags & MDP_EMULATED)) {
		xp->x_proc->p_md.md_flags |= MDP_EMULATED;
		error = load_emulator(xp);
	}
#endif

	return (error);
}

#ifdef COFF

static struct exec_load *
load_init(op, v, size, prot)
	int op;
	vm_offset_t v;
	vm_size_t size;
	vm_prot_t prot;
{
	struct exec_load *lp;

	MALLOC(lp, struct exec_load *, sizeof (*lp), M_TEMP, M_WAITOK);
	lp->el_op = op;
	lp->el_vnode = 0;
	lp->el_offset = 0;
	lp->el_address = v;
	lp->el_length = size;
	lp->el_prot = prot;
	lp->el_attr = MAP_COPY;
	lp->el_next = 0;
	return (lp);
}

static struct exec_load **
load_zero(v, size, lpp)
	vm_offset_t v;
	vm_size_t size;
	struct exec_load **lpp;
{

	*lpp = load_init(EXEC_ZERO, v, roundup(size, CLBYTES), VM_PROT_ALL);
	return (&(*lpp)->el_next);
}

static struct exec_load **
load_clear(vp, v, size, lpp)
	struct vnode *vp;
	vm_offset_t v;
	vm_size_t size;
	struct exec_load **lpp;
{
	struct exec_load *lp;

	lp = load_init(EXEC_CLEAR, v, size, VM_PROT_ALL);
	lp->el_vnode = vp;
	VREF(vp);
	*lpp = lp;
	return (&lp->el_next);
}

static struct exec_load **
load_file(off, vp, v, size, prot, lpp)
	off_t off;
	struct vnode *vp;
	vm_offset_t v;
	vm_size_t size;
	vm_prot_t prot;
	struct exec_load **lpp;
{
	struct exec_load *lp;

	lp = load_init(EXEC_MAP, v, roundup(size, CLBYTES), prot);
	lp->el_vnode = vp;
	VREF(vp);
	lp->el_offset = off;
	*lpp = lp;
	return (&lp->el_next);
}

#define	AOUT_SHLIB_DATA_OFFSET	0x400000

static int
exec_aout_shlib(xp)
	struct text *xp;
{
	struct exec *ep;
	struct exec_load **lpp;
	vm_offset_t base, taddr, daddr;
	off_t toff, doff;

	if (xp->x_vattr.va_uid != 0 ||
	    xp->x_vattr.va_mode & (VWRITE>>3|VWRITE>>6))
		/*
		 * We require that shared libraries be owned by root
		 * and unwritable by group or other.
		 */
		return (EACCES);
	if (xp->x_size < CLBYTES)
		return (ENOEXEC);
	ep = (struct exec *) xp->x_header;
	if (ep->a_text == 0 || xp->x_size < roundup(ep->a_text, CLBYTES))
		return (ENOEXEC);
	if (ep->a_data &&
	    xp->x_size < ep->a_data + roundup(ep->a_text, CLBYTES))
		return (ENOEXEC);

	/* ignored except for emulators */
	exec_set_entry(xp, ep->a_entry);

	lpp = &xp->x_load_commands;

	base = ep->a_entry & ~PDROFSET;
	taddr = base + ((ep->a_magic == QMAGIC) ? CLBYTES : 0);
	toff = (ep->a_magic == QMAGIC) ? 0 : CLBYTES;
	lpp = load_zero(taddr, ep->a_text, lpp);
	lpp = load_file(toff, xp->x_vnode, taddr, ep->a_text,
	    VM_PROT_READ|VM_PROT_EXECUTE, lpp);
	daddr = base + AOUT_SHLIB_DATA_OFFSET;
	doff = toff + roundup(ep->a_text, CLBYTES);
	lpp = load_zero(daddr, ep->a_data + ep->a_bss, lpp);
	lpp = load_file(doff, xp->x_vnode, daddr, ep->a_data, VM_PROT_ALL, lpp);
	*lpp = 0;
	return (0);
}

/*
 * Load a shared library given a COFF shared library section.
 * We recurse by calling analyze_exec_header().
 */
static struct exec_load **
load_shlib(xp, csp, lpp)
	struct text *xp;
	struct coff_scnhdr *csp;
	struct exec_load **lpp;
{
	struct exec_load *lp;
	struct shlib_entry *sebuf, *se, *last_se;
	struct text *lxp;
	size_t len;
	vm_offset_t o, v;
	int resid = 0;

	/* read the shared library section from the binary */
	MALLOC(sebuf, struct shlib_entry *, csp->s_size, M_TEMP, M_WAITOK);
	if (vn_rdwr(UIO_READ, xp->x_vnode, (caddr_t)sebuf, csp->s_size,
	    (off_t)csp->s_scnptr, UIO_SYSSPACE, (IO_UNIT|IO_NODELOCKED),
	    xp->x_proc->p_ucred, &resid, xp->x_proc))
	    	/* XXX how do we pass the errno back? */
	    	return (0);

	/* loop over library names, analyzing and collecting load commands */
	last_se = (struct shlib_entry *)((caddr_t)sebuf + csp->s_size);
	for (se = sebuf;
	    se < last_se;
	    se = (struct shlib_entry *)((long *)se + se->entsz)) {
		if (se->entsz <= 0 || se->pathndx < 2 ||
		    se->pathndx >= se->entsz) {
		    	lpp = 0;
		    	break;
		}

		MALLOC(lxp, struct text *, sizeof (*lxp), M_TEMP, M_WAITOK);
		bzero(lxp, sizeof (*lxp));
		len = (se->entsz - 2) * se->entsz * sizeof (long) + 5;
		MALLOC(lxp->x_path, char *, len, M_TEMP, M_WAITOK);
		bcopy("/sco", lxp->x_path, 4);
		bcopy((long *)se + se->pathndx, lxp->x_path + 4, len - 4);
		lxp->x_path[len - 1] = '\0';	/* paranoia */
		lxp->x_flags = X_PATH_SYSSPACE | X_LIBRARY;
		lxp->x_stack = xp->x_stack;		/* XXX */
		lxp->x_proc = xp->x_proc;

		if (exec_lookup(lxp) || analyze_exec_header(lxp)) {
			delete_text(lxp);
			lpp = 0;
			break;
		}
		if (lp = lxp->x_load_commands) {
			*lpp = lp;
			while (lp->el_next)
				lp = lp->el_next;
			lpp = &lp->el_next;
		}
		lxp->x_load_commands = 0;
		delete_text(lxp);
	}

	FREE(sebuf, M_TEMP);

	/* XXX we can lose errno info here */
	return (lpp);
}

static int
load_emulator(xp)
	struct text *xp;
{
	static char emulator[] = COFF_EMULATOR;
	struct text *lxp;
	struct exec_load *lp;
	vm_offset_t o, v;
	int error;

	MALLOC(lxp, struct text *, sizeof (*lxp), M_TEMP, M_WAITOK);
	bzero(lxp, sizeof (*lxp));
	MALLOC(lxp->x_path, char *, sizeof (emulator), M_TEMP, M_WAITOK);
	bcopy(emulator, lxp->x_path, sizeof (emulator));
	lxp->x_flags = X_PATH_SYSSPACE | X_LIBRARY;
	lxp->x_stack = xp->x_stack;		/* XXX */
	lxp->x_proc = xp->x_proc;

	error = exec_lookup(lxp);
	if (!error)
		error = analyze_exec_header(lxp);
	if (!error && (lp = lxp->x_load_commands) == 0)
		error = ENOEXEC;
	if (!error && lxp->x_flags & X_ENTRY) {
		/*
		 * Add a stack segment (not normally part of a shared library).
		 * XXX I suppose the emulator itself could do this...
		 */
		o = lxp->x_entry & ~PDROFSET;
		*load_zero(o - COFF_STACKSIZE, COFF_STACKSIZE,
		    &lxp->x_load_commands) = lp;
		/*
		 * We start in the emulator, not the emulated program.
		 * However, we provide the emulated program's start address.
		 */
		xp->x_save_entry = xp->x_entry;
		xp->x_entry = lxp->x_entry;
	}
	if (!error) {
		for (lp = xp->x_load_commands; lp->el_next; lp = lp->el_next)
			;
		lp->el_next = lxp->x_load_commands;
		lxp->x_load_commands = 0;
	}
	delete_text(lxp);
	return (error);
}

/*
 * Load a foreign SCO/Interactive COFF binary.
 * Such binaries are implicitly loaded with a shared library
 * that traps system calls and translates them to BSD calls.
 */
int
exec_coff_binary(xp)
	struct text *xp;
{
	struct coff_filehdr *cfp = (struct coff_filehdr *)xp->x_header;
	struct coff_aouthdr *cap = (struct coff_aouthdr *)&cfp[1];
	struct coff_scnhdr *csp, *last_csp;
	struct exec_load *lp, **lpp;
	vm_offset_t v;
	vm_size_t s;
	vm_prot_t prot;
	int error = 0;

	/*
	 * Sanity checking.
	 */
	if (xp->x_size < sizeof (*cfp))
	    	/* header not all present */
	    	return (ENOEXEC);
	if (cfp->f_nscns > (unsigned)0x80000000 / sizeof (struct coff_scnhdr))
		/* ridiculously large section count */
		return (ENOEXEC);
	if (cfp->f_nscns < 3)
		/* at least TEXT, DATA, BSS, per iBCS2 p5-1 */
		return (ENOEXEC);
	if (cfp->f_opthdr < sizeof (*cap))
		/* consistency check */
		return (ENOEXEC);
#if 0
	if (xp->x_size < sizeof (*cfp) + cfp->f_opthdr +
	    cfp->f_nscns * sizeof (struct coff_scnhdr))
	    	/* file too small to contain the full COFF header */
	    	return (ENOEXEC);
#endif
	if ((cfp->f_flags & COFF_F_EXEC) == 0)
		/* executables and shared libs must set this, iBCS2 p4-4 */
		return (ENOEXEC);

	/* XXX limitations of current implementation */
	if (sizeof (*cfp) + cfp->f_opthdr +
	    cfp->f_nscns * sizeof (struct coff_scnhdr) > CLBYTES)
	    	/* XXX header too big; should re-map x_header here */
	    	return (ENOEXEC);
	if (cap->magic != COFF_ZMAGIC && cap->magic != COFF_LMAGIC)
	    	/* XXX no support for COFF_OMAGIC or COFF_NMAGIC now */
	    	return (ENOEXEC);

	/*
	 * Loop over COFF sections and load them.
	 */
	csp = (struct coff_scnhdr *)
	    ((caddr_t)cfp + sizeof (*cfp) + cfp->f_opthdr);
	last_csp = &csp[cfp->f_nscns];
	lpp = &xp->x_load_commands;
	if (lp = *lpp) {
		/* XXX maybe it's time for bidirectional exec_load lists? */
		while (lp->el_next)
			lp = lp->el_next;
		lpp = &lp->el_next;
	}
	xp->x_dsize = 0;	/* build this incrementally */

	for (; csp < last_csp; ++csp) {

		if (csp->s_flags & COFF_STYP_FEATURES) {
			if (csp->s_flags & (COFF_STYP_DSECT|COFF_STYP_NOLOAD))
				/* dummy or unloadable section */
				continue;
			if (csp->s_flags & COFF_STYP_GROUP)
				/* XXX don't understand this yet */
				return (ENOEXEC);
			if (csp->s_flags & COFF_STYP_PAD)
				/* XXX is this the right way? */
				continue;
		}

		switch (csp->s_flags) {

		case COFF_STYP_TEXT:
			prot = VM_PROT_READ|VM_PROT_EXECUTE;
			goto textdata;

		case COFF_STYP_DATA:
			prot = VM_PROT_ALL;
		textdata:
			v = i386_trunc_page(csp->s_vaddr);
			s = i386_round_page(csp->s_vaddr + csp->s_size) - v;
			lpp = load_zero(v, s, lpp);
			lpp = load_file(i386_trunc_page(csp->s_scnptr),
			    xp->x_vnode, v, s, prot, lpp);
			if (prot == VM_PROT_ALL) {
				xp->x_daddr = (caddr_t)v;
				xp->x_dsize += howmany(s, CLBYTES);
			} else {
				xp->x_taddr = (caddr_t)v;
				xp->x_tsize = howmany(s, CLBYTES);
			}
			break;

		case COFF_STYP_BSS:
			/* XXX ISC binaries don't set s_vaddr for bss? */
			/* XXX requires data to precede bss */
			/* XXX assumes no overlap with a following section */
			v = i386_round_page(csp->s_vaddr);
			if (s = v - csp->s_vaddr)
				lpp = load_clear(xp->x_vnode, csp->s_vaddr,
				    s, lpp);
			s = i386_round_page(csp->s_vaddr + csp->s_size) - v;
			if (s) {
				lpp = load_zero(v, s, lpp);
				xp->x_dsize += howmany(s, CLBYTES);
			}
			break;

		case COFF_STYP_LIB:
			if (cap->magic == COFF_LMAGIC)
			    	/* XXX don't permit recursion (yet) */
			    	return (ENOEXEC);

			if (csp->s_size == 0)
				break;
			if ((unsigned)csp->s_size > NBPG)
				/* XXX guard against attack */
				return (ENOEXEC);

			if ((lpp = load_shlib(xp, csp, lpp)) == 0)
				return (ENOEXEC);
			break;

		case COFF_STYP_INFO:
			/* ignore it */
			break;

		default:
			return (ENOEXEC);
			break;
		}
	}

	if (cap->magic != COFF_LMAGIC)
		/* create a stack segment */
		lpp = load_zero(USRSTACK - INITSSIZ, INITSSIZ, lpp);
	if (!error)
		/* XXX honor setuid bits only on emulator file? */
		exec_set_entry(xp, cap->entry);
	return (error);
}

#endif /* COFF */

/*
 * Set up the initial register and signal state for the new image.
 * We don't yet support other OS emulations.
 */
void
exec_set_state(xp, retval)
	struct text *xp;
	int *retval;
{
	extern union descriptor ldt[];
	int argc = 0;
	struct proc *p = xp->x_proc;
	struct exec_arg *eap;
	struct syscframe *sfp = (struct syscframe *) p->p_md.md_regs;
	struct ps_strings pss;

#ifdef COPY_SIGCODE
	extern char sigcode[];
	extern int szsigcode;

	copyout(sigcode, xp->x_stack, szsigcode);
#endif

	/*
	 * Compute argc so we can put it on the stack.
	 * This is redundant -- the startup code computes this and
	 * throws it away!
	 */
	for (eap = xp->x_args; eap->ea_string; eap = eap->ea_next)
		++argc;

	pss.ps_argv = (char **)xp->x_stack_top;
	pss.ps_argc = argc;
	pss.ps_envp = pss.ps_argv + argc + 1;
	pss.ps_nenv = xp->x_arg_count - (argc + 2);
	copyout(&pss, (caddr_t)PS_STRINGS, sizeof (pss));

	xp->x_stack_top -= sizeof (argc);
	copyout(&argc, xp->x_stack_top, sizeof (argc));

#ifdef COFF
	/*
	 * Nasty hacks to pass information to an emulator.
	 * EDI tells the emulator whether it needs to emulate iBCS2 or BSD.
	 * ESI tells the emulator the start address of the program.
	 */
	sfp->sf_edi = (xp->x_flags & X_COFF ? 1 : 0);
	sfp->sf_esi = (xp->x_proc->p_md.md_flags & MDP_EMULATED ?
	    xp->x_save_entry : 0);
#else
	sfp->sf_edi = 0;
	sfp->sf_esi = 0;
#endif
	sfp->sf_ebp = (int) xp->x_stack_top;
	sfp->sf_ebx = (int) PS_STRINGS;
	sfp->sf_ecx = 0;
	sfp->sf_eip = xp->x_entry;
	sfp->sf_esp = (int) xp->x_stack_top;

	retval[0] = 0;		/* EAX */
	retval[1] = 0;		/* EDX */

	/*
	 * Initialize floating point state; use default on first access.
	 */
	p->p_addr->u_pcb.pcb_flags &= ~FP_WASUSED;

	/*
	 * Reset the default system call descriptor.
	 * We may have been using redirected system calls.
	 * XXX save a syscall and set up COFF binaries here?
	 */
	ldt[LDEFCALLS_SEL] = ldt[L43BSDCALLS_SEL];
	curpcb->pcb_ldtdefcall = ldt[LDEFCALLS_SEL];
}
