/*-
 * Copyright (c) 1992, 1993, 1994, 1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI $Id: st.c,v 2.7 1995/12/12 22:28:58 karels Exp $
 */

/*
 * Machine-independent SCSI tape driver
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/errno.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/malloc.h>
#include <sys/mtio.h>
#include <sys/proc.h>
#include <sys/tprintf.h>
#include <sys/tape.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/disktape.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/scsi_ioctl.h>
#include <dev/scsi/tape.h>

struct st_softc {
	struct	tpdevice sc_tp;	/* base tape device, must be first */
#define	TP_EXABYTE	0x8000	/* special Exabyte flag */
	struct	unit sc_unit;	/* scsi unit */

	dev_t	sc_devt;	/* for b_dev in tape op calls */
	long	sc_resid;	/* save hardware resid byte count on error */

	int	sc_ansi;	/* ANSI X3.131 conformance level */
	int	sc_reclen;	/* sticky value for fixed record length */

	/* should be in tpdevice?? */
	struct	buf sc_tab;	/* transfer queue */

	/* for user formatting and tape ops (?!) */
	struct	scsi_fmt sc_fmt;
};

void stattach __P((struct device *parent, struct device *self, void *aux));
static int stblkno __P((struct st_softc *sc));
static int stcache __P((struct st_softc *sc, int mode));
int stclose __P((dev_t dev, int flags, int ifmt, struct proc *p));
static int stcommand __P((struct st_softc *sc, struct scsi_cdb *cdb,
    caddr_t buf, int len));
static int stdensity __P((struct st_softc *sc, int flags, int d));
int stdump __P((dev_t dev, daddr_t blkoff, caddr_t addr, int len));
static int sterror __P((struct st_softc *sc, int stat));
void stgo __P((struct device *sc0, struct scsi_cdb *cdb));
void stigo __P((struct device *sc0, struct scsi_cdb *cdb));
void stintr __P((struct device *sc0, int stat, int resid));
int stioctl __P((dev_t dev, int cmd, caddr_t data, int flag, struct proc *p));
int stmatch __P((struct device *parent, struct cfdata *cf, void *aux));
static int stoffline __P((struct st_softc *sc));
int stopen __P((dev_t dev, int flags, int ifmt, struct proc *p));
void streset __P((struct unit *u));
static int strewind __P((struct st_softc *sc));
static int stspace __P((struct st_softc *sc, int mark, int count));
static void ststart __P((struct st_softc *sc, struct buf *bp));
void ststrategy __P((struct buf *bp));
static void sttype __P((struct st_softc *sc, char *vendor, char *rev));
static int stweof __P((struct st_softc *sc, int count));

#ifdef notyet
static struct sttpdriver = { ststrategy };
#endif

struct cfdriver stcd = 
    { NULL, "st", stmatch, stattach, DV_TAPE, sizeof(struct st_softc) };

static struct unitdriver stunitdriver = { /* stgo, stintr, */ streset };

struct devsw stsw = {
	&stcd,
	stopen, stclose, rawread, rawwrite, stioctl, seltrue, nommap,
	ststrategy, stdump, nopsize, B_TAPE,
	nostop
};

int
stmatch(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	struct scsi_attach_args *sa = aux;

	if (cf->cf_loc[0] != -1 && cf->cf_loc[0] != sa->sa_unit)
		return (0);
	if ((sa->sa_inq_status & STS_MASK) != STS_GOOD)
		return (0);
	if ((sa->sa_si.si_type & TYPE_TYPE_MASK) != TYPE_SAD)
		return (0);
	return (1);
}

/*
 * Classify different types of tape devices.
 * This routine currently has a rather naive view of the world.
 * Note that we get called once with a vendor name at attach time,
 * and we also get called at open time to get current parameters.
 */
static void
sttype(sc, vendor, rev)
	struct st_softc *sc;
	char *vendor;
	char *rev;
{
	struct scsi_ms_bd *mbd;
	struct tpdevice *tp = &sc->sc_tp;
	int i;
	int pc = PC_DEFAULT;
	union {
		struct {
			struct scsi_ms6 ms;
			struct scsi_ms_bd bd[1];
		} mode;
		char buf[128];
	} modebuf;

	sc->sc_reclen = 1024;

	/*
	 * If this device is SCSI-1, it won't respond to
	 * a MODE SENSE for its default parameters.
	 * If we recognize the vendor name, use that instead.
	 * Crude but fairly effective...
	 */
	if (sc->sc_ansi == 1) {
		if (strcmp(vendor, "EXABYTE") == 0) {
			tp->tp_flags |= TP_8MMVIDEO|TP_EXABYTE;
			return;
		}
		/* XXX should we test <= ? on revisions?  maybe others? */
		if (strcmp(vendor, "SANKYO") == 0 ||
		    strcmp(vendor, "WangDAT") == 0 ||
		    (strcmp(vendor, "WANGTEK") == 0 &&
		    rev != NULL && strcmp(rev, "C230") == 0)) {
			tp->tp_flags |= TP_NODENS;
			printf(" (no density support)");
		}
		/* the Archive has a special block address command */
		if (bcmp(vendor, "ARCHIVE VIPER", 13) == 0)
			tp->tp_flags |= TP_PHYS_ADDR;

		return;
	}
	if (strcmp(vendor, "EXABYTE") == 0) {
		/*
		 * Exabyte drives which claim to be SCSI-2
		 * are non-conformant in at least a couple of ways.
		 * (1) A page control value other than PC_CURRENT is illegal.
		 * (2) When reporting default values with PC_CURRENT,
		 * the drive sends a density code of 0 (grrr...),
		 * so the density/format detection code fails.
		 * We just hack in the correct format.
		 */
		pc = PC_CURRENT;
		tp->tp_flags |= TP_8MMVIDEO|TP_EXABYTE;
	}

	i = scsi_mode_sense(&sc->sc_unit, modebuf.buf, sizeof modebuf,
	    pc, 0);
	if ((i & STS_MASK) != STS_GOOD) {
#ifdef DEBUG
		printf(" (mode sense failed)");
#endif
		return;
	}

	if (modebuf.mode.ms.ms_bdl == 0) {
		printf(" (no block descriptor found)");
		return;
	}
	mbd = modebuf.mode.bd;
	tp->tp_reclen = (mbd->bd_blh << 16) | (mbd->bd_blm << 8) | mbd->bd_bll;
	if (tp->tp_reclen)
		tp->tp_flags |= TP_STREAMER;
	else
		tp->tp_flags &= ~TP_STREAMER;

	switch (mbd->bd_dc) {
	case SCSI_MS_DC_9T_800BPI:
		printf(" 800"); goto reel;
	case SCSI_MS_DC_9T_1600BPI:
		printf(" 1600"); goto reel;
	case SCSI_MS_DC_9T_3200BPI:
		printf(" 3200"); goto reel;
	case SCSI_MS_DC_9T_6250BPI:
		printf(" 6250");
	reel:
		printf(" bpi reel-to-reel");
		tp->tp_flags |= TP_DOUBLEEOF;
		break;
	case SCSI_MS_DC_HIC_XX5:
	case SCSI_MS_DC_HIC_XX6:
	case SCSI_MS_DC_HI_TC1:
	case SCSI_MS_DC_HI_TC2:
	case SCSI_MS_DC_HIC_XXB:
	case SCSI_MS_DC_HIC_XXC:
		printf(" 1/2-inch cartridge");
		break;
	case SCSI_MS_DC_QIC_XX1:
	case SCSI_MS_DC_QIC_XX2:
	case SCSI_MS_DC_QIC_XX3:
	case SCSI_MS_DC_QIC_XX7:
		printf(" QIC-11 or QIC-24 cartridge");
		break;
	case SCSI_MS_DC_QIC_120:
		printf(" QIC-120 cartridge");
		break;
	case SCSI_MS_DC_QIC_150:
		printf(" QIC-150 cartridge");
		break;
	case SCSI_MS_DC_QIC_320:
		printf(" QIC-320 cartridge");
		break;
	case SCSI_MS_DC_QIC_1350:
		printf(" QIC-1350 cartridge");
		break;
	case SCSI_MS_DC_CS_XX4:
		printf(" 4-track cassette");
		break;
	case SCSI_MS_DC_DDS:
		printf(" DDS cassette");
		tp->tp_flags |= TP_8MMVIDEO;
		break;
	case SCSI_MS_DC_DDS2:
		printf(" DDS2 cassette");
		tp->tp_flags |= TP_8MMVIDEO;
		break;
	case SCSI_MS_DC_8MM_XX9:
	case SCSI_MS_DC_8MM_XXA:
		printf(" 8mm video cassette");
		tp->tp_flags |= TP_8MMVIDEO;
		if (strcmp(vendor, "EXABYTE") == 0)
			tp->tp_flags |= TP_EXABYTE;
		break;
	case SCSI_MS_DC_QIC_MINI:	/* this might not be standard */
		printf(" QIC Minicartridge");
		tp->tp_flags |= TP_NODENS;	/* Nothing is selectable */
		break;
	case 0:
		/* some drives won't provide a default density code */
		break;
	default:
		printf(" unknown format 0x%x", mbd->bd_dc);
		break;
	}
	if (tp->tp_flags & TP_STREAMER)
		printf(" streaming (%d)", tp->tp_reclen);
	printf(" tape");
}

void
stattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct st_softc *sc = (struct st_softc *)self;
	struct scsi_attach_args *sa = aux;
	int ansi;
	char vendor[10], drive[17], rev[5];

	sc->sc_unit.u_driver = &stunitdriver;
	scsi_establish(&sc->sc_unit, &sc->sc_tp.tp_dev, sa->sa_unit);

	aprint_naive(": SCSI tape");
	ansi = (sa->sa_si.si_version >> VER_ANSI_SHIFT) & VER_ANSI_MASK;
	if (ansi == 1 || ansi == 2) {
		scsi_inq_ansi((struct scsi_inq_ansi *)&sa->sa_si,
		    vendor, drive, rev);
		aprint_normal(": %s %s", vendor, drive);
		if (rev[0])
			aprint_verbose(" rev %s", rev);
		if (ansi == 2)
			aprint_verbose(" (SCSI-2)");
		else if ((sa->sa_si.si_v2info & V2INFO_RDF_MASK) ==
		    V2INFO_RDF_CCS)
			aprint_verbose(" (CCS)");	/* XXX never seen one */
		else
			aprint_verbose(" (SCSI-1)");
		sc->sc_ansi = ansi;
	} else {
		bcopy("<unknown>", vendor, 10);
		bcopy("<unknown>", drive, 10);
		aprint_normal(": type 0x%x, qual 0x%x, ver %d",
		    sa->sa_si.si_type, sa->sa_si.si_qual,
		    sa->sa_si.si_version);
		sc->sc_ansi = 1;		/* XXX */    
	}

	sttype(sc, vendor, rev);
	printf("\n");
}

void
streset(u)
	struct unit *u;
{
	struct st_softc *sc = (struct st_softc*)u->u_dev;

	if (sc->sc_tp.tp_flags & TP_MOVED)
		sc->sc_tp.tp_flags |= TP_MUST_BOT;
}

/*
 * Execute a (possibly blocking) tape operation with a prepared cdb.
 * XXX do we need to pass a read/write indication too?
 */
static int
stcommand(sc, cdb, buf, len)
	struct st_softc *sc;
	struct scsi_cdb *cdb;
	caddr_t buf;
	int len;
{
	struct buf *bp;
	int error;

	MALLOC(bp, struct buf *, sizeof *bp, M_DEVBUF, M_WAITOK);
	bp->b_flags = B_BUSY;		/* XXX B_READ? */
	bp->b_error = 0;
	bp->b_dev = sc->sc_devt;
	bp->b_resid = 0;
	bp->b_proc = curproc;
	bp->b_iodone = 0;
	bp->b_chain = 0;
	bp->b_vp = 0;
	bp->b_un.b_addr = buf;
	bp->b_bcount = bp->b_iocount = len;
	bp->b_blkno = 0;

	sc->sc_fmt.sf_format_pid = curproc ? curproc->p_pid : -1;
	sc->sc_fmt.sf_cmd = *cdb;

	ststrategy(bp);
	error = biowait(bp);

	FREE(bp, M_DEVBUF);
	sc->sc_fmt.sf_format_pid = 0;

	return (error);
}

static int
strewind(sc)
	struct st_softc *sc;
{
	struct scsi_cdb cdb = { CMD_REWIND };

	sc->sc_tp.tp_flags &= ~(TP_WRITING|TP_MOVED|TP_SEENEOF|TP_MUST_BOT);
	sc->sc_tp.tp_fileno = 0;
	cdb.cdb_bytes[1] = sc->sc_unit.u_unit << 5;
	return (stcommand(sc, &cdb, (caddr_t)0, 0));
}

static int
stoffline(sc)
	struct st_softc *sc;
{
	struct scsi_cdb cdb = { CMD_LOAD_UNLOAD };

	sc->sc_tp.tp_flags &= ~(TP_WRITING|TP_MOVED|TP_SEENEOF|TP_MUST_BOT);
	sc->sc_tp.tp_fileno = 0;
	cdb.cdb_bytes[1] = sc->sc_unit.u_unit << 5;
	return (stcommand(sc, &cdb, (caddr_t)0, 0));
}

static int
stspace(sc, code, count)
	struct st_softc *sc;
	int code, count;
{
	struct scsi_cdb cdb = { CMD_SPACE };

	if (sc->sc_tp.tp_flags & TP_MUST_BOT)
		return (EIO);

	sc->sc_tp.tp_flags &= ~(TP_WRITING|TP_SEENEOF);
	/* XXX what if we backspace to BOT? */
	sc->sc_tp.tp_flags |= TP_MOVED;
	if (code == SCSI_CMD_SPACE_FMS)
		sc->sc_tp.tp_fileno += count;
	cdb.cdb_bytes[1] = sc->sc_unit.u_unit << 5 | code;
	cdb.cdb_bytes[2] = count >> 16;
	cdb.cdb_bytes[3] = count >> 8;
	cdb.cdb_bytes[4] = count;
	return (stcommand(sc, &cdb, (caddr_t)0, 0));
}

static int
stweof(sc, count)
	struct st_softc *sc;
	int count;
{
	struct scsi_cdb cdb = { CMD_WRITE_FILEMARK };

	if (sc->sc_tp.tp_flags & TP_MUST_BOT)
		return (EIO);

	sc->sc_tp.tp_flags |= TP_WRITING|TP_MOVED;
	sc->sc_tp.tp_fileno += count;
	cdb.cdb_bytes[1] = sc->sc_unit.u_unit << 5;
	cdb.cdb_bytes[2] = count >> 16;
	cdb.cdb_bytes[3] = count >> 8;
	cdb.cdb_bytes[4] = count;
	return (stcommand(sc, &cdb, (caddr_t)0, 0));
}

static int
stcache(sc, mode)
	struct st_softc *sc;
	int mode;
{
	union {
		/* assume exactly one block descriptor */
		struct {
			struct scsi_ms6 ms;
			struct scsi_ms_bd bd[1];
		} mode;
		char buf[128];
	} modebuf;
	int i, error;

resense:
	i = scsi_mode_sense(&sc->sc_unit, modebuf.buf, sizeof modebuf, 0, 0);
	if ((i & STS_MASK) != STS_GOOD && (error = sterror(sc, i))) {
		if (error == EAGAIN)
			goto resense;
		return (error);
	}

	modebuf.mode.ms.ms_dsp &= SCSI_MS_DSP_SPEED_MASK;
	modebuf.mode.ms.ms_dsp |= mode;

reselect:
	i = scsi_mode_select(&sc->sc_unit, modebuf.buf, sizeof modebuf, 0, 0);
	if ((i & STS_MASK) != STS_GOOD && (error = sterror(sc, i))) {
		if (error == EAGAIN)
			goto reselect;
		return (error);
	}
	return (0);
}

static int
stdensity(sc, flags, d)
	struct st_softc *sc;
	int flags, d;
{
	struct scsi_ms_bd *mbd;
	struct tpdevice *tp = &sc->sc_tp;
	union {
		struct {
			struct scsi_ms6 ms;
			struct scsi_ms_bd bd[1];
		} mode;
		char buf[128];
	} modebuf;
	int i, select_len;

	/*
	 * Get parameters from the drive.
	 * We do this regardless of whether we can set them.
	 */
	i = scsi_mode_sense(&sc->sc_unit, modebuf.buf, sizeof modebuf, 0, 0);
	if ((i & STS_MASK) != STS_GOOD)
		return (i);

	if (modebuf.mode.ms.ms_bdl == 0)
		return (-1);	/* XXX never happens? */
	mbd = modebuf.mode.bd;
	tp->tp_reclen = (mbd->bd_blh << 16) | (mbd->bd_blm << 8) | mbd->bd_bll;
	if (tp->tp_reclen)
		tp->tp_flags |= TP_STREAMER;
	else
		tp->tp_flags &= ~TP_STREAMER;

	if (tp->tp_flags & TP_NODENS)
		/* Tapes with bad density selection */
		return (STS_GOOD);

	mbd->bd_dc = (d & T_DENSEL) >> 1;

	select_len = sizeof(modebuf.mode.ms) + modebuf.mode.ms.ms_bdl;
	if (tp->tp_flags & TP_8MMVIDEO) {
		if (d & T_FIXED) {
			tp->tp_flags |= TP_STREAMER;
			mbd->bd_blh = sc->sc_reclen >> 16;
			mbd->bd_blm = (sc->sc_reclen >> 8) & 0xff;
			mbd->bd_bll = sc->sc_reclen & 0xff;
		} else {
			/* use variable length records */
			tp->tp_reclen = 0;
			tp->tp_flags &= ~TP_STREAMER;
			mbd->bd_blh = mbd->bd_blm = mbd->bd_bll = 0;
		}
		if (tp->tp_flags & TP_EXABYTE) {
			/* Exabyte insists on zero length in mode select */
			mbd->bd_nbh = mbd->bd_nbm = mbd->bd_nbl = 0;
			/* set NBE in Exabyte vendor-unique parameters page */
			modebuf.buf[12] |= 8;
			select_len = modebuf.mode.ms.ms_len +
			    sizeof(modebuf.mode.ms.ms_len);
		}
	}
	if (select_len > sizeof(modebuf))
		select_len = sizeof(modebuf);

	return (scsi_mode_select(&sc->sc_unit, modebuf.buf, select_len, 0, 0));
}

static int
stblkno(sc)
	struct st_softc *sc;
{
	struct scsi_cdb rp = { CMD_READ_POSITION };
	int i;

	if (sc->sc_ansi == 2) {
		struct scsi_rp pos;

		/*
		 * Report the logical block number.
		 */
		rp.cdb_bytes[1] = sc->sc_unit.u_unit << 5;
		i = (*sc->sc_unit.u_hbd->hd_icmd)(sc->sc_unit.u_hba,
		    sc->sc_unit.u_targ, &rp, (char *)&pos, sizeof pos,
		    B_READ);
		if ((i & STS_MASK) != STS_GOOD)
			return (-1);
		if (pos.rp_flags & SCSI_RP_BPU)
			return (-1);
		return ((pos.rp_fblh << 24) | (pos.rp_fblhm << 16) |
		    (pos.rp_fbllm << 8) | pos.rp_fbll);
	}

	/*
	 * If it's a Viper, opcode #2 returns the physical block number.
	 * Is this a CCS thing, an Archive thing, something else?
	 */
	if (sc->sc_tp.tp_flags & TP_PHYS_ADDR) {
		u_char posbuf[20];

		rp.cdb_bytes[0] = 0x2;
		i = (*sc->sc_unit.u_hbd->hd_icmd)(sc->sc_unit.u_hba,
		    sc->sc_unit.u_targ, &rp, (char *)posbuf, 3, B_READ);
		if ((i & STS_MASK) != STS_GOOD)
			return (-1);
		return ((posbuf[0] << 16) | (posbuf[1] << 8) | posbuf[0]);
	}

	return (-1);
}

int
stopen(dev, flags, ifmt, p)
	dev_t dev;
	int flags, ifmt;
	struct proc *p;
{
	struct st_softc *sc;
	int unit = tpunit(dev);
	int i;
	int error;

	if (unit >= stcd.cd_ndevs)
		return (ENXIO);
	if ((sc = stcd.cd_devs[unit]) == NULL)
		/* XXX try attaching on the fly? */
		return (ENXIO);
	if (sc->sc_tp.tp_flags & TP_OPEN)
		return (EBUSY);
	/* do this now, in case we block */
	sc->sc_tp.tp_flags |= TP_OPEN;

	/*
	 * Send most error messages to the controlling tty,
	 * since they typically don't require administrator attention.
	 * If there is no controlling tty, output goes to the console.
	 */
	sc->sc_tp.tp_ctty = tprintf_open(p);

again:
	i = scsi_test_unit_ready(sc->sc_unit.u_hba, sc->sc_unit.u_targ,
	    sc->sc_unit.u_unit);
	if ((i & STS_MASK) != STS_GOOD && (error = sterror(sc, i))) {
		/* XXX try a load command if the tape appears to be offline? */
		if (error == EAGAIN)
			goto again;
		tprintf_close(sc->sc_tp.tp_ctty);
		sc->sc_tp.tp_flags &= ~TP_OPEN;
		return (error);
	}
	sc->sc_devt = dev;

	/*
	 * Set tape density.  The SCSI-2 standard defines
	 * 24 different density codes; it's tough to map
	 * these onto small numbers for the set that any
	 * given drive supports (e.g. there are a half dozen
	 * different QIC formats) so we don't bother.
	 * The subunit from the minor device number
	 * contains 8 bits of density code.
	 * Zero is the default density.
	 */
density:
	i = stdensity(sc, flags, dv_subunit(dev));
	if ((i & STS_MASK) != STS_GOOD && (error = sterror(sc, i))) {
		if (error == EAGAIN)
			goto density;
		tprintf(sc->sc_tp.tp_ctty,
		    "%s: unsupported density selection\n",
		    sc->sc_tp.tp_dev.dv_xname);
		tprintf_close(sc->sc_tp.tp_ctty);
		sc->sc_tp.tp_flags &= ~TP_OPEN;
		return (error);
	}

	/*
	 * On a QIC streamer, we can only write from BOT or from EOD.
	 * If we need to write and we're not at BOT, move to EOD.
	 * XXX what if we want to read and then write?
	 */
	if (flags & FWRITE &&
	    (sc->sc_tp.tp_flags & (TP_STREAMER|TP_8MMVIDEO|TP_MOVED))
	     == (TP_STREAMER|TP_MOVED))
		stspace(sc, SCSI_CMD_SPACE_PEOD, 1);

	return (0);
}

int
stclose(dev, flags, ifmt, p)
	dev_t dev;
	int flags, ifmt;
	struct proc *p;
{
	struct st_softc *sc = stcd.cd_devs[tpunit(dev)];
	int error = 0;

	if (sc->sc_tp.tp_flags & TP_WRITING)
		error = stweof(sc, sc->sc_tp.tp_flags & TP_DOUBLEEOF ? 2 : 1);
	if (!tpnorew(dev))
		if (!error)
			error = strewind(sc);
		else
			(void)strewind(sc);
	tprintf_close(sc->sc_tp.tp_ctty);
	sc->sc_tp.tp_flags &= ~(TP_OPEN|TP_WRITING|TP_SEENEOF);
	sc->sc_fmt.sf_format_pid = 0;	/* XXX exclusive open... */
	/* XXX pick up sense data for 'mt status'? */
	return (error);
}

void
ststrategy(bp)
	struct buf *bp;
{
	struct st_softc *sc = stcd.cd_devs[tpunit(bp->b_dev)];
	struct buf *dp = &sc->sc_tab;
	int s;

	bp->av_forw = NULL;
	s = splbio();
	if (dp->b_actf == 0)
		dp->b_actf = bp;
	else
		dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	if (dp->b_active == 0) {
		dp->b_active = 1;
		ststart(sc, bp);
	}
	splx(s);
}

/*
 * Report errors to the user, rather than the console.
 * Tape errors are typically not critical system problems.
 */
static int
sterror(sc, stat)
	struct st_softc *sc;
	int stat;
{
	struct scsi_sense *sn;
	struct tpdevice *tp = &sc->sc_tp;
	char *s = tp->tp_dev.dv_xname;
	char *m;
	tpr_t t = tp->tp_ctty;
	int key;
	int asc, ascq, retries;

	sc->sc_resid = 0;
	sc->sc_fmt.sf_sense.status = stat;
	if ((stat & STS_MASK) == STS_CHECKCOND) {
		sn = (struct scsi_sense *)sc->sc_fmt.sf_sense.sense;
		stat = scsi_request_sense(sc->sc_unit.u_hba,
		    sc->sc_unit.u_targ, sc->sc_unit.u_unit,
		    (caddr_t)sn, sizeof *sn);
		if ((stat & STS_MASK) != STS_GOOD) {
			sc->sc_fmt.sf_sense.status = stat;	/* ??? */
			tprintf(t, "%s: sense failed, status %x\n", s, stat);
			return (EIO);
		}
		if (!SENSE_ISXSENSE(sn) || !XSENSE_ISSTD(sn)) {
			tprintf(t, "%s: scsi sense class %d, code %d\n",
			    s, SENSE_ECLASS(sn), SENSE_ECODE(sn));
			return (EIO);
		}
		if (XSENSE_IVALID(sn)) {
			sc->sc_resid = XSENSE_INFO(sn);
			if (tp->tp_flags & TP_STREAMER)
				sc->sc_resid *= tp->tp_reclen;
		}
		key = XSENSE_KEY(sn);
		if (key == SKEY_ATTENTION) {
			if (tp->tp_flags & TP_MOVED)
			    tprintf(t, "%s: tape changed without rewind\n", s);
			tp->tp_flags &= ~(TP_MOVED|TP_WRITING);
			tp->tp_fileno = 0;
			return (EAGAIN);
		}
		if (key != SKEY_NONE && key != SKEY_BLANK) {
			asc = XSENSE_HASASC(sn) ? XSENSE_ASC(sn) : 0;
			ascq = XSENSE_HASASCQ(sn) ? XSENSE_ASCQ(sn) : 0;
			if (XSENSE_HASRETRIES(sn))
				retries = XSENSE_RETRIES(sn);
			else
				retries = -1;

			if (asc == 0 && ascq == 0) {
				if (retries >= 0)
					tprintf(t, "%s: %s (%d retries)\n",
					    s, SCSISENSEKEYMSG(key), retries);
				else
					tprintf(t, "%s: %s\n",
					    s, SCSISENSEKEYMSG(key));
			} else {
				if (retries >= 0)
					tprintf(t, "%s: %s: %s (%d retries)\n",
					    s, SCSISENSEKEYMSG(key),
					    scsi_translate_asc(asc, ascq),
					    retries);
				else
					tprintf(t, "%s: %s: %s\n",
					    s, SCSISENSEKEYMSG(key),
					    scsi_translate_asc(asc, ascq));
			}
			if (key != SKEY_RECOVERED)
				return (EIO);
		}
		if (XSENSE_ILI(sn)) {
			/* XXX use the SILI bit? */
			if (tp->tp_flags & TP_STREAMER) {
				/* fixed length record mismatch */
				tprintf(t, "%s: short record (%d bytes)\n",
				    s, tp->tp_reclen - sc->sc_resid);
				return (EIO);
			}
			if (sc->sc_resid < 0) {
			    tprintf(t, "%s: overlength error, excess = %d\n",
				s, -sc->sc_resid);
				sc->sc_resid = 0;
				return (EIO);
			}
		}
		if (XSENSE_FM(sn)) {
			/* EOF on read */
			tp->tp_flags |= TP_SEENEOF;
			return (0);
		}
		if (XSENSE_EOM(sn)) {
			tprintf(t, "%s: end of medium\n", s);
			return (ENOSPC);
		}
		return (0);
	}
	if (sc->sc_fmt.sf_sense.status == -1)
		tprintf(t, "%s: host adapter error\n", s);
#ifdef DEBUG
	else if ((sc->sc_fmt.sf_sense.status & STS_MASK) != STS_GOOD)
		tprintf(t, "%s: scsi status %d\n", s,
		    sc->sc_fmt.sf_sense.status);
#endif
	return ((sc->sc_fmt.sf_sense.status & STS_MASK) != STS_GOOD ?
	    EAGAIN : 0);
}

static void
ststart(sc, bp)
	struct st_softc *sc;
	struct buf *bp;
{

	/*
	 * If we've loaded a SCSI command into sf_cmd and
	 * the table indicates that it completes quickly,
	 * use polling to get the result.
	 */
	if (sc->sc_fmt.sf_format_pid &&
	    scsi_legal_cmds[sc->sc_fmt.sf_cmd.cdb_bytes[0]] > 0) {
		(*sc->sc_unit.u_start)(sc->sc_unit.u_updev,
		    &sc->sc_unit.u_forw, (struct buf *)NULL,
		    stigo, &sc->sc_tp.tp_dev);
		return;
	}
	(*sc->sc_unit.u_start)(sc->sc_unit.u_updev, &sc->sc_unit.u_forw,
	    bp, stgo, &sc->sc_tp.tp_dev);
}

void
stigo(sc0, cdb)
	struct device *sc0;
	struct scsi_cdb *cdb;
{
	struct st_softc *sc = (struct st_softc *)sc0;
	struct buf *bp = sc->sc_tab.b_actf;
	int error;
	int stat;

again:
	sc->sc_resid = 0;
	if (sc->sc_tp.tp_flags & TP_MUST_BOT) {
		bp->b_flags |= B_ERROR;
		bp->b_error = EIO;
	} else { 
		stat = (*sc->sc_unit.u_hbd->hd_icmd)(sc->sc_unit.u_hba,
		    sc->sc_unit.u_targ, &sc->sc_cmd, bp->b_un.b_addr,
		    bp->b_iocount, bp->b_flags & B_READ);
		sc->sc_sense.status = stat;
		if ((stat & STS_MASK) != STS_GOOD && 
		    (error = sterror(sc, stat))) {
			if (error == EAGAIN)
				goto again;
			bp->b_flags |= B_ERROR;
			bp->b_error = error;
		}
	}

	(*sc->sc_unit.u_rel)(sc->sc_unit.u_updev);
	bp->b_resid = sc->sc_resid;
	sc->sc_tab.b_errcnt = 0;
	sc->sc_tab.b_actf = bp->b_actf;
	biodone(bp);
	/* XXX what happens if sf_format_pid is set? */
	if ((bp = sc->sc_tab.b_actf) == NULL)
		sc->sc_tab.b_active = 0;
	else
		ststart(sc, bp);
}

void
stgo(sc0, cdb)
	struct device *sc0;
	struct scsi_cdb *cdb;
{
	struct st_softc *sc = (struct st_softc *)sc0;
	struct buf *bp = sc->sc_tab.b_actf;
	int n, resid;

	if (sc->sc_format_pid)
		*cdb = sc->sc_fmt.sf_cmd;
	else {
		if (bp->b_flags & B_READ) {
			sc->sc_tp.tp_flags &= ~TP_WRITING;
			if (sc->sc_tp.tp_flags & TP_SEENEOF) {
				bp->b_resid = bp->b_iocount;
				sc->sc_tp.tp_flags &= ~TP_SEENEOF;
				++sc->sc_tp.tp_fileno;
				goto next;
			}
		} else
			sc->sc_tp.tp_flags |= TP_WRITING;
		sc->sc_tp.tp_flags |= TP_MOVED;
		sc->sc_tp.tp_flags &= ~TP_SEENEOF;
		cdb->cdb_bytes[0] = bp->b_flags & B_READ ?
		    CMD_READ : CMD_WRITE;
		cdb->cdb_bytes[1] = sc->sc_unit.u_unit << 5;
		n = bp->b_iocount;
		if (sc->sc_tp.tp_flags & TP_STREAMER) {
			/*
			 * Fixed block tape drives may begin with an
			 * initial residual count (of bytes not even
			 * attempted) due to a transfer being something
			 * other than the right block size.  We will
			 * adjust the transfer count downward here too.
			 * Thus, any residual count in stintr() must be
			 * added to bp->b_resid, rather than overwriting
			 * it (and we might need to bump b_bcount and
			 * b_iocount in stintr by any resid counted here,
			 * just to be nice...).
			 */
			cdb->cdb_bytes[1] |= 1;		/* fixed records */
			n /= sc->sc_tp.tp_reclen;
			resid = bp->b_iocount - (n * sc->sc_tp.tp_reclen);
			if (resid) {
				tprintf(sc->sc_tp.tp_ctty,
    "%s: record length %d is not a multiple of tape record length %d\n",
				    sc->sc_tp.tp_dev.dv_xname,
				    bp->b_iocount, sc->sc_tp.tp_reclen);
				bp->b_resid = resid;
				bp->b_iocount -= resid;
				bp->b_bcount -= resid;
			}
		}
		cdb->cdb_bytes[2] = n >> 16;
		cdb->cdb_bytes[3] = n >> 8;
		cdb->cdb_bytes[4] = n;
		cdb->cdb_bytes[5] = 0;
	}
	if ((!(sc->sc_tp.tp_flags & TP_MUST_BOT)) &&
	    ((*sc->sc_unit.u_go)(sc->sc_unit.u_updev, sc->sc_unit.u_targ,
	    stintr, (void *)sc, bp, 0) == 0))
		return;

	bp->b_flags |= B_ERROR;
	bp->b_error = EIO;
	bp->b_resid = 0;
next:
	(*sc->sc_unit.u_rel)(sc->sc_unit.u_updev);
	sc->sc_tab.b_errcnt = 0;
	sc->sc_tab.b_actf = bp->b_actf;
	biodone(bp);
	/* XXX what happens if sf_format_pid is set? */
	if ((bp = sc->sc_tab.b_actf) == NULL)
		sc->sc_tab.b_active = 0;
	else
		ststart(sc, bp);
}

void
stintr(sc0, stat, hbaresid)
	struct device *sc0;
	int stat, hbaresid;
{
	struct st_softc *sc = (struct st_softc *)sc0;
	struct buf *bp = sc->sc_tab.b_actf;
	int error, xsresid, resid, initresid;

	if (bp == NULL)
		panic("stintr");
	sc->sc_resid = 0;	/* can be set by sterror() */

	if (stat == HBAINTR_BUSRESET) {
		bp->b_flags |= B_ERROR;
		bp->b_error = EIO;
		stat = STS_GOOD;
	}
	if ((stat & STS_MASK) != STS_GOOD && (error = sterror(sc, stat))) {
		if (error == EAGAIN)
			goto again;
		bp->b_flags |= B_ERROR;
		bp->b_error = error;
	}
	xsresid = sc->sc_resid;
	resid = xsresid ? xsresid : hbaresid;
#if 1 /* ??? needs testing */
	if ((u_int)resid > bp->b_iocount) {
printf("%s: excessive resid, count = %d, hba resid = %d, xsense resid = %d\n",
		    sc->sc_tp.tp_dev.dv_xname, bp->b_iocount,
		    hbaresid, xsresid);
		resid = bp->b_iocount;
	}
#endif
	if (resid == bp->b_iocount) {
		/* we'll see it this time rather than next time */
		if (sc->sc_tp.tp_flags & TP_SEENEOF)
			++sc->sc_tp.tp_fileno;
		sc->sc_tp.tp_flags &= ~TP_SEENEOF;
	}
	initresid = bp->b_resid;
	bp->b_iocount += initresid;
	bp->b_bcount += initresid;
	bp->b_resid = initresid + resid;
	sc->sc_tab.b_errcnt = 0;
	sc->sc_tab.b_actf = bp->b_actf;
	biodone(bp);
	if ((bp = sc->sc_tab.b_actf) == NULL)
		sc->sc_tab.b_active = 0;
	else
again:
		ststart(sc, bp);
}

int
stioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	struct st_softc *sc = stcd.cd_devs[tpunit(dev)];
	u_char *sn;
	struct mtop *op;
	struct mtget *mtget;
	int cnt;
	int error;

	switch (cmd) {

	/*
	 * XXX ALL SCSI 'FORMAT' CODE SHOULD GET MERGED...
	 * XXX WE SHOULD ALSO CHANGE THE NAME TO SOMETHING OTHER THAN 'FORMAT'
	 * XXX SINCE IT DOES A LOT MORE THAN FORMATTING, EVEN WITH DISKS.
	 * XXX WE NEED TO CHANGE IOCTL NAMES TOO (SDIOC... => SCSIIOC...?).
	 */

	case SDIOCSFORMAT:
		if (suser(p->p_ucred, &p->p_acflag))
			return (EPERM);
		if (*(int *)data) {
			if (sc->sc_fmt.sf_format_pid)
				return (EPERM);
			sc->sc_fmt.sf_format_pid = p->p_pid;
		} else
			sc->sc_fmt.sf_format_pid = 0;
		break;

	case SDIOCGFORMAT:
		*(int *)data = sc->sc_fmt.sf_format_pid;
		break;

	case MTIOCSFBS:
		if ((!(sc->sc_tp.tp_flags & TP_8MMVIDEO)) ||
		    *(int *)data > MAXPHYS)
			return (EINVAL);
		sc->sc_reclen = *(int *)data;
		break;

	case MTIOCGFBS:
		*(int *)data = sc->sc_reclen;
		break;

	case SDIOCSCSICOMMAND:
#define cdb ((struct scsi_cdb *)data)
		if (sc->sc_fmt.sf_format_pid != p->p_pid)
			return (EPERM);
		if (scsi_legal_cmds[cdb->cdb_bytes[0]] == 0)
			return (EINVAL);
		sc->sc_fmt.sf_cmd = *cdb;
#undef	cdb
		break;

	case SDIOCSENSE:
		*(struct scsi_fmt_sense *)data = sc->sc_fmt.sf_sense;
		break;

	case MTIOCTOP:
		/*
		 * XXX a normal tape op mixed with SCSI ops will
		 * XXX wipe out the SCSI format_pid state...
		 */
		error = 0;
		op = (struct mtop *)data;
		cnt = op->mt_count;
		switch(op->mt_op) {

		case MTWEOF:
			error = stweof(sc, cnt);
			break;
		case MTFSR:
			error = stspace(sc, SCSI_CMD_SPACE_BLOCKS, cnt);
			break;
		case MTBSR:
			error = stspace(sc, SCSI_CMD_SPACE_BLOCKS, -cnt);
			break;
		case MTFSF:
			error = stspace(sc, SCSI_CMD_SPACE_FMS, cnt);
			break;
		case MTBSF:
			error = stspace(sc, SCSI_CMD_SPACE_FMS, -cnt);
			break;
		case MTREW:
			error = strewind(sc);
			break;
		case MTOFFL:
			error = stoffline(sc);
			break;
		case MTNOP:
			/* XXX get status? */
			break;
		case MTCACHE:
			error = stcache(sc, SCSI_MS_DSP_BUFFERED);
			break;
		case MTNOCACHE:
			error = stcache(sc, SCSI_MS_DSP_UNBUFFERED);
			break;

		default:
			return (EINVAL);
		}
		return (error);

	case MTIOCGET:
		mtget = (struct mtget *)data;
		sn = sc->sc_fmt.sf_sense.sense;
		mtget->mt_type = 0;		/* XXX */
		mtget->mt_dsreg = scsi_request_sense(sc->sc_unit.u_hba,
		    sc->sc_unit.u_targ, sc->sc_unit.u_unit,
		    (caddr_t)sn, sizeof(sc->sc_fmt.sf_sense.sense));
		mtget->mt_erreg = sn[0] << 8 | sn[2];
		mtget->mt_resid = sc->sc_resid;
		mtget->mt_fileno = sc->sc_tp.tp_fileno;
		mtget->mt_blkno = stblkno(sc);
		break;

	default:
		return (EINVAL);
	}

	return (0);
}

int
stdump(dev, blkoff, addr, len)
	dev_t dev;
	daddr_t blkoff;
	caddr_t addr;
	int len;
{
#ifdef notyet
	should be fairly similar to sddump()
#endif
	return (EIO);
}
