/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_shm.c,v 2.1 1995/02/03 15:15:17 polk Exp
 */

/*
 * Shared memory support for the iBCS2/SCO emulator.
 */

#include <sys/param.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <err.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "sco_ipc.h"

#ifndef IPC_LIBRARY

#include "emulate.h"
#include "sco.h"

/* iBCS2 p 3-36 */
#define	SCO_SHMAT	0
#define	SCO_SHMCTL	1
#define	SCO_SHMDT	2
#define	SCO_SHMGET	3

#endif

struct shaddr {
	struct shaddr	*next;
	void		*addr;
};

void *
shmat(id, addr, flags)
	int id;
	void *addr;
	int flags;
{
	char path[MAXPATHLEN];
	struct ipc_chain *c;
	struct ipc_object *o;
	struct shaddr *sp;
	unsigned long va;
	void *v;
	int save_errno;
	int f;
	int rdonly = flags & SHM_RDONLY;

	if ((c = __ipc_chain_by_id(SCO_IPC_SHM, id)) == 0)
		return ((void *)-1);
	o = c->obj;

	if (__ipc_perm(&o->p.perm, rdonly) == -1)
		return ((void *)-1);

	__ipc_path(SCO_IPC_SHM, id, path);
	if ((f = open(path, O_RDWR)) == -1) {
		if (errno == ENOENT)
			/* someone ran IPC_RMID on this segment */
			errno = EINVAL;
		return ((void *)-1);
	}

	va = (unsigned long)addr;
	if (va && flags & SHM_RND)
		va &= ~(SHMLBA - 1);
	if ((v = mmap((caddr_t)va, o->p.shm.shm_segsz,
	    PROT_READ | (rdonly ? 0 : PROT_WRITE) | PROT_EXEC,
	    MAP_FILE | MAP_SHARED | (addr ? MAP_FIXED : 0),
	    f, (off_t)NBPG)) == (caddr_t)-1) {
		save_errno = errno;
		close(f);
		errno = save_errno;
		return (v);
	}

	o->p.shm.shm_lpid = getpid();
	++o->p.shm.shm_nattch;
	time(&o->p.shm.shm_atime);
	msync((caddr_t)o, NBPG);

	if ((sp = malloc(sizeof (*sp))) == 0)
		err(1, "shmat shaddr malloc");

	sp->addr = v;
	sp->next = c->u.shmx.sha;
	c->u.shmx.sha = sp;

	close(f);

	return (v);
}

int
__ipc_shm_init(c, f, created, len)
	struct ipc_chain *c;
	int f, created, len;
{
	struct shmid_ds *dp = &c->obj->p.shm;
	off_t off;

	if (created) {
		off = NBPG + roundup(len, NBPG) - 1;
		if (lseek(f, off, SEEK_SET) == -1 || write(f, "", 1) == -1)
			return (-1);
		dp->shm_segsz = len;
		dp->shm_lpid = getpid();
		dp->shm_cpid = dp->shm_lpid;
		dp->shm_nattch = 0;
		dp->shm_cnattch = 0;
		dp->shm_atime = 0;
		dp->shm_dtime = 0;
		time(&dp->shm_ctime);
	}
	return (0);
}

int
__ipc_shm_work(c, indx, cookie, buf)
	struct ipc_chain *c;
	int indx, cookie;
	void *buf;
{
	struct shaddr *sp, *sp_next;
	struct ipc_object *o = c->obj;

	switch (cookie) {

	case IPC_SET:
		o->p.shm.shm_lpid = getpid();
		time(&o->p.shm.shm_ctime);
		break;

	case IPC_RMID:
		for (sp = c->u.shmx.sha; sp; sp = sp->next) {
			munmap(sp->addr, o->p.shm.shm_segsz);
			sp_next = sp->next;
			free(sp);
		}
		break;
	}
	return (0);
}

int
#ifdef __STDC__
shmctl(int id, int cookie, ...)
#else
shmctl(va_alist)
	va_dcl
#endif
{
	struct shmid_ds *sdp;
	va_list ap;
#ifndef __STDC__
	int id, cookie;

	va_start(ap);
	id = va_arg(ap, int);
	cookie = va_arg(ap, int);
#else
	va_start(ap, cookie);
#endif
	sdp = cookie == IPC_RMID ? 0 : va_arg(ap, struct shmid_ds *);
	va_end(ap);

	return (__ipc_ctl(SCO_IPC_SHM, id, 0, cookie, sdp));
}

int
shmdt(addr)
	void *addr;
{
	struct ipc_chain *c;
	struct shmid_ds *dp;
	struct shaddr *sp, **spp;

	for (c = __ipc_chains[SCO_IPC_SHM]; c; c = c->next)
		for (spp = &c->u.shmx.sha; *spp; spp = &(*spp)->next)
			if ((*spp)->addr == addr)
				goto out;
	errno = EINVAL;
	return (-1);

out:
	dp = &c->obj->p.shm;
	if (munmap(addr, dp->shm_segsz) == -1)
		return (-1);
	dp->shm_lpid = getpid();
	--dp->shm_nattch;
	dp->shm_dtime = time((time_t *)0);

	sp = *spp;
	*spp = sp->next;
	free(sp);

	return (0);
}

int
shmget(k, len, flags)
	key_t k;
	int len, flags;
{

	return (__ipc_get(SCO_IPC_SHM, k, len, flags));
}

#ifndef IPC_LIBRARY

int
sco_shm(cookie, a, b, c)
	int cookie, a, b, c;
{

	switch (cookie) {
	case SCO_SHMAT:
		return ((int)shmat(a, (void *)b, c));
	case SCO_SHMCTL:
		return (shmctl(a, b, (void *)c));
	case SCO_SHMDT:
		return (shmdt((void *)a));
	case SCO_SHMGET:
		return (shmget((key_t)a, b, c));
	}

	errno = EINVAL;
	return (-1);
}

#endif
