/*
 * Copyright (c) 1993,1994,1995 Berkeley Software Design, Inc.
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_sem.c,v 2.3 1995/12/19 22:49:31 donn Exp
 */

/*
 * Support for semaphore ops
 */

#include <sys/param.h>
#include <sys/mman.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "sco_ipc.h"

#ifndef IPC_LIBRARY

#include "emulate.h"
#include "sco.h"

/* iBCS2 p 3-35 */
#define	SCO_SEMCTL	0
#define	SCO_SEMGET	1
#define	SCO_SEMOP	2

#define	__ipc_init_library()

#else

#ifdef DEBUG

#define	debug	__ipc_debug

int debug;

#define	DEBUG_SEMAPHORES	0x03	/* XXX */

#endif

#endif

extern char **environ;

struct semadj {
	struct semadj	*next;
	unsigned short	num;
	unsigned short	seq;
	unsigned short	val;
};

void
__ipc_encode_semadj()
{
	struct ipc_chain *c;
	struct semadj **sapp, *sa;
	char **envp, **oenvp;
	char *p;
	int count = 0;

	/* estimate the number of enironment variables needed */
	for (envp = environ; *envp; ++envp) {
		if (**envp == '_' && strncmp(*envp, "_SEMADJ_", 8) == 0) {
			**envp = '\0';
			continue;
		}
		++count;
	}
	for (c = __ipc_chains[SCO_IPC_SEM]; c; c = c->next)
		if (c->u.semx.adj)
			++count;

	/* copy old environment variables, stripping old semadj variables */
	if ((envp = malloc((count + 1) * 4)) == 0)
		err(1, "__ipc_encode_semadj environ");

	for (oenvp = environ, environ = envp; *oenvp; ++oenvp)
		if (**oenvp)
			*envp++ = *oenvp;

	/* append new semadj environment variables */
	for (c = __ipc_chains[SCO_IPC_SEM]; c; c = c->next) {

		for (count = 0, sapp = &c->u.semx.adj; sa = *sapp; )
			if (sa->seq != c->u.semx.sems[sa->num]._semseq)
				/* eliminate stale semadj values */
				*sapp = sa->next;
			else {
				sapp = &sa->next;
				++count;
			}

		if (count == 0)
			continue;

		/*
		 * strlen("_SEMADJ_") + 8 hex bytes of id + strlen("=") +
		 * 3 * (4 hex bytes) * count + 1 nul byte
		 */
		if ((p = *envp++ = malloc(18 + 12*count)) == 0)
			err(1, "__ipc_encode_semadj string");

		p += sprintf(p, "_SEMADJ_%X=", c->obj->id);
		for (sa = c->u.semx.adj; sa; sa = sa->next)
			p += sprintf(p, " %x %x %x", sa->num,
			    sa->seq, sa->val);
		*p = '\0';
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_encode_semadj: %s", envp[-1]);
#endif
	}

	*envp = 0;
}

void
__ipc_decode_semadj()
{
	struct ipc_chain *c;
	struct semadj *sa, **sapp;
	char **envp;
	char *p, *e;
	int semid;

	for (envp = environ; *envp; ++envp) {
		if (**envp != '_' || strncmp(*envp, "_SEMADJ_", 8))
			continue;

		p = *envp + 8;
		semid = strtoul(p, &e, 16);
		if (e == p || *e != '=') {
			warnx("bad semadj environment variable: %s", *envp);
			continue;
		}
		p = e + 1;
		if (*p == '\0')
			continue;

		if ((c = __ipc_chain_by_id(SCO_IPC_SEM, semid)) == 0)
			continue;

		sapp = &c->u.semx.adj;
		do {
			if ((sa = malloc(sizeof *sa)) == 0)
				err(1, "__ipc_decode_semadj semadj");

			sa->num = strtoul(p, &e, 16);
			if (e == p)
				break;
			p = e + 1;
			sa->seq = strtoul(p, &e, 16);
			if (e == p)
				break;
			p = e + 1;
			sa->val = strtoul(p, &e, 16);
			if (e == p)
				break;
			p = e + 1;

#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("__ipc_decode_semadj: semadj(%d, %d, %hd)",
				    semid, sa->num, sa->val);

#endif
			*sapp = sa;
			sapp = &sa->next;
		} while (*p);

		if (e == p)
			free(sa);
		*sapp = 0;
	}
}

static void
destroy_semadj(c)
	struct ipc_chain *c;
{
	struct semadj *sa, *nsa;

	for (sa = c->u.semx.adj; sa; sa = nsa) {
		nsa = sa->next;
		free(sa);
	}
	c->u.semx.adj = 0;
}

void
__ipc_destroy_all_semadj()
{
	struct ipc_chain *c;

	for (c = __ipc_chains[SCO_IPC_SEM]; c; c = c->next)
		if (c->u.semx.adj)
			destroy_semadj(c);
}

void
__ipc_apply_semadj()
{
	struct ipc_chain *c;
	struct semadj *sa;

	for (c = __ipc_chains[SCO_IPC_SEM]; c; c = c->next) {
		if (c->u.semx.adj == 0)
			continue;
		__ipc_get_lock_nointr(c->obj);
		for (sa = c->u.semx.adj; sa; sa = sa->next) {
			if (sa->seq == c->u.semx.sems[sa->num]._semseq) {
#ifdef DEBUG
				if (debug & DEBUG_SEMAPHORES)
					warnx("__ipc_apply_semadj: apply(%d, %d, %hd) semval=%hd",
					    c->obj->id, sa->num, sa->val,
					    c->u.semx.sems[sa->num].semval);
#endif
				c->u.semx.sems[sa->num].semval += sa->val;
			}
		}
		__ipc_clear_lock(c->obj);
	}
}

static void
clear_semadj(c, n)
	struct ipc_chain *c;
	int n;
{
	struct semadj *sa, **sapp;

	for (sapp = &c->u.semx.adj; sa = *sapp; sapp = &sa->next)
		if (sa->num >= n) {
			if (sa->num == n) {
				*sapp = sa->next;
				free(sa);
			}
			break;
		}
}

static void
update_semadj(c, n, value)
	struct ipc_chain *c;
	int n;
	int value;
{
	struct semadj *sa, **sapp;

	for (sapp = &c->u.semx.adj; sa = *sapp; sapp = &sa->next)
		if (sa->num >= n)
			break;
	if (sa == 0 || sa->num > n) {
		if ((sa = malloc(sizeof *sa)) == 0)
			errx(1, "semadj malloc");
		sa->next = *sapp;
		sa->num = n;
		sa->val = 0;
		*sapp = sa;
	}
#ifdef DEBUG
	if (debug & DEBUG_SEMAPHORES)
		warnx("update_semadj: update(%d, %d, %hd) semadj=%hd",
		    c->obj->id, n, value, sa->val);
#endif
	sa->seq = c->u.semx.sems[n]._semseq;
	sa->val -= value;
}

#ifdef IPC_LIBRARY

/*
 * Since we don't get a hook into the startup code in BSD programs,
 * we must initialize before every semaphore operation.
 */
void
__ipc_init_library()
{
	static int initialized = 0;
	char *s;

	if (initialized == 0) {
		++initialized;
#ifdef DEBUG
		if (s = getenv("EMULATE_DEBUG"))
			debug = atoi(s);
#endif
		__ipc_decode_semadj();
	}
}

/*
 * A hack for a hack...
 */
#define	sig_pending_mask	__sig_pending_mask
int __sig_pending_mask;

#endif

int
__ipc_sem_check(c, indx, cookie, v)
	struct ipc_chain *c;
	int indx, cookie;
	void *v;
{
	struct ipc_perm *ip = &c->obj->p.perm;
	int rdonly = 1;

	switch (cookie) {

	case SETVAL:
	case SETALL:
		rdonly = 0;
		/* FALL THROUGH */

	case GETVAL:
	case GETPID:
	case GETNCNT:
	case GETZCNT:
	case GETALL:
		if (__ipc_perm(ip, rdonly) == -1) {
#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("__ipc_sem_check: op %d, perm(%d) failed",
				    cookie, c->obj->id);
#endif
			return (-1);
		}
		break;
	default:
		errno = EINVAL;
		return (-1);
	}

	/* check ranges */
	switch (cookie) {
	case GETVAL:
	case SETVAL:
	case GETPID:
	case GETNCNT:
	case GETZCNT:
		if (indx < 0 || indx >= c->obj->p.sem.sem_nsems) {
#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("__ipc_sem_check: op %d, range(%d) failed",
				    cookie, c->obj->id);
#endif
			errno = EINVAL;
			return (-1);
		}
		break;
	}
	if (cookie == SETVAL && ((int)v < 0 || (int)v >= USHRT_MAX)) {
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_check: SETVAL, range(%d) = %d failed",
			    c->obj->id, v);
#endif
		errno = ERANGE;
		return (-1);
	}

	return (0);
}

int
__ipc_sem_work(c, indx, cookie, buf)
	struct ipc_chain *c;
	int indx, cookie;
	void *buf;
{
	struct ipc_object *o = c->obj;
	unsigned short *v;
	struct sem *sm, *last_sm;

	switch (cookie) {

	case IPC_SET:
		time(&o->p.sem.sem_ctime);
		return (0);

	case IPC_RMID:
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_work: IPC_RMID(%u)", c->obj->id);
#endif
		destroy_semadj(c);
		/* try to unshare and free the mapped semaphore region */
		if (mmap((caddr_t)c->u.semx.sems,
		    roundup(o->p.sem.sem_nsems * sizeof (struct sem), NBPG),
		    PROT_READ|PROT_WRITE|PROT_EXEC,
		    MAP_ANON|MAP_FIXED|MAP_PRIVATE, -1, NBPG) != (caddr_t)-1)
			free(c->u.semx.sems);
		return (0);

	case GETVAL:
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_work: GETVAL(%d, %d) = %hd",
			    c->obj->id, indx, c->u.semx.sems[indx].semval);
#endif
		return (c->u.semx.sems[indx].semval);

	case SETVAL:
		__ipc_get_lock_nointr(o);
		c->u.semx.sems[indx].semval = (int)buf;
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_work: SETVAL(%d, %d) = %hd",
			    c->obj->id, indx, c->u.semx.sems[indx].semval);
#endif
		++c->u.semx.sems[indx]._semseq;
		__ipc_clear_lock(o);
		clear_semadj(c, indx);
		return (0);

	case GETPID:
		return (c->u.semx.sems[indx].sempid);

	case GETNCNT:
		return (c->u.semx.sems[indx].semncnt);

	case GETZCNT:
		return (c->u.semx.sems[indx].semzcnt);

	case GETALL:
		/* XXX take a lock on semaphore array? */
		/* XXX API does not specify atomic operation */
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_work: GETALL(%d)", c->obj->id);
#endif
		last_sm = &c->u.semx.sems[o->p.sem.sem_nsems];
		v = (unsigned short *)buf;
		for (sm = c->u.semx.sems; sm < last_sm; ++sm)
			*v++ = sm->semval;
		return (0);

	case SETALL:
#ifdef DEBUG
		if (debug & DEBUG_SEMAPHORES)
			warnx("__ipc_sem_work: SETALL(%d)", c->obj->id);
#endif
		last_sm = &c->u.semx.sems[o->p.sem.sem_nsems];
		__ipc_get_lock_nointr(o);
		v = (unsigned short *)buf;
		for (sm = c->u.semx.sems; sm < last_sm; ++sm) {
			++sm->_semseq;
			sm->semval = *v++;
		}
		__ipc_clear_lock(o);
		destroy_semadj(c);
		return (0);
	}

	errno = EINVAL;
	return (-1);
}

/*
 * XXX: System V makes some fairly interesting assumptions here
 * about passing unions as arguments vs. passing other parameters...
 */
int
#ifdef __STDC__
semctl(int id, int indx, int cookie, ...)
#else
semctl(va_alist)
	va_dcl
#endif
{
	void *v;
	va_list ap;
#ifndef __STDC__
	int id, indx, cookie;

	va_start(ap);
	id = va_arg(ap, int);
	indx = va_arg(ap, int);
	cookie = va_arg(ap, int);
#else
	va_start(ap, cookie);
#endif

	switch (cookie) {
	case SETVAL:
		v = (void *)va_arg(ap, int);
		break;
	case GETALL:
	case SETALL:
		v = va_arg(ap, unsigned short *);
		break;
	case IPC_STAT:
	case IPC_SET:
		v = va_arg(ap, struct semid_ds *);
		break;
	default:
		v = 0;
		break;
	}
	va_end(ap);

	__ipc_init_library();

	return (__ipc_ctl(SCO_IPC_SEM, id, indx, cookie, v));
}

int
__ipc_sem_init(c, f, created, n)
	struct ipc_chain *c;
	int f, created, n;
{
	struct semid_ds *dp = &c->obj->p.sem;
	int len;

	if (created)
		dp->sem_nsems = n;
	len = roundup(dp->sem_nsems * sizeof (struct sem), NBPG);
	if (created) {
		if (lseek(f, NBPG + len - 1, SEEK_SET) == -1 ||
		    write(f, "", 1) == -1)
			return (-1);
		dp->sem_otime = 0;
		time(&dp->sem_ctime);
		__ipc_init_lock(c->obj);
	}
	if ((c->u.semx.sems = valloc(len)) == 0)
		return (-1);
	if (mmap((caddr_t)c->u.semx.sems, len, PROT_READ|PROT_WRITE,
	    MAP_SHARED|MAP_FIXED, f, NBPG) == (caddr_t)-1) {
		free(c->u.semx.sems);
		return (-1);
	}
#ifdef DEBUG
	if (debug & DEBUG_SEMAPHORES)
		warnx("__ipc_sem_init: initialize(%d, %d)", c->obj->id,
		    dp->sem_nsems);
#endif
	return (0);
}

void
__ipc_sem_remove(c)
	struct ipc_chain *c;
{

	__ipc_clear_lock(c->obj);
}

int
semget(key, len, flags)
	key_t key;
	int len, flags;
{

	return (__ipc_get(SCO_IPC_SEM, key, len, flags));
}

/*
 * To handle SEM_UNDO, each semaphore ID is associated with a linked
 * list of pending semadj operations.  (We use a list because we don't
 * expect SEM_UNDO'ed semaphores to be dense; is this assumption
 * wrong?)  Each semadj operation contains a semaphore number, a semadj
 * value and a sequence number; the list is sorted by semaphore number.
 * When we alter a semaphore with a SEM_UNDO semop, we check to see
 * if the current semadj value for this semaphore is stale (if its
 * sequence number doesn't match, we clear it), then we add the current
 * sem_op to the semadj value in the semadj list.  When we (or the
 * image we exec!) exit, and the sequence number has not changed, we
 * subtract the semadj value from the semaphore.  Operations that
 * clear semadj values then simply increment the sequence number.
 *
 * PROBLEMS:
 *
 * +	If we die as a consequence of a fatal signal, we won't update
 *	semaphores with semadj values.  If this becomes a problem for
 *	support, we can arrange to catch (most) fatal signals and call
 *	exit(), or arrange to fork a process that waits for the child
 *	and then updates semaphores on child exit.
 *
 * +	If many semaphores are very active, the current BSD semphaore
 *	implementation will make us spin inefficiently.
 *
 * +	The current code depends on implementation details of the
 *	BSD semaphore implementation from Keith Bostic, most notably
 *	in the hack to wake up after anyone changes a semaphore.
 */
int
semop(id, sbuf, len)
	int id;
	struct sembuf *sbuf;
	size_t len;
{
	extern int sig_pending_mask;		/* XXX */
	struct ipc_chain *c;
	struct ipc_object *o;
	struct sem *sm;
	struct sembuf *sb;
	int rdonly = 1;
	int blocked;
	int save_errno = 0;

	__ipc_init_library();

	if ((c = __ipc_chain_by_id(SCO_IPC_SEM, id)) == 0)
		return (-1);
	o = c->obj;

#ifdef DEBUG
	if (debug & DEBUG_SEMAPHORES)
		warnx("semop: start(%d), count=%d", o->id, len);
#endif

	for (sb = sbuf; sb < &sbuf[len]; ++sb) {
		if (sb->sem_num >= o->p.sem.sem_nsems) {
#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("semop: range error, num = %d",
				    sb->sem_num);
#endif
			errno = EFBIG;
			return (-1);
		}
		if (sb->sem_op != 0)
			rdonly = 0;
	}

	if (__ipc_perm(&o->p.perm, rdonly) == -1)
		return (-1);

	if (__ipc_get_lock(o) == -1)
		return (-1);

	do {
		blocked = 0;

		/* If we'll block and shouldn't, bag it */
		for (sb = sbuf; sb < &sbuf[len]; ++sb) {
			sm = &c->u.semx.sems[sb->sem_num];
			if (sb->sem_flg & IPC_NOWAIT &&
			    ((sb->sem_op == 0 && sm->semval != 0) ||
			    (sb->sem_op < 0 && sm->semval < -sb->sem_op))) {
#ifdef DEBUG
				if (debug & DEBUG_SEMAPHORES)
					warnx("semop: nowait(%d, %d, %hd) semval=%hd",
					    o->id, sb->sem_num,
					    sb->sem_op, sm->semval);
#endif
				__ipc_yield_lock(o);
				errno = EAGAIN;
				return (-1);
			}
		}

		/* Try to get through the entire list without blocking */
		for (sb = sbuf; sb < &sbuf[len]; ++sb) {
			if (sb->sem_op > 0)
				continue;

			sm = &c->u.semx.sems[sb->sem_num];

			if (sb->sem_op == 0) {
				if (sm->semval == 0)
					continue;
				++sm->semzcnt;
			} else {
				if (sm->semval >= -sb->sem_op)
					continue;
				++sm->semncnt;
			}

			blocked = 1;

#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("semop: blocked(%d, %d, %hd) semval=%hd",
				    o->id, sb->sem_num, sb->sem_op, sm->semval);
#endif

			/*
			 * Let someone else use the object.
			 * We have to get the lock back later
			 * in spite of interruptions because
			 * we need to repair semzcnt/semncnt
			 * before leaving the emulator.
			 */
			if (__ipc_suspend_lock(o) == -1) {
				save_errno = errno;
				__ipc_get_lock_nointr(o);
			}

			if (sig_pending_mask && save_errno == 0)
				save_errno = EINTR;

			if (sb->sem_op == 0)
				--sm->semzcnt;
			else
				--sm->semncnt;

			if (save_errno) {
#ifdef DEBUG
				if (debug & DEBUG_SEMAPHORES)
					warnx("semop: error(%d): %s", o->id,
					    strerror(save_errno));
#endif
				__ipc_yield_lock(o);
				errno = save_errno;
				return (-1);
			}

			/* start over */
			break;
		}
	} while (blocked);

	/* We have the lock and all operations are ready; do the work */
	for (sb = sbuf; sb < &sbuf[len]; ++sb) {
		sm = &c->u.semx.sems[sb->sem_num];
		if (sb->sem_op > 0 ||
		   (sb->sem_op < 0 && sm->semval >= -sb->sem_op)) {
#ifdef DEBUG
			if (debug & DEBUG_SEMAPHORES)
				warnx("semop: apply(%d, %d, %hd) semval=%hd",
				    o->id, sb->sem_num, sb->sem_op, sm->semval);
#endif
			sm->semval += sb->sem_op;
			if (sb->sem_flg & SEM_UNDO)
				update_semadj(c, sb->sem_num, sb->sem_op);
		}
	}

	time(&o->p.sem.sem_otime);
	msync((caddr_t)o, NBPG);

#ifdef DEBUG
	if (debug & DEBUG_SEMAPHORES)
		warnx("semop: succeeded(%d)", o->id);
#endif

	__ipc_clear_lock(o);

	return (0);
}

#ifndef IPC_LIBRARY

int
sco_sem(cookie, a, b, c, d)
	int cookie, a, b, c, d;
{

	switch (cookie) {
	case SCO_SEMCTL:
		return (semctl(a, b, c, d));
	case SCO_SEMGET:
		return (semget((long)a, b, c));
	case SCO_SEMOP:
		return (semop(a, (struct sembuf *)b, (unsigned int)c));
	}

	errno = EINVAL;
	return (-1);
}

#endif
