/*
 * Copyright (c) 1993,1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_msg.c,v 2.4 1995/12/19 22:49:30 donn Exp
 */

/*
 * Support for message queue ops
 */

#include <sys/param.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "sco_ipc.h"

#ifdef IPC_LIBRARY

/*
 * A hack for a hack...
 */
#define	sig_pending_mask	__sig_pending_mask
int __sig_pending_mask;

#else

#include "emulate.h"
#include "sco.h"
#include "sco_errno.h"

/* iBCS2 p 3-35 */
#define	SCO_MSGGET	0
#define	SCO_MSGCTL	1
#define	SCO_MSGRCV	2
#define	SCO_MSGSND	3

#endif

/* XXX if message data average 2x size of overhead, this will cover the queue */
#define	DEFAULT_MSQ_LEN	98304

struct msg_head {
	long		qnext;	/* offset of next queue element */
	long		bprev;	/* offset of previous block */
	long		bnext;	/* offset of next block */
	long		msize;	/* message length */
	long		mtype;	/* message type */
	/* char		buf[]; */	/* message */
};

int
__ipc_msg_init(c, f, created, len)
	struct ipc_chain *c;
	int f, created, len;
{
	struct ipc_object *o = c->obj;
	struct msqid_ds *dp = &o->p.msg.msgd;
	struct msg_overhead *mo = &o->p.msg.msgo;
	struct msg_head *h;

	if (created) {
		if (lseek(f, NBPG + DEFAULT_MSQ_LEN - 1, SEEK_SET) == -1 ||
		    write(f, "", 1) == -1)
			return (-1);

		dp->msg_cbytes = 0;
		dp->msg_qnum = 0;
		dp->msg_qbytes = USHRT_MAX;
		dp->msg_lspid = 0;
		dp->msg_lrpid = 0;
		dp->msg_stime = 0;
		dp->msg_rtime = 0;
		time(&dp->msg_ctime);

		mo->len = DEFAULT_MSQ_LEN;
		mo->head = -1;
		mo->tail = -1;

		__ipc_init_lock(o);
	}

	if ((c->u.msgx.buf = valloc(DEFAULT_MSQ_LEN)) == 0)
		return (-1);

	if (mmap((caddr_t)c->u.msgx.buf, mo->len, PROT_READ|PROT_WRITE,
	    MAP_SHARED|MAP_FIXED, f, NBPG) == (caddr_t)-1) {
		free(c->u.msgx.buf);
		return (-1);
	}
	c->u.msgx.len = mo->len;

	if (created) {
		h = (struct msg_head *)c->u.msgx.buf;
		h->qnext = -1;
		h->bprev = 0;
		h->bnext = mo->len;
		h->msize = 0;
		h->mtype = 0;
	}

	return (0);
}

void
__ipc_msg_remove(c)
	struct ipc_chain *c;
{

	__ipc_clear_lock(c->obj);
}

int
msgget(key, flags)
	key_t key;
	int flags;
{

	return (__ipc_get(SCO_IPC_MSG, key, 0, flags));
}

#define	HEAD(c, n)	((struct msg_head *)((c)->u.msgx.buf + (n)))
#define	HOFF(c, h)	((char *)(h) - (c)->u.msgx.buf)
#define	RND(len)	(((len) + (sizeof (long) - 1) & ~(sizeof (long) - 1)) \
			    + sizeof (struct msg_head))

/*
 * If we got a request to send a message and
 * we can't satisfy it because the queue file is badly fragmented,
 * we must extend the queue file.
 * If len is -1, we assume that some other process has extended
 * the file and we merely need to re-map it.
 * We return the offset of the entry that contains the new space.
 * XXX With some work, we could avoid panicking on resource shortage
 * XXX and mark the msqid as 'broken', returning errors for every op...
 */
static long
extend_queue_file(c, len)
	struct ipc_chain *c;
	size_t len;
{
	char path[MAXPATHLEN];
	struct ipc_object *o = c->obj;
	struct msg_overhead *mo = &o->p.msg.msgo;
	size_t oldlen;
	long ohoff = 0, hoff;
	int f;

	/* (re-)open the queue file */
	__ipc_path(SCO_IPC_MSG, o->id, path);
	if ((f = open(path, O_RDWR|O_EXLOCK)) == -1)
		err(1, "extend_queue_file open");

	/* be polite and return old heap space to the pool */
	msync(c->u.msgx.buf, c->u.msgx.len);
	if (mmap((caddr_t)c->u.msgx.buf, c->u.msgx.len,
	    PROT_READ|PROT_WRITE|PROT_EXEC, MAP_ANON|MAP_FIXED|MAP_PRIVATE,
	    -1, 0) != (caddr_t)-1)
		free(c->u.msgx.buf);

	if (len != -1) {
		/* extend the file */
		do {
			oldlen = mo->len;
			mo->len *= 2;
		} while (oldlen < RND(len));

		if (lseek(f, mo->len - 1, SEEK_SET) == -1 ||
		    write(f, "", 1) == -1)
			err(1, "extend_queue_file message queue extension");
	}

	/* get new, bigger address space chunk */
	c->u.msgx.len = mo->len;
	if ((c->u.msgx.buf = valloc(c->u.msgx.len)) == 0)
		err(1, "extend_queue_file malloc");

	/* map extended file contents on top of it */
	if (mmap((caddr_t)c->u.msgx.buf, c->u.msgx.len, PROT_READ|PROT_WRITE,
	    MAP_SHARED|MAP_FIXED, f, NBPG) == (caddr_t)-1)
		err(1, "extend_queue_file mmap");

	if (len == -1) {
		/* extend the last entry */
		for (hoff = 0; hoff < mo->len; hoff = HEAD(c, hoff)->bnext)
			ohoff = hoff;
		HEAD(c, ohoff)->bnext = mo->len;
	}

	/* unlock the file */
	close(f);

	/* if len == -1, return offset of last entry */
	return (ohoff);
}

/*
 * Return true if the given block can store
 * a message of the given text length.
 */
static int
available(c, h, len)
	struct ipc_chain *c;
	struct msg_head *h;
	size_t len;
{

	if (h->mtype == 0)
		/*
		 * A special case: the first block doesn't
		 * currently contain a message, so we overlay the header.
		 */
		return (h->bnext - HOFF(c, h) >= RND(len));
	return (h->bnext - (HOFF(c, h) + RND(h->msize)) >= RND(len));
}

int
msgsnd(id, buf, len, flags)
	int id;
	const void *buf;
	size_t len;
	int flags;
{
	extern int sig_pending_mask;		/* XXX */
	struct ipc_chain *c;
	struct ipc_object *o;
	struct msg_overhead *mo;
	struct msqid_ds *dp;
	struct msg_head *t, *p, *n;
	long type;
	long noff;
	int first = 1;

	if ((c = __ipc_chain_by_id(SCO_IPC_MSG, id)) == 0)
		return (-1);
	o = c->obj;
	if (__ipc_perm(&o->p.perm, 0) == -1)
		return (-1);
	mo = &o->p.msg.msgo;
	dp = &o->p.msg.msgd;
	bcopy(buf, &type, sizeof type);
	if (type < 1 || len > dp->msg_qbytes) {
		errno = EINVAL;
		return (-1);
	}

	if (c->u.msgx.len < mo->len)
		extend_queue_file(c, -1);

	__ipc_get_lock_nointr(o);

	/*
	 * Wait for room on the queue.
	 */
	while (dp->msg_cbytes + len > dp->msg_qbytes) {
		if (flags & IPC_NOWAIT) {
			__ipc_yield_lock(o);
			errno = EAGAIN;
			return (-1);
		}
		if (__ipc_suspend_lock(o) == -1)
			return (-1);
		if (sig_pending_mask) {
			__ipc_yield_lock(o);
			errno = EINTR;
			return (-1);
		}
	}

	if (mo->tail == -1) {
		noff = 0;
		n = HEAD(c, 0);
	} else {
		/*
		 * Search for sufficiently large free space on the queue.
		 * Queue space is split into blocks, which start with
		 * a message and are padded with free space.
		 * The first block may omit the message
		 * (contains a dummy message).
		 * When sending, we start at the queue tail and wrap
		 * if we reach the end of the queue segment.
		 * (If we search the entire queue without finding
		 * sufficient space to store the new message,
		 * we extend the queue file and create a big new element.)
		 */
		for (noff = mo->tail, p = t = HEAD(c, noff);
		    (first || p != t) && !available(c, p, len);
		    noff = (p->bnext >= c->u.msgx.len ? 0 : p->bnext),
		    p = HEAD(c, noff))
			first = 0;

		if (!first && p == t)
			noff = extend_queue_file(c, len);
		else if (p->mtype > 0)
			/* skip increment if (first) block is empty */
			noff += RND(p->msize);
		t->qnext = noff;
		n = HEAD(c, noff);

		if (p != n) {
			n->bnext = p->bnext;
			n->bprev = HOFF(c, p);
			p->bnext = noff;
		}
		if (n->bnext < c->u.msgx.len)
			HEAD(c, n->bnext)->bprev = noff;
	}
	n->qnext = -1;
	n->msize = len;
	bcopy(buf, &n->mtype, len + sizeof (long));

	mo->tail = noff;
	if (mo->head == -1)
		mo->head = noff;

	dp->msg_cbytes += len;
	++dp->msg_qnum;
	dp->msg_lspid = getpid();
	time(&dp->msg_stime);

	__ipc_clear_lock(o);

	return (0);
}

/*
 * Message priority rules, based on message 'type'.
 * Don't blame me, I didn't make these up!
 */
static int
match_type(h, type)
	struct msg_head *h;
	long type;
{

	if (type == 0)
		return (1);
	if (type > 0)
		return (h->mtype == type);
	/* type < 0 */
	return (h->mtype <= -type);
}

int
msgrcv(id, buf, len, type, flags)
	int id;
	void *buf;
	size_t len;
	long type;
	int flags;
{
	extern int sig_pending_mask;		/* XXX */
	struct ipc_chain *c;
	struct ipc_object *o;
	struct msqid_ds *dp;
	struct msg_overhead *mo;
	struct msg_head *hprev, *h;
	size_t csize;
	long hoff;

	if ((c = __ipc_chain_by_id(SCO_IPC_MSG, id)) == 0)
		return (-1);
	o = c->obj;
	if (__ipc_perm(&o->p.perm, 1) == -1)
		return (-1);
	dp = &o->p.msg.msgd;
	mo = &o->p.msg.msgo;

	if (c->u.msgx.len < mo->len)
		extend_queue_file(c, -1);

	__ipc_get_lock_nointr(o);

	/*
	 * Loop waiting for an acceptable message.
	 */
	for (;;) {
		/*
		 * Scan the queue starting at the head,
		 * looking for a matching message type.
		 */
		for (hprev = 0, hoff = mo->head, h = HEAD(c, hoff);
		    hoff != -1 && !match_type(h, type);
		    hoff = h->qnext, h = HEAD(c, hoff))
			hprev = h;
		if (hoff != -1)
			break;

		/*
		 * No satisfactory message is on the queue.
		 * We either return now or wait for one to appear.
		 */
		if (flags & IPC_NOWAIT) {
			__ipc_yield_lock(o);
#ifdef SCO_ENOMSG
			errno = errno_in(SCO_ENOMSG);
#else
			errno = EAGAIN;
#endif
			return (-1);
		}
		if (__ipc_suspend_lock(o) == -1)
			return (-1);
		if (sig_pending_mask) {
			__ipc_yield_lock(o);
			errno = EINTR;
			return (-1);
		}
	}

	if (len < h->msize && (flags & MSG_NOERROR) == 0) {
		errno = E2BIG;
		return (-1);
	}

	/*
	 * Put the message in the user's buffer.
	 */
	csize = h->msize;
	if (len > csize)
		len = csize;
	bcopy(&h->mtype, buf, len + sizeof (long));

	/*
	 * Adjust queue pointers to account for the deleted message.
	 * If we deleted a queue head or tail, fix the head or tail pointer(s).
	 */
	if (hprev)
		hprev->qnext = h->qnext;
	else
		mo->head = h->qnext;
	if (mo->tail == hoff)
		if (hprev)
			mo->tail = HOFF(c, hprev);
		else
			mo->tail = -1;

	/*
	 * Return the block to free space.
	 */
	if (hoff > 0)
		HEAD(c, h->bprev)->bnext = h->bnext;
	if (h->bnext < c->u.msgx.len)
		HEAD(c, h->bnext)->bprev = h->bprev;

	h->msize = 0;
	h->mtype = 0;

	dp->msg_cbytes -= csize;
	--dp->msg_qnum;
	dp->msg_lrpid = getpid();
	time(&dp->msg_rtime);

	__ipc_clear_lock(o);

	return (len);
}

int
__ipc_msg_work(c, indx, cookie, buf)
	struct ipc_chain *c;
	int indx, cookie;
	void *buf;
{

	switch (cookie) {

	case IPC_SET:
		c->obj->p.msg.msgd.msg_qbytes =
		    ((struct msqid_ds *)buf)->msg_qbytes;
		time(&c->obj->p.msg.msgd.msg_ctime);
		break;

	case IPC_RMID:
		if (mmap((caddr_t)c->u.msgx.buf, c->u.msgx.len,
		    PROT_READ|PROT_WRITE|PROT_EXEC,
		    MAP_ANON|MAP_FIXED|MAP_PRIVATE, -1, NBPG) != (caddr_t)-1)
			free(c->u.msgx.buf);
		break;
	}
	return (0);
}

int
#ifdef __STDC__
msgctl(int id, int cookie, ...)
#else
msgctl(va_alist)
	va_dcl
#endif
{
	struct msqid_ds *sdp;
	va_list ap;
#ifndef __STDC__
	int id, cookie;

	va_start(ap);
	id = va_arg(ap, int);
	cookie = va_arg(ap, int);
#else
	va_start(ap, cookie);
#endif
	sdp = cookie == IPC_RMID ? 0 : va_arg(ap, struct msqid_ds *);
	va_end(ap);

	return (__ipc_ctl(SCO_IPC_MSG, id, 0, cookie, sdp));
}

#ifndef IPC_LIBRARY

int
sco_msg(cookie, a, b, c, d, e)
	int cookie, a, b, c, d, e;
{

	switch (cookie) {
	case SCO_MSGGET:
		return (msgget((long)a, b));
	case SCO_MSGSND:
		return (msgsnd(a, (const void *)b, (size_t)c, d));
	case SCO_MSGRCV:
		return (msgrcv(a, (void *)b, (size_t)c, (long)d, e));
	case SCO_MSGCTL:
		return (msgctl(a, b, (struct msqid_ds *)c));
	}

	errno = EINVAL;
	return (-1);
}

#endif
