/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_ipc_lock.c,v 2.2 1995/12/19 22:49:29 donn Exp
 */

#include <sys/param.h>
#include <err.h>
#include <errno.h>
#include <string.h>

#include "sco_ipc.h"
#include "sco_errno.h"

#ifndef SYS_sem_lock

#include <pthread.h>

/*
 * We assume a couple of extensions to Posix.4a pthread mutexes here.
 * (1) We assume that pthread_mutex_lock() is interruptible and
 * returns EINTR when a signal arrives while we're blocked.
 * (2) We assume that some other process can unlock a mutex
 * that we have locked, so that we can be signalled when it clears.
 */

void
__ipc_init_lock(o)
	struct ipc_object *o;
{
	static const pthread_mutexattr_t pma = {
	    MUTEX_TYPE_FAST,
	    MUTEX_FLAGS_INTR
	};

	bzero(&o->l, sizeof (o->l));
	pthread_mutex_init(&o->l.op, &pma);
	pthread_mutex_init(&o->l.yield, &pma);
}

/*
 * Get a lock, but fail if we get interrupted while blocking.
 */
int
__ipc_get_lock(o)
	struct ipc_object *o;
{

	if (errno = pthread_mutex_lock(&o->l.op))
		return (-1);
	return (0);
}

/*
 * Get a lock even if we're interrupted.
 */
void
__ipc_get_lock_nointr(o)
	struct ipc_object *o;
{

	while (errno = pthread_mutex_lock(&o->l.op))
		if (errno != EINTR)
			err(1, "__ipc_get_lock_nointr: bad lock");
}

/*
 * Clear a lock after doing work.
 */
void
__ipc_clear_lock(o)
	struct ipc_object *o;
{

	pthread_mutex_unlock(&o->l.op);

	/* announce that we have done some work */
	pthread_mutex_unlock(&o->l.yield);
}

/*
 * Give up a lock after failing to do work.
 */
void
__ipc_yield_lock(o)
	struct ipc_object *o;
{

	pthread_mutex_unlock(&o->l.op);
}

/*
 * Clear a lock and wait for someone else to do work,
 * after failing to do work ourselves.
 * Return the object locked; interruptible.
 */
int
__ipc_suspend_lock(o)
	struct ipc_object *o;
{

	/* set indication that work has not been done */
	pthread_mutex_trylock(&o->l.yield);

	/* clear the object lock and notify waiters */
	pthread_mutex_unlock(&o->l.op);

	/* wait interruptibly until someone has done some work */
	if (errno = pthread_mutex_lock(&o->l.yield))
		return (-1);

	/* get the object lock */
	errno = pthread_mutex_lock(&o->l.op);

	/* make sure the other yielding threads are awakened */
	pthread_mutex_unlock(&o->l.yield);

	return (errno ? -1 : 0);
}

#else

/*
 * Old code that uses Keith Bostic's Posix.4 (vs. Posix.4a) style semaphores.
 */

#include <sys/semaphore.h>

void
__ipc_init_lock(o)
	struct ipc_object *o;
{

	bzero(&o->l, sizeof (o->l));
}

/*
 * Get a lock, but fail if we get interrupted while blocking.
 */
int
__ipc_get_lock(o)
	struct ipc_object *o;
{
	int r;

	for (;;) {
		if (SEM_SPIN_LOCK(&o->l.op.lock))
			break;
		/* sem_lock() doesn't actually lock anything, so we loop */
		r = sem_lock(&o->l.op);
		if (o->removed) {
#ifndef IPC_LIBRARY
			errno = errno_in(SCO_EIDRM);
#else
			errno = EINVAL;
#endif
			return (-1);
		}
		if (r == -1)
			return (-1);
	}
	return (0);
}

/*
 * Get a lock even if we're interrupted.
 */
void
__ipc_get_lock_nointr(o)
	struct ipc_object *o;
{

	while (__ipc_get_lock(o) == -1) {
#ifndef IPC_LIBRARY
		if (errno == errno_in(SCO_EIDRM))
#else
		if (errno == EINVAL)
#endif
			/* it's dead, who cares about locking? */
			return;
		if (errno != EINTR)
			err(1, "__ipc_get_lock_nointr: bad lock");
	}
}

/*
 * Clear a lock after doing work.
 */
void
__ipc_clear_lock(o)
	struct ipc_object *o;
{

	SEM_SPIN_UNLOCK(&o->l.op.lock);

	/* send a wakeup if someone needs the object lock ... */
	if (o->l.op.want)
		sem_wakeup(&o->l.op);

	/* or if someone has been waiting for work to be done. */
	else if (SEM_SPIN_ISLOCKED(&o->l.yield.lock))
		sem_wakeup(&o->l.yield);
}

/*
 * Give up a lock after failing to do work.
 */
void
__ipc_yield_lock(o)
	struct ipc_object *o;
{

	SEM_SPIN_UNLOCK(&o->l.op.lock);
	if (o->l.op.want)
		sem_wakeup(&o->l.op);
}

/*
 * Clear a lock and wait for someone else to do work,
 * after failing to do work ourselves.
 * Return the object locked; interruptible.
 */
int
__ipc_suspend_lock(o)
	struct ipc_object *o;
{
	int r;

	/* set indication that work has not been done */
	SEM_SPIN_LOCK(&o->l.yield.lock);

	/* clear the object lock */
	SEM_SPIN_UNLOCK(&o->l.op.lock);

	/* if someone with work to do needs the object lock, wake them up now */
	if (o->l.op.want)
		sem_wakeup(&o->l.op);

	/* wait interruptibly until someone wakes us up */
	r = sem_lock(&o->l.yield);
	if (o->removed) {
#ifndef IPC_LIBRARY
		errno = errno_in(SCO_EIDRM);
#else
		errno = EINVAL;
#endif
		return (-1);
	}
	if (r == -1)
		return (-1);

	return (__ipc_get_lock(o));
}

#endif
