/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI authenticate.c,v 2.2 1995/10/12 17:27:04 prb Exp
 */
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>

#include <fcntl.h>
#include <paths.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "login_cap.h"

struct rmfiles {
	struct rmfiles	*next;
	char		*file;
};

#define	MAXSPOOLSIZE	(8*1024)	/* Spool up to 8K of back info */

extern char *__progname;

static char spool[MAXSPOOLSIZE];
static int spoolindex = 0;
static struct rmfiles *rmroot = NULL;

static void add_rmlist(char *);
static void spooldata(int);

int
authenticate(char *fullname, char *class, char *style, char *service)
{
	char path[MAXPATHLEN];
	int ret;

	if (!style)
		return (-1);
	if (!service)
		service = LOGIN_DEFSERVICE;

	snprintf(path, sizeof(path), _PATH_AUTHPROG "%s", style);
	if (auth_script(path, style, "-s", service, fullname, class, 0))
		return (0);
	return (auth_scan(0));
}

void
auth_env()
{
	char *line;
	char *name;

    	for (line = spool; line < spool + spoolindex;) {
		if (!strncasecmp(line, BI_SETENV, sizeof(BI_SETENV)-1)) {
			line += sizeof(BI_SETENV) - 1;
			if (*line == ' ' || *line == '\t') {
				name = line;
				while (*name == ' ' || *name == '\t')
					++name;
				line = name;
				while (*line && (*line != ' ' || *line == '\t'))
					++line;
				if (*line)
					*line++ = '\0';
				while (*line == ' ' || *line == '\t')
					++line;
				if (*line && setenv(name, line, 1))
					warn("setenv(%s, %s)", name, line);
			}
		}
		while (*line++)
			;
	}
}

void
auth_rmfiles()
{
	while (rmroot) {
		unlink(rmroot->file);
		rmroot = rmroot->next;
	}
}

int
auth_script(char *path, ...)
{
	va_list ap;

	pid_t pid;
	int status;
	int pfd[2];
	int argc;
	char *argv[64];	/* 64 args should more than enough */
#define	Nargc	(sizeof(argv)/sizeof(argv[0]))

	va_start(ap, path);

	argc = 0;
	while (argc < Nargc - 1 && (argv[argc] = va_arg(ap, char *)))
		++argc;
	argv[argc] = NULL;
	va_end(ap);

	spoolindex = 0;

	if (pipe(pfd) < 0) {
		syslog(LOG_ERR, "%s: unable to create backchannel %m",
		    __progname);
		return (-1);
	}


	switch (pid = fork()) {
	case -1:
		syslog(LOG_ERR, "%s: %m", path);
		return (-1);
	case 0:
		close(pfd[0]);
		if (pfd[1] != 3) {
			if (dup2(pfd[1], 3) < 0)
				err(1, "dup of backchannel");
			close(pfd[1]);
		}
		if (setenv("PATH", _PATH_DEFPATH, 1) ||
		    setenv("SHELL", _PATH_BSHELL, 1))
			err(1, "setenv");
		execv(path, argv);
		err(1, path);
	default:
		close(pfd[1]);
		spooldata(pfd[0]);
		if (waitpid(pid, &status, 0) < 0) {
			syslog(LOG_ERR, "%s: waitpid: %m", path);
			return (-1);
		}

		if (!WIFEXITED(status) || WEXITSTATUS(status))
			return (-1);
	}
	return (0);
}

int
auth_scan(int okay)
{
	char *line;

    	for (line = spool; line < spool + spoolindex;) {
		if (!strncasecmp(line, BI_REJECT, sizeof(BI_REJECT)-1))
			return (0);
		else if (!strncasecmp(line, BI_AUTH, sizeof(BI_AUTH)-1)) {
			line += sizeof(BI_AUTH) - 1;
			if (!*line || *line == ' ' || *line == '\t') {
				while (*line == ' ' || *line == '\t')
					++line;
				if (*line == '\0')
					okay |= AUTH_OKAY;
				else if (!strcasecmp(line, "root"))
					okay |= AUTH_ROOTOKAY;
				else if (!strcasecmp(line, "secure"))
					okay |= AUTH_SECURE;
			}
		}
		else if (!strncasecmp(line, BI_REMOVE, sizeof(BI_REMOVE)-1)) {
			line += sizeof(BI_REMOVE) - 1;
			while (*line == ' ' || *line == '\t')
				++line;
			if (*line)
				add_rmlist(line);
		}
		while (*line++)
			;
	}

	return (okay);
}

void
auth_checknologin(login_cap_t *lc)
{
        char *nologin;

        /*
         * If we fail to get the nologin file due to a database error,
         * assume there should have been one...
         */
        if ((nologin = login_getcapstr(lc, "nologin", "", NULL)) == NULL) {
                warn("Failure to retrieve nologins file");
                exit(1);
        }
        if (*nologin && auth_cat(nologin))
                exit(1);

        if (login_getcapbool(lc, "ignorenologin", 0))
                return;

        if (auth_cat(_PATH_NOLOGIN))
                exit(1);
}

int
auth_cat(char *file)
{
        int fd, nchars;
        char tbuf[8192];

        if ((fd = open(file, O_RDONLY, 0)) < 0)
                return(0);
        while ((nchars = read(fd, tbuf, sizeof(tbuf))) > 0)
                (void)write(fileno(stdout), tbuf, nchars);
        close(fd);
        return(1);
}

static void
spooldata(int fd)
{
	int r;
	char *b;

	while (spoolindex < sizeof(spool) - 1) {
		r = read(fd, spool + spoolindex, sizeof(spool)-spoolindex);
		if (r <= 0) {
			spool[spoolindex] = '\0';
			return;
		}
		b = spool + spoolindex;
		spoolindex += r;
		/*
		 * Go ahead and convert newlines into NULs to allow
		 * easy scanning of the file.
		 */
		while(r-- > 0)
			if (*b++ == '\n')
				b[-1] = '\0';
	}

	syslog(LOG_ERR, "%s: Overflowed backchannel spool buffer", __progname);
	err(1, "System error in authentication program");
}

static void
add_rmlist(char *file)
{
	struct rmfiles *rm;

	if ((rm = malloc(sizeof(struct rmfiles))) == NULL) {
		syslog(LOG_ERR, "%s: Failed to allocate rmfiles: %m",
		    __progname);
		return;
	}
	if (!(file = strdup(file))) {
		syslog(LOG_ERR, "%s: Failed to duplicate rmfile: %m",
		    __progname);
		return;
	}
	rm->file = file;
	rm->next = rmroot;
	rmroot = rm;
}
