/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI dlopen.c,v 1.2 1995/11/28 18:27:44 donn Exp
 */

#include <sys/param.h>
#include <sys/exec.h>
#include <machine/reloc.h>
#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#define	_AOUT_INCLUDE_		/* XXX */
#include <nlist.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "dl.h"

extern struct ps_strings *__ps_strings;

static struct dl_info *dlhead;
static struct dl_info myself = { 0, 0, 1, };

/*
 * Read in all relevant information from the object file.
 * Return 0 for success or an errno for failure.
 */
static int
dl_read(d)
	struct dl_info *d;
{
	size_t size;
	ssize_t n;
	struct nlist *nl, *end_sym;
	int error = 0;
	int f;

	d->code = 0;
	d->treloc = 0;
	d->dreloc = 0;
	d->syms = 0;
	d->strings = 0;

	if ((f = open(d->path, O_RDONLY)) == -1)
		return (errno);
	if ((n = read(f, &d->exec, sizeof d->exec)) != sizeof d->exec)
		error = n == -1 ? errno : EFTYPE;
	if (!error && (N_BADMAG(d->exec) || d->exec.a_syms == 0))
		error = EFTYPE;

	/* Read the symbol and string tables. */
	if (!error && (d->syms = malloc(d->exec.a_syms)) == 0)
		error = errno;
	if (!error && lseek(f, N_SYMOFF(d->exec), SEEK_SET) == -1)
		error = errno;
	if (!error &&
	    (n = read(f, d->syms, d->exec.a_syms)) != d->exec.a_syms)
		error = n == -1 ? errno : EFTYPE;
	if (!error && lseek(f, N_STROFF(d->exec), SEEK_SET) == -1)
		error = errno;
	if (!error && (n = read(f, &size, sizeof size)) != sizeof size)
		error = n == -1 ? errno : EFTYPE;
	if (!error && (d->strings = malloc(size)) == 0)
		error = errno;
	if (!error && (n = read(f, d->strings + 4, size - 4)) != size - 4)
		error = n == -1 ? errno : EFTYPE;

	if (!error && d == &myself)
		/* Our own text and data have already been loaded. */
		d->code = (caddr_t)N_TXTADDR(d->exec);
	else {
		if (!error && d->exec.a_magic != OMAGIC)
			error = EFTYPE;

		/* Try to guess the size of the eventual bss segment. */
		if (!error) {
			size = d->exec.a_text + d->exec.a_data + d->exec.a_bss;
			end_sym = &d->syms[d->exec.a_syms / sizeof *nl];
			for (nl = d->syms; nl < end_sym; ++nl)
				if ((nl->n_type & (N_TYPE|N_EXT)) ==
				    (N_UNDF|N_EXT))
					size += nl->n_value;

			if ((d->code = malloc(size)) == 0)
				error = errno;
		}

		/* Read the text and data, and create bss. */
		if (!error && lseek(f, N_TXTOFF(d->exec), SEEK_SET) == -1)
			error = errno;
		if (!error &&
		    (n = read(f, d->code, d->exec.a_text + d->exec.a_data)) !=
		    d->exec.a_text + d->exec.a_data)
			error = n == -1 ? errno : EFTYPE;
		if (!error)
			bzero(d->code + d->exec.a_text + d->exec.a_data,
			    size - (d->exec.a_text + d->exec.a_data));

		/* Read the text and data relocation records. */
		if (!error && (d->treloc =
		    malloc(d->exec.a_trsize + d->exec.a_drsize)) == 0)
			error = errno;
		if (!error)
			d->dreloc = (struct relocation_info *)
			    ((caddr_t)d->treloc + d->exec.a_trsize);
		if (!error && lseek(f, N_DATOFF(d->exec) + d->exec.a_data,
		    SEEK_SET) == -1)
			error = errno;
		if (!error && (n = read(f, d->treloc,
		    d->exec.a_trsize + d->exec.a_drsize)) !=
		    d->exec.a_trsize + d->exec.a_drsize)
			error = n == -1 ? errno : EFTYPE;
	}

	close(f);

	return (errno = error);
}

/*
 * Search a path for a file.
 */
static char *
check_path(path, variable, check_root)
	const char *path;
	const char *variable;
	int check_root;
{
	size_t pathlen;
	char *pathvar;
	char *s, *t;

	/*
	 * The conditions are described in the NOTES section
	 * of the dlopen(3X) man page in the SVr4 API.
	 */
	if ((pathvar = getenv(variable)) == 0 ||
	    strchr(path, '/') || (check_root && getuid() != geteuid()))
		return (strdup(path));

	if ((pathvar = strdup(pathvar)) == 0)
		return (0);

	pathlen = strlen(path);

	s = 0;
	for (t = strtok(pathvar, ":"); t; t = strtok(0, ":")) {
		if ((s = realloc(s, strlen(t) + 1 + pathlen + 1)) == 0)
			return (0);
		strcpy(s, t);
		strcat(s, "/");
		strcat(s, path);
		if (access(s, F_OK) == 0)
			break;
	}

	free(pathvar);

	if (t == 0) {
		if (s)
			free(s);
		errno = ENOENT;		/* it's the most that we can say */
		return (0);
	}

	return (s);
}

/*
 * Return a malloc()ed string containing the path to the object.
 * If $LD_LIBRARY_PATH is set, we check path components.
 */
static char *
ld_library_path(path)
	const char *path;
{

	return (check_path(path, "LD_LIBRARY_PATH", 1));
}

/*
 * Look for the given file on $PATH.
 */
static char *
exec_path(path)
	const char *path;
{

	return (check_path(path, "PATH", 0));
}

/*
 * If the user has loaded this object already,
 * return the existing relocated object information.
 * This implements the caching semantics in the SVr4 API.
 */
static struct dl_info *
dl_get(path)
	const char *path;
{
	struct dl_info *d;

	for (d = dlhead; d; d = d->next)
		if (strcmp(path, d->path) == 0) {
			++d->count;
			break;
		}
	return (d);
}

/*
 * Allocate space for a new dl_info structure and
 * put it on the chain.
 */
static struct dl_info *
dl_add(path)
	char *path;
{
	struct dl_info *d;

	if ((d = malloc(sizeof *d)) == 0)
		return (0);
	d->path = path;
	d->next = dlhead;
	dlhead = d;
	d->count = 1;
	d->code = 0;
	d->syms = 0;
	d->strings = 0;

	return (d);
}

/*
 * Free the space associated with a dl_info structure.
 */
void
dl_destroy(d)
	struct dl_info *d;
{
	struct dl_info **d_prev;

	if (d == &myself)
		return;
	if (--d->count > 0)
		return;
	for (d_prev = &dlhead; *d_prev; d_prev = &(*d_prev)->next)
		if (*d_prev == d) {
			*d_prev = d->next;
			break;
		}
	free(d->path);
	if (d->code)
		free(d->code);
	if (d->treloc)
		free(d->treloc);
	if (d->syms)
		free(d->syms);
	if (d->strings)
		free(d->strings);
	free(d);
}

/*
 * Read in the symbols and strings of the currently executing program.
 */
static int
read_current_file()
{
	struct nlist *nl, *end_sym;
	int error = 0;

	if ((myself.path = exec_path(__ps_strings->ps_argv[0])) == 0)
		return (errno);
	if (error = dl_read(&myself)) {
		free(myself.path);
		myself.path = 0;
		return (errno = error);
	}

	/* Do the relevant part of dl_link(). */
	end_sym = &myself.syms[myself.exec.a_syms / sizeof *nl];
	for (nl = myself.syms; nl < end_sym; ++nl)
		if (nl->n_type & N_EXT)
			nl->n_un.n_name = myself.strings + nl->n_un.n_strx;

	return (0);
}

/*
 * Compare pointers to two symbol table entries by name.
 */
static int
nlcompare(v1, v2)
	const void *v1, *v2;
{
	const struct nlist *n1 = *(const struct nlist **)v1;
	const struct nlist *n2 = *(const struct nlist **)v2;

	return (strcmp(n1->n_un.n_name, n2->n_un.n_name));
}

/*
 * Link against the currently executing binary.
 */
static int
dl_link(d)
	struct dl_info *d;
{
	struct nlist *nl;
	struct nlist *end_sym = &d->syms[d->exec.a_syms / sizeof *nl];
	struct nlist *cnl, *end_cur_file_sym;
	struct nlist **undef_tab, **u;
	caddr_t comm;
	unsigned int undefined = 0;
	unsigned long size;
	int error = 0;

	/*
	 * Pass one:
	 *	count undefined and global common symbols
	 *	fix up strings
	 *	relocate symbol values
	 */
	for (nl = d->syms; nl < end_sym; ++nl) {
		if (nl->n_type & N_EXT) {
			if ((nl->n_type & N_TYPE) == N_UNDF)
				++undefined;
			nl->n_un.n_name = d->strings + nl->n_un.n_strx;
		}
		if ((nl->n_type & N_TYPE) == N_TEXT ||
		    (nl->n_type & N_TYPE) == N_DATA ||
		    (nl->n_type & N_TYPE) == N_BSS)
			nl->n_value += (vm_offset_t)d->code;
	}

	if (undefined == 0)
		return (0);

	if (myself.path == 0 && (error = read_current_file()))
		return (error);

	if ((undef_tab = malloc(undefined * sizeof *undef_tab)) == 0)
		return (errno);

	/*
	 * Pass two: record undefined symbols and sort them.
	 */
	for (nl = d->syms, u = undef_tab; nl < end_sym; ++nl)
		if ((nl->n_type & (N_TYPE|N_EXT)) == (N_UNDF|N_EXT))
			*u++ = nl;
	qsort(undef_tab, undefined, sizeof *undef_tab, nlcompare); 

	/*
	 * Search the currently executing program's name list
	 * and resolve references to undefined symbols.
	 */
	end_cur_file_sym = &myself.syms[myself.exec.a_syms / sizeof *nl];
	for (cnl = myself.syms; cnl < end_cur_file_sym; ++cnl) {
		if ((cnl->n_type & N_EXT) == 0)
			continue;
		if ((u = (struct nlist **)bsearch(&cnl, undef_tab, undefined,
		    sizeof *u, nlcompare)) == 0)
			continue;
		nl = *u;
		nl->n_type = cnl->n_type;
		nl->n_value = cnl->n_value;
	}

	/*
	 * Pass three:
	 *	look for unresolved symbols
	 *	allocate common
	 */
	comm = d->code + d->exec.a_text + d->exec.a_data + d->exec.a_bss;
	for (u = undef_tab; undefined > 0; --undefined, ++u) {
		nl = *u;
		if ((nl->n_type & N_TYPE) != N_UNDF)
			continue;
		if ((size = nl->n_value) == 0) {
			error = EFTYPE;
			break;
		}
		nl->n_type |= N_BSS;
		nl->n_value = (unsigned long)comm;
		comm += size;
	}

	free(undef_tab);

	return (errno = error);
}

/*
 * Relocate a (text or data) segment.
 */
static void
relocate_segment(d, base, rip, nreloc)
	struct dl_info *d;
	caddr_t base;
	struct relocation_info *rip;
	size_t nreloc;
{
	long v;

#define	RELOCATE(type) \
	v = *(type *)&base[rip->r_address]; \
	if (rip->r_extern) \
		v += d->syms[rip->r_symbolnum].n_value; \
	else if (rip->r_symbolnum != N_ABS) \
		v += (long)d->code; \
	if (rip->r_pcrel) \
		v -= (long)base; \
	*(type *)&base[rip->r_address] = v;

	for (; nreloc > 0; --nreloc, ++rip) {
		switch (rip->r_length) {
		case 0:
			RELOCATE(char);
			break;
		case 1:
			RELOCATE(short);
			break;
		default:
			RELOCATE(long);
			break;
		}
	}
}

/*
 * Perform relocations.
 */
static void
dl_relocate(d)
	struct dl_info *d;
{

	relocate_segment(d, d->code, d->treloc,
	    d->exec.a_trsize / sizeof *d->treloc);
	relocate_segment(d, d->code + d->exec.a_text, d->dreloc,
	    d->exec.a_drsize / sizeof *d->dreloc);

	free(d->treloc);
	d->treloc = 0;
	d->dreloc = 0;
}

/*
 * Open an object file and load it.
 * The 'mode' parameter is currently ignored.
 */
void *
dlopen(input_path, mode)
	const char *input_path;
	int mode;
{
	struct dl_info *d;
	char *path;

	/* Handle a special case. */
	if (input_path == 0) {
		if (myself.path == 0 && read_current_file()) {
			dl_err("can't process symbols for current program");
			return (0);
		}
		return (&myself);
	}

	/* Look for the object file on the library path. */
	if ((path = ld_library_path(input_path)) == 0) {
		dl_err("can't open file");
		return (0);
	}

	/* If the object has already been loaded, return the loaded version. */
	if (d = dl_get(path)) {
		free(path);
		return (d);
	}

	if ((d = dl_add(path)) == 0) {
		dl_err("can't allocate space for new object");
		free(path);
		return (0);
	}

	if (dl_read(d)) {
		dl_err("can't process object file");
		dl_destroy(d);
		return (0);
	}

	if (dl_link(d)) {
		dl_err("can't resolve undefined symbols");
		dl_destroy(d);
		return (0);
	}

	dl_relocate(d);

	return (d);
}
