/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc.  All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI nlist.c,v 1.1 1995/12/18 21:49:48 donn Exp
 */

/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)nlist.c	8.1 (Berkeley) 6/4/93";
#endif /* LIBC_SCCS and not lint */

#include <sys/param.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <errno.h>
#include <elf.h>
#include <nlist.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

int
nlist(name, list)
	const char *name;
	struct nlist *list;
{
	int fd, n;

	fd = open(name, O_RDONLY, 0);
	if (fd < 0)
		return (-1);
	n = __fdnlist(fd, list);
	(void)close(fd);
	return (n);
}

#define	ISLAST(p)	(p->n_name == 0 || p->n_name[0] == 0)
#define	N_BADMAG(e) \
	((e)->e_ident[EI_MAG0] != ELFMAG0 || \
	 (e)->e_ident[EI_MAG1] != ELFMAG1 || \
	 (e)->e_ident[EI_MAG2] != ELFMAG2 || \
	 (e)->e_ident[EI_MAG3] != ELFMAG3 || \
	 ((e)->e_ident[EI_DATA] != ELFDATA2LSB && \
	  BYTE_ORDER == LITTLE_ENDIAN) || \
	 ((e)->e_ident[EI_DATA] != ELFDATA2MSB && \
	  BYTE_ORDER == BIG_ENDIAN))

static void
elf_to_nlist(base, s, n)
	caddr_t base;
	Elf32_Sym *s;
	struct nlist *n;
{

	n->n_value = s->st_value;
	switch (s->st_shndx) {
	case SHN_UNDEF:
		n->n_type = N_UNDF;
		break;
	case SHN_ABS:
		n->n_type = N_ABS;
		break;
	case SHN_COMMON:
		n->n_type = N_UNDF;
		n->n_value = s->st_size;
		break;
	default:
		switch (ELF32_ST_TYPE(s->st_info)) {
		case STT_NOTYPE:
			n->n_type = N_UNDF;
			break;
		case STT_OBJECT:
			if (s->st_shndx < SHN_LORESERVE &&
			    ((Elf32_Shdr *)(base +
			    ((Elf32_Ehdr *)base)->e_shoff +
			    s->st_shndx * ((Elf32_Ehdr *)base)->e_shentsize))
			    ->sh_type == SHT_NOBITS) {
				n->n_type = N_BSS;
				break;
			}
			/* Fall through...  */

		default:
			n->n_type = N_DATA;
			break;
		case STT_FUNC:
			n->n_type = N_TEXT;
			break;
		case STT_FILE:
			n->n_type = N_FN;
			break;
		}
	}
	if (ELF32_ST_BIND(s->st_info) == STB_GLOBAL)
		n->n_type |= N_EXT;
}

int
__fdnlist(fd, list)
	int fd;
	struct nlist *list;
{
	Elf32_Ehdr *e;
	Elf32_Shdr *sh, *strsh;
	Elf32_Sym *s;
	struct nlist *p;
	size_t size;
	struct stat st;
	caddr_t base, top;
	char *strtab;
	char *name;
	int nent;

	/*
	 * Clean out any left-over information for all valid entries.
	 * Type and value are defined to be 0 if not found.
	 */
	nent = 0;
	for (p = list; !ISLAST(p); ++p) {
		p->n_type = 0;
		p->n_value = 0;
		++nent;
	}
	if (nent == 0)
		return (0);

	/*
	 * Map the file.
	 * First we check whether the file is too big to map...
	 */
	if (fstat(fd, &st) < 0)
		return (-1);
	if (st.st_size > SIZE_T_MAX) {
		errno = EFBIG;
		return (-1);
	}
	size = (size_t)st.st_size;

	if ((base = mmap(NULL, size, PROT_READ, 0, fd, 0)) == (caddr_t)-1)
		return (-1);
	top = base + size;

	/*
	 * Perform some sanity checking on the file header.
	 */
	e = (Elf32_Ehdr *)base;
	if (size < sizeof (*e) ||
	    N_BADMAG(e) ||
	    e->e_shoff > size ||
	    e->e_shentsize < sizeof (Elf32_Shdr) ||
	    e->e_shoff + e->e_shentsize * e->e_shnum > size ||
	    e->e_shoff + e->e_shentsize * e->e_shnum < e->e_shoff) {
		errno = EFTYPE;
		return (-1);
	}
	if (e->e_shnum == 0)
		return (nent);

	/*
	 * Loop over symbol table sections.
	 * We ignore dynamic symbol sections.
	 */
	for (sh = (Elf32_Shdr *)(base + e->e_shoff);
	    sh < (Elf32_Shdr *)(base + e->e_shoff + e->e_shentsize*e->e_shnum);
	    sh = (Elf32_Shdr *)((caddr_t)sh + e->e_shentsize)) {
		if (sh->sh_type != SHT_SYMTAB)
			continue;

		/*
		 * More sanity checking.
		 * We're mainly looking for truncated files,
		 * but trashed files are a bit of a concern too.
		 */
		if (sh->sh_offset > size ||
		    sh->sh_entsize < sizeof (Elf32_Sym) ||
		    sh->sh_link >= e->e_shnum ||
		    sh->sh_offset + sh->sh_size > size ||
		    sh->sh_offset + sh->sh_size < sh->sh_offset) {
			errno = EFTYPE;
			return (-1);
		}
		strsh = (Elf32_Shdr *)
		    (base + e->e_shoff + sh->sh_link * e->e_shentsize);
		if (strsh->sh_offset > size ||
		    strsh->sh_offset + strsh->sh_size > size ||
		    strsh->sh_offset + strsh->sh_size < strsh->sh_offset) {
			errno = EFTYPE;
			return (-1);
		}
		if (strsh->sh_size <= 1)
			continue;
		strtab = base + strsh->sh_offset;

		/*
		 * Loop over symbols.
		 */
		for (s = (Elf32_Sym *)base + sh->sh_offset;
		    s < (Elf32_Sym *)(base + sh->sh_offset + sh->sh_size);
		    s = (Elf32_Sym *)((caddr_t)s + sh->sh_entsize)) {
			/* XXX We skip sanity checking in this inner loop.  */
			if (s->st_name == 0)
				continue;
			name = strtab + s->st_name;
			for (p = list; !ISLAST(p); p++)
				if (strcmp(name, p->n_name) == 0)
					elf_to_nlist(s, p);
		}
	}

	munmap(base, size);
	return (nent);
}
