/*-
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI i386.c,v 2.1 1995/03/20 05:15:04 torek Exp
 */

#include <sys/types.h>
#include <a.out.h>
#include <string.h>
#include "shlist.h"

#define	N_CODEOFF(e) \
	(N_TXTOFF(e) + (__HEADER_IN_TEXT(e) ? sizeof(e) : 0))

#define	NCODEBYTES	8

static char codebytes[NCODEBYTES];
static int ncodebytes;
static struct relocation_info ltreloc;
static long ltoffset;

/*
 * Given a pointer to the exec struct at the front of shlicrt0.o,
 * initialize the machine-dependent library table locator code.
 * Find the offset for library tables, and set things up so that
 * we can verify that a binary does in fact use shared libraries.
 * (The entire shlicrt0.o has been mmap'ed in.)
 */
char *
md_init(v)
	caddr_t v;
{
	struct exec *ep;
	struct nlist *sym, *sym_base, *sym_end;
	struct relocation_info *r, *r_base, *r_end;
	char *strings, *txt;
	int low_reloc, ltsym;

	/*
	 * Search the symbol table for __LIBRARY_TABLE.
	 */
	ep = (struct exec *)v;
	strings = (char *)v + N_STROFF(*ep);
	sym_base = (struct nlist *)(v + N_SYMOFF(*ep));
	sym_end = sym_base + ep->a_syms / sizeof *sym_base;
	ltsym = -1;
	for (sym = sym_base; sym < sym_end; ++sym) {
		if (sym->n_type & N_EXT && (sym->n_type & N_TYPE) == N_UNDF &&
		    strcmp(&strings[sym->n_un.n_strx], "___LIBRARY_TABLE")
			    == 0) {
			ltsym = sym - sym_base;
			break;
		}
	}
	if (ltsym == -1)
		return ("can't find ___LIBRARY_TABLE symbol");

	/*
	 * Find a text relocation corresponding to __LIBRARY_TABLE.
	 * Since shlicrt0.o must start a shared library binary,
	 * we can find the address of __LIBRARY_TABLE by examining
	 * the corresponding relocated value in the binary.
	 */
	r_base = (struct relocation_info *)(v + N_DATOFF(*ep) + ep->a_data);
	r_end = r_base + ep->a_trsize / sizeof *r_base;
	low_reloc = ep->a_text;
	for (r = r_base; r < r_end; ++r) {
		if (ltreloc.r_extern == 0 && r->r_extern &&
		    r->r_symbolnum == ltsym)
			ltreloc = *r;
		if (r->r_address < low_reloc)
			low_reloc = r->r_address;
	}
	if (ltreloc.r_extern == 0)
		return ("can't find ___LIBRARY_TABLE relocation");

	/*
	 * Fetch the offset from the actual __LIBRARY_TABLE address.
	 */
	txt = (char *)v + N_TXTOFF(*ep);
	switch (ltreloc.r_length) {
	case 0:
		ltoffset = *(char *)(txt + ltreloc.r_address);
		break;
	case 1:
		ltoffset = *(short *)(txt + ltreloc.r_address);
		break;
	default:
		ltoffset = *(long *)(txt + ltreloc.r_address);
		break;
	}
	ltreloc.r_address -= N_CODEOFF(*ep) - N_TXTOFF(*ep);

	/*
	 * Fill in initial opcode string and length.
	 */
	low_reloc -= N_CODEOFF(*ep) - N_TXTOFF(*ep);
	ncodebytes = low_reloc < NCODEBYTES ? low_reloc : NCODEBYTES;
	bcopy(v + N_CODEOFF(*ep), codebytes, ncodebytes);

	return (NULL);
}

/*
 * Given a pointer to the exec structure of a binary, find the `basic'
 * offset (the difference between a text virtual address and a file
 * offset) and the (text) virtual address of the library table.
 * Return 0 if the binary does not use shared libraries.
 * (The entire binary has been mmap'ed in.)
 */
int
md_ltoff(v, basep, addrp)
	caddr_t v;
	u_long *basep, *addrp;
{
	struct exec *ep;
	caddr_t ltpp;
	u_long ltp;

	/* Make sure it starts with shlicrt0.o. */
	ep = (struct exec *)v;
	if (bcmp(v + N_CODEOFF(*ep), codebytes, ncodebytes))
		return (0);

	ltpp = v + N_CODEOFF(*ep) + ltreloc.r_address;
	switch (ltreloc.r_length) {
	case 0:
		ltp = *(unsigned char *)ltpp - ltoffset;
		if (ltreloc.r_pcrel)
			ltp += ltreloc.r_address + sizeof (char);
		break;
	case 1:
		ltp = *(unsigned short *)ltpp - ltoffset;
		if (ltreloc.r_pcrel)
			ltp += ltreloc.r_address + sizeof (short);
		break;
	default:
		ltp = *(unsigned long *)ltpp - ltoffset;
		if (ltreloc.r_pcrel)
			ltp += ltreloc.r_address + sizeof (long);
		break;
	}
	*basep = N_TXTOFF(*ep) - N_TXTADDR(*ep);
	*addrp = ltp;
	return (1);
}
