/*-
//////////////////////////////////////////////////////////////////////////
//									//
//   Copyright (c) 1995 Migration Associates Corp. All Rights Reserved	//
//									//
// THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF MIGRATION ASSOCIATES	//
//	The copyright notice above does not evidence any   		//
//	actual or intended publication of such source code.		//
//									//
//	Use, duplication, or disclosure by the Government is		//
//	subject to restrictions as set forth in FAR 52.227-19,		//
//	and (for NASA) as supplemented in NASA FAR 18.52.227-19,	//
//	in subparagraph (c) (1) (ii) of Rights in Technical Data	//
//	and Computer Software clause at DFARS 252.227-7013, any		//
//	successor regulations or comparable regulations of other	//
//	Government agencies as appropriate.				//
//									//
//		Migration Associates Corporation			//
//			19935 Hamal Drive				//
//			Monument, CO 80132				//
//									//
//	A license is granted to Berkeley Software Design, Inc. by	//
//	Migration Associates Corporation to redistribute this software	//
//	under the terms and conditions of the software License		//
//	Agreement provided with this distribution.  The Berkeley	//
//	Software Design Inc. software License Agreement specifies the	//
//	terms and conditions for redistribution.			//
//									//
//	UNDER U.S. LAW, THIS SOFTWARE MAY REQUIRE A LICENSE FROM	//
//	THE U.S. COMMERCE DEPARTMENT TO BE EXPORTED.  IT IS YOUR	//
//	REQUIREMENT TO OBTAIN THIS LICENSE PRIOR TO EXPORTATION.	//
//									//
//////////////////////////////////////////////////////////////////////////
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>
#include <db.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "cryptodb.h"

#define	_CRYPTO_PATH "/etc/crypto.db"

static	DB	*cryptodb;

/*
 * Static function prototypes
 */

static	int	cryptodb_open(void);
static	void	cryptodb_close(void);

/*
 * Retrieve a user record from the token database file
 */

extern	int
cryptodb_getrec(char *username, CRYPTODB_Rec *cryptorec)
{
	DBT	key;
	DBT	data;
	int	status = 0;

	key.data = username;
	key.size = strlen(username) + 1;
	memset(&data, 0, sizeof(data));

	if (cryptodb_open())
		return(-1);

	status = (cryptodb->get)(cryptodb, &key, &data, 0);
	cryptodb_close();

	switch (status) {
	case 1:
		return(ENOENT);
	case -1:
		return(-1);
	}
	memcpy(cryptorec, data.data, sizeof(CRYPTODB_Rec));
	return (0);
}

/*
 * Put a user record to the token database file.
 */

extern	int
cryptodb_putrec(char *username, CRYPTODB_Rec *cryptorec)
{
	DBT	key;
	DBT	data;
	int	status = 0;

	key.data = username;
	key.size = strlen(username) + 1;

	data.data = cryptorec;
	data.size = sizeof(CRYPTODB_Rec);

	if (!cryptodb_open()) {
		if (flock((cryptodb->fd)(cryptodb), LOCK_EX)) {
			cryptodb_close();
			return (-1);
		}
		status = (cryptodb->put)(cryptodb, &key, &data, 0);
	}
	cryptodb_close();
	return (status);
}

/*
 * Remove a user record from the token database file.
 */

extern	int
cryptodb_delrec(char *username)
{
	DBT	key;
	DBT	data;
	int	status = 0;

	key.data = username;
	key.size = strlen(username) + 1;
	memset(&data, 0, sizeof(data));

	if (!cryptodb_open()) {
		if (flock((cryptodb->fd)(cryptodb), LOCK_EX)) {
			cryptodb_close();
			return (-1);
		}
		status = (cryptodb->del)(cryptodb, &key, 0);
	}
	cryptodb_close();
	return (status);
}

/*
 * Open the token database.  In order to expedite access in
 * heavily loaded conditions, we employ a N1 lock method.
 * Updates should be brief, so all locks wait infinitely.
 * Wait for a read (shared) lock as all updates read first.
 */

static	int
cryptodb_open(void)
{
	int	must_set_perms = 0;
	struct	stat	statb;

	if (stat(_CRYPTO_PATH, &statb) < 0) {
		if (errno != ENOENT)
			return (-1);
		must_set_perms++;
	} else {
		if (statb.st_uid != 0 || statb.st_gid != 0) {
#ifdef PARANOID
			printf("Authentication disabled\n");
			fflush(stdout);
			syslog(LOG_ALERT,
		"POTENTIAL COMPROMISE of CRYPTO.DB. Owner was %d, Group was %d",
					statb.st_uid, statb.st_gid);
			return (-1);
#else
			must_set_perms++;
#endif
		}
		if ((statb.st_mode & 0777) != 0600) {
#ifdef PARANOID
			printf("Authentication disabled\n");
			fflush(stdout);
			syslog(LOG_ALERT,
		"POTENTIAL COMPROMISE of CRYPTO.DB. Mode was %o", statb.st_mode);
			return (-1);
#else
			must_set_perms++;
#endif
		}
	}
	if (!(cryptodb =
	    dbopen(_CRYPTO_PATH, O_CREAT | O_RDWR, 0600, DB_BTREE, 0)) )
		return (-1);

	if (flock((cryptodb->fd)(cryptodb), LOCK_SH)) {
		(cryptodb->close)(cryptodb);
		return (-1);
	}
	if (must_set_perms && chown(_CRYPTO_PATH, 0, 0))
		syslog(LOG_INFO,
		    "Can't set owner/group of /etc/crypto.db errno=%m");

	return (0);
}

/*
 * Close the token database.  We are holding an unknown lock.
 * Release it, then close the db. Since little can be done 
 * about errors, we ignore them.
 */

static	void
cryptodb_close(void)
{
	(void)flock((cryptodb->fd)(cryptodb), LOCK_UN);
	(cryptodb->close)(cryptodb);
}

/*
 * Retrieve the first user record from the database, leaving the
 * database open for the next retrieval. If the march thru the
 * the database is aborted before end-of-file, the caller should
 * call cryptodb_close to release the read lock.
 */

extern	int
cryptodb_firstrec(int reverse_flag, CRYPTODB_Rec *cryptorec)
{
	DBT	key;
	DBT	data;
	int	status = 0;

	memset(&data, 0, sizeof(data));

	if (!cryptodb_open()) {
		status = (cryptodb->seq)(cryptodb, &key, &data,
				reverse_flag ? R_LAST : R_FIRST);
	}
	if (status) {
		cryptodb_close();
		return (status);
	}
	if (!data.data) {
		cryptodb_close();
		return (ENOENT);
	}
	memcpy(cryptorec, data.data, sizeof(CRYPTODB_Rec));
	return (0);
}

/*
 * Retrieve the next sequential user record from the database. Close
 * the database only on end-of-file or error.
 */


extern	int
cryptodb_nextrec(int reverse_flag, CRYPTODB_Rec *cryptorec)
{
	DBT	key;
	DBT	data;
	int	status;

	memset(&data, 0, sizeof(data));

	status = (cryptodb->seq)(cryptodb, &key, &data, 
		reverse_flag ? R_PREV : R_NEXT);

	if (status) {
		cryptodb_close();
		return (status);
	}
	if (!data.data) {
		cryptodb_close();
		return (ENOENT);
	}
	memcpy(cryptorec, data.data, sizeof(CRYPTODB_Rec));
	return (0);
}

/*
 * Retrieve and lock a user record.  Since there are no facilities in
 * BSD for record locking, we hack a bit lock into the user record.
 */

extern	int
cryptodb_lockrec(char *username, CRYPTODB_Rec *cryptorec, unsigned recflags)
{
	DBT	key;
	DBT	data;
	int	status;

	key.data = username;
	key.size = strlen(username) + 1;
	memset(&data, 0, sizeof(data));

	if (cryptodb_open())
		return(-1);

	if (flock((cryptodb->fd)(cryptodb), LOCK_EX)) {
		cryptodb_close();
		return(-1);
	}
	switch ((cryptodb->get)(cryptodb, &key, &data, 0)) {
	case 1:
		cryptodb_close();
		return (ENOENT);
	case -1:
		cryptodb_close();
		return(-1);
	}
	memcpy(cryptorec, data.data, sizeof(CRYPTODB_Rec));

	if ((cryptorec->flags & CRYPTO_LOCKED)||(cryptorec->flags & recflags)) {
		cryptodb_close();
		return(1);
	}
	data.data = cryptorec;
	data.size = sizeof(CRYPTODB_Rec);

	time(&cryptorec->lock_time);
	cryptorec->flags |= recflags;
	status = (cryptodb->put)(cryptodb, &key, &data, 0);
	cryptodb_close();
	if (status)
		return(-1);

	return(0);
}

