/*
 * Copyright (c) 1995 Berkeley Software Design, Inc. 
 * All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 * Audio mixer controller program
 *
 * BSDI mixer.c,v 1.3 1995/11/03 00:38:43 ewv Exp
 */
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <sys/soundcard.h>
#include <sys/fcntl.h>

typedef struct {
	int device;
	char *kword;
	char *desc;
} keytab_t;

void print_all __P((void));
keytab_t *keymatch __P((char *key));
void pdevs __P((int mask, char *sep));
void pval __P((keytab_t *kt));
void precsrc __P((void));
keytab_t *find_kte __P((int dev));
void usage __P((char *self));

#define RDEV	256

keytab_t keytab[] = {
	{ SOUND_MIXER_VOLUME,	"volume",	"Master volume" },
	{ SOUND_MIXER_BASS,	"bass",		"Bass" },
	{ SOUND_MIXER_TREBLE,	"treble",	"Treble" },
	{ SOUND_MIXER_SYNTH,	"synth",	"On-board Sythesizer" },
	{ SOUND_MIXER_PCM,	"pcm",		"Primary DAC" },
	{ SOUND_MIXER_SPEAKER,	"speaker",	"PC speaker" },
	{ SOUND_MIXER_LINE,	"line",		"Main line in" },
	{ SOUND_MIXER_MIC,	"microphone",	"Microphone" },
	{ SOUND_MIXER_CD,	"cd",		"CD Player" },
	{ SOUND_MIXER_IMIX,	"imix",		"Recording monitor" },
	{ SOUND_MIXER_ALTPCM,	"altpcm",	"Secondary DAC" },
	{ SOUND_MIXER_RECLEV,	"rec",		"Record level" },
	{ SOUND_MIXER_IGAIN,	"igain",	"Input gain" },
	{ SOUND_MIXER_OGAIN,	"ogain",	"Output gain" },
	{ SOUND_MIXER_LINE1,	"aux1",		"Aux input 1" },
	{ SOUND_MIXER_LINE2,	"aux2",		"Aux input 2" },
	{ SOUND_MIXER_LINE3,	"aux3",		"Aux input 3" },
	{ SOUND_MIXER_MUTE,	"mute",		"Mute output (0/1)" },
	{ SOUND_MIXER_ENHANCE,	"enhance",	"Enhance stereo field (0/1)" },
	{ SOUND_MIXER_LOUD,	"loudness",	"Loudness (0/1)" },
	{ RDEV,			"rdev",		"Recording devices" },
	{ -1,			NULL,		NULL }
};

int devmask;
int recmask;
int stereomask;
int fd = -1;
char	mixname[32] = "/dev/mixer";
int vflag;

int
main(int ac, char **av)
{
	int c;
	int mixer = 0;
	keytab_t *kt;
	keytab_t *kte;
	int val;
	int left, right;
	char *term = "";
	char *sep = "";
	char *p;
	char *ap;

	while ((c = getopt(ac, av, "m:v")) != EOF) {
		switch (c) {
		case 'm':
			mixer = atoi(optarg);
			break;
		case 'v':
			vflag = 1;
			break;
		default:
			usage(av[0]);
		}
	}
	ac -= optind;
	av += optind;

	if (mixer != 0)
		sprintf(mixname, "/dev/mixer%d", mixer);

	if ((fd = open(mixname, O_RDWR)) < 0)
		err(1, "Can't open %s", mixname);

	/* Get device and record mask */
	if (ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devmask) < 0)
		err(1, "SOUND_MIXER_READ_DEVMASK failed");
	if (ioctl(fd, SOUND_MIXER_READ_RECMASK, &recmask) < 0)
		err(1, "SOUND_MIXER_READ_RECMASK failed");
	if (ioctl(fd, SOUND_MIXER_READ_STEREODEVS, &stereomask) < 0)
		err(1, "SOUND_MIXER_READ_STEREODEVS failed");

	if (ac == 0) {
		print_all();
		exit(0);
	}

	while (ac) {
		kt = keymatch(av[0]);
		if (!kt) {
			fprintf(stderr, "Invalid keyword: %s\n", av[0]);
			exit(1);
		}
		if (!(devmask & 1 << kt->device)) {
			fprintf(stderr, "Unsupported device: %s\n", av[0]);
			exit(1);
		}

		/*
		 * Sort of hokey: if there are no arguments left or the
		 * next argument isn't a number and the current argument
		 * isn't the record device keyword then print info on the
		 * channel.
		 */
		if (ac == 1 || (!strchr("0123456789", *av[1]) && 
		    (kt->device != RDEV))) {
			fputs(sep, stdout);
			pval(kt);
			term = "\n";
			sep = " ";
			ac--;
			av++;
			continue;
		}

		/* Set a channel */
		if (kt->device == RDEV) {
			/* Recording devices */
			val = 0;
			p = av[1];

			while ((ap = strsep(&p, ",")) != NULL) {
				kte = keymatch(ap);
				if (!kte || kte->device == RDEV) {
					fprintf(stderr, "Unknown recording "
					    "source: %s\n", ap);
					exit(1);
				}
				left = 1 << kte->device;
				if ((recmask & left) == 0) {
					fprintf(stderr, "Not able to record "
					    "from %s\n", kte->kword);
					exit(1);
				}
				val |= left;
			}
			if (ioctl(fd, SOUND_MIXER_WRITE_RECSRC, &val) < 0)
				err(1, "SOUND_MIXER_WRITE_RECSRC failed");
		} else {
			/* Numeric value channels */
			if (strchr(av[1], ':'))
				sscanf(av[1], "%d:%d", &left, &right);
			else {
				sscanf(av[1], "%d", &left);
				right = left;
			}
			if (left < 0 || left > 100 ||
			    right < 0 || right > 100) {
				fprintf(stderr, "Value out of range "
				    "(0 ... 100)\n");
				exit(1);
			}
			val = left | right << 8;
			if (ioctl(fd, MIXER_WRITE(kt->device), &val) < 0)
				err(1, "MIXER_WRITE(%d) failed", kt->device);
		}
		ac -= 2;
		av += 2;
	}
	fputs(term, stdout);
	if (vflag)
		print_all();
	exit(0);
}

/*
 * Print all settings and available devices
 */
void
print_all()
{
	int idx;
	int mask;
	int val;
	int left, right;
	keytab_t *kte;

	printf("%-10s %-30s %s\n", "Device", "Description", "Value");
	idx = 0;
	mask = devmask;
	while (mask != 0) {
		if ((mask & 0x1) != 0) {
			if (ioctl(fd, MIXER_READ(idx), &val) < 0)
				err(1, "MIXER_READ(%d) failed", idx);
			left = val & 0xff;
			right = val >> 8 & 0xff;
			kte = find_kte(idx);
			printf("%-10s %-30s ", kte->kword, kte->desc);
			if ((stereomask & 1 << idx) == 0)
				printf("%d\n", left);
			else
				printf("%d:%d\n", left, right);
		}
		mask >>= 1;
		idx++;
	}
	printf("%-10s %-30s ", "rdev", "Recording source");
	precsrc();
	printf("\n\nValid recording devices: ");
	pdevs(recmask, ", ");
	printf("\n");
}

/*
 * Find a device number based on a (possibly partial) name.
 */
keytab_t *
keymatch(char *key)
{
	int hit;
	keytab_t *kt;
	keytab_t *kth;

	hit = 0;
	kt = keytab;
	kth = NULL;
	while (kt->device >= 0) {
		if (strncmp(key, kt->kword, strlen(key)) == 0) {
			hit++;
			kth = kt;
		}
		kt++;
	}
	if (hit > 1) {
		fprintf(stderr, "\"%s\" is not unique\n", key);
		return (NULL);
	}
	return (kth);
}

/*
 * Print all devices specified in the passed bit vector
 */
void
pdevs(int mask, char *sep)
{
	int idx = 0;
	char *tsep = "";
	keytab_t *kte;

	while (mask != 0) {
		if (mask & 0x1) {
			kte = find_kte(idx);
			printf("%s%s", tsep, kte->kword);
			tsep = sep;
		}
		mask >>= 1;
		idx++;
	}
}

/*
 * Print the current value of a channel
 */
void
pval(keytab_t *kt)
{
	int val;
	int left, right;

	if (kt->device == RDEV) {
		printf("%s ", kt->kword);
		precsrc();
		return;
	}
	if (ioctl(fd, MIXER_READ(kt->device), &val) < 0)
		err(1, "MIXER_READ(%d) failed", kt->device);
	left = val & 0xff;
	right = val >> 8 & 0xff;
	if (stereomask & 1 << kt->device)
		printf("%s %d:%d", kt->kword, left, right);
	else
		printf("%s %d", kt->kword, left);
}

/*
 * Print current recording sources
 */
void
precsrc()
{
	int recsrc;

	if (ioctl(fd, SOUND_MIXER_READ_RECSRC, &recsrc) < 0)
		err(1, "SOUND_MIXER_READ_RECSRC failed");
	pdevs(recsrc, ",");
}

/*
 * Find an entry in the keytab with the given device number
 */
keytab_t *
find_kte(int dev)
{
	keytab_t *kt;

	kt = keytab;
	while (kt->device >= 0) {
		if (kt->device == dev)
			return (kt);
		kt++;
	}
	return (0);
}

void
usage(char *self)
{
	fprintf(stderr,"Usage: %s [-v] [-m mixer-num] [device [val]] [...]\n",
	    self);
	exit(1);
}

