/*	BSDI ppcbsdi-kgdb.c,v 1.2 1996/01/13 21:04:11 donn Exp	*/

/*
   Power PC kgdb support.  Based on:

   Intel 386 target-dependent stuff.
   Copyright (C) 1988, 1989, 1991 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


/*
 * Machine-dependent kernel debugging support for BSD/386.
 * Mainly taken from sparcbsd-tdep.c from LBL.
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/proc.h>
#include <vm/vm.h>
#include <machine/frame.h>
#include <machine/reg.h>
#include <machine/pcb.h>
#include <machine/vmparam.h>

#include <kvm.h>

int kernel_debugging;

/*
 * Read the "thing" at address 'addr' into the space pointed to by P.
 * The length of the "thing" is determined by the type of P.
 * Result is non-zero if transfer fails.
 */
#define READMEM(addr, p) \
    (target_read_memory((CORE_ADDR)(addr), (char *)(p), sizeof(*(p))))

#include "defs.h"
#include "frame.h"
#include "value.h"
#include "target.h"
#include "gdbcore.h"

static CORE_ADDR leave_kernel;

/*
 * The code below implements kernel debugging of crashdumps (or /dev/kmem)
 * or remote systems (via a serial link).  For remote kernels, the remote
 * context does most the work, so there is very little to do -- we just
 * manage the kernel stack boundaries so we know where to stop a backtrace.
 *
 * The crashdump/kmem (kvm) support is a bit more grungy, but thanks to
 * libkvm (see kcore.c) not too bad.  The main work is kvm_fetch_registers
 * which sucks the register state out of the current processes pcb.
 * There is a command that let's you set the current process -- hopefully,
 * to something that's blocked (in the live kernel case).
 */
 
static CORE_ADDR kernbase;
static struct proc *masterprocp;		/* kernel virtual address */
static struct proc fakeproc;
void set_curproc();

int
inside_kernel(CORE_ADDR pc)
{

	return ((unsigned long)pc >= (unsigned long)kernbase);
}

/*
 * Return true if ADDR is a valid stack address according to the
 * current boundaries (which are determined by the currently running 
 * user process).
 */
int
inside_kernstack(CORE_ADDR addr)
{

	/* currently, any kernel address works as a kernel stack */
	return (inside_kernel(addr));
}

/*
 * (re-)set the variables that make inside_kernstack() work.
 */
static void
set_kernel_boundaries(struct proc *p)
{

	kernbase = ksym_lookup("SYSTEM");
}

/*
 * Return the current proc.  masterprocp points to
 * current proc which points to current u area.
 */
static struct proc *
fetch_curproc()
{
	struct proc *p;
	static CORE_ADDR curproc_addr;
	static CORE_ADDR prevproc_addr;

	if (curproc_addr == 0)
		curproc_addr = ksym_lookup("curproc");
	if (READMEM(curproc_addr, &p))
		error("cannot read curproc pointer at 0x%x\n", curproc_addr);
	if (p == 0) {
		if (prevproc_addr == 0)
			prevproc_addr = ksym_lookup("prevproc");
		if (READMEM(prevproc_addr, &p))
			error("cannot read prevproc pointer at 0x%x\n",
				prevproc_addr);
		if (p == 0)
			p = (struct proc *)ksym_lookup("proc0");
	}
	return (p);
}

/*
 * All code below is exclusively for support of kernel core files.
 */

/*
 * Fetch registers from a crashdump or /dev/kmem.
 */
static void
kvm_fetch_registers(p)
	struct proc *p;
{
	struct pcb *ppcb, pcb;
	int saved_lr;
	int i;

	/* find the pcb for the current process */
	if (READMEM(&p->p_addr, &ppcb))
		error("cannot read pcb pointer at 0x%x", &p->p_addr);
	if (READMEM(ppcb, &pcb))
		error("cannot read pcb at 0x%x", ppcb);
	if (READMEM((int *)pcb.pcb_sp + 1, &saved_lr))
		error("cannot read saved lr at 0x%x\n", (int *)pcb.pcb_sp + 1);

	/*
	 * Invalidate all the registers then fill in the ones we know about.
	 */
	registers_changed();

	supply_register(PC_REGNUM, (char *)&pcb.pcb_lr);
	supply_register(CR_REGNUM, (char *)&pcb.pcb_cr);
	supply_register(LR_REGNUM, (char *)&saved_lr);
	supply_register(SP_REGNUM, (char *)&pcb.pcb_sp);

	for (i = 0; i < 18; ++i)
		supply_register(GP0_REGNUM + i + 14, (char *)&pcb.pcb_gpr[i]);
}

/*
 * Called from remote_wait, after the remote kernel has stopped.
 * Look up the current proc, and set up boundaries.
 * This is for active kernels only.
 */
void
set_curproc()
{
	masterprocp = fetch_curproc();
	set_kernel_boundaries(masterprocp);
}

/*
 * Set the process context to that of the proc structure at
 * system address paddr.  Read in the register state.
 */
int
set_procaddr(CORE_ADDR paddr)
{
	struct pcb *ppcb;
	CORE_ADDR kernel_map_addr;
	struct vm_map *kernel_map;
	struct pmap *kernel_pmap;

	if (paddr == 0)
		masterprocp = fetch_curproc();
	else if (paddr != (CORE_ADDR)masterprocp) {
		if ((unsigned)paddr < KERNBASE)
			return (1);
		masterprocp = (struct proc *)paddr;
	}
	if (READMEM(&masterprocp->p_vmspace, &fakeproc.p_vmspace))
		error("cannot read vmspace pointer at 0x%x",
		    &masterprocp->p_vmspace);
	if (fakeproc.p_vmspace == 0) {
		kernel_map_addr = ksym_lookup("kernel_map");
		if (READMEM(kernel_map_addr, &kernel_map))
			error("cannot get kernel_map pointer at 0x%x\n",
			    kernel_map_addr);
		if (READMEM(&kernel_map->pmap, &kernel_pmap))
			error("cannot get kernel_pmap pointer at 0x%x\n",
			    &kernel_map->pmap);
		fakeproc.p_vmspace = (struct vmspace *)
		    ((CORE_ADDR)kernel_pmap - sizeof (struct vm_map));
	}
	set_kernel_boundaries(masterprocp);
	kvm_fetch_registers(masterprocp);
	return (0);
}

/*
 * Get the registers out of a crashdump or /dev/kmem.
 * XXX This somehow belongs in kcore.c.
 *
 * We just get all the registers, so we don't use regno.
 */
void
kernel_core_registers(int regno)
{
	/*
	 * Need to find current u area to get kernel stack and pcb
	 * where "panic" saved registers.
	 * (libkvm also needs to know current u area to get user
	 * address space mapping).
	 */
	(void)set_procaddr((CORE_ADDR)masterprocp);
}

extern kvm_t *kd;

int
kernel_xfer_memory(addr, cp, len, write, target)
     CORE_ADDR addr;
     char *cp;
     int len;
     int write;
     struct target_ops *target;
{
	if (write)
		return kvm_write(kd, addr, cp, len);
	if (addr < (CORE_ADDR)KERNBASE) {
		if (masterprocp == 0 && set_procaddr(0))
			return (-1);
		return kvm_uread(kd, &fakeproc, addr, cp, len);
	}
	return kvm_read(kd, addr, cp, len);
}

CORE_ADDR
kernel_frame_chain(struct frame_info *next)
{
	CORE_ADDR v;

	if (leave_kernel == 0)
		leave_kernel = ksym_lookup("leave_kernel");
	if (frame_saved_pc(next) == leave_kernel) {
		v = rs6000_frame_chain(next) + 16;
		v = (CORE_ADDR)&((struct exception_frame *)v)->ef_gpr[1];
		return (read_memory_integer(v, 4));
	}
	return (rs6000_frame_chain(next));
}

int
kernel_frame_chain_valid(CORE_ADDR v, struct frame_info *fr)
{

	if (v >= kernbase)
		return (1);
	return (0);
}

void
init_frame_pc_first(int leaf, struct frame_info *fr)
{
	struct rs6000_framedata rf;
	CORE_ADDR v;

	if (fr->next == 0) {
		fr->pc = read_pc();
		return;
	}

	if (kernel_debugging) {

		if (leave_kernel == 0)
			leave_kernel = ksym_lookup("leave_kernel");

		/*
		 * All exception vectors call their exception handlers
		 * with LR set to leave_kernel.  If we see leave_kernel
		 * as a saved LR, it means that a function took an exception.
		 * We recover the PC from SRR0 in the exception_frame.
		 */
		if (frame_saved_pc(fr->next) == leave_kernel) {
			v = rs6000_frame_chain(fr->next) + 16;
			v = (CORE_ADDR)
			    &((struct exception_frame *)v)->ef_srr0;
			fr->pc = read_memory_integer(v, 4);
			return;
		}

		/*
		 * If the next frame down represents a function that
		 * took an exception, and that function didn't save LR,
		 * look for the LR value in the exception_frame instead.
		 */
		if (fr->next->next &&
		    frame_saved_pc(fr->next->next) == leave_kernel) {
			v = get_pc_function_start(fr->next->pc) +
			    FUNCTION_START_OFFSET;
			skip_prologue(v, &rf);
			if (rf.nosavedpc) {
				v = rs6000_frame_chain(fr->next->next) + 16;
				v = (CORE_ADDR)
				    &((struct exception_frame *)v)->ef_lr;
				fr->pc = read_memory_integer(v, 4);
				return;
			}
		}
	}

	if (leaf)
		fr->pc = SAVED_PC_AFTER_CALL(fr->next);
	else
		fr->pc = FRAME_SAVED_PC(fr->next);
}

void
init_extra_frame_info(int leaf, struct frame_info *fr)
{
	extern struct obstack frame_cache_obstack;
	struct frame_saved_regs *fregs;
	struct exception_frame *ef;
	int i;

	fr->initial_sp = 0;
	fr->cache_fsr = 0;

	if (!kernel_debugging)
		return;

	if (leave_kernel == 0)
		leave_kernel = ksym_lookup("leave_kernel");
	if (frame_saved_pc(fr) != leave_kernel)
		return;

	fregs = fr->cache_fsr = (struct frame_saved_regs *)
	    obstack_alloc(&frame_cache_obstack,
	    sizeof (struct frame_saved_regs));
	memset(fr->cache_fsr, '\0', sizeof (struct frame_saved_regs));

	ef = (struct exception_frame *)(rs6000_frame_chain(fr) + 16);

	fregs->regs[PC_REGNUM] = (CORE_ADDR)&ef->ef_srr0;
	fregs->regs[PS_REGNUM] = (CORE_ADDR)&ef->ef_srr1;
	fregs->regs[CR_REGNUM] = (CORE_ADDR)&ef->ef_cr;
	fregs->regs[LR_REGNUM] = (CORE_ADDR)&ef->ef_lr;
	fregs->regs[CTR_REGNUM] = (CORE_ADDR)&ef->ef_ctr;
	fregs->regs[XER_REGNUM] = (CORE_ADDR)&ef->ef_xer;

	for (i = 0; i < 32; ++i)
		fregs->regs[GP0_REGNUM + i] = (CORE_ADDR)&ef->ef_gpr[i];
}
