/*	BSDI i386bsdi-kgdb.c,v 1.1 1995/11/26 18:00:01 donn Exp	*/

/* Intel 386 target-dependent stuff.
   Copyright (C) 1988, 1989, 1991 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


/*
 * Machine-dependent kernel debugging support for BSD/386.
 * Mainly taken from sparcbsd-tdep.c from LBL.
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/proc.h>
#include <vm/vm.h>
#include <machine/frame.h>
#include <machine/reg.h>
#include <machine/pcb.h>
#include <machine/vmparam.h>

#include <kvm.h>

int kernel_debugging;

/*
 * Read the "thing" at address 'addr' into the space pointed to by P.
 * The length of the "thing" is determined by the type of P.
 * Result is non-zero if transfer fails.
 */
#define READMEM(addr, p) \
    (target_read_memory((CORE_ADDR)(addr), (char *)(p), sizeof(*(p))))

#include "defs.h"
#include "frame.h"
#include "value.h"
#include "target.h"
#include "gdbcore.h"

static CORE_ADDR tstart, tend, istart, iend;

static void
init_translate_limits()
{

	tstart = ksym_lookup("START_TRAPS");
	tend = ksym_lookup("END_TRAPS");
	istart = ksym_lookup("START_INTRS");
	iend = ksym_lookup("END_INTRS");
}

/*
 * The code below implements kernel debugging of crashdumps (or /dev/kmem)
 * or remote systems (via a serial link).  For remote kernels, the remote
 * context does most the work, so there is very little to do -- we just
 * manage the kernel stack boundaries so we know where to stop a backtrace.
 *
 * The crashdump/kmem (kvm) support is a bit more grungy, but thanks to
 * libkvm (see kcore.c) not too bad.  The main work is kvm_fetch_registers
 * which sucks the register state out of the current processes pcb.
 * There is a command that let's you set the current process -- hopefully,
 * to something that's blocked (in the live kernel case).
 */
 
static CORE_ADDR kstack;
static struct proc *masterprocp;		/* kernel virtual address */
static struct proc fakeproc;
void set_curproc();

int
inside_kernel(CORE_ADDR pc)
{

	return ((unsigned long)pc >= (unsigned long)kstack);
}

/*
 * Return true if ADDR is a valid stack address according to the
 * current boundaries (which are determined by the currently running 
 * user process).
 */
int
inside_kernstack(CORE_ADDR addr)
{

	/* currently, any kernel address works as a kernel stack */
	return (inside_kernel(addr));
}

/*
 * (re-)set the variables that make inside_kernstack() work.
 */
static void
set_kernel_boundaries(struct proc *p)
{

	kstack = ksym_lookup("kstack");
}

/*
 * Return the current proc.  masterprocp points to
 * current proc which points to current u area.
 */
static struct proc *
fetch_curproc()
{
	struct proc *p;
	static CORE_ADDR curproc_addr;
	static CORE_ADDR prevproc_addr;

	if (curproc_addr == 0)
		curproc_addr = ksym_lookup("curproc");
	if (READMEM(curproc_addr, &p))
		error("cannot read curproc pointer at 0x%x\n", curproc_addr);
	if (p == 0) {
		if (prevproc_addr == 0)
			prevproc_addr = ksym_lookup("prevproc");
		if (READMEM(prevproc_addr, &p))
			error("cannot read prevproc pointer at 0x%x\n",
				prevproc_addr);
		if (p == 0)
			p = (struct proc *)ksym_lookup("proc0");
	}
	return (p);
}

/*
 * All code below is exclusively for support of kernel core files.
 */

/*
 * Fetch registers from a crashdump or /dev/kmem.
 */
static void
kvm_fetch_registers(p)
	struct proc *p;
{
	struct pcb *ppcb, pcb;

	/* find the pcb for the current process */
	if (READMEM(&p->p_addr, &ppcb))
		error("cannot read pcb pointer at 0x%x", &p->p_addr);
	if (READMEM(ppcb, &pcb))
		error("cannot read pcb at 0x%x", ppcb);

	/*
	 * Invalidate all the registers then fill in the ones we know about.
	 */
	registers_changed();

	supply_register(PC_REGNUM, (char *)&pcb.pcb_pc);
	supply_register(FP_REGNUM, (char *)&pcb.pcb_fp);
	supply_register(SP_REGNUM, (char *)&pcb.pcb_tss.tss_esp);
	supply_register(PS_REGNUM, (char *)&pcb.pcb_psl);

	supply_register(0, (char *)&pcb.pcb_tss.tss_eax);
	supply_register(1, (char *)&pcb.pcb_tss.tss_ecx);
	supply_register(2, (char *)&pcb.pcb_tss.tss_edx);
	supply_register(3, (char *)&pcb.pcb_tss.tss_ebx);
	supply_register(6, (char *)&pcb.pcb_tss.tss_esi);
	supply_register(7, (char *)&pcb.pcb_tss.tss_edi);
	supply_register(10, (char *)&pcb.pcb_tss.tss_cs);
	supply_register(11, (char *)&pcb.pcb_tss.tss_ss);
	supply_register(12, (char *)&pcb.pcb_tss.tss_ds);
	supply_register(13, (char *)&pcb.pcb_tss.tss_es);
	supply_register(14, (char *)&pcb.pcb_tss.tss_fs);
	supply_register(15, (char *)&pcb.pcb_tss.tss_gs);
}

/*
 * Called from remote_wait, after the remote kernel has stopped.
 * Look up the current proc, and set up boundaries.
 * This is for active kernels only.
 */
void
set_curproc()
{
	masterprocp = fetch_curproc();
	set_kernel_boundaries(masterprocp);
}

/*
 * Set the process context to that of the proc structure at
 * system address paddr.  Read in the register state.
 */
int
set_procaddr(CORE_ADDR paddr)
{
	struct pcb *ppcb;
	CORE_ADDR kernel_map_addr;
	struct vm_map *kernel_map;
	struct pmap *kernel_pmap;

	if (paddr == 0)
		masterprocp = fetch_curproc();
	else if (paddr != (CORE_ADDR)masterprocp) {
		if ((unsigned)paddr < KERNBASE)
			return (1);
		masterprocp = (struct proc *)paddr;
	}
	if (READMEM(&masterprocp->p_vmspace, &fakeproc.p_vmspace))
		error("cannot read vmspace pointer at 0x%x",
		    &masterprocp->p_vmspace);
	if (fakeproc.p_vmspace == 0) {
		kernel_map_addr = ksym_lookup("kernel_map");
		if (READMEM(kernel_map_addr, &kernel_map))
			error("cannot get kernel_map pointer at 0x%x\n",
			    kernel_map_addr);
		if (READMEM(&kernel_map->pmap, &kernel_pmap))
			error("cannot get kernel_pmap pointer at 0x%x\n",
			    &kernel_map->pmap);
		fakeproc.p_vmspace = (struct vmspace *)
		    ((CORE_ADDR)kernel_pmap - sizeof (struct vm_map));
	}
	set_kernel_boundaries(masterprocp);
	kvm_fetch_registers(masterprocp);
	return (0);
}

/*
 * Get the registers out of a crashdump or /dev/kmem.
 * XXX This somehow belongs in kcore.c.
 *
 * We just get all the registers, so we don't use regno.
 */
void
kernel_core_registers(int regno)
{
	/*
	 * Need to find current u area to get kernel stack and pcb
	 * where "panic" saved registers.
	 * (libkvm also needs to know current u area to get user
	 * address space mapping).
	 */
	(void)set_procaddr((CORE_ADDR)masterprocp);
}

extern kvm_t *kd;

/*
 * We have to handle kernel stack references specially since they
 * fall in the user part of the page table.
 */
int
kernel_xfer_memory(addr, cp, len, write, target)
     CORE_ADDR addr;
     char *cp;
     int len;
     int write;
     struct target_ops *target;
{
	if (write)
		return kvm_write(kd, addr, cp, len);
	if (addr < (CORE_ADDR)KERNBASE) {
		if (masterprocp == 0 && set_procaddr(0))
			return (-1);
		return kvm_uread(kd, &fakeproc, addr, cp, len);
	}
	return kvm_read(kd, addr, cp, len);
}

CORE_ADDR
kernel_translate_saved_pc(fr)
	struct frame_info *fr;
{
	CORE_ADDR pc;
	unsigned long addr;

	if (tstart == 0)
		init_translate_limits();
	pc = read_memory_integer(fr->frame + 4, 4);
	if (tstart <= pc && pc < tend) {
		struct trapframe *tfr = (struct trapframe *)
			(fr->frame + 8);
		return (read_memory_integer((CORE_ADDR)&tfr->tf_eip, 4));
	}
	if (istart <= pc && pc < iend) {
		struct intrframe *ifr = (struct intrframe *)
			(fr->frame + 8);
		return (read_memory_integer((CORE_ADDR)&ifr->if_eip, 4));
	}
	return (pc);
}

void
kernel_frame_find_saved_regs(fr, fregs)
	struct frame_info *fr;
	struct frame_saved_regs *fregs;
{
	CORE_ADDR pc;

	if (tstart == 0)
		init_translate_limits();
	pc = read_memory_integer(fr->frame + 4, 4);
	if (tstart <= pc && pc < tend) {
		struct trapframe *tfr = (struct trapframe *)
			(fr->frame + 8);

		bzero(fregs, sizeof *fregs);
		fregs->regs[PC_REGNUM] = (CORE_ADDR)fr->frame + 4;
		fregs->regs[FP_REGNUM] = (CORE_ADDR)fr->frame;
		fregs->regs[SP_REGNUM] = (CORE_ADDR)&tfr[1];
		fregs->regs[PS_REGNUM] = (CORE_ADDR)&tfr->tf_eflags;
		fregs->regs[0] = (CORE_ADDR)&tfr->tf_eax;
		fregs->regs[1] = (CORE_ADDR)&tfr->tf_ecx;
		fregs->regs[2] = (CORE_ADDR)&tfr->tf_edx;
		fregs->regs[3] = (CORE_ADDR)&tfr->tf_ebx;
		fregs->regs[6] = (CORE_ADDR)&tfr->tf_esi;
		fregs->regs[7] = (CORE_ADDR)&tfr->tf_edi;
		return;
	}
	if (istart <= pc && pc < iend) {
		struct intrframe *ifr = (struct intrframe *)
			(fr->frame + 8);

		bzero(fregs, sizeof *fregs);
		fregs->regs[PC_REGNUM] = (CORE_ADDR)fr->frame + 4;
		fregs->regs[FP_REGNUM] = (CORE_ADDR)fr->frame;
		fregs->regs[SP_REGNUM] = (CORE_ADDR)&ifr[1];
		fregs->regs[PS_REGNUM] = (CORE_ADDR)&ifr->if_eflags;
		fregs->regs[0] = (CORE_ADDR)&ifr->if_eax;
		fregs->regs[1] = (CORE_ADDR)&ifr->if_ecx;
		fregs->regs[2] = (CORE_ADDR)&ifr->if_edx;
		fregs->regs[3] = (CORE_ADDR)&ifr->if_ebx;
		fregs->regs[6] = (CORE_ADDR)&ifr->if_esi;
		fregs->regs[7] = (CORE_ADDR)&ifr->if_edi;
		return;
	}
	i386_frame_find_saved_regs(fr, fregs);
}
