/*
 * Copyright (c) 1995 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI ipcs.c,v 1.1 1995/12/19 23:56:20 donn Exp
 */

#include <sys/param.h>
#include <sys/mman.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <err.h>
#include <fcntl.h>
#include <glob.h>
#include <grp.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "sco_ipc.h"

#define MAXFIELD	32
#define	MAXHEADING	160

#define	T_Q	1
#define	T_M	2
#define	T_S	4

struct ipc_object o;

enum ipc_field { NONE=0, T, ID, KEY, MODE, OWNER, GROUP, CREATOR,
    CGROUP, CBYTES, QNUM, QBYTES, LSPID, LRPID, STIME, RTIME,
    QCTIME, NATTCH, SEGSZ, CPID, LPID, ATIME, DTIME, MCTIME, NSEMS,
    OTIME, SCTIME };

struct ipcs {
	enum ipc_field field;
	const char *name;
	int width;
	void (*format)(int, const struct ipcs *);
	void *value;
};

void
dotype(int type, const struct ipcs *ip)
{
	static const char typestring[] = "xqmxs";

	putchar(typestring[type]);
}

void
doint(int type, const struct ipcs *ip)
{

	printf("%*d", ip->width, *(int *)ip->value);
}

void
doushort(int type, const struct ipcs *ip)
{

	printf("%*hu", ip->width, *(unsigned short *)ip->value);
}

void
dohex(int type, const struct ipcs *ip)
{

	printf("%#*lx", ip->width, *(long *)ip->value);
}

void
domode(int type, const struct ipcs *ip)
{
	int mode, mask;

	/*
	 * first byte:
	 *	C if shm segment is to be cleared on attach;
	 *	D if shm segment has been removed
	 *	- otherwise
	 * Since we don't produce or need this sort of state,
	 * we always print a '-'.
	 */
	putchar('-');

	/*
	 * second byte:
	 *	R if waiting on msg receive
	 *	S if waiting on msg send
	 *	- otherwise
	 */
	if (type == T_Q && o.l.yield.want) {
		struct msqid_ds *dp = &o.p.msg.msgd;

		/* a fairly poor heuristic; better than nothing? */
		if (dp->msg_cbytes < dp->msg_qbytes / 2)
			putchar('R');
		else
			putchar('S');
	} else
		putchar('-');

	mode = o.p.perm.mode;
	for (mask = 0400; mask; mask >>= 1) {
		if (mode & mask)
			putchar('r');
		else
			putchar('-');
		mask >>= 1;
		if (mode & mask)
			if (type == T_S)
				putchar('a');
			else
				putchar('w');
		else
			putchar('-');
		mask >>= 1;
		putchar('-');		/* currently unused */
	}
}

void
douser(int type, const struct ipcs *ip)
{
	uid_t uid = *(unsigned short *)ip->value;

	printf("%*.*s", ip->width, ip->width, user_from_uid(uid, 0));
}

void
dogroup(int type, const struct ipcs *ip)
{
	gid_t gid = *(unsigned short *)ip->value;

	printf("%*.*s", ip->width, ip->width, group_from_gid(gid, 0));
}

void
dotime(int type, const struct ipcs *ip)
{
	time_t t = *(unsigned int *)ip->value;
	char buf[9];

	if (t == 0)
		printf("no-entry");
	else {
		strftime(buf, 9, "%T", localtime(&t));
		printf("%s", buf);
	}
}

struct ipcs ipcstab[] = {
	{ T, "T", 1, dotype, 0 },
	{ ID, "ID", 6, doint, &o.id },
	{ KEY, "KEY   ", 10, dohex, &o.key },
	{ MODE, "MODE   ", 11, domode, &o },
	{ OWNER, "OWNER", 8, douser, &o.p.perm.uid },
	{ GROUP, "GROUP", 8, dogroup, &o.p.perm.gid },
	{ CREATOR, "CREATOR", 8, douser, &o.p.perm.cuid },
	{ CGROUP, "CGROUP", 8, dogroup, &o.p.perm.cgid },
	{ CBYTES, "CBYTES", 6, doushort, &o.p.msg.msgd.msg_cbytes },
	{ QNUM, "QNUM", 5, doushort, &o.p.msg.msgd.msg_qnum },
	{ QBYTES, "QBYTES", 6, doushort, &o.p.msg.msgd.msg_qbytes },
	{ LSPID, "LSPID", 5, doushort, &o.p.msg.msgd.msg_lspid },
	{ LRPID, "LRPID", 5, doushort, &o.p.msg.msgd.msg_lrpid },
	{ STIME, "STIME ", 8, dotime, &o.p.msg.msgd.msg_stime },
	{ RTIME, "RTIME ", 8, dotime, &o.p.msg.msgd.msg_rtime },
	{ QCTIME, "CTIME ", 8, dotime, &o.p.msg.msgd.msg_ctime },
	{ NATTCH, "NATTCH", 6, doushort, &o.p.shm.shm_nattch },
	{ SEGSZ, "SEGSZ", 6, doint, &o.p.shm.shm_segsz },
	{ CPID, "CPID", 5, doushort, &o.p.shm.shm_cpid },
	{ LPID, "LPID", 5, doushort, &o.p.shm.shm_lpid },
	{ ATIME, "ATIME ", 8, dotime, &o.p.shm.shm_atime },
	{ DTIME, "DTIME ", 8, dotime, &o.p.shm.shm_dtime },
	{ MCTIME, "CTIME ", 8, dotime, &o.p.shm.shm_ctime },
	{ NSEMS, "NSEMS", 5, doushort, &o.p.sem.sem_nsems },
	{ OTIME, "OTIME ", 8, dotime, &o.p.sem.sem_otime },
	{ SCTIME, "CTIME ", 8, dotime, &o.p.sem.sem_ctime },
	{ 0 }
};

enum ipc_field mvec[MAXFIELD], qvec[MAXFIELD], svec[MAXFIELD];

void
addfield(enum ipc_field *vec, ...)
{
	va_list ap;
	enum ipc_field f, *v;

	va_start(ap, vec);
	while (f = va_arg(ap, enum ipc_field)) {
		for (v = vec; *v && *v != f; ++v)
			;
		*v = f;
	}
	va_end(ap);
}

void
heading(enum ipc_field *vec)
{
	struct ipcs *ip;
	enum ipc_field *v;
	char *sep = "";
	int n = 0;
	static char curheading[MAXHEADING], lastheading[MAXHEADING];

	/* inefficient but correct */
	for (ip = ipcstab; ip->field != 0; ++ip)
		for (v = vec; *v; ++v)
			if (*v == ip->field) {
				n += sprintf(&curheading[n], "%s%*.*s", sep,
				    ip->width, ip->width, ip->name);
				sep = " ";
				break;
			}
	if (strncmp(curheading, lastheading, n) == 0)
		return;
	strcpy(lastheading, curheading);
	puts(curheading);
}

const char *gpat[] = {
0,
"/tmp/.msg.[0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f]",
"/tmp/.shm.[0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f]",
0,
"/tmp/.sem.[0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f][0-9a-f]",
};

void
dovec(int type, enum ipc_field *vec)
{
	glob_t g;
	struct ipcs *ip;
	enum ipc_field *v;
	char **pathp;
	caddr_t buf;
	char sep;
	int f;

	bzero(&g, sizeof g);
	if (glob(gpat[type], 0, 0, &g) != 0)
		err(1, "glob");
	if (g.gl_matchc == 0)
		return;

	for (pathp = g.gl_pathv; *pathp; ++pathp) {
		if ((f = open(*pathp, O_RDONLY)) == -1)
			continue;
		if ((buf = mmap(0, NBPG, PROT_READ, MAP_SHARED, f, 0))
		    == (caddr_t)-1) {
			warn("%s: mmap", *pathp);
			continue;
		}

		o = *(struct ipc_object *)buf;
		munmap(buf, NBPG);
		close(f);

		sep = '\0';
		for (ip = ipcstab; ip->field; ++ip)
			for (v = vec; *v; ++v)
				if (*v == ip->field) {
					if (sep)
						putchar(sep);
					sep = ' ';
					(*ip->format)(type, ip);
					break;
				}
		putchar('\n');
	}
}

extern char *__progname;

void
usage()
{

	errx(1, "usage: %s [-abcmopqst]", __progname);
}

int
main(int argc, char **argv)
{
	time_t t;
	int c;
	int type = 0;

	addfield(mvec, T, ID, KEY, MODE, OWNER, GROUP, 0);
	addfield(qvec, T, ID, KEY, MODE, OWNER, GROUP, 0);
	addfield(svec, T, ID, KEY, MODE, OWNER, GROUP, 0);

	while ((c = getopt(argc, argv, "abcmopqstC:N:X")) != EOF)
		switch (c) {

		case 'a':
			addfield(qvec, CREATOR, CGROUP, CBYTES, QNUM,
			    QBYTES, LSPID, LRPID, STIME, RTIME, QCTIME, 0);
			addfield(mvec, CREATOR, CGROUP, NATTCH, SEGSZ,
			    CPID, LPID, ATIME, DTIME, MCTIME, 0);
			addfield(svec, CREATOR, CGROUP, NSEMS, OTIME,
			    SCTIME, 0);
			break;
		case 'b':
			addfield(qvec, QBYTES, 0);
			addfield(mvec, SEGSZ, 0);
			addfield(svec, NSEMS, 0);
			break;
		case 'c':
			addfield(qvec, CREATOR, CGROUP, 0);
			addfield(mvec, CREATOR, CGROUP, 0);
			addfield(svec, CREATOR, CGROUP, 0);
			break;
		case 'm':
			type |= T_M;
			break;
		case 'o':
			addfield(qvec, CBYTES, QNUM, 0);
			addfield(mvec, NATTCH, 0);
			break;
		case 'p':
			addfield(qvec, LSPID, LRPID, 0);
			break;
		case 'q':
			type |= T_Q;
			break;
		case 's':
			type |= T_S;
			break;
		case 't':
			addfield(qvec, STIME, RTIME, QCTIME, 0);
			addfield(mvec, ATIME, DTIME, MCTIME, 0);
			addfield(svec, OTIME, SCTIME, 0);
			break;

		case 'C':
		case 'N':
		case 'X':
			warnx("-%c: option not supported", optopt);
			break;

		default:
			usage();
			break;
		}

	if (optind < argc)
		usage();

	time(&t);
	printf("IPC status from /tmp as of %s", ctime(&t));

	if (type == 0 || type & T_Q) {
		heading(qvec);
		printf("Message Queues:\n");
		dovec(T_Q, qvec);
	}
	if (type == 0 || type & T_M) {
		heading(mvec);
		printf("Shared Memory:\n");
		dovec(T_M, mvec);
	}
	if (type == 0 || type & T_S) {
		heading(svec);
		printf("Semaphores:\n");
		dovec(T_S, svec);
	}

	return (0);
}
