/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_ipc.h,v 2.3 1995/12/19 22:49:27 donn Exp
 */

/*
 * Common definitions for IPC object management.
 */

#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/msg.h>

#include <sys/syscall.h>	/* XXX */

#ifdef SYS_sem_lock
#include <sys/semaphore.h>
#else
#include <pthread.h>
#endif

#ifndef IPC_LIBRARY
/*
 * If we're executing in the emulator,
 * we know that our malloc() will give us page-aligned memory.
 * We can't guarantee that for user binaries.
 */
#define	valloc		malloc
#endif

enum ipc_type { SCO_IPC_SHM, SCO_IPC_SEM, SCO_IPC_MSG };

struct msg_overhead {
	long	len;		/* length of queue segment */
	long	head;		/* offset of queue head */
	long	tail;		/* offset of queue tail */
};

/*
 * An IPC object is a file in /tmp that is logically divided into two parts.
 * The first page contains object management information.
 * This information begins with an ipc_perm structure and is followed
 * by standard information specific to the IPC type (shm, sem or msg),
 * which is in turn followed by implementation-specific information.
 * The rest of the file is shared data that implements the object's semantics.
 * Both parts of the file are mapped into the current process,
 * although not contiguously.
 */
struct ipc_object {
	union {
		struct ipc_perm	perm;
		struct {
			struct msqid_ds		msgd;
			struct msg_overhead	msgo;
		} msg;
		struct shmid_ds	shm;
		struct semid_ds	sem;
	} p;
	key_t	key;
	int	id;
	struct ipc_lock {
#ifdef SYS_sem_lock
		sem_t		op;
		sem_t		yield;
#else
		pthread_mutex_t	op;
		pthread_mutex_t	yield;
#endif
	} l;
	int	removed;
};

/*
 * An IPC chain structure contains information about an IPC object
 * that is specific to the current process, rather than shared with others.
 * This includes a link to other chain structures, a pointer to
 * the shared, mapped ipc_object, and other overhead.
 */
struct ipc_chain {
	struct ipc_chain	*next;
	struct ipc_object	*obj;
	union {
		struct {
			char		*buf;
			long		len;
		} msgx;
		struct {
			struct sem	*sems;
			struct semadj	*adj;
		} semx;
		struct {
			struct shaddr	*sha;
		} shmx;
	} u;
};

void __ipc_apply_semadj __P((void));
extern struct ipc_chain *__ipc_chains[];
struct ipc_chain *__ipc_chain_by_id  __P((enum ipc_type, int));
extern int (*__ipc_check[]) __P((struct ipc_chain *, int, int, void *));
int __ipc_check_uid __P((const struct ipc_perm *));
void __ipc_clear_lock __P((struct ipc_object *));
int __ipc_ctl __P((enum ipc_type, int, int, int, void *));
void __ipc_decode_semadj __P((void));
void __ipc_destroy_all_semadj __P((void));
void __ipc_encode_semadj __P((void));
int __ipc_get __P((enum ipc_type, key_t, int, int));
int __ipc_get_lock __P((struct ipc_object *));
void __ipc_get_lock_nointr __P((struct ipc_object *));
extern int (*__ipc_init[]) __P((struct ipc_chain *, int, int, int));
void __ipc_init_lock __P((struct ipc_object *));
int __ipc_msg_init __P((struct ipc_chain *, int, int, int));
void __ipc_msg_remove __P((struct ipc_chain *));
int __ipc_msg_work __P((struct ipc_chain *, int, int, void *));
extern const char *__ipc_name[];
void __ipc_path __P((enum ipc_type, int, char *));
int __ipc_perm __P((const struct ipc_perm *, int));
extern void (*__ipc_remove_wakeup[]) __P((struct ipc_chain *));
int __ipc_sem_check __P((struct ipc_chain *, int, int, void *));
int __ipc_sem_init __P((struct ipc_chain *, int, int, int));
void __ipc_sem_remove __P((struct ipc_chain *));
int __ipc_sem_work __P((struct ipc_chain *, int, int, void *));
int __ipc_shm_init __P((struct ipc_chain *, int, int, int));
int __ipc_shm_work __P((struct ipc_chain *, int, int, void *));
extern const int __ipc_sizes[];
int __ipc_suspend_lock __P((struct ipc_object *));
extern int (*__ipc_work[]) __P((struct ipc_chain *, int, int, void *));
void __ipc_yield_lock __P((struct ipc_object *));
