/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 *
 *	BSDI sco_ipc.c,v 2.5 1996/01/08 03:19:03 donn Exp
 */

/*
 * Common code for emulation of iBCS2 IPC routines.
 */

#include <sys/param.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <err.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "sco_ipc.h"

static int __ipc_get_by_id __P((enum ipc_type, int, key_t, int, int));

struct ipc_chain *__ipc_chains[3];

struct ipc_chain *
__ipc_chain_by_id(type, id)
	enum ipc_type type;
	int id;
{
	struct ipc_chain *c;

	do {
		for (c = __ipc_chains[type]; c; c = c->next)
			if (c->obj->id == id) {
				if (c->obj->removed) {
					errno = EINVAL;
					return (0);
				}
				return (c);
			}
	} while (__ipc_get_by_id(type, id, 0, 0, 0) != -1);

	errno = EINVAL;
	return (0);
}

/*
 * XXX writing implies reading for this implementation...
 */
int
__ipc_perm(ip, rdonly)
	const struct ipc_perm *ip;
	int rdonly;
{
	int m = ip->mode;
	long uid, gid;

	/* do the easy one first */
	if (m & S_IROTH && (rdonly || m & S_IWOTH))
		return (0);

	uid = geteuid();

	/* root always gets permission */
	if (uid == 0)
		return (0);

	if ((uid == ip->uid || uid == ip->cuid) &&
	    m & S_IRUSR && (rdonly || m & S_IWUSR))
	    	return (0);

	gid = getegid();
	if ((gid == ip->gid || gid == ip->cgid) &&
	    m & S_IRGRP && (rdonly || m & S_IWGRP))
	    	return (0);

	errno = EACCES;
	return (-1);
}

int
__ipc_check_uid(ip)
	const struct ipc_perm *ip;
{
	long uid = geteuid();

	if (uid == 0 || uid == ip->uid || uid == ip->cuid)
		return (0);

	errno = EPERM; 
	return (-1);
}

void
__ipc_path(type, id, path)
	enum ipc_type type;
	int id;
	char *path;
{

	sprintf(path, "/tmp/.%s.%08x", __ipc_name[type], id);
}

int
__ipc_get(type, key, len, flags)
	enum ipc_type type;
	key_t key;
	int len, flags;
{
	char name[MAXPATHLEN];
	char name2[MAXPATHLEN];
	struct stat st;
	struct ipc_chain *c;
	int f;
	int id;
	int r;
#ifdef IPC_LIBRARY
	sigset_t s, os;
#endif

#ifdef IPC_LIBRARY
	sigfillset(&s);
	sigprocmask(SIG_BLOCK, &s, &os);
#endif

	if (key != IPC_PRIVATE)
		for (c = __ipc_chains[type]; c; c = c->next)
			if (c->obj->key == key) {
				if (c->obj->removed)
					continue;
#ifdef IPC_LIBRARY 
				sigprocmask(SIG_SETMASK, &os, 0);
#endif
				if ((flags & (IPC_CREAT|IPC_EXCL)) ==
				    (IPC_CREAT|IPC_EXCL)) {
					errno = EEXIST;
					return (-1);
				}
				return (c->obj->id);
			}

	/* get the key-to-ID mapping file */
	if (key == IPC_PRIVATE)
		sprintf(name, "/tmp/.key.x.%d", getpid());
	else
		sprintf(name, "/tmp/.key.%s.%08lx", __ipc_name[type], key);
	if (stat(name, &st) == -1) {
		/* try creating the file */
		if ((f = open(name, O_WRONLY|O_CREAT|O_EXCL,
		    S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) == -1)
			err(1, "%sget open: %s", __ipc_name[type], name);
		if (fstat(f, &st) == -1)
			err(1, "%sget fstat: %s", __ipc_name[type], name);
		close(f);
	}

	/* the inode number is guaranteed to be unique (and positive?) */
	id = (int)st.st_ino;
	if (key == IPC_PRIVATE) {
		/* reserve a unique ID */
		sprintf(name2, "/tmp/.key.p.%08x", id);
		rename(name, name2);
		/* always permit creation */
		flags |= IPC_CREAT;
	}

	r = __ipc_get_by_id(type, id, key, len, flags);
#ifdef IPC_LIBRARY 
	sigprocmask(SIG_SETMASK, &os, 0);
#endif
	return (r);
}

static int
__ipc_get_by_id(type, id, key, len, flags)
	enum ipc_type type;
	int id;
	key_t key;
	int len, flags;
{
	char name[MAXPATHLEN];
	struct ipc_chain *c;
	struct ipc_object *o;
	mode_t mode = 0;
	int save_errno;
	int oflags;
	int f;

	/* open an object file */
	oflags = O_RDWR | O_EXLOCK;
	if (flags & IPC_CREAT) {
		oflags |= O_CREAT | O_EXCL;
		mode = flags & (S_IRWXU|S_IRWXG|S_IRWXO);
	}
	__ipc_path(type, id, name);
	while ((f = open(name, oflags, mode)) == -1) {
		if (errno == EINTR)
			continue;
		if (errno != EEXIST ||
		    (flags & (IPC_CREAT|IPC_EXCL)) == (IPC_CREAT|IPC_EXCL))
			return (-1);
		oflags &= ~(O_CREAT | O_EXCL);
	}
	if (oflags & O_CREAT) {
		/* override umask() */
		if (fchmod(f, mode) == -1)
			err(1, "%sget fchmod", __ipc_name[type]);

		/* force allocation of first page */
		if (lseek(f, NBPG-1, SEEK_SET) == -1 || write(f, "", 1) == -1) {
			save_errno = errno;
			close(f);
			errno = save_errno;
			return (-1);
		}
	}

	/* create an object cache entry */
	if ((c = malloc(sizeof (*c))) == 0)
		err(1, "%sget malloc cache", __ipc_name[type]);
	bzero(c, sizeof (*c));
	if ((o = valloc(NBPG)) == 0)
		err(1, "%sget malloc object", __ipc_name[type]);
	if (mmap((caddr_t)o, NBPG, PROT_READ|PROT_WRITE, MAP_SHARED|MAP_FIXED,
	    f, 0) == (caddr_t)-1) {
		save_errno = errno;
		close(f);
		errno = save_errno;
		return (-1);
	}
	c->obj = o;

	/* some default initializations */
	if (oflags & O_CREAT) {
		o->key = key;
		o->id = id;
		o->p.perm.cuid = geteuid();
		o->p.perm.cgid = getegid();
		o->p.perm.uid = o->p.perm.cuid;
		o->p.perm.gid = o->p.perm.cgid;
		o->p.perm.mode = mode;
	}

	if ((*__ipc_init[type])(c, f, oflags & O_CREAT, len) == -1) {
		save_errno = errno;
		close(f);
		free(c);
		if (mmap((caddr_t)o, NBPG, PROT_READ|PROT_WRITE|PROT_EXEC,
		    MAP_ANON|MAP_FIXED|MAP_PRIVATE, -1, 0) != (caddr_t)-1)
			free(o);
		errno = save_errno;
		return (-1);
	}

	msync((caddr_t)o, NBPG);
	close(f);

	c->next = __ipc_chains[type];
	__ipc_chains[type] = c;

	return (o->id);
}

static int
__ipc_remove(type, id)
	enum ipc_type type;
	int id;
{
	char name[MAXPATHLEN], name2[MAXPATHLEN];
	struct ipc_chain *c, **cp;

	for (cp = &__ipc_chains[type]; *cp; cp = &(*cp)->next)
		if ((*cp)->obj->id == id)
			break;
	if (*cp == 0) {
		errno = EINVAL;
		return (-1);
	}
	c = *cp;
	*cp = c->next;

	/*
	 * Delete both the IPC object and the key-to-ID mapping file.
	 * To avoid races, we first rename the key file so
	 * that the id remains reserved but no key translation
	 * can be done; then we remove the IPC object, and
	 * finally we remove the renamed key file,
	 * making the ID available for allocation again.
	 */
	if (c->obj->key == IPC_PRIVATE)
		sprintf(name, "/tmp/.key.p.%08x", id);
	else
		sprintf(name, "/tmp/.key.%s.%08lx", __ipc_name[type],
		    c->obj->key);
	sprintf(name2, "/tmp/.lock.%08x", id);
	if (rename(name, name2) == -1)
		err(1, "ipc_remove rename: %s, %s", name, name2);
	sprintf(name, "/tmp/.%s.%08x", __ipc_name[type], id);
	unlink(name);
	unlink(name2);

	c->obj->removed = 1;
	msync((caddr_t)c->obj, NBPG);

	if (__ipc_remove_wakeup[type])
		(*__ipc_remove_wakeup[type])(c);

	/* remove shared page and reallocate an unshared page */
	if (mmap((caddr_t)c->obj, NBPG, PROT_READ|PROT_WRITE|PROT_EXEC,
	    MAP_ANON|MAP_FIXED|MAP_PRIVATE, -1, 0) != (caddr_t)-1)
		free(c->obj);
	free(c);

	return (0);
}

int
__ipc_ctl(type, id, indx, cookie, buf)
	enum ipc_type type;
	int id, indx, cookie;
	void *buf;
{
	struct ipc_chain *c;
	struct ipc_object *o;
	struct ipc_perm *p;
	int r = 0;
#ifdef IPC_LIBRARY
	sigset_t s, os;
#endif

	if ((c = __ipc_chain_by_id(type, id)) == 0)
		return (-1);
	o = c->obj;

	/* check permissions */
	switch (cookie) {
	case IPC_STAT:
		if (__ipc_perm(&o->p.perm, 1) == -1)
			return (-1);
		break;
	case IPC_SET:
	case IPC_RMID:
		if (__ipc_check_uid(&o->p.perm) == -1)
			return (-1);
		break;
	default:
		if ((*__ipc_check[type])(c, indx, cookie, buf) == -1) {
			errno = EINVAL;
			return (-1);
		}
		break;
	}

#ifdef IPC_LIBRARY
	sigfillset(&s);
	sigprocmask(SIG_BLOCK, &s, &os);
#endif

	/* do the work */
	switch (cookie) {

	case IPC_STAT:
		bcopy(&o->p, buf, __ipc_sizes[type]);
		break;

	case IPC_SET:
		p = (struct ipc_perm *)buf;
		o->p.perm.uid = p->uid;
		o->p.perm.gid = p->gid;
		o->p.perm.mode = p->mode;
		r = (*__ipc_work[type])(c, indx, cookie, buf);
		msync((caddr_t)o, NBPG);
		break;

	case IPC_RMID:
		if ((*__ipc_work[type])(c, indx, cookie, buf) == -1 ||
		    __ipc_remove(type, id) == -1)
			r = -1;
		break;

	default:
		r = (*__ipc_work[type])(c, indx, cookie, buf);
		break;
	}

#ifdef IPC_LIBRARY
	sigprocmask(SIG_SETMASK, &os, 0);
#endif
	return (r);
}
