/*
 * Copyright (c) 1994 Berkeley Software Design, Inc. All rights reserved.
 * The Berkeley Software Design Inc. software License Agreement specifies
 * the terms and conditions for redistribution.
 */
/*	BSDI disk.h,v 2.2 1995/10/11 18:52:14 prb Exp	*/

#include "partition.h"
#include "filesys.h"
#ifdef	i386
#include <machine/bootblock.h>
#endif

class Disk;

struct BootBlock {
    unsigned char	data[512];

    int Write(int fd);
    int Read(int fd);
    int Read(char *);
    void Clean()	{ memset(data, 0, sizeof(data)); }
#ifdef	i386
    void MergePartitions(mbpart *);

    int Signature()	{ return((data[511] << 8) | data[510]); }
    unsigned char *Part(int i)
			{ return(data + 512 - (5 - i) * 16 - 2); }
    unsigned long Length(int i)
			{ return(*(unsigned long *)(Part(i)+12)); }
    unsigned long Offset(int i)
			{ return(*(unsigned long *)(Part(i)+8)); }

    unsigned long BTrack(int i) {
			unsigned char *p = Part(i) + 1;
			return(p[0]); }
    unsigned long BCyl(int i) {
			unsigned char *p = Part(i) + 1;
			return(p[2] | (int(p[1] & 0xc0) << 2)); }
    unsigned long BSector(int i) {
			unsigned char *p = Part(i) + 1;
			return(p[1] & 0x3f); }

    unsigned long ETrack(int i) {
			unsigned char *p = Part(i) + 5;
			return(p[0]); }
    unsigned long ECyl(int i) {
			unsigned char *p = Part(i) + 5;
			return(p[2] | (int(p[1] & 0xc0) << 2)); }
    unsigned long ESector(int i) {
			unsigned char *p = Part(i) + 5;
			return(p[1] & 0x3f); }

    unsigned char Type(int i)
			{ return(Part(i)[4]); }
    unsigned char Active(int i)
			{ return(Part(i)[0]); }
#endif
};

struct DiskLabel : public disklabel {
    int Read(int fd, off_t, off_t * = 0);
    int Internal(int);
    int WriteInternal(int);
    int ST506(int);
    int SCSI(char *);
    int Disktab(char *, char * = 0);
    void Fixup();
    void Clean();
    int Valid();
    int Soft();
    void ComputeSum(u_long = 0);
#ifdef	sparc
    int Sun(int fd);
    int WriteSun(int fd);
#endif
};

struct Geometry {
private:
    int			heads;
    int			sectors;
    int			cylinders;
    quad_t		size;
    int	  		secpercyl;
public:
    Geometry()			{ secpercyl = heads =
				  sectors = cylinders = size = 0; }
    void Zero()			{ secpercyl = heads =
				  sectors = cylinders = size = 0; }
    int Valid()			{ return(cylinders && heads && sectors); }

    quad_t Sectors()		{ return(size); }
    quad_t Sectors(quad_t i)	{ return(size = i); }
    int Heads()			{ return(heads); }
    int Heads(int i)		{ return(heads = i); }
    int SecPerTrack()		{ return(sectors); }
    int SecPerTrack(int i)	{ return(sectors = i); }
    int Cyls()			{ return(cylinders); }
    int Cyls(int i)		{ return(cylinders = i); }
    int SecPerCyl(int i)	{ return(secpercyl = i); }
    int SecPerCyl()		{ return(secpercyl
					 ? secpercyl
					 : (secpercyl = heads * sectors)); }

    Geometry &operator =(Geometry &i)
				{ heads = i.heads; sectors = i.sectors;
				  cylinders = i.cylinders;
				  secpercyl = i.SecPerCyl();
				  if (i.size) size = i.size;
				  return(*this); }
    int operator ==(Geometry &i)
				{ return(heads == i.heads &&
					 sectors == i.sectors &&
					 cylinders == i.cylinders &&
					 SecPerCyl() == i.SecPerCyl()); }
    int Match(Geometry &i)	{ return(heads == i.heads &&
					 sectors == i.sectors &&
					 SecPerCyl() == i.SecPerCyl()); }
};

#if 0
#define	SECSIZE	disk.secsize
#else
#define	SECSIZE	DEV_BSIZE
#endif

struct Disk {
public:
    int			dfd;
    char		device[PATH_MAX];
    char		*path;
    int			d_type;

    off_t		bsdoff;			// Offset to BSD label

    int			secsize;		// # Bytes/sector
    int			badblock;
    Geometry		bsdg;

    BootBlock		bootblock;
    DiskLabel		label_original;
    DiskLabel		label_template;		// Filled in by scsicmd
    DiskLabel		label_new;		// The label we will write out
    static DiskLabel	empty_label;

    FileSystem		bsd[8];			// Where we build BSD parts
    static FileSystem	empty_filsys;
    unsigned		bsd_modified : 1;	// Original BSD part changed

#ifdef	sparc
    DiskLabel		label_sun;		// Any SUN Label found
#endif
#ifdef	i386
    int			bsdbootpart;		// Bootable BSD partition
    Geometry		cmos;
    Geometry		bios;
    Geometry		fdisk;
    Geometry		dosg;

    unsigned		use_bsdg : 1;		// Only use bsd geometry
    unsigned		use_bsd : 1;		// if we are doing bsd
    unsigned		has_fdisk : 1;		// if we have an fdisk label
    unsigned		use_fdisk : 1;		// if we want an fdisk label

    unsigned		part_modified : 1;	// Original FDISK part changed
    mbpart		part_table[4];

    Partition		part[4];		// Where we build FDISK parts
    static Partition	empty_partition;

    unsigned char	active;			// partition # which is active
#endif

    Disk();
    int Init(char *);
    int Type();
    int FSys();
    int EditBSDPart(int);
    void Sort();
    void BDraw();
    int DefaultSwap(int = -1);
    int DefaultRoot(int = 0);
    FileSystem &RootFilesystem();
    FileSystem &FindFileSystem(int);
    off_t LabelLocation();
    int FindFSys(int);
    int FindFreeSpace(int pt, int &offset, int &length);

    quad_t Sectors()		{ return(bsdg.Sectors()); }
    quad_t UseSectors()		{ return(bsdg.Sectors() - badblock); }
    int MBs()			{ return(int(ToMB(Sectors()))); }
    double ToMB(int i)		{ return(i / (1024.0 * 1024.0 / secsize)); }
    double ToCyl(int i)		{ return(i / double(SecPerCyl())); }
    int FromMB(double i);
    int FromCyl(double i);
    int FromTrack(double i);
    int FromSector(double i);
    int Max(int i)		{ return(i > UseSectors() ? UseSectors() : i); }

    void AddNumber(int &v, char *b)
	{ AddNumber(v, b, FromSector); }
    void AddNumber(int &, char *, int (Disk::*func)(double));
    int AddPartition(int = 0);

    void WriteEnable();
    void WriteDisable();

#ifdef	sparc
    int SecPerTrack()		{ return(bsdg.SecPerTrack()); }
    int Heads()			{ return(bsdg.Heads()); }
    int Cyls()			{ return(bsdg.Cyls()); }
    int SecPerCyl()		{ return(bsdg.SecPerCyl()); }
#endif

#ifdef	i386
    int FDisk();
    int EditPart(int);
    void FSort();
    void FDraw();
    void ComputeBSDBoot();
    mbpart *PartTable()		{ UpdatePartTable();
				  return((has_fdisk||use_fdisk)?part_table:0); }

    Partition &FindPartition(int);

    int FindPartForBSD(int);
    int FindPartForBSD(FileSystem &);
    int FindFPart(int);

    void SwitchToCMOS()		{ if (!dosg.Valid()) dosg = fdisk;
				  use_bsdg = !dosg.Valid(); }
    void SwitchToBSD()		{ use_bsdg = bsdg.Valid(); }

    void UpdatePart();
    void UpdatePartTable();

    int SecPerTrack()		{ return(use_bsdg ? bsdg.SecPerTrack()
						  : dosg.SecPerTrack()); }
    int Heads()			{ return(use_bsdg ? bsdg.Heads()
						  : dosg.Heads()); }
    int Cyls()			{ return(use_bsdg ? bsdg.Cyls()
						  : dosg.Cyls()); }
    int SecPerCyl()		{ return(use_bsdg ? bsdg.SecPerCyl()
						  : dosg.SecPerCyl()); }
#endif
};

#ifdef	i386
inline
BSDType(int t)
{
    return(t == 0x9f);
}
#endif

extern Disk disk;

#ifdef	i386
extern "C" void disklabel_display(char *, FILE *, disklabel *, mbpart * = 0);
extern "C" int disklabel_getasciilabel(FILE *, disklabel *, mbpart * = 0);
#else
extern "C" void disklabel_display(char *, FILE *, disklabel *);
extern "C" int disklabel_getasciilabel(FILE *, disklabel *);
#endif
extern "C" int disklabel_checklabel(disklabel *);
extern "C" u_short dkcksum(struct disklabel *);
