/*	BSDI getgrent.c,v 2.7 1995/11/02 21:35:32 bostic Exp	*/

/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)getgrent.c	8.2 (Berkeley) 3/21/94";
#endif /* LIBC_SCCS and not lint */

#include <sys/types.h>

#include <grp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static FILE *gr_fp;
static int gr_stayopen;

static struct group *grscan __P((int, gid_t, const char *));
static int grstart __P((void));

struct group *
getgrent()
{
	if (gr_fp == NULL && !grstart())
		return (NULL);
	return (grscan(0, 0, NULL));
}

struct group *
getgrnam(name)
	const char *name;
{
	struct group *rval;

	if (!grstart())
		return (NULL);
	rval = grscan(1, 0, name);
	if (!gr_stayopen)
		endgrent();
	return (rval);
}

struct group *
getgrgid(gid)
	gid_t gid;
{
	struct group *rval;

	if (!grstart())
		return (NULL);
	rval = grscan(1, gid, NULL);
	if (!gr_stayopen)
		endgrent();
	return (rval);
}

static int
grstart()
{
	if (gr_fp != NULL) {
		rewind(gr_fp);
		return (1);
	}
	return ((gr_fp = fopen(_PATH_GROUP, "r")) ? 1 : 0);
}

int
setgrent()
{
	return (setgroupent(0));
}

int
setgroupent(stayopen)
	int stayopen;
{
	if (!grstart())
		return (0);
	gr_stayopen = stayopen;
	return (1);
}

void
endgrent()
{
	if (gr_fp != NULL) {
		(void)fclose(gr_fp);
		gr_fp = NULL;
	}
}

/*
 * Need space to store the entries read from the group file.  The members
 * list also needs space per member, and the strings making up the user
 * names must be allocated somewhere.  Rather than doing lots of small
 * allocations, we keep one buffer and resize it as needed.
 */
static struct group gr_group;
static char **members;
static size_t nmemb;
static char *membuf;
static size_t membufsize;

/*
 * Read target line from the group file.  Return NULL at EOF.
 * Target line is next line if search==0, otherwise the given
 * group name or gid.
 */
static struct group *
grscan(search, gid, name)
	int search;
	gid_t gid;
	const char *name;
{
	size_t linelen, n;
	char *bp, *lp, **m, *namep, *p, *passwdp;

#define	INITIAL_NMEMB	30			/* about 120 bytes */
#define	INITIAL_BUFSIZ	(INITIAL_NMEMB * 8)	/* about 240 bytes */

	/* Read lines until we find one that matches our search criteria. */
	for (;;) {
		if ((lp = fgetln(gr_fp, &linelen)) == NULL)
			return (0);
		lp[linelen - 1] = '\0';
		bp = lp;

		/* Optimize the usual case of searching for a name. */
		namep = strsep(&bp, ":\n");
		if (search && name != NULL && strcmp(namep, name) != 0)
			continue;
		passwdp = strsep(&bp, ":\n");
		if ((p = strsep(&bp, ":\n")) == NULL)
			continue;
		gr_group.gr_gid = atoi(p);
		if (search && name == NULL && gr_group.gr_gid != gid)
			continue;
		break;
	}

	/*
	 * Must save the line we read above, as it lives inside stdio.
	 * We only need the name, passwd, and members, but getting the
	 * gid too is not worth worrying about, so just save the whole
	 * thing.  To avoid lots of little expansions, allocate largest
	 * of INITIAL_BUFSIZE, or 2*linelen.
	 */
	if ((n = linelen) > membufsize || membuf == NULL) {
		if ((n *= 2) < INITIAL_BUFSIZ)
			n = INITIAL_BUFSIZ;
		if ((p = realloc(membuf, n)) == NULL)
			return (NULL);		/* ??? */
		membuf = p;
		membufsize = n;
	}

	/*
	 * Copy the line, and set the name and passwd fields (the gid
	 * field has already been set), then switch to the new line.
	 */
	memmove(membuf, lp, linelen);
	gr_group.gr_name = membuf + (namep - lp);
	gr_group.gr_passwd = membuf + (passwdp - lp);
	bp = membuf + (bp - lp);

	/*
	 * Count commas to find out how many members there might be.
	 * Note that commas separate, so if there is one comma there
	 * can be two members (group:*:id:user1,user2).  Add another
	 * to account for the NULL terminator.  As above, allocate
	 * largest of INITIAL_NMEMB, or 2*n.
	 */
	for (n = 2, p = bp; (p = strpbrk(p, ", ")) != NULL; ++p, ++n);
	if (n > nmemb || members == NULL) {
		if ((n *= 2) < INITIAL_NMEMB)
			n = INITIAL_NMEMB;
		if ((m = realloc(members, n * sizeof(*m))) == NULL)
			return (NULL);		/* ??? */
		members = m;
		nmemb = n;
	}

	/* Set the name pointers. */
	for (m = members; (p = strsep(&bp, ", \n")) != NULL;)
		if (p[0] != '\0')
			*m++ = p;
	*m = NULL;

	gr_group.gr_mem = members;
	return (&gr_group);
}
