// pitchtracker.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// PitchTracker determines the frequency of a frame's worth of sound and puts
// the result into a PCHData object (via the 'frame' pointer).

#ifndef PITCHTRACKER_H
#ifdef __GNUG_
#pragma interface
#endif
#define PITCHTRACKER_H

#include "arrayfunction.h"
#include "processfun.h"
#include "range.h"

#define maxFrameSize 1010	  /* maximum sample frame size */
#define MAXMM ((maxFrameSize / 10 + 1) / 2)

#define TWO_PI	6.28318530


class PitchTracker : public ArrayFunction {
	typedef ArrayFunction Super;
	typedef float P3DA5[50][5][MAXMM];
	typedef float P2DA5[50][5];
	typedef float P3DA6[50][6][MAXMM];
	typedef float P2DA6[50][6];
	typedef float PA5MAX[5][MAXMM];
	typedef float PA6MAX[6][MAXMM];
public:
	PitchTracker(Data* data, const Range& freqs, int framesize, int frmoffset);
	PitchTracker(Data* data) : ArrayFunction(data) {}
	virtual ~PitchTracker() {}
	redefined const char* message() { return "Running pitchtrack:  "; }
	redefined int operator () (double *input, Data *frame);
	void filter(double *array, int size=0);	
protected:
	redefined void initialize();
	redefined boolean setValues(Request& r);
	redefined boolean confirmValues(class Controller*);
	redefined int doApply(Data *);
	redefined int maxFrameLength() { return maxFrameSize; }
	Modifier* createFilter(Data *);
	double getPitch(double *array);
	double getRMS(double *array);
private:
	redefined const QueryInfo *requestInfo();
	double search(double qsum, float *g, float *h);
	void filfun(double qsum, float *fun, float *g, float *h);
	double nyquist() { return sampRate()/2.0; }
	void trigpo(double omega, PA5MAX phi, PA6MAX psi,
		float *gampsi, float *gamphi, int n);
private:
	float freq[50];
	P3DA5 tphi;
	P3DA6 tpsi;
	P2DA5 tgamph;
	P2DA6 tgamps;
	Range freqRange;
	int JMAX;
	int MM;
	enum FilterType { BandPass = 0x1, LowPass = 0x2 } filterType;
};

// this low-pass filters sound in preparation for pitch tracking

class PitchTrackFilter : public SimpleFunction {
public:
	PitchTrackFilter(Data *d);
	virtual ~PitchTrackFilter() {}
	redefined boolean configure(Controller *) { return true; }
	redefined const char* message() {
		return "Running low pass filter...";
	}
	redefined double operator () (double input);
protected:
	redefined void initialize();
	redefined boolean setValues(Request &) { return true; }
private:
	redefined const QueryInfo *requestInfo() { return nil; }
	double w1,w11,w12,w2,w21,w22,w3,w31,w32,w4,w41,w42;
};
	
#endif

