// optionsetter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include <String.h>
#undef String
#include "application.h"
#include "dialogbox.h"
#include "fileselector.h"
#include "optionsetter.h"
#include "query.h"
#include "request.h"
#include "lpcdata.h"
#include "smartmem.h"
#include "sound.h"
#include "soundheader.h"

const QueryInfo *
GlobalOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set Global Program Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
GlobalOptionSetter::configureRequest(Request &request) {
	request.appendValue("Alert beep volume: ", DialogBox::beepLevel(), Range(0, 100));
	request.appendChoice("Dialog Panels Ignore Window Manager: ", "|Yes|No|",
		DialogBox::overrideWindowManager() ? 0x1 : 0x2, true);
	request.appendChoice("Auto-Place Windows on Screen: ", "|Yes|No|",
		Application::globalResourceIsTrue("AutoPlaceWindows") ? 0x1 : 0x2, true);
}

boolean
GlobalOptionSetter::setValues(Request &request) {
	QueryValue beepLevel;
	request.retrieveValues(beepLevel);
	DialogBox::setBeepLevel(beepLevel);
	QueryChoice choices[2];
	request.retrieveChoices(choices, 2);
	DialogBox::overrideWindowManager(choices[0] == 0x1);
	Application::setGlobalResource(
		"AutoPlaceWindows", (choices[1] == 0x1) ? "true" : "false"
	);
	return true;
}

//********

const QueryInfo *
MemoryOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set Data Memory Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "Use to adjust memory limits for data only.", nil },
		{ nil }
	};
	return info;
}

void
MemoryOptionSetter::configureRequest(Request &request) {
	static char label[120];
	sprintf(label, "Total memory currently in use for data:  %.2f mB",
		SmartMemory::totalBytesAllocated()/1.0e+06);
	request.appendLabel(label);
	request.appendLabel("----------------");
	request.appendValue("Maximum Single Memory Allocation (mB.):",
		SmartMemory::maxSingleAllocation()/1.0e+06,  Range(1.0, 1000.0));
	request.appendValue("Maximum Total Memory Allocation (mB.):",
		SmartMemory::maxTotalAllocation()/1.0e+06, Range(10.0, 5000.0));
}

boolean
MemoryOptionSetter::setValues(Request &request) {
	const int nValues = 2;
	QueryValue v[nValues];
	request.retrieveValues(v, nValues);
	int maxSingle = v[0];
	int maxTotal = v[1];
	if(maxTotal <= maxSingle) {
		Application::alert("Total Allocation limit must be greater than",
			"the Single Allocation limit.  Please recheck settings.");
		return false;
	}
	else {
		SmartMemory::setMaxSingleAllocation(unsigned(maxSingle * 1.0e+06));
		SmartMemory::setMaxTotalAllocation(unsigned(maxTotal * 1.0e+06));
	}
	return true;
}

//********

const QueryInfo *
FileOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set File Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
FileOptionSetter::configureRequest(Request &request) {
	request.appendChoice("Read Raw (Headerless) Files:", "|Yes|No|",
		Application::globalResourceIsTrue("ReadRawFiles") ? 0x1 : 0x2, true);
	request.appendChoice("Store/Recall Browser Path:", "|Yes|No|",
		Application::globalResourceIsTrue("BrowserUseLastPath") ? 0x1 : 0x2, true);
	request.appendChoice("Browser Shows Invisible Files:", "|Yes|No|",
		FileSelector::showDotFiles() ? 0x1 : 0x2, true);
}

boolean
FileOptionSetter::setValues(Request &request) {
	const int nChoices = 3;
	QueryChoice c[nChoices];
	request.retrieveChoices(c, nChoices);
	Application::setGlobalResource(
		"ReadRawFiles", (c[0] == 0x1) ? "true" : "false"
	);
	Application::setGlobalResource(
		"BrowserUseLastPath", (c[1] == 0x1) ? "true" : "false"
	);
	FileSelector::showDotFiles(c[2] == 0x1);
	return true;
}

//********

const QueryInfo *
SoundOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set Sound Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "", nil },
		{ nil }
	};
	return info;
}

void
SoundOptionSetter::configureRequest(Request &request) {
	request.appendValue("Default Sampling Rate:", Sound::defaultSampleRate(),
		CharCheck::posIntsOnly);
	request.appendChoice("Default Sound Sample Format:",
		"|8-bit linear|8-bit mu-law|16-bit linear|32-bit floating point|",
		linearEnumToPowerOfTwo(Sound::defaultDataType()),
		true
	);
	request.appendChoice("Default Sound Header Format:",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIF-C|WAV|",
		SoundHeader::defaultHeaderType(),
		true
	);
}

boolean
SoundOptionSetter::setValues(Request &request) {
	QueryValue srate;
	request.retrieveValues(srate);
	Sound::setDefaultSampleRate(srate);
	const int nChoices = 2;
	QueryChoice c[nChoices];
	request.retrieveChoices(c, nChoices);
	int state = c[0];
	Sound::setDefaultDataType(DataType(powerOfTwoToLinearEnum(state)));
	SoundHeader::setDefaultHeaderType(SoundHeader::Type(c[1].state()));
	return true;
}

//********

const QueryInfo *
LPCOptionSetter::requestInfo() {
	static QueryLabelInfo labelInfo[] = {
		{ "Set LPC Data Options: " },
		{ nil }
	};      
	static QueryInfo info[] = {
		{ labelInfo, "Default samp rate = default sound samp rate.", nil },
		{ nil }
	};
	return info;
}

void
LPCOptionSetter::configureRequest(Request &request) {
	request.appendValue("Default Frame Rate: ", LPCData::defaultFrameRate(),
		CharCheck::posNumsOnly);
	request.appendValue("Default Number of Filter Poles:",
		LPCData::defaultNumberOfPoles(), Range(12, 64));
	request.appendChoice("Default LPC Header Format: ",
		"|Raw (No Header)|Csound-format Header|",
		LPCHeader::defaultHeaderType(),
		true
	);
}

boolean
LPCOptionSetter::setValues(Request &request) {
	const int nValues = 2;
	QueryValue values[nValues];
	request.retrieveValues(values, nValues);
	LPCData::setDefaultFrameRate(values[0]);
	LPCData::setDefaultNumberOfPoles(values[1]);
	QueryChoice headerFormat;
	request.retrieveChoices(headerFormat);
	LPCHeader::setDefaultHeaderType(LPCHeader::Type(headerFormat.state()));
	return true;
}

enum OptionType { StringType, IntType, DoubleType, BooleanType };

struct OptionEntry {
	const char* optionName;
	OptionType optionType;
};

static OptionEntry optionEntries[] = {
	// Global Options
	{ "AlertBeepVolume", IntType },
	{ "DialogIgnoreWindowManager", BooleanType },
	{ "AutoPlaceWindows", BooleanType },
	// Memory Options
	{ "MaximumSingleAllocation", DoubleType },
	{ "MaximumTotalAllocation", DoubleType },
	// File Options
	{ "ReadRawFiles", BooleanType },
	{ "BrowserUseLastPath", BooleanType },
	{ "BrowserShowDotFiles", BooleanType },
	{ "DefaultSoundFileDir", StringType },
	{ "DefaultLPCFileDir", StringType },
	{ "DefaultPitchFileDir", StringType },
	{ "DefaultFFTFileDir", StringType },
	{ "DefaultEnvelopeFileDir", StringType },
	{ "DefaultPvocFileDir", StringType },
	// Data Options
	{ "DefaultSoundSampleRate", IntType },
	{ "DefaultSoundSampleFormat", StringType },
	{ "DefaultSoundHeaderType", StringType },
	{ "DefaultLPCFrameRate", DoubleType },
	{ "DefaultLPCNumberOfPoles", IntType },
	{ "DefaultLPCHeaderType", StringType },
	{ nil }
};

#ifndef __GNUG__
extern "C" char* getenv(const char*);
#endif

void read_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "r");
	if(file == nil) return;
	char oname[80];
	char ovalue[1024];
	while(fscanf(file, "%s %s", oname, ovalue) != EOF) {
		String optionName(oname);
		for(OptionEntry* entry = optionEntries;
				entry->optionName != nil; entry++) {
			if(fcompare(optionName, String(entry->optionName)) == 0) {
				Application::setGlobalResource(entry->optionName, ovalue);
			}
		}
	}
	fclose(file);
}

void write_mxvrc() {
	char mxvrc_path[1024];
	const char* homeDir = getenv("HOME");
	if(!homeDir) homeDir = ".";
	sprintf(mxvrc_path, "%s/.mxvrc", homeDir);
	FILE* file = fopen(mxvrc_path, "w+");
	if(file != nil) {
		for(OptionEntry* entry = optionEntries;
				entry->optionName != nil; entry++) {
			const char* resource =
				Application::getGlobalResource(entry->optionName);
			if(resource && strlen(resource))
				fprintf(file, "%s\t\t%s\n", entry->optionName, resource);
		}
		fclose(file);
	}
	else {
		Application::error("Unable to open .mxvrc file for writing.",
			mxvrc_path);
	}
}

