// envelope.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// Envelope functions as a generic data class for editing and storage of
// contiguous binary floating point data.  Envelope values are used as
// multipliers in many operations in which a time-variable value is desired.
// The added functionality beyond FrameData is:  automatic interpolated lookup
// and customizable alternate range displays and labels.
// Envelope is a restricted version of Data: always type float, always one
// channel, for use as an envelope for such things as pitch, amplitude, and
// other one-dimensional arrays used by functions.

#ifndef ENVELOPE_H
#ifdef __GNUG__
#pragma interface
#endif
#define ENVELOPE_H

#include "framedata.h"
#include "range.h"

class Controller;
class Header;

class Envelope : public FrameData {
	typedef FrameData Super;
	enum Size { DefaultEnvelopeSize = 1024 };
public:
	Envelope();
	Envelope(int len);
	virtual ~Envelope();
	redefined Data *newData();
	redefined Data *newData(int len);
	redefined Data *clone(const Range &r);
	redefined Data *clone(const Range &r, const Range &c);
	virtual double next();
	virtual void reset();
	void setMappingLength(int mlength);
	redefined int channels() const { return 1; }
	
	redefined void information(Controller *);
	// information methods for display objects
	redefined const char* windowClassName() const;
	redefined const char* channelDisplayAttribute() const;
	redefined const char* horizontalScaleModeAttribute() const;
	redefined const char* defaultDirAttribute() const;
	void setFrameRangeLabel(const char*);
	void setRangeFactor(double fac, double offset = 0);
	void setChannelName(const char*);
    redefined Range frameRange(RangeUnit units) const;
	redefined const char* frameRangeLabel(RangeUnit units) const;
	redefined const char* channelName(int chan) const;
	redefined FileType fileType() const { return EVP_Data; }
	redefined const char *fileSuffix() const { return ".evp"; }
protected:
	redefined Header* createHeader(DataFile *, boolean isNew);
	redefined void readFromHeader(Header *);
	redefined void checkValues();
	Envelope(const Envelope *e) : FrameData(e) { init(); }
	Envelope(const Envelope *e, int newlen) : FrameData(e, newlen) { init(); }
	Envelope(const Envelope *e, const Range &r) : FrameData(e, r) { init(); }
	Envelope(const Envelope *e, const Range &r, const Range &c)
		: FrameData(e, r, c) { init(); }
private:
	int mappinglength;		// number of values to interp to
	double currentIndex;	// current location in array
	int oldindex;
	double	increment,		// these are all used for interpolated lookup
			fraction,
			currentVal,
			nextVal;
	char* frameLabel;		// custom labels for visual display
	char* channelLabel;
	double	rangeFactor,	// modifiers for custom horiz range
			rangeOffset;	
	void init();
};

inline void
Envelope::setMappingLength(int mlength) {
	mappinglength = mlength;
	increment = double(length()) / mappinglength;
}

inline const char*
Envelope::windowClassName() const { return "EnvelopeWindow"; }

inline const char*
Envelope::channelDisplayAttribute() const {
	return "EnvelopeWindowDisplayChannels";
}

inline const char*
Envelope::horizontalScaleModeAttribute() const {
	return "EnvelopeWindowHorizontalScale";
}

inline const char*
Envelope::defaultDirAttribute() const {
	return "DefaultEnvelopeFileDir";
}

inline void
Envelope::setRangeFactor(double fac, double offset) {
	rangeFactor = fac; rangeOffset = offset;
}

#endif
