// diskfile.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <std.h>
#include <stdarg.h>
#include <sys/file.h>           // needed to determine values of O_RDONLY...
#include <sys/stat.h>
#include "diskfile.h"

#if defined(NeXT)
#include <errno.h>
extern "C" {
extern int  unlink(const char *);
}
#elif defined(sgi)
#ifdef __GNUG__
#include <sys/errno.h>
#else
#include <errno.h>
#endif
#endif

#if defined(linux)
extern "C" void setbuffer(FILE*, char*, int);
#else
#ifndef __bsdi__
#ifndef sgi
extern "C" int setbuffer(FILE*, char*, int);
#endif
#endif
#endif

#ifdef __bsdi__
#define	_IOAPPEND	0
#define _IOREAD		4
#define _IOWRT		8
#define _IORW		0x10
#else
#ifndef _IOAPPEND
#define _IOAPPEND	00	/* not used in this version of stdio -- DAS */
#endif
#ifndef _IOREAD
#define _IOREAD 01
#endif
#ifndef _IOWRT
#define _IOWRT 02
#endif
#ifndef _IORW
#define	_IORW 0400
#endif
#endif

// error handlers

void verbose_File_error_handler(const char* msg)
{
  perror(msg);
  errno = 0;
}

void quiet_File_error_handler(const char*)
{
  errno = 0;
}

void fatal_File_error_handler(const char* msg)
{
  perror(msg);
  exit(1);
}

one_arg_error_handler_t File_error_handler = verbose_File_error_handler;


one_arg_error_handler_t set_File_error_handler(one_arg_error_handler_t f)
{
  one_arg_error_handler_t old = File_error_handler;
  File_error_handler = f;
  return old;
}


/*

 Opening files. 

 open(filename, io_mode, access_mode) is done via system open 
 command since fopen doesn't handle all of the cases possible 
 with sys open. After a successful open, fdopen is called to 
 attach an _iobuf to the file descriptor.

 All this requires a few decoding routines that can translate among our
 enumerated types, system flags, and fopen modes.

*/


enum sys_open_cmd_io_mode  // These should be correct for most systems
{                        
  sio_read      = O_RDONLY,
  sio_write     = O_WRONLY,
  sio_readwrite = O_RDWR,
  sio_append    = O_APPEND
};

enum sys_open_cmd_access_mode
{
  sa_create     = O_CREAT,
  sa_truncate   = O_TRUNC,
  sa_createonly = O_EXCL | O_CREAT
};

  
static int open_cmd_arg(io_mode i, access_mode a) // decode modes
{
  int arg;
  switch(i)
  {
  case io_readonly:   arg = sio_read;                   break;
  case io_writeonly:  arg = sio_write;                  break;
  case io_readwrite:  arg = sio_readwrite;              break;
  case io_appendonly: arg = sio_append | sio_write;     break;
  case io_append:     arg = sio_append | sio_readwrite; break;
  default:            return -1;
  };
  switch(a)
  {
  case a_createonly:  return arg | sa_createonly;
  case a_create:      return arg | sa_create | sa_truncate;
  case a_useonly:     return arg;
  case a_use:         return arg | sa_create;
  default:            return -1;
  }
}

char *
DiskFile::fopen_cmd_arg(io_mode i) {
  switch(i) {
  case io_readonly:  return "r";
  case io_writeonly: return "w";
  case io_readwrite: return "r+";
  case io_appendonly:return "a";
  case io_append:    return "a+";
  default:           return 0;
  }
}

DiskFile::DiskFile() { 
	initialize(); 
}

DiskFile::~DiskFile() {
	delete(nm);
	close();
}

void DiskFile::initialize() 
{ 
  fp = 0; nm = 0; stat = 0; state = _bad; rw = 0;
}

// reset class vars after open

void DiskFile::reinitialize(const char* filename)
{
  if (filename != 0)     setname(filename);
  else setname(0);

  if (fp != 0)
  {
    state = _good;
#if defined(linux)
    rw = 01|02;
    if (fp->_flags & _IO_NO_READS)
      rw &= ~01;
    if (fp->_flags & _IO_NO_WRITES)
      rw &= ~02;
#else
#ifdef __bsdi__
    if (fp->_flags & (_IOREAD|_IORW))
      rw |= 01;
    if (fp->_flags & (_IOWRT|_IORW|_IOAPPEND))
      rw |= 02;
#else
    if (fp->_flag & (_IOREAD|_IORW))
      rw |= 01;
    if (fp->_flag & (_IOWRT|_IORW|_IOAPPEND))
      rw |= 02;
#endif
#endif
    check_state();
  }
  else
  {
    set(_fail); set(_bad);
    error();
  }
}

void DiskFile::check_state() // ensure fp & state agree about eof
{
  if (fp != 0)
  {
    if (feof(fp))
      set(_eof);
    else
      unset(_eof);
    if (ferror(fp))
      set(_bad);
  }
}

DiskFile& DiskFile::open(const char* filename, io_mode mode, access_mode a) {                                   
	close();
	int open_arg = open_cmd_arg(mode, a);
	if (open_arg != -1) {
		int fd = ::open(filename, open_arg, 0666);
		if (fd >= 0)
			fp = fdopen(fd, fopen_cmd_arg(mode));
	}
	reinitialize(filename);
	return *this;
}

DiskFile&
DiskFile::open(const char* filename, const char* m) {
	close();
	fp = fopen(filename, m);
	reinitialize(filename);
	return *this;
}

DiskFile&
DiskFile::reOpen(const char* mode) {
	return open(DiskFile::name(), mode);	// force use of "real" filename
}

DiskFile&
DiskFile::close() {
	if (fp != 0)
		fclose(fp);
	fp = 0;
	rw = 0;
	set(_bad);
	return *this;
}

DiskFile&
DiskFile::remove() {
	close();
	return failif (nm == 0 || unlink(nm) != 0);
}

DiskFile&
DiskFile::rename(const char* newname) {
	if(::rename(name(), newname) >= 0)
		setname(newname);
	else fail();
	return *this;
}

void DiskFile::setname(const char* newname) {
	if (nm != newname) {
		delete [] nm;
		nm = newstr(newname);
	}
}

DiskFile& DiskFile::setbuf(int buffer_kind) {                  
  if (!is_open())
  {
    set(_fail);
    return *this;
  }
  switch(buffer_kind)
  {
  case _IOFBF:       
#ifdef HAVE_SETVBUF
    setvbuf(fp, 0, _IOFBF, 0);
#endif
    break;           
  case _IONBF:       
    ::setbuf(fp, 0); 
    break;
  case _IOLBF:
#ifdef HAVE_SETLINEBUF
    setlinebuf(fp);
#else
#ifdef HAVE_SETVBUF
    setvbuf(fp, 0, _IOLBF, 0);
#endif
#endif    
    break;
  default:
    break;
  }
  return *this;
}

DiskFile& DiskFile::setbuf(int size, char* buf) {
  if (!is_open())
  {
    set(_fail);
    return *this;
  }
#ifdef HAVE_SETVBUF
  setvbuf(fp, buf, _IOFBF, size);
#else
  setbuffer(fp, buf, size);
#endif
  return *this;
}

// static function

int
DiskFile::exists(const char* filename) {
	int doesExist = true;
	static struct stat s;
	if(::stat(filename, &s) < 0) {
		switch(errno) {
		case ENOENT:
			doesExist = false;
			break;
		default:
			doesExist = false;	// assume false for now until better way found
			break;
		}
	}
	return doesExist;
}

void DiskFile::error() {
  check_state();
  set(_fail);
  if (errno != 0) {
    char error_string[400];
    sprintf(error_string, "File error: %s:", (nm != 0) ? nm : "");
    (*File_error_handler)(error_string);
  }
}

