// dataview.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/border.h>
#include <InterViews/box.h>
#include <InterViews/canvas.h>
#include <InterViews/cursor.h>
#include <InterViews/event.h>
#include <InterViews/glue.h>
#include <InterViews/message.h>
#include <InterViews/perspective.h>
#include "application.h"
#include "block.h"
#include "controller.h"
#include "data.h"
#include "dataview.h"
#include "graph.h"
#include "range.h"
#include "scale.h"
#include "scroller.h"
#include "statuspanel.h"
#include "viewchanger.h"
#include <X11/keysym.h>

// if these are not defined, this program cannot be run correctly on R6 displays
// I assume that if one is not defined, then none are

#ifndef XK_KP_Left
#define XK_KP_Left      0xFF96
#define XK_KP_Right     0xFF98
#define XK_KP_Up        0xFF97
#define XK_KP_Down      0xFF99
#endif

ViewInfo::ViewInfo() 
	: rangeUnits(FrameUnit), frameRange(-1, -1), channelRange(-1, -1) {
}

ViewInfo::ViewInfo(RangeUnit units)
	: rangeUnits(units), frameRange(-1, -1), channelRange(-1, -1) {
}

ViewInfo::ViewInfo(RangeUnit units, const Range& frange, const Range& chrange)
	: rangeUnits(units), frameRange(frange), channelRange(chrange) {
}

//********

DataView::DataView(Controller *c, ViewInfo& info)
		: dataptr(c->model()), controller(c), 
		cursorLoc(0), graphsShown(0), maxGraphs(0), perspectiveSet(false) {
	Init(info);
}

void
DataView::Init(ViewInfo &info) {
	perspective = new Perspective;
	hscale = new HScale(
		dataptr->frameRangeLabel(info.rangeUnits),
		dataptr->frameRange(info.rangeUnits),
		dataptr->frameRangeDisplay(info.rangeUnits),
		Scale::IsAbove
	);
	setHorizRangeUnits(info.rangeUnits);
	graphs = nil;
	expandGraphArray();
	dataptr->Attach(this);	// this is view of Data
	interior = new VBox(
			hscale,
			graphBox = new VBox,		// box for scales and graphs
			new HBorder,
			new HBox(
				new ViewScaler(this),
				new VBorder,
				makeEditStatusPanel()
			),
			new HBorder,
			viewScroller = new HorizontalViewScroller(this)
	);
	Insert(interior);
	// initialize internal copy of perspective
	shown = new Perspective;
}

Interactor *
DataView::makeEditStatusPanel() {
	StatusPanel* start = nil;
	StatusPanel* end = nil;
	StatusPanel* duration = nil;
	VBox* panel = new VBox(
		new VGlue(3, 0, vfil),
		new HBox(
			start = new StatusPanel("Edit Start: ",
				"                         "),
			new HGlue(10, 0, hfil),
			end = new StatusPanel("Edit End: ",
				"                         ")
		),
		new VGlue(0, 0),
		new HBox(
			duration = new StatusPanel("Edit Dur: ",
				"                         "),
			new HGlue(20, 0, hfil)
		),
		new VGlue(3, 0, vfil)
	);
	// this delegate is passed to the controller
	editDelegate = new EditStatusDelegate(this, start, end, duration);
	return panel;
}

DataView::~DataView() {
	// all graphs and scales are destroyed by the MonoScene subclass
	delete [] graphs;
	dataptr->Detach(this);	// detach this view from Data
	Resource::unref(shown);
	Resource::unref(perspective);
}

void
DataView::Adjust(Perspective &p) {
	BUG("DataView::Adjust()");
	if(p != *perspective) {
		Application::busy(true);
		*perspective = constrainView(p);
		struct DoAdjuster : public Block {
			Perspective *newView;
			DoAdjuster(Perspective *p) : newView(p) {}
			redefined void doIt( Interactor *i ) {
				i->Adjust(*newView);
			}
		} adjuster(perspective);
		doForGraphs(adjuster);
		p = *perspective;
		perspective->Update();	// updates scales and sliders
		Application::busy(false);
	}
}

// all Events handled by the DataView are passed to each of the Graph objects
// to see if it was generated by it.  If the shift key is down, the event is
// only handed to the Graph that generated it.  If not, the event is passed to
// all the Graphs.  This allows the Graphs to react in sync to events only
// generated in one.

void
DataView::Handle(Event& e) {
	graphEvent(e.shift ? e.target : nil, e);
}

void
DataView::graphEvent(Interactor *target, Event &e) {
	if(e.shift_is_down()) {		// if selecting single channel
		switch(e.eventType) {
		case DownEvent:
			unselectOthers(target);
			cursorLoc = e.x;		
			break;
		case UpEvent:
			if(e.leftmouse || e.middlemouse)
				cursorLoc = e.x;
			break;
		case MotionEvent:
			if(e.leftmouse)		// if tracking insert point
				cursorLoc = e.x;
			break;
		case EnterEvent:		// if tracking insert or region between graphs
			if(e.leftmouse  || e.middlemouse) {
				unselectOthers(target);
				((Graph *) e.target)->setInsertCoord(cursorLoc);
			}
			break;
		default:
			break;
		}
	}
    struct EventSender : public TargetBlock {
		Event &event;
		EventSender(Interactor *target, Event &e) 
			: TargetBlock(target), event(e) {}
		redefined void doThis( Interactor *i ) {
			Graph *g = (Graph *) i;
			g->handleEvent(event);
		}
    } sender(target, e);
    doForGraphs( sender );
}

void
DataView::Reconfig() {
	BUG("DataView::Reconfig()");
	*shown = *perspective;
	hscale->setViewPerspective(perspective);	// ??? OK HERE ???
	viewScroller->Update();
	// set each graph's plot edges to match horiz scale's borders
    struct Margins : public Block {
		int left, right;
		Margins(int l, int r) : left(l), right(r) {}
		redefined void doIt( Interactor *i ) {
			BorderedArea *b = (BorderedArea *) i;
			b->setHorizMargins(left, right);
		}
    } margins(hscale->topMargin(), hscale->bottomMargin());
    doForGraphs( margins );
	MonoScene::Reconfig();
}

void
DataView::Resize() {
	hscale->setSpacers(horizScaleLeftSpacer(), horizScaleRightSpacer());
	MonoScene::Resize();	// this resizes all interior Interactors
}

void
DataView::Update() {
	BUG("DataView::Update()");
	Application::busy(true);
	checkPerspective();
	updateGraphs();
	perspective->Update();
	Application::busy(false);
}

boolean
DataView::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	// map arrow keys from keypad and non-keypad to same methods
	case XK_Left:
	case XK_KP_Left:
		horizontalZoom();
		break;
	case XK_Up:
	case XK_KP_Up:
		verticalUnZoom();
		break;
	case XK_Right:
	case XK_KP_Right:
		horizontalUnZoom();
		break;
	case XK_Down:
	case XK_KP_Down:
		verticalZoom();
		break;
	case XK_f:
		setVisibleFrameRange();
		break;
	case XK_V:
		resetVerticalRange();
		break;
	case XK_equal:
		setChannelRange();
		break;
	case XK_greater:
		shiftChannelViewUp();
		break;
	case XK_less:
		shiftChannelViewDown();
		break;
	case XK_j:
		setInsertPoint();
		break;
	case XK_J:
		setEditRegion();
		break;
	case XK_percent:
	case XK_asciicircum:
		toggleGraphMode();
		break;
	case XK_KP_Add:
		showVerticalScales(true);
		break;
	case XK_KP_Subtract:
		showVerticalScales(false);
		break;
	default:
		interested = false;
		break;
	}
	return interested;
}

boolean
DataView::doForGraphs(Block &block) {
    int i = 0;
	while(1)
		if(block.doAndDone( graphs[i++] ))
			return true;
	return false;
}

void
DataView::updateGraphs() {
    struct Updater : public Block {
		redefined void doIt( Interactor *i ) {
			i->Update();
		}
    } updater;
    doForGraphs( updater );
}

void
DataView::unselect() {
	unselectOthers(nil);	// unselects all channels
}

void
DataView::unselectOthers(Interactor *target) {
    struct Unselector : public DoExceptBlock {
		Unselector(Interactor *notThis) : DoExceptBlock(notThis) {}
		redefined void doThis( Interactor *i ) {
			Graph *g = (Graph *) i;
			g->unSelect();
		}
    } unselector(target);
    doForGraphs( unselector );
}

Perspective &
DataView::constrainView(Perspective& np) {
	BUG("DataView::constrainView()");
	if(np == *shown) return np;
	*shown = np;
	constrainHorizontalView(*shown);
	constrainVerticalView(*shown);
	return *shown;
}

void
DataView::constrainHorizontalView(Perspective& np) {
	const int minwidth = 8;
	register Perspective* p = perspective;
	np.height = p->height;
	np.curx = max(np.curx, 0);						// always
	np.curwidth = max(np.curwidth, minwidth);		// always
	int nx = np.curx;
	int nw = np.curwidth;
	if(nw == p->curwidth && np.width == p->width)
		np.curx = min(nx, max(0, np.width - nw));
	else if(nx + nw >= np.width) {					// horizontal scale change
		np.curwidth = min(nw, np.width);
		np.curx = min(nx, np.width - np.curwidth);
	}
	np.lx = np.curwidth;
	np.sx = np.lx/4;				// scroll incr. is 1/4 screen
}

Range
DataView::getDisplayedHorizRange() {
	return hscale->getRange();
}

boolean
DataView::horizScaleInFrames() {
	return horizRangeUnits() != FrameTimeUnit
		&& horizRangeUnits() != ChannelSpecialUnit;
}

void
DataView::setHorizScaleTo(RangeUnit units) {
	setHorizRangeUnits(units);
	hscale->setNumberDisplay(dataptr->frameRangeDisplay(units));
	hscale->setRange(dataptr->frameRange(units));
	setHorizScaleLabel(dataptr->frameRangeLabel(units));
	hscale->Update();
	getEditDelegate()->update();	// update the edit panel display
}

double
DataView::horizontalScalingFactor() {
	return getHorizScaleRange().max() / dataptr->length();
}

int
DataView::horizScaleLeftSpacer() {
	Canvas *c = topGraph()->vScale()->GetCanvas();
	return c != nil ? c->Width() : topGraph()->vScale()->GetShape()->width;
}

void
DataView::setHorizScaleLabel(const char *name) {
	hscale->setLabel(name);
}

void
DataView::showVerticalScales(boolean show) {
    struct ScaleShow : public Block {
		boolean show;
		ScaleShow(boolean doshow) : show(doshow) {}
		redefined void doIt( Interactor *i ) {
			Graph *g = (Graph *) i;
			g->vScale()->showScale(show);
		}
    } scaleshow(show);
    doForGraphs( scaleshow );
	doChange();
}

void
DataView::doChange() {
	if(canvas != nil) {
		topGraph()->Parent()->Change(nil);
	}
}

void
DataView::addGraph(Graph *graph) {
	if(graphsShown == maxGraphs)
		expandGraphArray();
	graphs[graphsShown] = graph;	// add to array
	Scale *vscale = new VScale(
		graph->verticalScaleLabel(),
		graph->verticalScaleRange()
	);
	vscale->setViewPerspective(getVerticalPerspective());
	graph->setViewPerspective(getHorizontalPerspective());
	graph->attachVScale(vscale);	// establish connection
	graphBox->Insert(new HBox(vscale, graph));
	incrementGraphsVisible();
}

// graph array is expanded by 4 plus an additional zeroed end-of-array member

void
DataView::expandGraphArray() {
	int newMax = maxGraphs + 4;
	Graph **newGraphs = new Graph *[newMax + 1];	// plus 1 for EOA
	int i;
	for(i=0; i < maxGraphs; i++)
		newGraphs[i] = graphs[i];	// copy existing graphs if any
	for(i = maxGraphs; i <= newMax; i++)
		newGraphs[i] = nil;
	delete [] graphs;
	graphs = newGraphs;
	maxGraphs = newMax;
}

// remove one graph and its associated scale from frame 
// the channel removed is always the highest channel displayed

void
DataView::removeGraph(boolean chng) {
	int topGraph = graphsVisible() - 1;
	if(topGraph > 0) {				// dont remove last graph!
		Graph *graph = graphs[topGraph];
		Scale *scale = graph->vScale();
		graph->removeVScale(scale);
		// graph and scale are enclosed in Scene, so remove this too
		Scene* interior = graph->Parent();
		if(interior != nil) {
			graphBox->Remove(interior);
			interior->Remove(scale);
			interior->Remove(graph);
			delete interior;
		}
		delete scale;
		delete graph;
		graphs[topGraph] = nil;
		decrementGraphsVisible();
		if(chng)
			doChange();
	}
}

void
DataView::horizontalZoom() {
	Perspective np = *perspective;
	int width = np.curwidth;
	np.curx += width/4;
	np.curwidth /= 2;
	Adjust(np);
}

void
DataView::horizontalUnZoom() {
	Perspective np = *perspective;
	int width = np.curwidth;
	np.curx -= width/2;
	np.curwidth *= 2;
	Adjust(np);
}

void
DataView::verticalZoom() {
	Perspective np = *perspective;
	np.cury = 0;
	np.curheight = int(np.curheight * 0.5);
	Adjust(np);
}

void
DataView::verticalUnZoom() {
	Perspective np = *perspective;
	np.cury = 0;
	np.curheight *= 2;
	Adjust(np);
}

void
DataView::resetVerticalRange() {
	Perspective np = *perspective;
	np.cury = 0;
	np.curheight = perspective->height;
	Adjust(np);
}

void
DataView::setVisibleFrameRange() {
	VisibleFrameRange changer(this);
	if(changer.configure(controller))
		changer.apply();
}

void
DataView::setInsertPoint() {
	InsertPoint changer(this);
	if(changer.configure(controller))
		changer.apply();
}

void
DataView::setEditRegion() {
	EditRegion changer(this);
	if(changer.configure(controller))
		changer.apply();
}

//********

EditStatusDelegate::EditStatusDelegate(DataView *view, StatusPanel* start, 
		StatusPanel* end, StatusPanel* dur)
		: myStart(start), myEnd(end), myDur(dur), myView(view) {
	myStart->ref();
	myEnd->ref();
	myDur->ref();
}

EditStatusDelegate::~EditStatusDelegate() {
	Resource::unref(myDur);
	Resource::unref(myEnd);
	Resource::unref(myStart);
}

// this converts range in frames into range in time when necessary

void
EditStatusDelegate::update(const Range& current) {
	editRange = current;
	if(!myView->horizScaleInFrames()) {	// convert to time range
		Range values(editRange.min(), editRange.max() + 1);
		values *= myView->horizontalScalingFactor();
		myStart->updateValue(values.min());
		myEnd->updateValue(values.max());
		myDur->updateValue(values.spread());
	}
	else {
		myStart->updateValue(current.intMin());
		myEnd->updateValue(current.intMax());
		myDur->updateValue(current.size());
	}
}

