/************************************************************************
 * This program is Copyright (C) 1986-1994 by Jonathan Payne.  JOVE is  *
 * provided to you without charge, and with no warranty.  You may give  *
 * away copies of JOVE, including sources, provided that this notice is *
 * included in all the files.                                           *
 ************************************************************************/

/* Contains the main loop initializations, and some system dependent
   type things, e.g. putting terminal in CBREAK mode, etc. */

#include "jove.h"
#include "fp.h"
#include "termcap.h"
#include "jctype.h"
#include "chars.h"
#include "disp.h"
#include "re.h"	/* for dosearch() */
#include "reapp.h"	/* for find_tag(), UseRE */
#include "sysprocs.h"
#include "rec.h"
#include "ask.h"
#include "extend.h"
#include "fmt.h"
#include "msgetch.h"
#include "macros.h"
#include "marks.h"
#include "mouse.h"
#include "paths.h"
#include "proc.h"
#include "screen.h"
#include "term.h"
#include "version.h"
#include "wind.h"
#ifdef IPROCS
# include "iproc.h"
#endif
#ifdef USE_SELECT
#  include <sys/time.h>
#  include "select.h"
#endif

#ifdef SCO	/* ??? what is this for? */
# include <sys/stream.h>
# include <sys/ptem.h>
#endif

#include <signal.h>
#include <errno.h>

#ifdef MAC

# include "mac.h"
# define	WINRESIZE	1

#else /* !MAC */

# include <sys/stat.h>

#endif /* !MAC */

#ifdef MSDOS
# include <bios.h>
# include <dos.h>
# include <time.h>
# include <process.h>
# define SIGHUP	99
private	void
	break_off proto((void)),
	break_rst proto((void));
# if defined(__WATCOMC__)
#  include <malloc.h>	/* for _heapgrow */
# endif
#endif /* MSDOS */

#ifdef STACK_DECL	/* provision for setting up appropriate stack */
STACK_DECL
#endif

private void
	ttysetattr proto((bool n)),
	UnsetTerm proto((bool)),
	DoKeys proto((bool firsttime));

#ifdef NONBLOCKINGREAD
private void	setblock proto((bool on));
#endif

#ifdef UNIX

# include "ttystate.h"

# if defined(TIOCGWINSZ) && defined(SIGWINCH)
#  define	WINRESIZE	1
# endif

# ifdef BIFF
bool	DisBiff = NO;		/* VAR: turn off/on biff with entering/exiting jove */
# endif /* BIFF */

/* set things up to update the modeline every UpdFreq seconds */

int	UpdFreq = 30;	/* VAR: how often to update modeline */
bool	inIOread = NO;

#define SetClockAlarm()	{ \
		if (UpdFreq != 0) { \
			(void) signal(SIGALRM, updmode); \
			(void) alarm((unsigned) (UpdFreq - (time((time_t *)NULL) % UpdFreq))); \
		} \
	}

/*ARGSUSED*/
private SIGRESTYPE
updmode(junk)
int	junk;	/* passed in on signal; of no interest */
{
	int save_errno = errno;	/* Subtle, but necessary! */

	UpdModLine = YES;
	if (inIOread)
		redisplay();
	SetClockAlarm();
	errno = save_errno;
	return SIGRESVALUE;
}

#endif /* UNIX */

bool	errormsg;

char	NullStr[] = "";
jmp_buf	mainjmp;

#ifdef USE_SELECT
fd_set	global_fd;	/* set of file descriptors of interest (for select) */
int	global_maxfd;
#endif

/* paths */

/* path of machine-independent library */
#ifdef SHAREDIR
char	ShareDir[FILESIZE] = SHAREDIR;
#else
char	ShareDir[FILESIZE];
#endif

/* VAR: directory/device to store tmp files */
#ifdef TMPDIR
char	TmpDir[FILESIZE] = TMPDIR;
#else
char	TmpDir[FILESIZE];
#endif

#ifdef SUBSHELL
char
	Shell[FILESIZE] = DFLTSHELL,	/* VAR: shell to use */
# ifdef MSDOS
	ShFlags[sizeof(ShFlags)] = "/c";	/* VAR: flags to shell */
# else
	ShFlags[sizeof(ShFlags)] = "-c";	/* VAR: flags to shell */
# endif /* MSDOS */
#endif

/* LibDir: path of machine-dependent library (for Portsrv and Recover) */

#if defined(SUBSHELL) || defined(PIPEPROCS)
# define NEED_LIBDIR	1
private char	LibDir[FILESIZE] = LIBDIR;
#endif

/* finish: handle bad-news signals.
 * It also handles internally generated requests for termination.
 * For most values of code (signal types) an attempt
 * is made to save the buffers for recovery by "recover".
 * For most values of code, this routine stops JOVE
 * by calling abort, which causes a core dump under UNIX.
 *
 * - code -1 is an internally generated request to die with an
 *   attempt to save the buffers.  It had better not be the code
 *   for some real signal (it cannot be one under UNIX).
 *
 * - code 0 is an internally generated request to die quietly.
 *   It had better not be the code for some real signal
 *   (it cannot be one under UNIX).  This is the only code
 *   for which buffers are not saved.
 *
 * - SIGINT is caused by the user hitting the INTR key.
 *   We give him a choice of death or continuation.
 *
 * - SIGHUP is caused by loss of connection.  This code and
 *   code 0 are the only ones which don't cause an abort.
 *   Generated by OS and by JOVE itself.
 */

SIGRESTYPE
finish(code)
int	code;
{
	int save_errno = errno;	/* Subtle, but necessary! */
	bool	DelTmps = YES;		/* Usually we delete them. */

	if (code == SIGINT) {
		char	c;
#ifdef PIPEPROCS
		bool	started;
#endif

		(void) signal(code, finish);
		f_mess("Abort (Type 'n' if you're not sure)? ");
#ifdef UNIX
# ifdef PIPEPROCS
		started = kbd_stop();
# endif
		/*
		 * Yuk!  This doesn't deal with all cases, we really need a
		 * standard jove input routine that's lower than kbd_getch so
		 * that this can use it.  The code that this replaces was even
		 * more ugly.  What about nonblocking reads? -- MM.
		 */
# ifdef NONBLOCKINGREAD
		setblock(YES);	/* turn blocking on (in case it was off) */
# endif
		for (;;) {
			c = 'n';
			if (read(0, (UnivPtr) &c, sizeof(c)) < 0) {
				switch (errno) {
				case EINTR:
# if EWOULDBLOCK != EAGAIN	/* aliases in SvR4 */
				case EWOULDBLOCK:
# endif
				case EAGAIN:
					continue;
				}
			}
			break;
		}
# ifdef PIPEPROCS
		if (started)
			kbd_strt();
# endif /* PIPEPROCS */
#endif /* UNIX */
#ifdef MSDOS
		c = getrawinchar();
#endif /* MSDOS */
		message(NullStr);
		if (c != 'y') {
			redisplay();
			errno = save_errno;
			return SIGRESVALUE;
		}
	}
	DisabledRedisplay = YES;
#ifndef MAC
	UnsetTerm(NO);
#endif
#ifdef PIPEPROCS
	kbd_kill();		/* kill the keyboard process */
#endif
#ifdef RECOVER
	if (code != 0) {
		static bool	Crashing = NO;	/* we are in the middle of crashing */

		if (!Crashing) {
			Crashing = YES;
			lsave();
			SyncRec();
			writef("JOVE CRASH!! (code %d; last errno %d)\n",
				code, save_errno);
			if (ModBufs(YES)) {
				writef("Your buffers have been saved.\n");
				writef("Use \"jove -r\" to have a look at them.\n");
				DelTmps = NO;	/* Don't delete anymore. */
			} else {
				writef("No buffers needed saving: you didn't lose any work.\n");
			}
		} else {
			writef("\r\nYou may have lost your work!\n");
		}
	}
#endif /* RECOVER */
	flushscreen();
	if (DelTmps) {
		tmpremove();
#ifdef RECOVER
		recremove();
#endif /* RECOVER */
	}
#ifdef UNIX
	if (code != 0 && code != SIGHUP)
		abort();
# ifdef USE_EXIT
	exit(0);
# else
	_exit(0);
# endif
#else /* !UNIX */
# ifdef MSDOS
	break_rst();	/* restore previous ctrl-c handling */
# endif
	exit(0);
#endif /* !UNIX */
	/*NOTREACHED*/
}

private char	smbuf[20],
		*bp = smbuf;
private int	nchars = 0;

#ifdef NONBLOCKINGREAD

# include <fcntl.h>

private void
setblock(on)	/* turn blocking on or off */
bool	on;
{
	static int blockf, nonblockf;
	static bool	first = YES;

	if (first) {
		int flags;

		first = NO;
		if ((flags = fcntl(0, F_GETFL, 0)) == -1)
			finish(SIGHUP);
		blockf = flags & ~O_NDELAY;	/* make sure O_NDELAY is off */
		nonblockf = flags | O_NDELAY;	/* make sure O_NDELAY is on */
	}
	if (fcntl(0, F_SETFL, on ? blockf : nonblockf) == -1)
		finish(SIGHUP);
}

#endif /* NONBLOCKINGREAD */

bool	InputPending = NO;

/* Inputp is used to jam a NUL-terminated string into JOVE's input stream.
 * It is used to feed each line of the joverc file, to fill in the default
 * make_cmd in compile-it, and to fill in the default i-search string.
 * ??? Probably this means that i-search or compile-it cannot be done from
 * a joverc -- DHR.
 */
char	*Inputp = NULL;

/* kbd_ungetch must only be used to push back a character that
 * was just returned by kbd_getch.
 */
private ZXchar	kbdpeek = EOF;

void
kbd_ungetch(c)
ZXchar	c;
{
	InputPending = YES;
	kbdpeek = c;
}

ZXchar
kbd_getch()
{
	register ZXchar	c;

	if (kbdpeek != EOF) {
		c = kbdpeek;
		kbdpeek = EOF;
		InputPending = nchars > 0;
		return c;
	}
#if NCHARS != UCHAR_ROOF
	do {
#endif
		while (nchars <= 0) {
			bp = smbuf;
#ifdef MSDOS
			*bp = getrawinchar();
			nchars = 1;
#else /* !MSDOS */
# ifdef PTYPROCS
			/* Get a character from the keyboard, first checking for
			   any input from a process.  Handle that first, and then
			   deal with the terminal input. */
			{
				fd_set	reads;
				int
					nfds,
					fd;

				bp = smbuf;
				for (;;) {
					while (procs_to_reap)
							reap_procs();	/* synchronous process reaping */
					reads = global_fd;
					nfds = select(global_maxfd,
						&reads, (fd_set *)NULL, (fd_set *)NULL,
						(struct timeval *)NULL);
					if (nfds >= 0)
						break;

					if (errno != EINTR) {
						complain("\rerror in select: %s", strerror(errno));
						/* NOTREACHED */
					}
				}

				if (FD_ISSET(0, &reads)) {
					do {
						nchars = read(0, (UnivPtr) smbuf, sizeof(smbuf));
					} while (nchars < 0 && errno == EINTR);
					if (nchars <= 0)
						finish(SIGHUP);
					nfds -= 1;
				}
				for (fd=1; nfds != 0; fd += 1) {
					if (FD_ISSET(fd, &reads)) {
						nfds -= 1;
						read_pty_proc(fd);
						if (UpdModLine)
							redisplay();
					}
				}
			}
# else /* !PTYPROCS */
#  ifdef PIPEPROCS
			if (NumProcs > 0) {
				/* Handle process input until kbd input arrives */
				struct header	header;
				size_t	n = f_readn(ProcInput, (char *) &header, sizeof(header));

				if (n != sizeof(header)) {
					raw_complain("\r\nError reading kbd process, expected %d, got %d bytes", sizeof header, n);
					finish(SIGHUP);
				}
				if (header.pid == kbd_pid) {
					/* data is from the keyboard process */
					nchars = f_readn(ProcInput, smbuf, (size_t)header.nbytes);
					if (nchars != header.nbytes) {
						raw_complain("\r\nError reading kbd process, expected %d, got %d bytes.", header.nbytes, nchars);
						finish(SIGHUP);
					}
				} else {
					/* data is from an interactive process */
					read_pipe_proc(header.pid, header.nbytes);
					if (NumProcs == 0)
						(void) kbd_stop();
					if (UpdModLine)
						redisplay();
				}
			} else /*...*/
#  endif /* PIPEPROCS */
			/*...*/ {
				do {
					nchars = read(0, (UnivPtr) smbuf, sizeof smbuf);
				} while (nchars < 0 && errno == EINTR);
				if (nchars <= 0)
					finish(SIGHUP);
			}
# endif /* !PTYPROCS */
#endif /* !MSDOS */
		}
		c = ZXRC(*bp++);
		if ((c & 0200) && MetaKey) {
			*--bp = c & ~0200;
			nchars += 1;
			c = ESC;
		}
		InputPending = --nchars > 0;
#if NCHARS != UCHAR_ROOF
	} while (c >= NCHARS);	/* discard c if it is a bad char */
#endif
	return c;
}

/* Returns non-zero if a character waiting */

bool
charp()
{
	if (InJoverc != 0 || kbdpeek != EOF || nchars > 0 || Inputp != NULL)
		return YES;
#ifdef FIONREAD
	{
		/*
		 * Some manual pages, notably SunOS4.1.3 say 'c' should be
		 * 'long', but that's a lie -- it's an 'int' according to all
		 * kernels I've seen (including SunOS4.1.3) and most other
		 * manual pages.  At any rate, 'int' works correctly on 32- and
		 * 64-bit architectures, whereas long breaks on the 64
		 * bitters. -- MM.
		 */
		int c;

		if (ioctl(0, FIONREAD, (UnivPtr) &c) == -1)
			c = 0;
		return c > 0;
	}
#else /* !FIONREAD */
# ifdef NONBLOCKINGREAD
	setblock(NO);		/* turn blocking off */
	nchars = read(0, (UnivPtr) smbuf, sizeof smbuf);	/* Is anything there? */
	setblock(YES);		/* turn blocking on */
	bp = smbuf;			/* make sure bp points to it */
	return nchars > 0;	/* just say we found something */
# else /* !NONBLOCKINGREAD */
#  ifdef MSDOS
	return rawkey_ready();
#  else /* !MSDOS */
#   ifdef MAC
	return rawchkc();
#   else
	return NO;	/* who knows? */
#   endif
#  endif /* !MSDOS */
# endif /* !NONBLOCKINGREAD */
#endif /* !FIONREAD */
}

/*
 * Tries to pause for delay/10 seconds OR until a character is typed at the
 * keyboard.  This works well on systems with select() and not so well on the
 * rest.
 */

void
SitFor(delay)
int	delay;
{
#ifdef MAC
	long
		start,
		end;

#define Ticks ((long *) 0x16A)	/* 1/60 sec */
	Keyonly = YES;
	redisplay();
	start = *Ticks;

	end = start + delay * 6;
	do ; while ((InputPending = charp()) == NO && *Ticks < end);
#undef	Ticks

#else /* !MAC */

# ifndef MSDOS
	if (!charp()) {
#  ifdef USE_SELECT
		struct timeval	timer;
		fd_set	readfds;

		/* So messages that aren't error messages don't
		 * hang around forever.
		 * Gross that I had to snarf this from getch()
		 */
		if (!UpdMesg && !Asking && mesgbuf[0] && !errormsg)
			message(NullStr);
		redisplay();

		timer.tv_sec = (delay / 10);
		timer.tv_usec = (delay % 10) * 100000;
		FD_ZERO(&readfds);
		FD_SET(0, &readfds);
		(void) select(1,
			&readfds, (fd_set *)NULL, (fd_set *)NULL,
			&timer);
#  else /* ! USE_SELECT */
		/* Pause by spitting NULs at the terminal.  Ugh! */
		static const int cps[] = {
			0,
			5,
			7,
			11,
			13,
			15,
			20,
			30,
			60,
			120,
			180,
			240,
			480,
			960,
			1920,
			1920,
		};
		register int	nchars,
				check_cnt;

		nchars = (delay * cps[ospeed]) / 10;
		check_cnt = ScrBufSize;
		redisplay();
		if (!NP) {
			while ((--nchars > 0) && !InputPending) {
				scr_putchar(PC);
				if (--check_cnt == 0) {
					check_cnt = ScrBufSize;
					InputPending = charp();
				}
			}
		}
#  endif /* !USE_SELECT */
	}
# else /* MSDOS */

	long	start,
		end;
#  ifndef IBMPC
	struct dostime_t tc;
#  endif

	redisplay();
#  ifdef IBMPC
	_bios_timeofday(_TIME_GETCLOCK, &start);
#  else
	_dos_gettime(&tc);
	start = (long)(tc.hour*60L*60L*10L)+(long)(tc.minute*60L*10L)+
	    (long)(tc.second*10)+(long)(tc.hsecond/10);
#  endif
	end = (start + delay);
	do  {
		if ((InputPending = charp()) != NO)
			break;
#  ifdef IBMPC
		if (_bios_timeofday(_TIME_GETCLOCK, &start))
			break;	/* after midnight */
#  else
		start = (long)(tc.hour*60L*60L*10L)+(long)(tc.minute*60L*10L)+
			(long)(tc.second*10)+(long)(tc.hsecond/10);
#  endif
	} while (start < end);
# endif /* MSDOS */
#endif /* !MAC */
}

char
	key_strokes[100],
	*keys_p = key_strokes;

void
pp_key_strokes(buffer, size)
char	*buffer;
size_t	size;
{
	char	*buf_end = buffer + size - 1,
		*kp = key_strokes,
		c;

	*buffer = '\0';
	while ((c = *kp++) != '\0') {
		swritef(buffer, (size_t) (buf_end-buffer), "%p ", c);
		buffer += strlen(buffer);
	}
}

private bool	*slowp = NULL;	/* for waitchar() */

/*ARGSUSED*/
private SIGRESTYPE
slowpoke(junk)
int	junk;
{
	int save_errno = errno;	/* Subtle, but necessary! */
	char	buffer[100];

	if (slowp)
		*slowp = YES;
	pp_key_strokes(buffer, sizeof (buffer));
	f_mess(buffer);
	errno = save_errno;
	return SIGRESVALUE;
}

#define N_SEC	1	/* will be precisely 1 second on 4.2 */

ZXchar
waitchar(slow)
bool	*slow;
{
	ZXchar	c;

	slowp = slow;

	if (InJoverc || (!Interactive && in_macro()))		/* make macros faster ... */
		return getch();

	/* If slow is a valid pointer and it's value is yes, then
	   we know we have already been slow during this sequence,
	   so we just wait for the character and then echo it. */
	if (slow != NULL && *slow) {
		c = getch();
		slowpoke(0);
		return c;
	}
#ifdef UNIX
	(void) signal(SIGALRM, slowpoke);
	(void) alarm((unsigned) N_SEC);
	c = getch();
	alarm((unsigned)0);	/* suppress alarm, in case it hasn't fired */
	SetClockAlarm();

	if (slow != NULL && *slow)
		slowpoke(0);
	return c;

#else /* !UNIX */
# ifdef MAC
	Keyonly = YES;
	if (charp()) {
		c = getch();	/* to avoid flicker */
		if (slow != NULL && *slow)
			slowpoke(0);
		return c;
	}
# endif
	/* NOTE: busy wait! */
	{
		time_t sw = N_SEC + time((time_t *)NULL);

		while (time((time_t *)NULL) <= sw)
			if (charp())
				return getch();
	}
# ifdef MAC
	menus_off();
# endif
	slowpoke(0);
	c = getch();
	slowpoke(0);

	return c;
#endif /* !UNIX */
}

#ifdef MSDOS
# include "pcscr.h"
#endif

private void
SetTerm()
{
#ifdef IBMPC
	pcSetTerm();
#endif
	ttysetattr(YES);
#ifdef TERMCAP
	putpad(TI, 1);	/* Cursor addressing start */
	putpad(VS, 1);	/* Visual start */
	putpad(KS, 1);	/* Keypad mode start */
# ifdef MOUSE
	MouseOn();
# endif
#endif
#ifdef UNIX
	(void) chkmail(YES);	/* force it to check so we can be accurate */
#endif
}

private void
UnsetTerm(WarnUnwritten)
bool	WarnUnwritten;
{
#ifdef TERMCAP
# ifdef ID_CHAR
	INSmode(NO);
# endif /* ID_CHAR */
# ifdef MOUSE
	MouseOff();
# endif
	putpad(KE, 1);
	putpad(VE, 1);
	Placur(ILI, 0);
	putpad(CE, 1);
	if (TE)
		putpad(TE, 1);
#else /* !TERMCAP */
	Placur(ILI, 0);
	clr_eoln();
#endif /* !TERMCAP */
	flushscreen();
#ifdef IBMPC
	pcUnsetTerm();
#endif /* IBMPC */
	ttysetattr(NO);
	if (WarnUnwritten && ModBufs(NO))
		raw_complain("[There are modified buffers]");
}

#ifdef JOB_CONTROL
void
PauseJove()
{
	UnsetTerm(YES);
	(void) kill(0, SIGTSTP);
	SetTerm();
	ClAndRedraw();
}
#endif /* JOB_CONTROL */

#ifdef SUBSHELL

# ifndef MSDOS
void
jcloseall()
{
	tmpclose();
#ifdef RECOVER
	recclose();
#endif
#ifdef IPROCS
	closeiprocs();
#endif
}
# endif /* !MSDOS */

void
Push()
{
	SIGRESTYPE	(*old_int) ptrproto((int)) = signal(SIGINT, SIG_IGN);
# ifdef MSDOS

	UnsetTerm(YES);
	(void) signal(SIGINT, SIG_DFL);
	break_rst();
	if (spawnl(0, Shell, basename(Shell), (char *)NULL) == -1)
		s_mess("[Spawn failed %d]", errno);
	SetTerm();
	ClAndRedraw();
	break_off();
	getCWD();
	(void) signal(SIGINT, old_int);
# else /* !MSDOS */
	int	forkerr = 0;
#  ifdef PIPEPROCS
	bool	started = kbd_stop();
#  endif

#  ifdef WINRESIZE
	SigHold(SIGWINCH);
#  endif
	alarm((unsigned)0);
	UnsetTerm(YES);
	switch (ChildPid = fork()) {
	case -1:
		/* parent, fork failed */
		forkerr = errno;
		break;

	default:
		/* parent, fork worked */
		dowait((wait_status_t *) NULL);
		break;

	case 0:
		/* child */
#  ifdef WINRESIZE
		(void) signal(SIGWINCH, SIG_DFL);
		SigRelse(SIGWINCH);
#  endif
		/* (void) signal(SIGTERM, SIG_DFL); */
		(void) signal(SIGINT, SIG_DFL);
		jcloseall();
		/* note that curbuf->bfname may be NULL */
		execl(Shell, basename(Shell), "-is", pr_name(curbuf->b_fname, NO),
			(char *)NULL);
		raw_complain("[Execl failed: %s]", strerror(errno));
		_exit(1);
		/*NOTREACHED*/
	}
	SetTerm();
#  ifdef WINRESIZE
	SigRelse(SIGWINCH);
#  endif
	ClAndRedraw();
	(void) signal(SIGINT, old_int);
	SetClockAlarm();
#  ifdef PIPEPROCS
	if (started)
		kbd_strt();
#  endif
	if (forkerr != 0)
		complain("[Fork failed: %s]", strerror(errno));
# endif /* !MSDOS */
}

#endif /* SUBSHELL */

#ifdef UNIX
bool	OKXonXoff = NO;	/* VAR: XON/XOFF can be used as ordinary chars */
ZXchar	IntChar = CTL(']');	/* VAR: ttysetattr sets this to generate SIGINT */
#endif

private void
ttsize()
{
#ifdef UNIX
# ifdef TIOCGWINSZ
	struct winsize win;

	if (ioctl(0, TIOCGWINSZ, (UnivPtr) &win) == 0) {
		if (win.ws_col != 0) {
			CO = win.ws_col;
			if (CO > MAXCOLS)
				CO = MAXCOLS;
		}
		if (win.ws_row != 0)
			LI = win.ws_row;
	}
# else /* !TIOCGWINSZ */
#  ifdef BTL_BLIT
	struct jwinsize jwin;

	if (ioctl(0, JWINSIZE, (UnivPtr) &jwin) == 0) {
		if (jwin.bytesx != 0) {
			CO = jwin.bytesx;
			if (CO > MAXCOLS)
				CO = MAXCOLS;
		}
		if (jwin.bytesy != 0)
			LI = jwin.bytesy;
	}
#  endif /* BTL_BLIT */
# endif /* !TIOCGWINSZ */
#endif /* UNIX */
#ifdef MAC
	CO = getCO();	/* see mac.c */
	if (CO > MAXCOLS)
		CO = MAXCOLS;
	LI = getLI();
	Windchange = YES;
	clr_page();
#endif
	ILI = LI - 1;
}

/* adjust the tty to reflect possible change to JOVE variables */
void
tty_adjust()
{
	ttysetattr(YES);
#ifdef MOUSE
	MouseOn();	/* XtermMouse might have changed */
#endif
}

/* Set tty to original (if !n) or JOVE (if n) modes.
 * This is designed to be idempotent: it can be called
 * several times with the same argument without damage.
 */

private void
ttysetattr(n)
bool	n;	/* also used as subscript! */
{
	static bool	prev_n = NO;

	if (!prev_n) {
#ifdef UNIX
		/* Previously, the tty was not in JOVE mode.
		 * Find out the current settings:
		 * do the ioctls or whatever to fill in NO half
		 * of each appropriate tty state pair.
		 * NOTE: the nested tangle of ifdefs is intended to follow
		 * the structure of the definitions in ttystate.c.
		 */
# ifdef SGTTY
		(void) gtty(0, &sg[NO]);
# endif

# ifdef TERMIO
		(void) ioctl(0, TCGETA, (UnivPtr) &sg[NO]);
# endif

# ifdef TERMIOS
		(void) tcgetattr(0, &sg[NO]);
# endif

# ifdef USE_TIOCSLTC
		(void) ioctl(0, TIOCGLTC, (UnivPtr) &ls[NO]);
# endif /* USE_TIOCSLTC */

# ifdef SGTTY

#  ifdef TIOCGETC
		(void) ioctl(0, TIOCGETC, (UnivPtr) &tc[NO]);
#  endif

#  ifdef LPASS8	/* use 4.3BSD's LPASS8 instead of raw for meta-key */
		(void) ioctl(0, TIOCLGET, (UnivPtr) &lmword[NO]);
#  endif

# endif /* SGTTY */

/* extract some info from results */

#  if defined(TERMIO) || defined(TERMIOS)
#   ifdef TAB3
		TABS = (sg[NO].c_oflag & TABDLY) != TAB3;
#   endif
#   ifdef TERMIOS
		ospeed = cfgetospeed(&sg[NO]);
#   else /* ! TERMIOS */
#    ifdef CBAUD
		ospeed = sg[NO].c_cflag & CBAUD;
#    else /* ! CBAUD */
		ospeed = B9600;	/* XXX */
#    endif /* CBAUD */
#   endif /* TERMIOS */
#  endif /* defined(TERMIO) || defined(TERMIOS) */

#  ifdef SGTTY
		TABS = !(sg[NO].sg_flags & XTABS);
		ospeed = sg[NO].sg_ospeed;
#  endif /* SGTTY */

#endif /* UNIX */

#ifdef MSDOS
		TABS = NO;
#endif /* MSDOS */
	}

#ifdef UNIX

	/* Fill in YES half of each appropriate tty state pair.
	 * They are filled in as late as possible so that each will
	 * reflect the latest settings of controling variables.
	 * NOTE: the nested tangle of ifdefs is intended to follow
	 * the structure of the definitions in ttystate.c.
	 */

	sg[YES] = sg[NO];

# ifdef SGTTY
	sg[YES].sg_flags &= ~(XTABS|ECHO|CRMOD);
#  ifdef LPASS8
	sg[YES].sg_flags |= CBREAK;
#  else
	sg[YES].sg_flags |= (MetaKey ? RAW : CBREAK);
#  endif
# endif

# if defined(TERMIO) || defined(TERMIOS)
	if (OKXonXoff)
		sg[YES].c_iflag &= ~(IXON | IXOFF);
	sg[YES].c_iflag &= ~(INLCR|ICRNL|IGNCR | (MetaKey? ISTRIP : 0));
	sg[YES].c_lflag &= ~(ICANON|ECHO);
	sg[YES].c_oflag &= ~(OPOST);
	sg[YES].c_cc[VINTR] = IntChar;

#  ifndef VDISABLE
#   ifdef _POSIX_VDISABLE
#    define VDISABLE	_POSIX_VDISABLE
#   else /* !_POSIX_VDISABLE */
#    define VDISABLE	0
#   endif /* !_POSIX_VDISABLE */
#  endif /* VDISABLE */

#  ifdef VQUIT
	sg[YES].c_cc[VQUIT] = VDISABLE;
#  endif
	/* VERASE, VKILL, VEOL2 irrelevant */
	/* Beware aliasing! VMIN is VEOF and VTIME is VEOL */
#  ifdef VSWTCH
	sg[YES].c_cc[VSWTCH] = VDISABLE;
#  endif

	/* Under at least one system (SunOS 4.0), <termio.h>
	 * mistakenly defines the extra V symbols of <termios.h>
	 * without extending the c_cc array in struct termio
	 * to hold them!  This is why the following goo is doubly
	 * ifdefed.  It turns out that we don't use <termio.h>
	 * on SunOS 4.0, so the problem may be moot.
	 */

#  ifdef TERMIOS
#   ifdef VSUSP
	sg[YES].c_cc[VSUSP] = VDISABLE;
#   endif
#   ifdef VDSUSP
	sg[YES].c_cc[VDSUSP] = VDISABLE;
#   endif
#   ifdef VDISCARD
	/* ??? Under Solaris 2.1 needs VDISCARD disabled, or it will
	 * be processed by the tty driver, but not under SVR4!
	 */
	sg[YES].c_cc[VDISCARD] = VDISABLE;	/* flush output */
#   endif
#   ifdef VLNEXT
	sg[YES].c_cc[VLNEXT] = VDISABLE;	/* literal next char */
#   endif
#  endif /* TERMIOS */

	sg[YES].c_cc[VMIN] = 1;
	sg[YES].c_cc[VTIME] = 1;
# endif /* defined(TERMIO) || defined(TERMIOS) */

# ifdef USE_TIOCSLTC
	ls[YES] = ls[NO];
	ls[YES].t_suspc = (char) -1;
	ls[YES].t_dsuspc = (char) -1;
	ls[YES].t_flushc = (char) -1;
	ls[YES].t_lnextc = (char) -1;
# endif /* USE_TIOCSLTC */

# ifdef SGTTY

#  ifdef TIOCGETC
	tc[YES] = tc[NO];
	tc[YES].t_intrc = IntChar;
	tc[YES].t_quitc = (char) -1;
	if (OKXonXoff) {
		tc[YES].t_stopc = (char) -1;
		tc[YES].t_startc = (char) -1;
	}
#  endif

#  ifdef LPASS8	/* use 4.3BSD's LPASS8 instead of raw for meta-key */
	lmword[YES] = lmword[NO];

	if (MetaKey)
		lmword[YES] |= LPASS8;

#   ifdef LLITOUT
	/* ??? under what conditions should we turn on LLITOUT flag? */
#   endif /* LLITOUT */

#   ifdef LTILDE
	if (Hazeltine)
		lmword[YES] &= ~LTILDE;
#   endif /* LTILDE */

#  endif /* LPASS8 */

# endif /* SGTTY */

	/* Set tty state according to appropriate entry of each state pair.
	 * NOTE: the nested tangle of ifdefs is intended to follow
	 * the structure of the definitions in ttystate.c.
	 */

# ifdef SGTTY
#   ifdef TIOCSETN
	(void) ioctl(0, TIOCSETN, (UnivPtr) &sg[n]);
#   else
	(void) stty(0, &sg[n]);
#   endif
# endif

# ifdef TERMIO
	do ; while (ioctl(0, TCSETAW, (UnivPtr) &sg[n]) < 0 && errno == EINTR);
# endif

# ifdef TERMIOS
	do ; while (tcsetattr(0, TCSADRAIN, &sg[n]) < 0 && errno == EINTR);
# endif

# ifdef USE_TIOCSLTC
	(void) ioctl(0, TIOCSLTC, (UnivPtr) &ls[n]);
# endif /* USE_TIOCSLTC */

# ifdef SGTTY

#  ifdef TIOCGETC
	(void) ioctl(0, TIOCSETC, (UnivPtr) &tc[n]);
#  endif

#  ifdef LPASS8	/* use 4.3BSD's LPASS8 instead of raw for meta-key */
	(void) ioctl(0, TIOCLSET, (UnivPtr) &lmword[n]);	/* local mode word */
#  endif

# endif /* SGTTY */

#endif /* UNIX */

#ifdef MSDOS
# ifndef IBMPC
	/* ??? this seems peculiar: surely the mode won't carry over to child or parent. */
	setmode(1, n? 0x8000 : 0x4000);	/* ??? set stdout to O_BINARY or O_TEXT mode */
# endif
#endif /* MSDOS */

#ifdef BIFF

	/* biff state is an honorary part of the tty state.
	 * On the other hand, it is different from the rest of the state
	 * since we only want to examine the setting if DisBiff
	 * has been set by the user.  For this reason, the code is
	 * somewhat more intricate.
	 */
	{
#		define BS_UNEXAMINED	0	/* we don't know if biff is enabled */
#		define BS_DISABLED	1	/* we have disabled biff */
#		define BS_UNCHANGED	2	/* we didn't disable biff */
		static int	biff_state = BS_UNEXAMINED;

		static struct stat	tt_stat;
# if !defined(USE_FSTAT) || !defined(USE_FCHMOD)
		static char	*tt_name = NULL;	/* name of the control tty */
		extern char	*ttyname proto((int));	/* for systems w/o fstat */
# endif

		if (n && DisBiff) {
			/* biff supression is our business */
			if (biff_state == BS_UNEXAMINED) {
				/* and we haven't looked after it */
				biff_state = BS_UNCHANGED;	/* at least so far */
				if (
# ifdef USE_FSTAT
					fstat(0, &tt_stat) != -1
# else
					((tt_name != NULL) || (tt_name = ttyname(0)) != NULL)
					&& stat(tt_name, &tt_stat) != -1
# endif
				&& (tt_stat.st_mode & S_IEXEC))
				{
					/* so let's suppress it */
# ifdef USE_FCHMOD
					(void) fchmod(0, tt_stat.st_mode & ~S_IEXEC);
					biff_state = BS_DISABLED;
# else
					if ((tt_name != NULL || (tt_name = ttyname(0)) != NULL)
					&& chmod(tt_name, tt_stat.st_mode & ~S_IEXEC) != -1)
					{
						/* Note: only change biff_state if we were able to
						 * get the tt_name -- this prevents the other
						 * chmod from blowing up.
						 */
						biff_state = BS_DISABLED;
					}
# endif
				}
			}
		} else {
			/* any biff suppression should be undone */
			if (biff_state == BS_DISABLED) {
				/* and we did suppress it, so we enable it */
# ifdef USE_FCHMOD
				(void) fchmod(0, tt_stat.st_mode);
# else
				(void) chmod(tt_name, tt_stat.st_mode);
# endif
			}
			biff_state = BS_UNEXAMINED;	/* it's out of our hands */
		}
#		undef BS_UNEXAMINED
#		undef BS_DISABLED
#		undef BS_UNCHANGED
	}

#endif /* BIFF */
	prev_n = n;
}

bool	Interactive = NO;	/* True when we invoke with the command handler? */

ZXchar
	peekchar = EOF,	/* holds pushed-back getch output */
	LastKeyStruck;	/* used by SelfInsert and friends */

bool
	MetaKey = NO;		/* VAR: this terminal has a meta key */

void
Ungetc(c)
ZXchar	c;
{
	peekchar = c;
}

ZXchar
getch()
{
	register ZXchar	c;

	if (Inputp) {
		if ((c = ZXC(*Inputp++)) != '\0')
			return LastKeyStruck = c;
		Inputp = NULL;
	}

	if (InJoverc) {
		/* somethings wrong if Inputp runs out while
		   we're reading a .joverc file. */
		complain("[command line too short]");
	}

#ifdef RECOVER
	if (ModCount >= SyncFreq) {
		ModCount = 0;
		SyncRec();
	}
#endif /* RECOVER */

	if ((c = peekchar) != EOF) {
		/* got input from pushback */
		peekchar = EOF;
	} else {
		if (!Interactive && (c = mac_getc()) != EOF) {
			/* got input from macro */
		} else {
			/* So messages that aren't error messages don't
			 * hang around forever.
			 * Note: this code is duplicated in SitFor()!
			 */
			if (!UpdMesg && !Asking && mesgbuf[0] != '\0' && !errormsg)
				message(NullStr);
			redisplay();
#ifdef UNIX
			inIOread = YES;
#endif
			c = kbd_getch();
#ifdef UNIX
			inIOread = NO;
#endif
			if (!Interactive && InMacDefine)
				mac_putc(c);
		}
		add_stroke(c);
	}
	return LastKeyStruck = c;
}

#if defined(SUBSHELL) && defined(RECOVER)
private void
dorecover()
{
	char	Recover[FILESIZE];	/* path to recover program (in LibDir) */

	/* Since recover is a normal cooked mode program, reset the terminal */
	UnsetTerm(NO);
# ifdef PIPEPROCS
	kbd_kill();		/* kill the keyboard process */
# endif
	swritef(Recover, sizeof(Recover), "%s/recover", LibDir);
	execl(Recover, "recover", "-d", TmpDir, (char *) NULL);
	writef("%s: execl failed! %s\n", Recover, strerror(errno));
	flushscreen();
	_exit(-1);
	/* NOTREACHED */
}
#endif /* defined(SUBSHELL) && defined(RECOVER) */

void
ShowVersion()
{
	s_mess("Jonathan's Own Version of Emacs (%s)", version);
}

private void
UNIX_cmdline(argc, argv)
int	argc;
char	*argv[];
{
	int	lineno = 0,
		nwinds = 1;
	char	*pattern = NULL;
	Buffer	*b;

	while (argc > 1) {
		switch (argv[1][0]) {
		case '+':
			if ('0' <= argv[1][1] && argv[1][1] <= '9') {
				(void) chr_to_int(&argv[1][1], 10, NO, &lineno);
				break;
			} else switch (argv[1][1]) {
			case '\0':
				/* goto end of file just like some people's favourite editor */
				lineno = -1;
				break;
 			case '/':	/* search for pattern */
 				/* check if syntax is +/pattern or +/ pattern */
 				if (argv[1][2] != 0) {
 					pattern = &argv[1][2];
 				} else {
 					argv += 1;
 					argc -= 1;
 					if (argv[1] != 0)
 						pattern = &argv[1][0];
 				}
 				break;
 			default:
 				error("Invalid switch %s",argv[1]);
 				break;
			}
			break;
		case '-':
			switch (argv[1][1]) {
			case 'd':	/* Ignore current directory path */
			case 'l':	/* Ignore libdir path */
			case 's':	/* Ignore sharedir path */
				argv += 1;
				argc -= 1;
				break;

			case 'J':	/* Ignore jove.rc in SHAREDIR */
			case 'j':	/* Ignore ~/.joverc */
				break;
			case 'p':	/* parse errors in file */
				argv += 1;
				argc -= 1;
				if (argv[1] != NULL) {
					SetBuf(do_find(curwind, argv[1], YES, YES));
					ErrParse();
					nwinds = 0;
				}
				break;
			case 't':	/* find tag */
				/* check if syntax is -tTag or -t Tag */
				if (argv[1][2] != '\0') {
					find_tag(&(argv[1][2]), YES);
				} else {
					argv += 1;
					argc -= 1;
					if (argv[1] != NULL)
						find_tag(argv[1], YES);
				}
				break;

			case 'w':	/* multiple windows */
				if (argv[1][2] == '\0')
					nwinds += 1;
				else {
					int	n;

					(void) chr_to_int(&argv[1][2], 10, NO, &n);
					nwinds += -1 + n;
				}
				(void) div_wind(curwind, nwinds - 1);
				break;
 			default:
 				error("Invalid switch %s",argv[1]);
 				break;
			}
			break;
		default:
			{
			bool	force = (nwinds > 0 || lineno != 0 || pattern != NULL);

#ifdef MSDOS
			strlwr(argv[1]);
#endif
			minib_add(argv[1], force);
			b = do_find(nwinds > 0 ? curwind : (Window *) NULL,
				    argv[1], force, YES);
			if (force) {
				SetABuf(curbuf);
				SetBuf(b);
				if (lineno > 0)
					SetLine(next_line(curbuf->b_first, lineno));
				else if (lineno == -1)
					SetLine(curbuf->b_last);
				if (pattern != NULL) {
					Bufpos	*bufp;

					if ((bufp = dosearch(pattern, FORWARD, UseRE)) != NULL
					|| (bufp = dosearch(pattern, BACKWARD, UseRE)) != NULL)
						SetDot(bufp);
					else
						complain("Couldn't match pattern in file.");
				}
				if (nwinds > 1)
					NextWindow();
				if (nwinds > 0)
					nwinds -= 1;
			}
			lineno = 0;
			pattern = NULL;
			}
			break;
		}
		argv += 1;
		argc -= 1;
	}
}

#ifdef STDARGS
void
error(const char *fmt, ...)
#else
/*VARARGS1*/ void
error(fmt, va_alist)
	const char	*fmt;
	va_dcl
#endif
{
	va_list	ap;

	if (fmt) {
		va_init(ap, fmt);
		format(mesgbuf, sizeof mesgbuf, fmt, ap);
		va_end(ap);
		UpdMesg = YES;
	}
	rbell();
	longjmp(mainjmp, ERROR);
}

#ifdef STDARGS
void
complain(const char *fmt, ...)
#else
/*VARARGS1*/ void
complain(fmt, va_alist)
	const char	*fmt;
	va_dcl
#endif
{
	va_list	ap;

	if (fmt) {
		va_init(ap, fmt);
		format(mesgbuf, sizeof mesgbuf, fmt, ap);
		va_end(ap);
		UpdMesg = YES;
	}
	rbell();
	longjmp(mainjmp, COMPLAIN);
}

/* format and display a message without using the normal display mechanisms */

#ifdef STDARGS
void
raw_complain(const char *fmt, ...)
#else
/*VARARGS1*/ void
raw_complain(fmt, va_alist)
	const char	*fmt;
	va_dcl
#endif
{
	char	buf[MESG_SIZE];
	va_list	ap;

	va_init(ap, fmt);
	format(buf, sizeof(buf) - 2, fmt, ap);
	va_end(ap);
	strcat(buf, "\r\n");	/* \r *may* be redundant */
	write(2, (UnivConstPtr)buf, strlen(buf));
}

#ifdef STDARGS
void
confirm(const char *fmt, ...)
#else
/*VARARGS1*/ void
confirm(fmt, va_alist)
	const char	*fmt;
	va_dcl
#endif
{
	va_list	ap;

	va_init(ap, fmt);
	format(mesgbuf, sizeof mesgbuf, fmt, ap);
	va_end(ap);
	if (!yes_or_no_p("%s", mesgbuf))
		longjmp(mainjmp, COMPLAIN);
}

/* Recursive edit.
 * Guarantee: current buffer will still be current and
 * it will be in the current window.  If not, complain!
 */

int	RecDepth = 0;

void
Recur()
{
	Buffer	*b = curbuf;
	Mark	*m;

	m = MakeMark(curline, curchar);

	RecDepth += 1;
	UpdModLine = YES;
	DoKeys(NO);	/* NO means not first time */
	UpdModLine = YES;
	RecDepth -= 1;
	if (!valid_bp(b))
		complain("Buffer gone!");
	tiewind(curwind, b);
	SetBuf(b);
	if (!is_an_arg())
		ToMark(m);
	DelMark(m);
}

private int	iniargc;	/* main sets these for DoKeys() */
private char	**iniargv;

private void
DoKeys(firsttime)
bool	firsttime;
{
	jmp_buf	savejmp;

	push_env(savejmp);

	switch (setjmp(mainjmp)) {
	case 0:
		if (firsttime)
			UNIX_cmdline(iniargc, iniargv);
		break;

	case QUIT:
		if (RecDepth == 0) {
			if (ModMacs()) {
				rbell();
				if (!yes_or_no_p("Some MACROS haven't been saved; leave anyway? "))
					break;
			}
			if (ModBufs(NO)) {
				rbell();
				if (!yes_or_no_p("Some buffers haven't been saved; leave anyway? "))
					break;
			}
#ifdef IPROCS
			if (!KillProcs())
				break;	/* user chickened out */
#endif
		}
		pop_env(savejmp);
		return;

	case ERROR:
		getDOT();	/* God knows what state linebuf was in */
		/*FALLTHROUGH*/
	case COMPLAIN:
		{
		gc_openfiles();		/* close any files we left open */
		errormsg = YES;
		unwind_macro_stack();
		Asking = NO;
		curwind->w_bufp = curbuf;
		DisabledRedisplay = NO;
		redisplay();
		break;
		}
	}

#ifdef SIGWINCH
	/* Note: Under at least some versions of UNIX, the signal mask
	 * is restored by a longjmp.  For this reason, the SigRelse must
	 * be placed before the setjmp or on each path after it.
	 * For some reason that I don't remember, we wish to process
	 * the command line before releasing SIGWINCH, so we must do so
	 * here, on all paths leading out of the setjmp.
	 */
	(void) SigRelse(SIGWINCH);
#endif
	this_cmd = last_cmd = OTHER_CMD;

	for (;;) {
#ifdef MAC
		setjmp(auxjmp);
#endif
		if (this_cmd != ARG_CMD) {
			clr_arg_value();
			last_cmd = this_cmd;
			init_strokes();
		}
#ifdef MAC
		HiliteMenu(0);
		EventCmd = NO;
		menus_on();
#endif
		dispatch(getch());
	}
}

private char **
scanvec(args, str)
register char	**args,
		*str;
{
	while (*args) {
		if (strcmp(*args, str) == 0)
			return args;
		args += 1;
	}
	return NULL;
}

#ifdef WINRESIZE

# ifdef UNIX
private jmp_buf winresizejmp;

/*ARGSUSED*/
private SIGRESTYPE
restart_win_reshape(junk)
int	junk;	/* passed in when invoked by a signal; of no interest */
{
	longjmp(winresizejmp, 1);
	/* NOTREACHED */
}
# endif

/*ARGSUSED*/
SIGRESTYPE
win_reshape(junk)
int	junk;	/* passed in when invoked by a signal; of no interest */
{
	int save_errno = errno;	/* Subtle, but necessary! */
	register int	oldLI;
	register int newsize, total;
	register Window *wp;

# ifdef UNIX
	/*
	 * Programs such as xjove (even shelltool) can generate a large
	 * number of SIGWINCHes on startup.  It is important that we act on,
	 * and do not miss, the last one.  We therefore arrange that each
	 * SIGWINCH interrupts whatever its predecessor was doing, and
	 * starts over.  However, we do not want to get interrupted whilst
	 * adjusting the window sizes.  Neither do we want a redisplay()
	 * caused by an updmode() triggered by a SIGALRM.
	 */

	setjmp(winresizejmp);	/* the start over point */

	signal(SIGWINCH, restart_win_reshape);
	(void) SigHold(SIGWINCH);	/* but don't allow recursive call just yet */
	(void) SigHold(SIGALRM);
	/* Note that a SigHold is ineffective on a signal that was going
	   to be ignored anyway */
# endif
	/*
	 * Save old number of lines.
	 */
	oldLI = LI;

	/*
	 * Get new line/col info.
	 */
	ttsize();

	/*
	 * LI has changed, and now holds the
	 * new value.
	 */
	/*
	 *  Go through the window list, changing each window size in
	 *  proportion to the resize. If a window becomes too small,
	 *  delete it. We keep track of all the excess lines (caused by
	 *  roundoff!), and give them to the current window, as a sop -
	 *  can't be more than one or two lines anyway. This seems fairer
	 *  than just resizing the current window.
	 */
	wp = fwind;
	total = 0;
	do {
		newsize = LI * wp->w_height / oldLI;

		if (newsize < 2) {
			total += wp->w_height;
			wp = wp->w_next;
			del_wind(wp->w_prev);
		} else {
			wp->w_height = newsize;
			total += newsize;
			wp = wp->w_next;
		}
	} while (wp != fwind);

	curwind->w_height += ILI - total;

	/* Make a new screen structure */
	make_scr();

# ifdef UNIX
	(void) SigRelse(SIGWINCH);
# endif
	/* Do a 'hard' update on the screen - clear and redraw */
	cl_scr(YES);
	flushscreen();
	redisplay();

# ifdef UNIX
	(void) signal(SIGWINCH, win_reshape);
	(void) SigRelse(SIGALRM);
# endif
	errno = save_errno;
	longjmp(mainjmp, COMPLAIN);		/* safer to start over after all that messing */
	/* NOT REACHED */
}
#endif /* WINRESIZE */

private bool
carefulcpy(to, from, maxsize, mess, raw)
char	*to,*from;
size_t	maxsize;
char	*mess;
bool	raw;
{
	if (from != NULL) {
		char	*ugh;

		if (strlen(from) >= maxsize)
			ugh = "too long";
		else if (*from == '\0')
			ugh = "empty";
		else {
			strcpy(to, from);
			return YES;
		}
		if (raw) {
			raw_complain("\r\n%s %s", mess, ugh);
		} else {
			swritef(mesgbuf, sizeof mesgbuf, "%s %s", mess, ugh);
			message(mesgbuf);
		}
	}
	return NO;
}

private void
dojovercs(dosys, dousr)
bool	dosys;
bool	dousr;
{
	char	Joverc[FILESIZE];

	if (dosys) {
		swritef(Joverc, sizeof(Joverc), "%s/jove.rc", ShareDir);
		(void) joverc(Joverc);	/* system wide jove.rc */
	}

	if (dousr) {
#ifdef MSDOS
		/* We don't want to run the same rc file twice */
		if (!dosys || strcmp(HomeDir, ShareDir) != 0) {
			swritef(Joverc, sizeof(Joverc), "%s/jove.rc", HomeDir);
			(void) joverc(Joverc);	/* jove.rc in home directory */
		}
#else
		swritef(Joverc, sizeof(Joverc), "%s/.joverc", HomeDir);
		(void) joverc(Joverc);	/* .joverc in home directory */
#endif
	}
}

int
#ifdef MAC	/* will get args from user, if option key held during launch */
main()
{
	int argc;
	char **argv;
#else /* !MAC */
main(argc, argv)
int	argc;
char	*argv[];
{
#endif /* !MAC */
	char	**argp;
#ifdef pdp11
	/* On the PDP-11, UNIX allocates at least 8K for the stack.
	 * In order not to waste this space, we allocate
	 * a bunch of buffers as autos.
	 */

	char	s_iobuff[LBSIZE],
		s_genbuf[LBSIZE],
		s_linebuf[LBSIZE];

	iobuff = s_iobuff;
	genbuf = s_genbuf;
	linebuf = s_linebuf;
#endif

#ifdef MAC
	MacInit();		/* initializes all */
	argc = getArgs(&argv);
#endif /* MAC */

#if defined(__WATCOMC__) && defined(FAR_LINES)
	/* Watcom C under DOS won't grow the near heap after any far
	 * allocation, so we must bump it up to the full 64K now.
	 */
	_heapgrow();
#endif

	iniargc = argc;
	iniargv = argv;

	if (setjmp(mainjmp)) {
		writef("\rAck! I can't deal with error \"%s\" now.\n\r", mesgbuf);
		finish(-1);
	}

#if defined(USE_CTYPE) && !defined(NO_SETLOCALE)
	/* make ctype reflect "native environment" */
	locale_adjust();
#endif

	getTERM();	/* Get terminal. */
	if (getenv("METAKEY"))
		MetaKey = YES;
	ttysetattr(YES);
	ttsize();

#ifdef UNIX
# ifdef WINRESIZE
	(void) signal(SIGWINCH, win_reshape);
	(void) SigHold(SIGWINCH);	/* we must not miss any, but we cannot
					   handle them until DoKeys */
# endif
#endif

#ifdef MAC
	InitEvents();
#endif

	d_cache_init();		/* initialize the disk buffer cache */

	make_scr();
	mac_init();	/* Initialize Macros */
	winit();	/* Initialize Window */
#ifdef PTYPROCS
# ifdef SIGCHLD
	(void) signal(SIGCHLD, sigchld_handler);
# endif
#endif
#ifdef USE_SELECT
	FD_ZERO(&global_fd);
	FD_SET(0, &global_fd);
	global_maxfd = 1;
#endif
	buf_init();

	if ((argp = scanvec(argv, "-d")) != NULL && chkCWD(argp[1]))
		setCWD(argp[1]);
	else
		getCWD();	/* After we setup curbuf in case we have to getwd() */

	HomeDir = getenv("HOME");
	if (HomeDir == NULL) {
#ifdef MSDOS
		HomeDir = copystr(pwd());	/* guess at current (initial) directory */
#else
		HomeDir = "/";
#endif
	}
	HomeLen = strlen(HomeDir);

	InitKeymaps();

	settout();	/* not until we know baudrate */
	SetTerm();

	/* Handle overrides for ShareDir and LibDir.
	 * We take care to use the last specification.
	 * Even if we don't use LibDir, we accept it.
	 */
	 {
		char
			*so = getenv("JOVESHARE");
#ifdef NEED_LIBDIR
		char
			*lo = getenv("JOVELIB");
#endif

		for (argp = argv; argp[0] != NULL && argp[1] != NULL; argp++) {
			if (strcmp(*argp, "-s") == 0)
				so = *++argp;
#ifdef NEED_LIBDIR
			else if (strcmp(*argp, "-l") == 0)
				lo = *++argp;
			else if (strcmp(*argp, "-ls") == 0 || strcmp(*argp, "-sl") == 0)
				lo = so = *++argp;
#endif
		}
		if (so != NULL)
			if (!carefulcpy(ShareDir, so, sizeof(ShareDir)-9, "ShareDir", YES))
				finish(0);
#ifdef NEED_LIBDIR
		if (lo != NULL)
			if (!carefulcpy(LibDir, lo, sizeof(LibDir)-9, "LibDir", YES))
				finish(0);
# ifdef PIPEPROCS
		swritef(Portsrv, sizeof(Portsrv), "%s/portsrv", LibDir);
# endif
#endif /* NEED_LIBDIR */
	}

	ShowVersion();	/* but the 'carefulcpy's which follow might overwrite it */

#ifdef MSDOS
	/* import the temporary file path from the environment
	   and fix the string, so that we can append a slash
	   safely	*/
	{
		char	*cp;

		if (((cp = getenv("TMPDIR"))!=NULL || (cp = getenv("TEMP"))!=NULL)
		&& carefulcpy(TmpDir, cp, sizeof(TmpDir), "TMPDIR or TEMP", NO))
		{
			cp = &TmpDir[strlen(TmpDir)];
			while (cp != TmpDir && (*--cp == '/' || *cp == '\\'))
				*cp = '\0';
		}
	}
#endif /* MSDOS */

#ifdef SUBSHELL
# ifdef MSDOS
	carefulcpy(Shell, getenv("COMSPEC"), sizeof(Shell), "COMSPEC", NO);
	/* SHELL, if present in DOS environment, will take precedence over COMSPEC */
# endif /* MSDOS */
	carefulcpy(Shell, getenv("SHELL"), sizeof(Shell), "SHELL", NO);
#endif /* SUBSHELL */

#ifdef UNIX
	carefulcpy(Mailbox, getenv("MAIL"), sizeof(Mailbox), "MAIL", NO);
#endif

	dojovercs(scanvec(argv, "-J") == NULL, scanvec(argv, "-j") == NULL);

#if defined(SUBSHELL) && defined(RECOVER)
	if (scanvec(argv, "-r") != NULL)
		dorecover();
#endif

#ifdef MSDOS
	(void) signal(SIGINT, SIG_IGN);
	break_off();	/* disable ctrl-c checking */
#endif /* MSDOS */
#ifdef UNIX
# ifndef DEBUGCRASH
	(void) signal(SIGHUP, finish);
	(void) signal(SIGINT, finish);
#  ifdef SIGBUS
	(void) signal(SIGBUS, finish);
#  endif /* SIGBUS */
	(void) signal(SIGSEGV, finish);
	(void) signal(SIGPIPE, finish);
	/* ??? Why should we ignore SIGTERM? */
	/* (void) signal(SIGTERM, SIG_IGN); */
# endif /* DEBUGCRASH */
	SetClockAlarm();
#endif /* UNIX */
	cl_scr(YES);
	flushscreen();
	RedrawDisplay();	/* start the redisplay process. */
	DoKeys(YES);
	finish(0);
	/* NOTREACHED*/
}

#ifdef MSDOS

private	char break_state;

/* set the break state to off */
private void
break_off()
{
	union REGS regs;

	regs.h.ah = 0x33;		/* break status */
	regs.h.al = 0x00;		/* request current state */
	intdos(&regs, &regs);
	break_state = regs.h.dl;
	bdos(0x33, 0, 1);	/* turn off break */
}

/* reset the break state */
private void
break_rst()
{
	bdos(0x33, break_state, 1);
}
#endif /* MSDOS */
