/*
 * Copyright (c) 1994 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Definitions of general values and related routines used by the calculator.
 */

#ifndef	VALUE_H
#define	VALUE_H

#include "cmath.h"
#include "config.h"

#define MAXDIM		4	/* maximum number of dimensions in matrices */
#define USUAL_ELEMENTS	4	/* usual number of elements for objects */


/*
 * Flags to modify results from the printvalue routine.
 * These flags are OR'd together.
 */
#define	PRINT_NORMAL	0x00	/* print in normal manner */
#define	PRINT_SHORT	0x01	/* print in short format (no elements) */
#define	PRINT_UNAMBIG	0x02	/* print in non-ambiguous manner */


/*
 * Definition of values of various types.
 */
typedef struct value VALUE;
typedef struct object OBJECT;
typedef struct matrix MATRIX;
typedef struct list LIST;
typedef	struct assoc ASSOC;
typedef	long FILEID;
typedef	struct rand RAND;
typedef	struct random RANDOM;
typedef	struct hashstate HASH;
typedef struct block BLOCK;


/*
 * calc values
 *
 * See below for information on what needs to be added for a new type.
 */
struct value {
	short v_type;			/* type of value */
	short v_subtype;		/* other data related to some types */
	union {				/* types of values (see V_XYZ below) */
		long vv_int;		/* 1: small integer value */
		NUMBER *vv_num;		/* 2: arbitrary sized numeric value */
		COMPLEX *vv_com;	/* 3: complex number */
		VALUE *vv_addr;		/* 4: address of variable value */
		char *vv_str;		/* 5: string value */
		MATRIX *vv_mat;		/* 6: address of matrix */
		LIST *vv_list;		/* 7: address of list */
		ASSOC *vv_assoc;	/* 8: address of association */
		OBJECT *vv_obj;		/* 9: address of object */
		FILEID vv_file;		/* 10: id of opened file */
		RAND *vv_rand;		/* 11: additive 55 random state */
		RANDOM *vv_random;	/* 12: Blum random state */
		CONFIG *vv_config;	/* 13: configuration state */
		HASH *vv_hash;		/* 14: hash state */
		BLOCK *vv_block;	/* 15: memory block */
	} v_union;
};


/*
 * For ease in referencing
 */
#define v_int	v_union.vv_int
#define	v_file	v_union.vv_file
#define v_num	v_union.vv_num
#define v_com	v_union.vv_com
#define v_addr	v_union.vv_addr
#define v_str	v_union.vv_str
#define v_mat	v_union.vv_mat
#define	v_list	v_union.vv_list
#define	v_assoc	v_union.vv_assoc
#define v_obj	v_union.vv_obj
#define	v_valid	v_union.vv_int
#define v_rand	v_union.vv_rand
#define v_random v_union.vv_random
#define v_config v_union.vv_config
#define v_hash	v_union.vv_hash
#define v_block	v_union.vv_block


/*
 * Value types.
 *
 * NOTE: The following files should be checked/adjusted for a new type:
 *
 *	quickhash.c
 *	shs.c
 *	value.c
 *
 * There may be others, but at is at least a start.
 */
#define V_NULL	0	/* null value */
#define V_INT	1	/* normal integer */
#define V_NUM	2	/* number */
#define V_COM	3	/* complex number */
#define V_ADDR	4	/* address of variable value */
#define V_STR	5	/* address of string */
#define V_MAT	6	/* address of matrix structure */
#define	V_LIST	7	/* address of list structure */
#define	V_ASSOC	8	/* address of association structure */
#define V_OBJ	9	/* address of object structure */
#define	V_FILE	10	/* opened file id */
#define V_RAND  11	/* address of additive 55 random state */
#define V_RANDOM 12	/* address of Blum random state */
#define V_CONFIG 13	/* configuration state */
#define V_HASH	14	/* hash state */
#define V_BLOCK	15	/* memory block */
#define V_MAX	15	/* highest legal value */

#define V_NOSUBTYPE	0	/* subtype has no meaning */
#define V_STRLITERAL	1	/* string subtype for literal str */
#define V_STRALLOC	2	/* string subtype for allocated str */

#define TWOVAL(a,b) ((a) * (V_MAX+1) + (b))	/* for switch of two values */

#define	NULL_VALUE	((VALUE *) 0)


extern void freevalue PROTO((VALUE *vp));
extern void copyvalue PROTO((VALUE *vp, VALUE *vres));
extern void negvalue PROTO((VALUE *vp, VALUE *vres));
extern void addvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void addnumeric PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void subvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void mulvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void squarevalue PROTO((VALUE *vp, VALUE *vres));
extern void invertvalue PROTO((VALUE *vp, VALUE *vres));
extern void roundvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern void broundvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern void apprvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern void intvalue PROTO((VALUE *vp, VALUE *vres));
extern void fracvalue PROTO((VALUE *vp, VALUE *vres));
extern void incvalue PROTO((VALUE *vp, VALUE *vres));
extern void decvalue PROTO((VALUE *vp, VALUE *vres));
extern void conjvalue PROTO((VALUE *vp, VALUE *vres));
extern void sqrtvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern void rootvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3,
	VALUE *vres));
extern void absvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void normvalue PROTO((VALUE *vp, VALUE *vres));
extern void shiftvalue PROTO((VALUE *v1, VALUE *v2, BOOL rightshift,
	VALUE *vres));
extern void scalevalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void powivalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void powervalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3,
	VALUE *vres));
extern void divvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void quovalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern void modvalue PROTO((VALUE *v1, VALUE *v2, VALUE *v3, VALUE *vres));
extern BOOL testvalue PROTO((VALUE *vp));
extern BOOL comparevalue PROTO((VALUE *v1, VALUE *v2));
extern void relvalue PROTO((VALUE *v1, VALUE *v2, VALUE *vres));
extern void sgnvalue PROTO((VALUE *vp, VALUE *vres));
extern QCKHASH hashvalue PROTO((VALUE *vp, QCKHASH val));
extern void printvalue PROTO((VALUE *vp, int flags));
extern BOOL precvalue PROTO((VALUE *v1, VALUE *v2));



/*
 * Structure of a matrix.
 */
struct matrix {
	long m_dim;		/* dimension of matrix */
	long m_size;		/* total number of elements */
	long m_min[MAXDIM];	/* minimum bound for indices */
	long m_max[MAXDIM];	/* maximum bound for indices */
	VALUE m_table[1];	/* actually varying length table */
};

#define matsize(n) (sizeof(MATRIX) - sizeof(VALUE) + ((n) * sizeof(VALUE)))


extern MATRIX *matadd PROTO((MATRIX *m1, MATRIX *m2));
extern MATRIX *matsub PROTO((MATRIX *m1, MATRIX *m2));
extern MATRIX *matmul PROTO((MATRIX *m1, MATRIX *m2));
extern MATRIX *matneg PROTO((MATRIX *m));
extern MATRIX *matalloc PROTO((long size));
extern MATRIX *matcopy PROTO((MATRIX *m));
extern MATRIX *matinit PROTO((MATRIX *m, VALUE *v1, VALUE *v2));
extern MATRIX *matsquare PROTO((MATRIX *m));
extern MATRIX *matinv PROTO((MATRIX *m));
extern MATRIX *matscale PROTO((MATRIX *m, long n));
extern MATRIX *matshift PROTO((MATRIX *m, long n));
extern MATRIX *matmulval PROTO((MATRIX *m, VALUE *vp));
extern MATRIX *matpowi PROTO((MATRIX *m, NUMBER *q));
extern MATRIX *matconj PROTO((MATRIX *m));
extern MATRIX *matquoval PROTO((MATRIX *m, VALUE *vp, VALUE *v3));
extern MATRIX *matmodval PROTO((MATRIX *m, VALUE *vp, VALUE *v3));
extern MATRIX *matint PROTO((MATRIX *m));
extern MATRIX *matfrac PROTO((MATRIX *m));
extern MATRIX *matappr PROTO((MATRIX *m, VALUE *v2, VALUE *v3));
extern MATRIX *mattrans PROTO((MATRIX *m));
extern MATRIX *matcross PROTO((MATRIX *m1, MATRIX *m2));
extern BOOL mattest PROTO((MATRIX *m));
extern void matsum PROTO((MATRIX *m, VALUE *vres));
extern BOOL matcmp PROTO((MATRIX *m1, MATRIX *m2));
extern long matsearch PROTO((MATRIX *m, VALUE *vp, long index));
extern long matrsearch PROTO((MATRIX *m, VALUE *vp, long index));
extern VALUE matdet PROTO((MATRIX *m));
extern VALUE matdot PROTO((MATRIX *m1, MATRIX *m2));
extern void matfill PROTO((MATRIX *m, VALUE *v1, VALUE *v2));
extern void matfree PROTO((MATRIX *m));
extern void matprint PROTO((MATRIX *m, long max_print));
extern VALUE *matindex PROTO((MATRIX *mp, BOOL create, long dim,
	VALUE *indices));
extern void matreverse PROTO((MATRIX *m));
extern void matsort PROTO((MATRIX *m));
extern BOOL matisident PROTO((MATRIX *m));
extern MATRIX *matround PROTO((MATRIX *m, VALUE *v2, VALUE *v3));
extern MATRIX *matbround PROTO((MATRIX *m, VALUE *v2, VALUE *v3));



/*
 * List definitions.
 * An individual list element.
 */
typedef struct listelem LISTELEM;
struct listelem {
	LISTELEM *e_next;	/* next element in list (or NULL) */
	LISTELEM *e_prev;	/* previous element in list (or NULL) */
	VALUE e_value;		/* value of this element */
};


/*
 * Structure for a list of elements.
 */
struct list {
	LISTELEM *l_first;	/* first list element (or NULL) */
	LISTELEM *l_last;	/* last list element (or NULL) */
	LISTELEM *l_cache;	/* cached list element (or NULL) */
	long l_cacheindex;	/* index of cached element (or undefined) */
	long l_count;		/* total number of elements in the list */
};


extern void insertlistfirst PROTO((LIST *lp, VALUE *vp));
extern void insertlistlast PROTO((LIST *lp, VALUE *vp));
extern void insertlistmiddle PROTO((LIST *lp, long index, VALUE *vp));
extern void removelistfirst PROTO((LIST *lp, VALUE *vp));
extern void removelistlast PROTO((LIST *lp, VALUE *vp));
extern void removelistmiddle PROTO((LIST *lp, long index, VALUE *vp));
extern void listfree PROTO((LIST *lp));
extern void listprint PROTO((LIST *lp, long max_print));
extern long listsearch PROTO((LIST *lp, VALUE *vp, long index));
extern long listrsearch PROTO((LIST *lp, VALUE *vp, long index));
extern BOOL listcmp PROTO((LIST *lp1, LIST *lp2));
extern VALUE *listfindex PROTO((LIST *lp, long index));
extern LIST *listalloc PROTO((void));
extern LIST *listcopy PROTO((LIST *lp));
extern void listreverse PROTO((LIST *lp));
extern void listsort PROTO((LIST *lp));
extern LIST *listappr PROTO((LIST *lp, VALUE *v2, VALUE *v3));
extern LIST *listround PROTO((LIST *m, VALUE *v2, VALUE *v3));
extern LIST *listbround PROTO((LIST *m, VALUE *v2, VALUE *v3));
extern LIST *listquo PROTO((LIST *lp, VALUE *v2, VALUE *v3));
extern LIST *listmod PROTO((LIST *lp, VALUE *v2, VALUE *v3));
extern BOOL evp PROTO((LISTELEM *cp, LISTELEM *x, VALUE *vres));
extern BOOL evalpoly PROTO((LIST *clist, LISTELEM *x, VALUE *vres));
extern void insertitems PROTO((LIST *lp1, LIST *lp2));


/*
 * Structures for associations.
 * Associations are "indexed" by one or more arbitrary values, and are
 * stored in a hash table with their hash values for quick indexing.
 */
typedef	struct assocelem ASSOCELEM;
struct assocelem {
	ASSOCELEM *e_next;	/* next element in list (or NULL) */
	long e_dim;		/* dimension of indexing for this element */
	QCKHASH e_hash;		/* hash value for this element */
	VALUE e_value;		/* value of association */
	VALUE e_indices[1];	/* index values (variable length) */
};


struct assoc {
	long a_count;		/* number of elements in the association */
	long a_size;		/* current size of association hash table */
	ASSOCELEM **a_table;	/* current hash table for elements */
};


extern ASSOC *assocalloc PROTO((long initsize));
extern ASSOC *assoccopy PROTO((ASSOC *ap));
extern void assocfree PROTO((ASSOC *ap));
extern void assocprint PROTO((ASSOC *ap, long max_print));
extern long assocsearch PROTO((ASSOC *ap, VALUE *vp, long index));
extern long assocrsearch PROTO((ASSOC *ap, VALUE *vp, long index));
extern BOOL assoccmp PROTO((ASSOC *ap1, ASSOC *ap2));
extern VALUE *assocfindex PROTO((ASSOC *ap, long index));
extern VALUE *associndex PROTO((ASSOC *ap, BOOL create, long dim,
	VALUE *indices));


/*
 * Object actions.
 */
#define OBJ_PRINT	0	/* print the value */
#define OBJ_ONE		1	/* create the multiplicative identity */
#define OBJ_TEST	2	/* test a value for "zero" */
#define OBJ_ADD		3	/* add two values */
#define OBJ_SUB		4	/* subtrace one value from another */
#define OBJ_NEG		5	/* negate a value */
#define OBJ_MUL		6	/* multiply two values */
#define OBJ_DIV		7	/* divide one value by another */
#define OBJ_INV		8	/* invert a value */
#define OBJ_ABS		9	/* take absolute value of value */
#define OBJ_NORM	10	/* take the norm of a value */
#define OBJ_CONJ	11	/* take the conjugate of a value */
#define OBJ_POW		12	/* take the power function */
#define OBJ_SGN		13	/* return the sign of a value */
#define OBJ_CMP		14	/* compare two values for equality */
#define OBJ_REL		15	/* compare two values for inequality */
#define OBJ_QUO		16	/* integer quotient of values */
#define OBJ_MOD		17	/* remainder of division of values */
#define OBJ_INT		18	/* integer part of */
#define OBJ_FRAC	19	/* fractional part of */
#define OBJ_INC		20	/* increment by one */
#define OBJ_DEC		21	/* decrement by one */
#define OBJ_SQUARE	22	/* square value */
#define OBJ_SCALE	23	/* scale by power of two */
#define OBJ_SHIFT	24	/* shift left (or right) by number of bits */
#define OBJ_ROUND	25	/* round to specified decimal places */
#define OBJ_BROUND	26	/* round to specified binary places */
#define OBJ_ROOT	27	/* take nth root of value */
#define OBJ_SQRT	28	/* take square root of value */
#define OBJ_MAXFUNC	28	/* highest function */


/*
 * Definition of an object type.
 * This is actually a varying sized structure.
 */
typedef struct {
	char *name;			/* name of object */
	int count;			/* number of elements defined */
	long actions[OBJ_MAXFUNC+1];	/* function indices for actions */
	int elements[1];		/* element indexes (MUST BE LAST) */
} OBJECTACTIONS;

#define objectactionsize(elements) \
	(sizeof(OBJECTACTIONS) + ((elements) - 1) * sizeof(int))


/*
 * Structure of an object.
 * This is actually a varying sized structure.
 * However, there are always at least USUAL_ELEMENTS values in the object.
 */
struct object {
	OBJECTACTIONS *o_actions;	/* action table for this object */
	VALUE o_table[USUAL_ELEMENTS];	/* object values (MUST BE LAST) */
};

#define objectsize(elements) \
	(sizeof(OBJECT) + ((elements) - USUAL_ELEMENTS) * sizeof(VALUE))


extern OBJECT *objcopy PROTO((OBJECT *op));
extern OBJECT *objalloc PROTO((long index));
extern VALUE objcall PROTO((int action, VALUE *v1, VALUE *v2, VALUE *v3));
extern void objfree PROTO((OBJECT *op));
extern void objuncache PROTO((void));
extern int addelement PROTO((char *name));
extern void defineobject PROTO((char *name, int indices[], int count));
extern int checkobject PROTO((char *name));
extern void showobjfuncs PROTO((void));
extern int findelement PROTO((char *name));
extern int objoffset PROTO((OBJECT *op, long index));


/*
 * hashstate - state of a hash system
 */
struct hashstate {
	int type;		/* hash type (see XYZ_HASH_TYPE below) */
	BOOL prevstr;		/* TRUE=>previous value hashed was a string */
	union {
		int hh_shs;	/* XXX - place holder, not really used */
	} h_union;
};

/* For ease in referencing */
#define h_shs   h_union.hh_shs

/* we support these hash types - must start with 0 */
#define SHS_HASH_TYPE 0
#define HASH_TYPE_MAX 0		/* must be number of XYZ_HASH_TYPE values */


/*
 * block - dynamic of fixed memory block
 *
 * There are two types of memory blocks: fixed memory blocks are fixed
 * in size and dynamic memory blocks can grow in size.  The max length 
 * (x.max) may be >= current (x.len), even in the fixed case.  A fixed block
 * can be shrunk instead of realloced.  The (x.max) refers to the number
 * of bytes malloced and 0 <= (x.len) <= (x.max).  If (x.max) == 0, then
 * (x.data) does not point to malloced storage.
 */
struct block {
	int type;		/* block type */
	int len;		/* current block length in USB8's */
	int max;		/* malloced block length in USB8's */
	USB8 *data;		/* start of data block if max > 0 */
};

#define V_FIXEDBLOCK	1	/* memory block is fixed in size */
#define V_DYNAMBLOCK	2	/* memory block size is dynamic */

#define is_fixedblock(x) (((x)->type) == V_FIXEDBLOCK)
#define is_dynamblock(x) (((x)->type) == V_DYNAMBLOCK)

#endif
