/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991, 1994 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/*
 * taper.c - writes files to tapes using buffering between reader and writer.
 */
#include "amanda.h"
#include "conffile.h"
#include "tapefile.h"
#include "clock.h"
#include "stream.h"
#include "infofile.h"
#include "logfile.h"
#include "tapeio.h"
#include "changer.h"
#include "version.h"
#include "arglist.h"

#define MAX_LINE 	1024
#define MAX_ARGS 	7
#define NBUFS		20
#define CONNECT_TIMEOUT 2*60

/* parameters controlling highwater position */

#define HIGHWATER_MAX	(NBUFS-1)
#define HIGHWATER_MIN	7
#define HIGHWATER_INC	3
#define HIGHWATER_DEC	1

typedef enum { BOGUS, START_TAPE, FILE_WRITE, PORT_WRITE, QUIT } cmd_t;

typedef enum { EMPTY, FILLING, FULL } status_t;

typedef struct buffer_s {
    status_t status;
    char buffer[BUFFER_SIZE];
} buffer_t;

#define nextbuf(p)    ((p) == buftable+NBUFS-1? buftable : (p)+1)
#define prevbuf(p)    ((p) == buftable? buftable+NBUFS-1 : (p)-1)

buffer_t *buftable;

char *pname = "taper";
int interactive;
char line[MAX_LINE];
char *hostname, *diskname;
int level, filenum;

char *argv[MAX_ARGS];
int argc;

int p2c[2], c2p[2];	/* parent-to-child, child-to-parent pipes */

char datestamp[80];
char label[80];

/* local functions */
void main P((int ext_argc, char **ext_argv));
static cmd_t getcmd P((void));
static void reader P((int fd, char *handle, int getp, int putp));
static void writer P((int getp, int putp));
static int start_tape P((char *new_datestamp));
static void putresult P((char *format, ...));
static buffer_t *attach_buffers P((void));
static void detach_buffers P((buffer_t *bufp));
static void destroy_buffers P((void));
static end_tape P((void));
void record_success P((char *str));
void record_failure P((char *str));
char get P((void));
char *getstr P((void));
void put P((int ch));
void putstr P((char *str));
int count P((status_t status));
int fillbuffer P((int fd, char *buffer, int size));


/*
 * ========================================================================
 * MAIN PROGRAM
 *
 */

void main(ext_argc, ext_argv)
int ext_argc;
char **ext_argv;
{
    int fd, data_port, data_socket, writerpid, retstat, wpid;
    cmd_t cmd;
    times_t total_wait;
    
    interactive = (ext_argc > 1 && !strcmp(ext_argv[1],"-t"));

    if(interactive)
	erroutput_type = ERR_INTERACTIVE;
    else
	erroutput_type = ERR_AMANDALOG;

    fprintf(stderr, "taper: pid %d version %d.%d.%d\n", 
	    getpid(), VERSION_MAJOR, VERSION_MINOR, VERSION_PATCH);

    if(read_conffile(CONFFILE_NAME))
	error("parse error in %s", CONFFILE_NAME);

    if(interactive)
	fprintf(stderr,"taper: running in interactive test mode\n");

    /* must get START_TAPE before beginning */

    startclock();
    cmd = getcmd();
    total_wait = stopclock();

    assert(cmd == START_TAPE && argc == 2);

    if(start_tape(argv[1])) 
	exit(1);

    /* create read/write syncronization pipes */

    if(pipe(p2c) || pipe(c2p))
	error("creating sync pipes: %s", strerror(errno));

    /* create shared memory segment */

    buftable = attach_buffers();

    /* fork off child writer */
    switch(writerpid = fork()) {
    case -1: 
	error("fork: %s", strerror(errno));
    case 0: 	/* child */
	close(p2c[1]);
	close(c2p[0]);
	writer(p2c[0], c2p[1]);
	error("writer() returned");
    default:	/* parent */
	close(p2c[0]);
	close(c2p[1]);
	break;
    }

    filenum = 0;

    while(1) {
	startclock();
	cmd = getcmd();
	total_wait = timesadd(total_wait, stopclock());
	filenum += 1;
	switch(cmd) {
	case PORT_WRITE:
	    assert(argc == 5);	/* PORT-WRITE <handle> <host> <disk> <lev> */

	    hostname = argv[2];
	    diskname = argv[3];
	    level = atoi(argv[4]);

	    data_port = 0;
	    data_socket = stream_server(&data_port);
	    putresult("PORT %d\n", data_port);

	    if((fd = stream_accept(data_socket, CONNECT_TIMEOUT,
				   DEFAULT_SIZE, BUFFER_SIZE)) == -1) {
		putresult("TAPE-ERROR %s [port connect timeout]\n", argv[1]);
		close(data_socket);
		break;
	    }
	    reader(fd, argv[1], c2p[0], p2c[1]);
	    close(data_socket);
	    break;

	case FILE_WRITE:
	    assert(argc == 6);	/* FILE-WRITE <sn> <fname> <hst> <dsk> <lev> */

	    hostname = argv[3];
	    diskname = argv[4];
	    level = atoi(argv[5]);

	    if((fd = open(argv[2], O_RDONLY)) == -1) {
		putresult("TAPE-ERROR %s [%s]\n", argv[1], strerror(errno));
		break;
	    }
	    reader(fd, argv[1], c2p[0], p2c[1]);
	    break;
	case QUIT:
	    putresult("QUITTING\n");
	    fprintf(stderr,"taper: DONE [idle wait: %s secs]\n",
		    walltime_str(total_wait));
	    fflush(stderr);
	    close(p2c[1]);	/* equiv to put('Q') */
	    end_tape();		/* !!! check results of end tape ?? */

	    if((wpid = wait(&retstat)) != writerpid) {
		fprintf(stderr, 
			"taper: writer wait returned %d instead of %d: %s\n",
			wpid, writerpid, strerror(errno));
	    }

	    detach_buffers(buftable);
	    destroy_buffers();
	    exit(0);

	default:
	    putresult("BAD-COMMAND %s\n", argv[0]);
	    break;
	}
    }
}


static cmd_t getcmd()
{
    char *p;
    int arg;


    if(fgets(line, MAX_LINE, stdin) == NULL)
	return QUIT;

    p = line;
    argc = 0;
    while(*p) {
	while(isspace(*p)) p++;
	if(argc < MAX_ARGS) argv[argc++] = p;
	while(*p && !isspace(*p)) p++;
	if(*p) *p++ = '\0';
    }
    for(arg = argc; arg < MAX_ARGS; arg++) argv[arg] = "";

#ifdef DEBUG
    printf("argc = %d\n", argc);
    for(arg = 0; arg < MAX_ARGS; arg++)
	printf("argv[%d] = \"%s\"\n", arg, argv[arg]);
#endif

    /* not enough commands for a table lookup */

    if(!strcmp(argv[0],"START-TAPE")) return START_TAPE;
    else if(!strcmp(argv[0],"FILE-WRITE")) return FILE_WRITE;
    else if(!strcmp(argv[0],"PORT-WRITE")) return PORT_WRITE;
    else if(!strcmp(argv[0],"QUIT")) return QUIT;
    return BOGUS;
}


arglist_function(static void putresult, char *, format)
{
    va_list argp;
    char result[MAX_LINE];

    arglist_start(argp, format);
    vsprintf(result, format, argp);
    arglist_end(argp);
    write(1, result, strlen(result));
}

/* ---------------------------- */

#ifdef HAVE_SYSVSHM

int shmid = -1;

static buffer_t *attach_buffers()
{
    long result;

    shmid = shmget(IPC_PRIVATE, sizeof(buffer_t)*NBUFS, IPC_CREAT|0700);
    if(shmid == -1)
	error("shmget: %s", strerror(errno));

    if((result = (long)shmat(shmid, NULL, 0)) == -1)
	error("shmat: %s", strerror(errno));

    return (buffer_t *) result;
}


static void detach_buffers(bufp)
buffer_t *bufp;
{
    if(shmdt(bufp) == -1)
	error("shmdr: %s", strerror(errno));
}

static void destroy_buffers()
{
    if(shmid == -1) return;	/* nothing to destroy */
    if(shmctl(shmid, IPC_RMID, NULL) == -1)
	error("shmctl: %s", strerror(errno));
}

#else
#ifdef HAVE_MMAP

#ifndef MAP_ANON
#  ifdef MAP_ANONYMOUS			/* OSF/1-style */
#    define MAP_ANON MAP_ANONYMOUS
#  else					/* SunOS4-style */
#    define MAP_ANON 0
#    define ZERO_FILE "/dev/zero"
#  endif
#endif

int shmfd = -1;

static buffer_t *attach_buffers()
{
    buffer_t *shmbuf;

#ifdef ZERO_FILE
    shmfd = open(ZERO_FILE, O_RDWR);
    if(shmfd == -1)
	error("attach_buffers: could not open %s: %s", 
	      ZERO_FILE, strerror(errno));
#endif

    shmbuf = (buffer_t *) mmap((void *) 0, 
			       sizeof(buffer_t)*NBUFS, 
			       PROT_READ|PROT_WRITE,
			       MAP_ANON|MAP_SHARED,
			       shmfd, 0);
    if((long)shmbuf == -1)
	error("attach_buffer: mmap: %s", strerror(errno));

    return shmbuf;
}

static void detach_buffers(bufp)
buffer_t *bufp;
{
    if(munmap((void *)bufp, sizeof(buffer_t)*NBUFS) == -1)
	error("detach_buffers: munmap: %s", strerror(errno));

    if(shmfd != -1) close(shmfd);
}

static void destroy_buffers()
{
}

#else
!!! error: must define either HAVE_SYSVSHM or HAVE_MMAP
#endif
#endif

/* -------- */

void record_success(str)
char *str;
{
    info_t record;

    /* update info record */

    if(open_infofile(getconf_str(CNF_INFOFILE)))
       error("could not open infofile: %s", strerror(errno));
    get_info(hostname, diskname, &record);
    strcpy(record.inf[level].label, label);
    record.inf[level].filenum = filenum;
    record.command = NO_COMMAND;
    if(interactive)
	fprintf(stderr,"taper: not updating database record for test\n");
    else
	put_info(hostname, diskname, &record);
    close_infofile();

    /* log results */

    log(L_SUCCESS, "%s %s %d %s", hostname, diskname, level, str);

}

void record_failure(str)
char *str;
{
    /* We don't have to update the info record.  Since dumper already did */

    /* log results */

    log(L_FAIL, "%s %s %d %s", hostname, diskname, level, str);
}


/* -------- */

int getpipe, putpipe;

char get()
{
    int rc;
    char buf[1];

    rc = read(getpipe, buf, 1);
    if(rc == 0) return 'Q';	/* EOF */
    else if(rc < 0) error("getpipe: %s", strerror(errno));

    return buf[0];
}

char *getstr()
{
    int rc;
    int len;
    static char str[MAX_LINE];

    rc = read(getpipe, &len, sizeof(int));
    if(rc < 0) error("getpipe read error: %s", strerror(errno));
    assert(rc == sizeof(int));

    rc = read(getpipe, str, len);
    if(rc < 0) error("getpipe read error: %s", strerror(errno));
    assert(rc == len);

    return str;
}


void put(ch)
char ch;
{
    int rc;

    rc = write(putpipe, &ch, 1);
    if(rc == 0) error("put %c: zero-len write", ch);
    else if(rc < 0) error("put %c: %s", ch, strerror(errno));
}

void putstr(str)
char *str;
{
    int rc, len;

    len = strlen(str)+1;	/* send \0 too */
    if((rc = write(putpipe, &len, sizeof(int))) < sizeof(int))
	error("putpipe: len: short write");
    if((rc = write(putpipe, str, len)) < len)
	error("putpipe: str: short write");
}

int count(status)
status_t status;
{
    int i, cnt;
    cnt = 0;
    for(i=0;i<NBUFS;i++) if(buftable[i].status == status) cnt++;
    return cnt;
}

int fillbuffer(fd, buffer, size)
int fd, size;
char *buffer;
{
    char *curptr;
    int spaceleft, cnt;
    
    curptr = buffer;
    spaceleft = size;

    do {
	cnt = read(fd, curptr, spaceleft);
	switch(cnt) {
	case 0:	/* eof */
	    if(spaceleft == size) {
		return 0;
	    }
	    else {
		/* partial buffer, zero rest */
		memset(curptr, '\0', spaceleft);
		return size;
	    }
	case -1:	/* error on read, punt */
	    return -1;
	default:
	    spaceleft -= cnt;
	    curptr += cnt;
	}

    } while(spaceleft > 0);
if(interactive)write(2,"R",1);
    return size;
}


static void reader(fd, handle, getp, putp)
int fd, getp, putp;
char *handle;
{
    buffer_t *bp;
    char ch;
    int rc, err, opening, closing;
    long filesize;
    times_t runtime;
    char errstr[256];

    /* initialize */

    getpipe = getp;
    putpipe = putp;
    filesize = 0;
    closing = 0;
    err = 0;

    /* set up shm buffers */

    for(bp = buftable; bp < buftable + NBUFS; bp++)
	bp->status = EMPTY;
    bp = buftable;

    /* tell writer to open tape */

    opening = 1;
    put('O');

    startclock();

    /* read file in loop */

    while(1) {
	ch = get();
	switch(ch) {

	case 'O':
	    assert(opening);
	    opening = 0;
	    err = 0;
	    break;

	case 'R':
	    if(closing) break;	/* ignore extra read tokens */

	    assert(!opening);
	    assert(bp->status == EMPTY);

	    bp->status = FILLING;
	    if((rc = fillbuffer(fd, bp->buffer, BUFFER_SIZE)) <= 0) {
		err = (rc < 0)? errno : 0;
		closing = 1;
		put('C');
	    }
	    else {
		bp->status = FULL;
		filesize += BUFFER_SIZE;
		bp = nextbuf(bp);
		put('W');
	    }
	    break;

	case 'E':
	    close(fd);
	    if(!interactive)
		sprintf(errstr, "[%s]", getstr());
	    else {
		runtime = stopclock();
		filesize = (filesize+1023)/1024;   /* switch to KB, round up */
		sprintf(errstr, "[sec %s kb %ld kps %3.1f {%s}]",
		       walltime_str(runtime), filesize,
		       filesize/(runtime.r.tv_sec+runtime.r.tv_usec/1000000.0),
		       getstr());
	    }

	    putresult("TAPE-ERROR %s %s\n", handle, errstr);
	    record_failure(errstr);
	    return;

	case 'C':
	    assert(!opening);
	    assert(closing);

	    close(fd);
	    runtime = stopclock();
	    if(err) {
		sprintf(errstr, "[input: %s]", strerror(err));
		putresult("TAPE-ERROR %s %s\n", handle, errstr);
		record_failure(errstr);
	    }
	    else {
		filesize = (filesize+1023)/1024;   /* switch to KB, round up */
		sprintf(errstr, "[sec %s kb %ld kps %3.1f %s]",
		       walltime_str(runtime), filesize, 
		       filesize/(runtime.r.tv_sec+runtime.r.tv_usec/1000000.0),
		       getstr());
		putresult("DONE %s %s\n", handle, errstr);
		record_success(errstr);
	    }
	    return;
	default:
	    assert(0);
	}
    }
}




/*
 * ========================================================================
 * TAPE WRITER SUBSYSTEM
 *
 */


buffer_t *bp, *prev;
int out_open, outfd, rc, full_buffers;
char tok, errstr[MAX_LINE];
times_t rdwait, wrwait, clwait;
long totwr, totfull;
int highwater;
char *tapedev;

/* local functions */
void highwater_inc P((void));
void highwater_dec P((void));
void writer_open P((void));
void writer_close P((void));
void writer_flush P((void));

void highwater_inc()
{
    highwater += HIGHWATER_INC;
    if(highwater > HIGHWATER_MAX) highwater = HIGHWATER_MAX;
    if(interactive) {
	fprintf(stderr," +(%d) ", highwater); fflush(stderr);
    }
}

void highwater_dec()
{
    highwater -= HIGHWATER_DEC;
    if(highwater < HIGHWATER_MIN) highwater = HIGHWATER_MIN;
    if(interactive) {
	fprintf(stderr," -(%d) ", highwater); fflush(stderr);
    }
}


void writer_open()
{
    assert(!out_open);
 
    if((outfd = tape_open(tapedev, O_WRONLY)) == -1) {
	sprintf(errstr, "taper: %s", strerror(errno));
	put('E');
	putstr(errstr);
    }
    else {
	out_open = 1;
	put('O');	/* tell reader tape is open */
	for(rc = 0; rc < NBUFS; rc++) 	/* and give it all the bufs */
	    put('R');

	/* re-initialize */
	bp = buftable;
	totwr = totfull = 0;
	rdwait = wrwait = times_zero;
	highwater = HIGHWATER_MIN;
    }
}

void writer_close()
{
    assert(out_open);
    out_open = 0;

    startclock();
    if(close(outfd) == -1) {
	sprintf(errstr, "taper: close: %s", strerror(errno));
	stopclock();
	put('E');
	putstr(errstr);
    }
    else {
	char rdstr[80], wrstr[80];
	
	clwait = stopclock();
	
	strcpy(rdstr, walltime_str(rdwait));
	strcpy(wrstr, walltime_str(wrwait));

	sprintf(errstr,
		"{wr: avgfull %5.2f writes %ld rdwait %s wrwait %s close %s }",
		totfull*1.0/totwr, totwr, rdstr, wrstr,
		walltime_str(clwait));
	put('C');
	putstr(errstr);
    }
}

void writer_flush()
{
    int nfull;

    while(bp->status == FULL) {

	startclock();
	rc = write(outfd, bp->buffer, BUFFER_SIZE);
	if(rc == BUFFER_SIZE) {
	    wrwait = timesadd(wrwait, stopclock());
	    bp->status = EMPTY;
	    nfull = count(FULL);
	    totfull += nfull;
	    totwr += 1;
	    bp = nextbuf(bp);
	    if(nfull == NBUFS) highwater_dec();
	    
	    full_buffers -= 1;	/* can go negative */
	    put('R');
	}
	else {
	    if(rc < 0)
		sprintf(errstr, "taper: write: %s", strerror(errno));
	    else sprintf(errstr, "taper: short write");
	    wrwait = timesadd(wrwait, stopclock());
	    put('E');
	    putstr(errstr);
	}
if(interactive)write(2, "W", 1);
    }
}

static void writer(getp, putp)
int getp, putp;
{

    getpipe = getp;
    putpipe = putp;
    out_open = 0;
    bp = buftable;
    totwr = totfull = 0;
    wrwait = rdwait = times_zero;
    full_buffers = 0;

    while(1) {
	startclock();
	tok = get();
	rdwait = timesadd(rdwait, stopclock());
	
	switch(tok) {
	case 'Q':
	    exit(0);
	case 'O': 
	    writer_open();
	    break;
	case 'C':
	    writer_flush();
	    writer_close();
	    break;
	case 'W':
	    full_buffers += 1;
	    if(full_buffers < highwater) break;
	    if(interactive)write(2," H",2);

	    /* 
	     * We've hit the high-water mark: write out all the full
	     * buffers.  Keep going until we hit one that isn't full.
	     * Note that the reader hopefully will have read in a few
	     * more buffers before we're done, so it's possible for
	     * full_buffers to go negative, and for some W tokens to
	     * collect in the sync pipe.  Rather than stopping to read
	     * these, we just trust the buffer status flag.  Once we
	     * stop, the main writer loop will automagically drain the
	     * W's from the sync pipe as full_buffers becomes positive
	     * again.
	     */

	    writer_flush();
	    highwater_inc();
	    if(interactive)write(2,"L ",2);
	    break;
	default:
	    assert(0);
	}
    }
}



/*
 * ========================================================================
 * TAPE MANIPULATION SUBSYSTEM
 *
 */

int nslots, backwards, found, got_match, tapedays;
char first_match_label[64], first_match[64], found_device[1024];
char *searchlabel, *labelstr;
tape_t *tp;

/* local functions */
int scan_init P((int rc, int ns, int bk));
int match P((char *regex, char *str));
int taperscan_slot P((int rc, char *slotstr, char *device));
char *taper_scan P((void));


int scan_init(rc, ns, bk)
int rc, ns, bk;
{
    if(rc)
	error("could not get changer info: %s", changer_resultstr);
	
    nslots = ns;
    backwards = bk;

    return 0;
}

int taperscan_slot(rc, slotstr, device)
int rc;
char *slotstr, *device;
{
    char *errstr;

    if(rc == 2) {
	fprintf(stderr, "%s: fatal slot %s: %s\n", 
		pname, slotstr, changer_resultstr);
	return 1;
    }
    else if(rc == 1) {
	fprintf(stderr, "%s: slot %s: %s\n",pname,slotstr,changer_resultstr);
	return 0;
    }
    else {
	if((errstr = tape_rdheader(device, datestamp, label)) != NULL)
	    fprintf(stderr, "%s: slot %s: %s\n", pname, slotstr, errstr);
	else {
	    /* got an amanda tape */
	    fprintf(stderr, "%s: slot %s: date %-8s label %s",
		    pname, slotstr, datestamp, label);
	    if(searchlabel != NULL && !strcmp(label, searchlabel)) {
		/* it's the one we are looking for, stop here */
		fprintf(stderr, " (exact label match)\n");
		strcpy(found_device, device);
		found = 1;
		return 1;
	    }
	    else if(!match(labelstr, label)) 
		fprintf(stderr, " (no match)\n");
	    else {
		/* not an exact label match, but a labelstr match */
		/* check against tape list */
		tp = lookup_tapelabel(label);
		if(tp != NULL && tp->position < tapedays)
		    fprintf(stderr, " (active tape)\n");
		else if(got_match)
		    fprintf(stderr, " (labelstr match)\n");
		else {
		    got_match = 1;
		    strcpy(first_match, slotstr);
		    strcpy(first_match_label, label);
		    fprintf(stderr, " (first labelstr match)\n");
		    if(!backwards || !searchlabel) {
			found = 2;
			strcpy(found_device, device);
			return 1;
		    }
		}
	    }
	}
    }
    return 0;
}

char *taper_scan()
{
    char outslot[32];

    if((tp = lookup_tapepos(getconf_int(CNF_TAPECYCLE))) == NULL)
	searchlabel = NULL;
    else
	searchlabel = tp->label;

    found = 0;
    got_match = 0;

    changer_scan(scan_init, taperscan_slot);

    if(found == 2)
	searchlabel = first_match_label;
    else if(!found && got_match) {
	searchlabel = first_match_label;
	if(changer_loadslot(first_match, outslot, found_device) == 0)
	    found = 1;
    }
    else if(!found) {
	if(searchlabel)
	    sprintf(changer_resultstr, 
		    "label %s or new tape not found in rack", searchlabel);
	else
	    sprintf(changer_resultstr, "new tape not found in rack");
    }

    return found? found_device : NULL;
}

static int start_tape(new_datestamp)
char *new_datestamp;
{ 
    char oldtapefilename[1024];
    char errstr[256], *result;
    tape_t *tp;

    char *tapefilename	= getconf_str(CNF_TAPELIST);

    tapedev	= getconf_str(CNF_TAPEDEV);
    tapedays	= getconf_int(CNF_TAPECYCLE);
    labelstr	= getconf_str(CNF_LABELSTR);

    if(read_tapelist(tapefilename))
	error("parse error in %s", tapefilename);

    if(changer_init() && (tapedev = taper_scan()) == NULL) {
	strcpy(errstr, changer_resultstr);
	goto tape_error;
    }

    if((result = tape_rdheader(tapedev, datestamp, label)) != NULL) {
	strcpy(errstr, result);
	goto tape_error;
    }

    fprintf(stderr, "taper: read label `%s' date `%s'\n", label, datestamp);
    fflush(stderr);

    /* check against tape list */
    tp = lookup_tapelabel(label);
    if(tp != NULL && tp->position < tapedays) {
	sprintf(errstr, "cannot overwrite active tape %s", label);
	goto tape_error;
    }

    if(!match(labelstr, label)) {
	sprintf(errstr,"label %s doesn't match labelstr \"%s\"",
		label, labelstr);
	goto tape_error;
    }

    strcpy(datestamp, new_datestamp);
    if((result = tape_wrheader(tapedev, datestamp, label)) != NULL) {
	strcpy(errstr, result);
	goto tape_error;
    }

    fprintf(stderr, "taper: wrote label `%s' date `%s'\n", label, datestamp);
    fflush(stderr);

    /* check against tape list */

    if(interactive) 
	fprintf(stderr,"taper: not writing new tapelist for test.\n");
    else {
	shift_tapelist(atoi(datestamp), label, tapedays);

	sprintf(oldtapefilename, "%s.yesterday", tapefilename);
	rename(tapefilename, oldtapefilename);
	if(write_tapelist(tapefilename))
	    error("couldn't write tapelist: %s", strerror(errno));
    }

    putresult("TAPE-OK\n");
    log(L_START, "datestamp %s label %s", datestamp, label);
    return 0;

tape_error:
    putresult("TAPE-ERROR [%s]\n", errstr);
    log(L_ERROR,"no-tape [%s]", errstr);
    return 1;
}


static int end_tape()
{ 
    char errstr[MAX_LINE];
    char *result;


    fprintf(stderr, "taper: writing end marker.\n"); fflush(stderr);

    if((result = tape_wrendmark(tapedev, datestamp)) != NULL) {
	strcpy(errstr, result);
	goto tape_error;
    }

    if((result = tape_rewind(tapedev)) != NULL) {
	strcpy(errstr, result);
	goto tape_error;
    }

    return 0;

tape_error:
    putresult("TAPE-ERROR [%s]\n", errstr);
    return 1;
}

int match(regex, str)
char *regex, *str;
{
    char *result, *re_comp();

    if((result = re_comp(regex)) != NULL)
        return 0;
    return re_exec(str);
}
