/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/*
 * amadmin.c - controlling process for the Amanda backup system
 */
#include "amanda.h"
#include "conffile.h"
#include "diskfile.h"
#include "tapefile.h"
#include "infofile.h"
#include "logfile.h"

char *pname = "amadmin";
disklist_t *diskqp;

void main P((int argc, char **argv));
void usage P((void));
void force P((int argc, char **argv));
void force_one P((disk_t *dp));
void unforce P((int argc, char **argv));
void unforce_one P((disk_t *dp));
void info P((int argc, char **argv));
void info_one P((disk_t *dp));
void find P((int argc, char **argv));
void delete P((int argc, char **argv));
void delete_one P((disk_t *dp));
void balance P((void));
void tape P((void));
void bumpsize P((void));
void diskloop P((int argc, char **argv, char *cmdname,
		 void (*func) P((disk_t *dp))));
char *seqdatestr P((int seq));
static int next_level0 P((disk_t *dp, info_t *ip));
void find P((int argc, char **argv));
int match P((char *host, char *disk));
char *nicedate P((int datestamp));
int bump_thresh P((int level));
void search_logfile P((char *label, int datestamp, char *logfile));
int get_logline P((FILE *logf));

void main(argc, argv)
int argc;
char **argv;
{
    char confdir[80];

    erroutput_type = ERR_INTERACTIVE;

    if(argc < 3) usage();

    sprintf(confdir, "%s/%s", CONFIG_DIR, argv[1]);
    if(chdir(confdir)) {
	fprintf(stderr,"%s: could not find config dir %s\n", pname, confdir);
	usage();
    }

    if(read_conffile(CONFFILE_NAME))
        error("could not find \"%s\" in this directory.\n", CONFFILE_NAME);

    if((diskqp = read_diskfile(getconf_str(CNF_DISKFILE))) == NULL)
        error("could not load \"%s\"\n", getconf_str(CNF_DISKFILE));

    if(read_tapelist(getconf_str(CNF_TAPELIST)))
        error("could not load \"%s\"\n", getconf_str(CNF_TAPELIST));

    if(open_infofile(getconf_str(CNF_INFOFILE)))
        error("could not open info db \"%s\"\n", getconf_str(CNF_INFOFILE));

    if(!strcmp(argv[2],"force")) force(argc, argv);
    else if(!strcmp(argv[2],"unforce")) unforce(argc, argv);
    else if(!strcmp(argv[2],"info")) info(argc, argv);
    else if(!strcmp(argv[2],"find")) find(argc, argv);
    else if(!strcmp(argv[2],"delete")) delete(argc, argv);
    else if(!strcmp(argv[2],"balance")) balance();
    else if(!strcmp(argv[2],"tape")) tape();
    else if(!strcmp(argv[2],"bumpsize")) bumpsize();
    else {
	fprintf(stderr, "%s: unknown command \"%s\"\n", pname, argv[2]);
	usage();
    }
    close_infofile();
}


void usage()
{
    fprintf(stderr, "\nUsage: %s <conf> <command> {<args>} ...\n", pname);
    fprintf(stderr, "    Valid <command>s are:\n");
    fprintf(stderr,
	    "\tforce <hostname> <disks> ...\t# Force level 0 tonight.\n");
    fprintf(stderr,
	    "\tunforce <hostname> <disks> ...\t# Clear force command.\n");
    fprintf(stderr,
    "\tfind <hostname> <disks> ...\t# Show which tapes these dumps are on.\n");
    fprintf(stderr,
	    "\tdelete <hostname> <disks> ...\t# Delete from database.\n");
    fprintf(stderr,
	    "\tinfo <hostname> <disks> ...\t# Show current info records.\n");
    fprintf(stderr,
	    "\tbalance\t\t\t\t# Show nightly dump size balance.\n");
    fprintf(stderr,
	    "\ttape\t\t\t\t# Show which tape is due next.\n");
    fprintf(stderr,
	    "\tbumpsize\t\t\t# Show current bump thresholds.\n");
    exit(1);
}


/* ----------------------------------------------- */

void diskloop(argc, argv, cmdname, func)
int argc; char **argv;
char *cmdname;
void (*func) P((disk_t *dp));
{
    host_t *hp;
    disk_t *dp;
    char *diskname;
    int count;

    if(argc < 4) {
	fprintf(stderr,"%s: expecting \"%s <hostname> {<disks> ...}\"\n",
		pname, cmdname);
	usage();
    }

    if((hp = lookup_host(argv[3])) == NULL) {
	fprintf(stderr, "%s: host %s not in current disklist database.\n",
		pname, argv[3]);
	exit(1);
    }
    if(argc < 5) {
	for(dp = hp->disks; dp != NULL; dp = dp->hostnext)
	    func(dp);
    }
    else {
	for(argc -= 4, argv += 4; argc; argc--, argv++) {
	    count = 0;
	    diskname = *argv;
	    for(dp = hp->disks; dp != NULL; dp = dp->hostnext) {
		if(!strncmp(dp->name, diskname, strlen(diskname))) {
		    count++;
		    func(dp);
		}
	    }
	    if(count == 0)
		fprintf(stderr, "%s: host %s has no disks that match \"%s\"\n",
			pname, hp->hostname, diskname);
	}
    }
}

/* ----------------------------------------------- */


void force_one(dp)
disk_t *dp;
{
    char *hostname = dp->host->hostname;
    char *diskname = dp->name;
    info_t inf;

    get_info(hostname, diskname, &inf);
    inf.command = PLANNER_FORCE;
    put_info(hostname, diskname, &inf);
    printf("%s: %s:%s is set to a forced level 0 tonight.\n", 
	   pname, hostname, diskname);
}


void force(argc, argv)
int argc; char **argv;
{
    diskloop(argc, argv, "force", force_one);
}


/* ----------------------------------------------- */


void unforce_one(dp)
disk_t *dp;
{
    char *hostname = dp->host->hostname;
    char *diskname = dp->name;
    info_t inf;

    get_info(hostname, diskname, &inf);
    if(inf.command == PLANNER_FORCE) {
	inf.command = NO_COMMAND;
	put_info(hostname, diskname, &inf);
	printf("%s: force command for %s:%s cleared.\n",
	       pname, hostname, diskname);
    }
    else {
	printf("%s: no force command outstanding for %s:%s, unchanged.\n",
	       pname, hostname, diskname);
    }
}

void unforce(argc, argv)
int argc; char **argv;
{
    diskloop(argc, argv, "unforce", unforce_one);
}


/* ----------------------------------------------- */

static int deleted;

void delete_one(dp)
disk_t *dp;
{
    char *hostname = dp->host->hostname;
    char *diskname = dp->name;
    info_t inf;

    if(get_info(hostname, diskname, &inf)) {
	printf("%s: %s:%s NOT currently in database.\n",
	       pname, hostname, diskname);
	return;
    }

    deleted++;
    if(del_info(hostname, diskname))
	error("couldn't delete %s:%s from database: %s", strerror(errno));
    else
	printf("%s: %s:%s deleted from curinfo database.\n",
	       pname, hostname, diskname);
}

void delete(argc, argv)
int argc; char **argv;
{
    deleted = 0;
    diskloop(argc, argv, "delete", delete_one);
 
   if(deleted)	
       printf(
	 "%s: NOTE: you'll have to remove these from the disklist yourself.\n",
	 pname);
}

/* ----------------------------------------------- */

void info_one(dp)
disk_t *dp;
{
    info_t inf;
    int lev, lev0date;
    struct tm *tm;
    stats_t *sp;

    get_info(dp->host->hostname, dp->name, &inf);

    printf("\nCurrent info for %s %s:\n", dp->host->hostname, dp->name);
    if(inf.command) printf("  (Forcing to level 0 dump on next run)\n");
    printf("  Stats: dump rates (kps), Full:  %5.1f, %5.1f, %5.1f\n",
	   inf.full.rate[0], inf.full.rate[1], inf.full.rate[2]);
    printf("                    Incremental:  %5.1f, %5.1f, %5.1f\n",
	   inf.incr.rate[0], inf.incr.rate[1], inf.incr.rate[2]);
    printf("          compressed size, Full: %5.1f%%,%5.1f%%,%5.1f%%\n",
	   inf.full.comp[0]*100, inf.full.comp[1]*100, inf.full.comp[2]*100);
    printf("                    Incremental: %5.1f%%,%5.1f%%,%5.1f%%\n",
	   inf.incr.comp[0]*100, inf.incr.comp[1]*100, inf.incr.comp[2]*100);

    printf("  Dumps: lev datestmp  tape   file  origK  compK secs\n");
    lev0date = inf.inf[0].date;
    for(lev = 0, sp = &inf.inf[0]; lev < 9; lev++, sp++) {
	if(sp->date == EPOCH) continue;
	tm = localtime(&sp->date);
	printf("          %d  %04d%02d%02d  %-6s  %3d %6d %6d %4d\n",
	       lev, tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
	       sp->label, sp->filenum, sp->size, sp->csize, sp->secs);
    }
}


void info(argc, argv)
int argc; char **argv;
{
    diskloop(argc, argv, "info", info_one);
}

/* ----------------------------------------------- */

void tape()
{
    tape_t *tp;

    printf("The next Amanda run should go onto ");

    if((tp = lookup_tapepos(getconf_int(CNF_TAPECYCLE))) != NULL)
	printf("tape %s or ", tp->label);
    printf("a new tape.\n");
}

/* ----------------------------------------------- */

#define SECS_PER_DAY (24*60*60)
time_t today;
int runtapes, dumpcycle;

char *seqdatestr(seq)
int seq;
{
    static char str[16];
    static char *dow[7] = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
    time_t t = today + seq*SECS_PER_DAY;
    struct tm *tm;

    tm = localtime(&t);

    sprintf(str, "%2d/%02d %3s", tm->tm_mon+1, tm->tm_mday, dow[tm->tm_wday]);
    return str;
}


/* when is next level 0 due? 0 = tonight, 1 = tommorrow, etc*/
static int next_level0(dp, ip)
disk_t *dp;
info_t *ip;
{
    if(dp->dtype->no_full) 
	return 1;	/* fake it */
    else if(ip->inf[0].date == EPOCH)
	return 0;	/* new disk */
    else
	return dp->dtype->dumpcycle - days_diff(ip->inf[0].date, today);
}


void balance()
{
    disk_t *dp;
    struct balance_stats {
	int disks;
	long origsize, outsize;
    } *sp;
    int seq, max_seq, total, balanced, runs_per_cycle;
    info_t inf;

    total = getconf_int(CNF_TAPECYCLE);
    max_seq = getconf_int(CNF_DUMPCYCLE)-1;     /* print at least this many */
    time(&today);
    runtapes = getconf_int(CNF_RUNTAPES);
    dumpcycle = getconf_int(CNF_DUMPCYCLE);

    runs_per_cycle = guess_runs_from_tapelist();

    sp = (struct balance_stats *) 
	alloc(sizeof(struct balance_stats) * (total+1));

    for(seq=0; seq <= total; seq++)
	sp[seq].disks = sp[seq].origsize = sp[seq].outsize = 0;

    for(dp = diskqp->head; dp != NULL; dp = dp->next) {
	if(get_info(dp->host->hostname, dp->name, &inf)) {
	    printf("new disk %s:%s ignored.\n", dp->host->hostname, dp->name);
	    continue;
	}
	seq = next_level0(dp, &inf);
	if(seq > max_seq) max_seq = seq;
	if(seq >= total)
	    error("bogus seq number %d for %s:%s", seq,
		  dp->host->hostname, dp->name);

	sp[seq].disks++;
	sp[seq].origsize += inf.inf[0].size;
	sp[seq].outsize += inf.inf[0].csize;

	sp[total].disks++;
	sp[total].origsize += inf.inf[0].size;
	sp[total].outsize += inf.inf[0].csize;
    }

    if(sp[total].outsize == 0) {
	printf("\nNo data to report on yet.\n");
	return;
    }

    balanced = sp[total].outsize / runs_per_cycle;

    printf("\n due-date  #fs   orig KB    out KB  balance\n");
    printf("-------------------------------------------\n");
    for(seq = 0; seq <= max_seq; seq++) {
	printf("%-9.9s  %3d  %8ld  %8ld  ",
	       seqdatestr(seq), sp[seq].disks,
	       sp[seq].origsize, sp[seq].outsize);
	if(!sp[seq].outsize) printf("   --- \n");
	else printf("%+6.1f%%\n",
		    (sp[seq].outsize-balanced)*100.0/(double)balanced);
    }
    printf("-------------------------------------------\n");
    printf("TOTAL      %3d  %8ld  %8ld  %7d", sp[total].disks,
	   sp[total].origsize, sp[total].outsize, balanced);
    printf("  (estimated %d runs per dumpcycle)\n", runs_per_cycle);
}


/* ----------------------------------------------- */

#define MAX_LINE 2048


typedef enum program_e {
    P_UNKNOWN, P_PLANNER, P_DRIVER, P_REPORTER, P_DUMPER, P_TAPER, P_AMFLUSH
} program_t;
#define P_LAST P_AMFLUSH

char *program_str[] = {
    "UNKNOWN", "planner", "driver", "reporter", "dumper", "taper", "amflush"
};

typedef struct line_s {
    struct line_s *next;
    char *str;
} line_t;

int curlinenum;
logtype_t curlog;
program_t curprog;
char *curstr;
char logline[MAX_LINE], tmpstr[MAX_LINE];

char *find_hostname;
char **find_diskstrs;
int  find_ndisks;

void find(argc, argv)
int argc; char **argv;
{
    char *conflog, logfile[1024];
    host_t *hp;
    int tape, maxtape, len, seq, logs;
    tape_t *tp;

    if(argc < 4) {
	fprintf(stderr,
		"%s: expecting \"find <hostname> [<disk> ...]\"\n", pname);
	usage();
    }

    find_hostname = argv[3];
    find_diskstrs = &argv[4];
    find_ndisks = argc - 4;

    if((hp = lookup_host(find_hostname)) == NULL)
	printf("Warning: host %s not in disklist.\n", find_hostname);

    conflog = getconf_str(CNF_LOGFILE);
    maxtape = getconf_int(CNF_TAPECYCLE);

    len = strlen(find_hostname)-4;
    printf("date        host%*s disk lv tape  file status\n", 
	   len>0?len:0,"");
    for(tape = 1; tape <= maxtape; tape++) {
	tp = lookup_tapepos(tape);
	if(tp == NULL) continue;

	/* search log files */

	logs = 0;

	/* new-style log.<date>.<seq> */

	for(seq = 0; 1; seq++) {
	    sprintf(logfile, "%s.%d.%d", conflog, tp->datestamp, seq);
	    if(access(logfile, R_OK) != 0) break;
	    search_logfile(tp->label, tp->datestamp, logfile);
	    logs += 1;
	}

	/* search old-style amflush log, if any */

	sprintf(logfile, "%s.%d.amflush", conflog, tp->datestamp);
	if(access(logfile,R_OK) == 0) {
	    search_logfile(tp->label, tp->datestamp, logfile);
	    logs += 1;
	}

	/* search old-style main log, if any */

	sprintf(logfile, "%s.%d", conflog, tp->datestamp);
	if(access(logfile,R_OK) == 0) {
	    search_logfile(tp->label, tp->datestamp, logfile);
	    logs += 1;
	}
	if(logs == 0)
	    printf("Warning: no log files found for date %s\n", 
		   nicedate(tp->datestamp));
    }
}

int match(host, disk)
char *host, *disk;
{
    int d;

    if(strcmp(host, find_hostname)) return 0;
    if(find_ndisks == 0) return 1;

    for(d = 0; d < find_ndisks; d++) {
	if(!strncmp(disk, find_diskstrs[d], strlen(find_diskstrs[d])))
	    return 1;
    }
    return 0;
}

char *nicedate(datestamp)
int datestamp;
{
    static char nice[20];
    int year, month, day;

    year  = datestamp / 10000;
    month = (datestamp / 100) % 100;
    day   = datestamp % 100;

    sprintf(nice, "%4d-%02d-%02d", year, month, day);

    return nice;
}

void search_logfile(label, datestamp, logfile)
char *label, *logfile;
int datestamp;
{
    FILE *logf;
    char host[80], disk[80], rest[MAX_LINE];
    int level, rc, filenum;

    if((logf = fopen(logfile, "r")) == NULL)
	error("could not open logfile %s: %s", logfile, strerror(errno));

    filenum = 0;
    while(get_logline(logf)) {
	if(curlog == L_SUCCESS && curprog == P_TAPER) filenum ++;

	if(curlog == L_SUCCESS || curlog == L_FAIL) {
	    rc =sscanf(curstr,"%s %s %d %[^\n]", host, disk, &level, rest);
	    if(rc != 4) {
		printf("strange log line \"%s\"\n", curstr);
		continue;
	    }
	    if(match(host, disk)) {
		if(curprog == P_TAPER) {
		    printf("%s  %-4s %-4s %d  %-6s %3d", nicedate(datestamp),
			   host, disk, level, label, filenum);
		    if(curlog == L_SUCCESS) printf(" OK\n");
		    else printf(" FAILED %s\n", rest);
		}
		else if(curlog == L_FAIL) {	/* print other failures too */
		    printf("%s  %-4s %-4s %d  %-10s FAILED (%s) %s\n", 
			   nicedate(datestamp), host, disk, level, 
			   "---", program_str[(int)curprog], rest);
		}
	    }
	}
    }
}


/* shared code with reporter.c -- should generalize into log-reading library */

int get_logline(logf)
FILE *logf;
{
    char *cp, *logstr, *progstr;

    if(fgets(logline, MAX_LINE, logf) == NULL)
	return 0;
    curlinenum++;
    cp = logline;

    /* continuation lines are special */

    if(*cp == ' ') {
	curlog = L_CONT;
	/* curprog stays the same */
	curstr = cp + 2;
	return 1;
    }

    /* isolate logtype field */

    logstr = cp;
    while(*cp && *cp != ' ') cp++;
    if(*cp) *cp++ = '\0';

    /* isolate program name field */

    while(*cp == ' ') cp++;	/* skip blanks */
    progstr = cp;
    while(*cp && *cp != ' ') cp++;
    if(*cp) *cp++ = '\0';

    /* rest of line is logtype dependent string */

    curstr = cp;

    /* lookup strings */

    for(curlog = L_MARKER; curlog != L_BOGUS; curlog--)
	if(!strcmp(logtype_str[curlog], logstr)) break;

    for(curprog = P_LAST; curprog != P_UNKNOWN; curprog--)
	if(!strcmp(program_str[curprog], progstr)) break;

    return 1;
} 

/* ------------------------ */


/* shared code with planner.c */

int bump_thresh(level)
int level;
{
    int bump = getconf_int(CNF_BUMPSIZE);
    double mult = getconf_real(CNF_BUMPMULT);

    while(--level) bump = (int) bump * mult;
    return bump;
}
     
void bumpsize()
{
    int l;

    printf("Current bump parameters:\n");
    printf("  bumpsize %5d KB\t- minimum savings (threshold) to bump level 1 -> 2\n",
	   getconf_int(CNF_BUMPSIZE));
    printf("  bumpdays %5d\t- minimum days at each level\n",
	   getconf_int(CNF_BUMPDAYS));
    printf("  bumpmult %5.5g\t- threshold = bumpsize * (level-1)**bumpmult\n\n",
	   getconf_real(CNF_BUMPMULT));

    printf("      Bump -> To  Threshold\n");
    for(l = 1; l < 9; l++)
	printf("\t%d  ->  %d  %9d KB\n", l, l+1, bump_thresh(l));
    putchar('\n');
}
