/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991,1993 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/* 
 * sendsize-dump.c - send estimated backup sizes using dump
 */
#define DEBUG_CODE
#undef FAST_SIZE_CHECK		/* XXX not ready for prime-time yet */

#include "amanda.h"
#include "statfs.h"

#ifdef SYSV_SETPGRP
#  define SETPGRP	setpgrp()
#else
#  define SETPGRP	setpgrp(0,getpid())
#endif

#define MAXLINE 4096

typedef struct regex_s {
    char *regex;
    int scale;
} regex_t;

regex_t re_size[] = {
    {"  DUMP: estimated [0-9][0-9]* tape blocks", 1024},
    {"  DUMP: [Ee]stimated [0-9][0-9]* blocks", 512},
    {"dump: Estimate: [0-9][0-9]* tape blocks", 1024},		    /* OSF/1 */
    {"backup: There are an estimated [0-9][0-9]* tape blocks.",1024}, /* AIX */
    {"backup: [0-9][0-9]* tape blocks on [0-9][0-9]* tape(s)",1024},

    { NULL, 0 }
};


char line[MAXLINE];
char *pname = "sendsize-dump";

/* local functions */
void main P((int argc, char **argv));
int getsize P((char *disk, int level));
int handle_dumpline P((char *str));
int first_num P((char *str));
int fast_size P((char *str));


void main(argc, argv)
int argc;
char **argv;
{
    int size, level;
    char disk[256];

    /* initialize */

    chdir("/tmp");
    erroutput_type = (ERR_INTERACTIVE|ERR_SYSLOG);
    umask(0);
    dbopen("/tmp/sendsize.debug");

    /* handle all service requests */

    while(fgets(line, MAXLINE, stdin)) {
	if(!strncmp(line, "OPTIONS", 7)) {
	    /* we don't recognize any options yet */
	    printf("OPTIONS ;\n");
	    continue;
	}

	if(sscanf(line, "%s %d\n", disk, &level) != 2) goto err;

	size = getsize(disk, level);
	/* XXX handle errors differently */
	printf("%s %d SIZE %d\n", disk, level, size);
    }

    dbclose();
    exit(0);
 err:
    printf("FORMAT ERROR IN REQUEST PACKET\n");
    dbprintf(("REQ packet is bogus\n"));
    dbclose();
    exit(1);
}

int getsize(disk, level)
char *disk;
int level;
{
    int pipefd[2], nullfd, size, dumppid;
    FILE *dumpout;
    char dumpkeys[10], device[256];

    if(disk[0] == '/') strcpy(device, disk);
    else sprintf(device, "%s%s", DEV_PREFIX, disk);

#ifdef FAST_SIZE_CHECK
    if(level == 0)				/* shortcut for level 0s */
	if((size = fast_size(device)) != -1)
	    return size;
#endif

    nullfd = open("/dev/null", O_RDWR);
    pipe(pipefd);
    sprintf(dumpkeys, "%dsf", level);

    dbprintf(("sendsize: running \"%s %s 100000 - %s\"\n",
	      DUMP, dumpkeys, device));

    switch(dumppid = fork()) {
    case -1: return -1;
    default: break; 
    case 0:	/* child process */
	if(SETPGRP == -1)
	    dbprintf(("setpgrp(0,%d) failed: %s\n",getpid(),strerror(errno)));

	dup2(nullfd, 0);
	dup2(nullfd, 1);
	dup2(pipefd[1], 2);
	close(pipefd[0]);

#ifndef AIX_BACKUP
	sprintf(dumpkeys, "%dsf", level);
	execl(DUMP, "dump", dumpkeys, "100000", "-", device, 0);
#else
	sprintf(dumpkeys, "-%df", level);
	execl(DUMP, "backup", dumpkeys, "-", device, 0);
#endif
	exit(1);
    }
    close(pipefd[1]);
    dumpout = fdopen(pipefd[0],"r");

    size = -1;
    while(fgets(line,MAXLINE,dumpout) != NULL) {
	dbprintf(("%s",line));
	size = handle_dumpline(line);
	if(size > -1) {
	    if(fgets(line, MAXLINE, dumpout) != NULL)
		dbprintf(("%s",line));
	    break;
	}
    }

    dbprintf((".....\n"));
    if(size == -1)
	dbprintf(("(no size line match in above dump output)\n.....\n"));

#ifdef OSF1_HANG_BUG
    sleep(5);
#endif

    kill(-dumppid, SIGTERM);

    close(nullfd);
    fclose(dumpout);

    return size;
}


int first_num(str)
char *str;
/*
 * Returns the value of the first integer in a string.
 */
{
    char tmp[16], *tp;

    tp = tmp;
    while(*str && !isdigit(*str)) str++;
    while(*str && isdigit(*str)) *tp++ = *str++;
    *tp = '\0';

    return atoi(tmp);
}


int handle_dumpline(str)
char *str;
/*
 * Checks the dump output line against the error and size regex tables.
 */
{
    regex_t *rp;
    char *errs, *re_comp();
    
    /* check for size match */
    for(rp = re_size; rp->regex != NULL; rp++) {
	errs = re_comp(rp->regex);
	/* check for error return: errs != NULL */
	if(re_exec(str) == 1) 
	    return (first_num(str)*rp->scale+1023)/1024;
    }
    return -1;
}

#ifdef FAST_SIZE_CHECK

#include <fstab.h>

int fast_size(str)
char *str;
{
    generic_fs_stats_t fs_stats;
    struct stat dev_stat;
    struct fstab *fsent;
    char *fsname, device[256];
    int size;

    strcpy(device, str);

    if(stat(device, &dev_stat) == -1) {
	dbprintf(("sendsize: got error from stat(%s): %s\n",
		  device, strerror(errno)));
	return -1;
    }

    if((dev_stat.st_mode & S_IFBLK) || (dev_stat.st_mode & S_IFCHR)) {
	/* look up in /etc/fstab */
	setfsent();
	fsname = NULL;
	while((fsent = getfsent()) != NULL) {
	    if(!strcmp(fsent->fs_spec, device)) {
		/* XXX copy out the string */
		fsname = fsent->fs_file;
		break;
	    }
	}
	endfsent();
	
	if(fsname == NULL) {
	    dbprintf(("sendsize: %s: blk or chr, and can't find in fstab\n",
		      device));
	    return -1;
	}

	dbprintf(("sendsize: using dir \"%s\" from fstab, for device %s\n",
		  fsname, device));
	strcpy(device, fsname);
    }

    if(get_fs_stats(device, &fs_stats) == -1) {
	dbprintf(("sendsize: hmmm, got error from get_fs_stats(%s): %s\n",
		  device, strerror(errno)));
	return -1;
    }
    size = fs_stats.total - fs_stats.free;	/* close enough! */
    dbprintf(("sendsize: device %s: level 0 size vis statfs: %dK\n", 
	      device, size));
    return size;
}

#endif /* FAST_SIZE_CHECK */
