/*
 * Amanda, The Advanced Maryland Automatic Network Disk Archiver
 * Copyright (c) 1991,1994 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: James da Silva, Systems Design and Analysis Group
 *			   Computer Science Department
 *			   University of Maryland at College Park
 */
/* 
 * sendbackup-dump.c - send backup data using BSD dump
 */
#define DEBUG_CODE

#include "amanda.h"
#include "stream.h"
#include "arglist.h"

#define MAX_LINE 1024
#define TIMEOUT 30

#define BUFFER_SIZE	32*1024

/*
 * If we don't have the new-style wait access functions, use our own,
 * compatible with old-style BSD systems at least.  Note that we don't
 * care about the case w_stopval == WSTOPPED since we don't ask to see
 * stopped processes, so should never get them from wait.
 */
#ifndef WEXITSTATUS
#   define WEXITSTATUS(r)	(((union wait *) &(r))->w_retcode)
#   define WTERMSIG(r)		(((union wait *) &(r))->w_termsig)

#   undef  WIFSIGNALED
#   define WIFSIGNALED(r)	(((union wait *) &(r))->w_termsig != 0)
#endif

int comppid, dumppid, encpid;
char thiserr[80], errorstr[256];

int data_socket, data_port, dataf;
int mesg_socket, mesg_port, mesgf;

char line[MAX_LINE];
char *pname = "sendbackup";
int compress, no_record, bsd_auth;
#define COMPR_FAST 1
#define COMPR_BEST 2

#ifdef KRB4_SECURITY
#include "sendbackup-krb4.c"
#else					/* I'd tell you what this does */
#define NAUGHTY_BITS			/* but then I'd have to kill you */
#endif


int dump_size = -1;

/* local functions */
void main P((int argc, char **argv));
void parse_options P((char *str));
char *optionstr P((void));
char *childstr P((int pid));
int check_status P((int pid, int w));

void write_tapeheader P((char *host, char *disk, int level, int compress, \
			 char *datestamp, int outf));
int pipespawn P((char *prog, int *stdinfd, int stdoutfd, int stderrfd, ...));
int pipefork P((void (*func) P((void)), char *fname, int *stdinfd, \
		int stdoutfd, int stderrfd));
void start_backup P((char *disk, int level, char *datestamp, \
		     int dataf, int mesgf));
void parse_backup_messages P((int mesgin));
void add_msg_data P((char *str, int len));

void parse_options(str)
char *str;
{
    /* only a few options, no need to get fancy */

    if(strstr(str, "compress") != NULL) {
	if(strstr(str, "compress-best") != NULL)
	    compress = COMPR_BEST;
	else
	    compress = COMPR_FAST;	/* the default */
    }

    no_record = strstr(str, "no-record") != NULL;
    bsd_auth = strstr(str, "bsd-auth") != NULL;
#ifdef KRB4_SECURITY
    krb4_auth = strstr(str, "krb4-auth") != NULL;
    kencrypt = strstr(str, "kencrypt") != NULL;
#endif
}

char *optionstr()
{
    static char optstr[256];

    strcpy(optstr,";");
    if(compress == COMPR_BEST)
	strcat(optstr, "compress-best;");
    else if(compress == COMPR_FAST)
	strcat(optstr, "compress-fast;");

    if(no_record) strcat(optstr, "no-record;");
    if(bsd_auth) strcat(optstr, "bsd-auth;");
#ifdef KRB4_SECURITY
    if(krb4_auth) strcat(optstr, "krb4-auth;");
    if(kencrypt) strcat(optstr, "kencrypt;");
#endif

    return optstr;
}

void main(argc, argv)
int argc;
char **argv;
{
    int level, mesgpipe[2];
    char disk[256], options[80], datestamp[80];

    /* initialize */

    chdir("/tmp");
    erroutput_type = (ERR_INTERACTIVE|ERR_SYSLOG);
    umask(0);
    dbopen("/tmp/sendbackup.debug");
    {
	extern int db_file;
	dup2(db_file, 2);
    }

    /* parse dump request */

    if(fgets(line, MAX_LINE, stdin) == NULL)
	goto err;
    dbprintf(("%s: got input request: %s", pname, line));
    if(sscanf(line, "%s %d DATESTAMP %s OPTIONS %[^\n]\n", 
	      disk, &level, datestamp, options) != 4)
	goto err;
    dbprintf(("  parsed request as: disk `%s' lev %d stamp `%s' opt `%s'\n",
	      disk, level, datestamp, options));

    parse_options(options);

#ifdef KRB4_SECURITY
    if(krb4_auth) {
	if(read(KEY_PIPE, session_key, sizeof session_key) 
	   != sizeof session_key) {
	    printf("ERROR [%s: could not read session key]\n", pname);
	    exit(0);
	}
    }
#endif

    data_socket = stream_server(&data_port);
    mesg_socket = stream_server(&mesg_port);

    switch(fork()) {
    case -1:	/* fork error */
	printf("ERROR [%s: could not fork: %s]\n", pname, strerror(errno));
	exit(0);
    default:	/* parent */
	printf("CONNECT DATA %d MESG %d\n", data_port, mesg_port);
	printf("OPTIONS %s\n", optionstr());
	exit(0);
    case 0:	/* child, keep going */
	break;
    }

    dbprintf(("  waiting for connect on %d, then %d\n", data_port, mesg_port));
    
    dataf = stream_accept(data_socket, TIMEOUT, DEFAULT_SIZE, DEFAULT_SIZE);
    if(dataf == -1)
	dbprintf(("%s: timeout on data port %d\n", pname, data_port));
    mesgf = stream_accept(mesg_socket, TIMEOUT, DEFAULT_SIZE, DEFAULT_SIZE);
    if(mesgf == -1)
	dbprintf(("%s: timeout on mesg port %d\n", pname, mesg_port));

    if(dataf == -1 || mesgf == -1) {
	dbclose();
	exit(1);
    }

    dbprintf(("  got both connections\n"));

#ifdef KRB4_SECURITY
    if(kerberos_handshake(dataf, session_key) == 0) {
	dbprintf(("kerberos_handshake on data socket failed\n"));
	dbclose();
	exit(1);
    }

    if(kerberos_handshake(mesgf, session_key) == 0) {
	dbprintf(("kerberos_handshake on mesg socket failed\n"));
	dbclose();
	exit(1);
    }

    dbprintf(("%s: kerberos handshakes succeeded!\n", pname));
#endif
	
    /* redirect stderr */
    if(dup2(mesgf, 2) == -1) {
	dbprintf(("error redirecting stderr: %s\n", strerror(errno)));
	dbclose();
	exit(1);
    }

    if(pipe(mesgpipe) == -1)
	error("error [opening mesg pipe: %s]", strerror(errno));


    start_backup(disk, level, datestamp, dataf, mesgpipe[1]);
    parse_backup_messages(mesgpipe[0]);

    dbclose();
    exit(0);

 err:
    printf("FORMAT ERROR IN REQUEST PACKET\n");
    dbprintf(("REQ packet is bogus\n"));
    dbclose();
    exit(1);
}

char *childstr(pid)
int pid;
/*
 * Returns a string for a child process.  Checks the saved dump and
 * compress pids to see which it is.
 */
{
    if(pid == dumppid) return "dump";
    if(pid == comppid) return "compress";
    if(pid == encpid)  return "kencrypt";
    return "unknown";
}


int check_status(pid, w)
int pid, w;
/*
 * Determine if the child return status really indicates an error.
 * If so, add the error message to the error string; more than one
 * child can have an error.
 */
{
    char *str;
    int ret, sig;

    if(WIFSIGNALED(w))	ret = 0, sig = WTERMSIG(w);
    else sig = 0, ret = WEXITSTATUS(w);

    str = childstr(pid);

#ifndef HAVE_GZIP
    if(pid == comppid) 	/* compress returns 2 sometimes; it's ok */
	if(ret == 2) return 0;
#endif

#ifdef DUMP_RETURNS_1
    if(pid == dumppid) /* Ultrix dump returns 1 sometimes; it's ok too */
        if(ret == 1) return 0;
#endif

    if(ret == 0) sprintf(thiserr, "%s got signal %d", str, sig);
    else sprintf(thiserr, "%s returned %d", str, ret);

    if(*errorstr != '\0') strcat(errorstr, ", ");
    strcat(errorstr, thiserr);
    return 1;
}


void write_tapeheader(host, disk, level, compress, datestamp, outf)
char *host, *disk, *datestamp;
int outf, compress, level;
/*
 * writes an Amanda tape header onto the output file.
 */
{
    char line[128], unc[256], buffer[BUFFER_SIZE];
    int len, rc;

    sprintf(buffer, "AMANDA: FILE %s %s %s lev %d comp %s\n",
            datestamp, host, disk, level, 
	    compress? COMPRESS_SUFFIX : "N");
    

    strcat(buffer,"To restore, position tape at start of file and run:\n");

    if(compress) sprintf(unc, " %s |", UNCOMPRESS_CMD);
    else strcpy(unc, "");

    sprintf(line,"\tdd if=<tape> bs=%dk skip=1 |%s restore ...f -\n\014\n",
            BUFFER_SIZE/1024, unc);
    strcat(buffer, line);

    len = strlen(buffer);
    memset(buffer+len, '\0', BUFFER_SIZE-len);
    if((rc = write(outf, buffer, BUFFER_SIZE)) < BUFFER_SIZE)
	error("error [write header: %s]", strerror(errno));
}

#ifdef __STDC__
int pipespawn(char *prog, int *stdinfd, int stdoutfd, int stderrfd, ...)
#else
int pipespawn(prog, stdinfd, stdoutfd, stderrfd, va_alist)
char *prog;
int *stdinfd;
int stdoutfd, stderrfd;
va_dcl
#endif
{
    va_list ap;
    char *environ[1], *argv[16];
    int pid, i, inpipe[2];

    dbprintf(("%s: spawning \"%s\" in pipeline\n", pname, prog));

    if(pipe(inpipe) == -1) 
	error("error [open pipe to %s: %s]", prog, strerror(errno));

    switch(pid = fork()) {
    case -1: error("error [fork %s: %s]", prog, strerror(errno));
    default:	/* parent process */
	close(inpipe[0]);	/* close input side of pipe */
	*stdinfd = inpipe[1];
	break;
    case 0:		/* child process */
	close(inpipe[1]);	/* close output side of pipe */

	if(dup2(inpipe[0], 0) == -1)
	    error("error [spawn %s: dup2 in: %s]", prog, strerror(errno));
	if(dup2(stdoutfd, 1) == -1)
	    error("error [spawn %s: dup2 out: %s]", prog, strerror(errno));
	if(dup2(stderrfd, 2) == -1)
	    error("error [spawn %s: dup2 err: %s]", prog, strerror(errno));

	arglist_start(ap, stderrfd);		/* setup argv */
	for(i=0; i<16 && (argv[i]=arglist_val(ap, char *)) != (char *)0; i++);
	if(i == 16) argv[15] = (char *)0;
	arglist_end(ap);

	environ[0] = (char *)0;	/* pass empty environment */

	execve(prog, argv, environ);
	error("error [exec %s: %s]", prog, strerror(errno));
	/* NOTREACHED */
    }
    return pid;
}


int pipefork(func, fname, stdinfd, stdoutfd, stderrfd)
void (*func) P((void));
char *fname;
int *stdinfd;
int stdoutfd, stderrfd;
{
    int pid, inpipe[2];

    dbprintf(("%s: forking function %s in pipeline\n", pname, fname));

    if(pipe(inpipe) == -1) 
	error("error [open pipe to %s: %s]", fname, strerror(errno));

    switch(pid = fork()) {
    case -1: error("error [fork %s: %s]", fname, strerror(errno));
    default:	/* parent process */
	close(inpipe[0]);	/* close input side of pipe */
	*stdinfd = inpipe[1];
	break;
    case 0:		/* child process */
	close(inpipe[1]);	/* close output side of pipe */

	if(dup2(inpipe[0], 0) == -1)
	    error("error [fork %s: dup2 in: %s]", fname, strerror(errno));
	if(dup2(stdoutfd, 1) == -1)
	    error("error [fork %s: dup2 out: %s]", fname, strerror(errno));
	if(dup2(stderrfd, 2) == -1)
	    error("error [fork %s: dup2 err: %s]", fname, strerror(errno));

	func();
	exit(0);
	/* NOTREACHED */
    }
    return pid;
}

void start_backup(disk, level, datestamp, dataf, mesgf)
char *disk, *datestamp;
int level, dataf, mesgf;
{
    int dumpin, dumpout, nullfd;
    char host[80], dumpkeys[10];
    char device[80], *domain;

    if(gethostname(host, sizeof host) == -1)
        error("error [gethostname: %s]", strerror(errno));
    if((domain = strchr(host, '.'))) *domain++ = '\0';

    fprintf(stderr, "%s: start [%s:%s level %d datestamp %s]\n",
	    pname, host, disk, level, datestamp);

    if((nullfd = open("/dev/null", O_RDWR)) == -1)
	error("error [open /dev/null: %s]", strerror(errno));

    NAUGHTY_BITS;

    write_tapeheader(host, disk, level, compress, datestamp, dataf);

    if(compress)
	comppid = pipespawn(COMPRESS_PATH, &dumpout, dataf, mesgf,
			    COMPRESS_PATH, 
			    compress == COMPR_BEST? 
			        COMPRESS_BEST_OPT : COMPRESS_FAST_OPT,
			    (char *)0);
    else {
	dumpout = dataf;
	comppid = -1;
    }

    /* invoke dump */

    if(disk[0] == '/')
	strcpy(device, disk);
    else
	sprintf(device, "%s%s", DEV_PREFIX, disk);

#ifndef AIX_BACKUP
    /* normal dump */
    sprintf(dumpkeys, "%d%ssf", level, no_record ? "" : "u");

    dumppid = pipespawn(DUMP, &dumpin, dumpout, mesgf, 
			"dump", dumpkeys, "100000", "-", device, (char *)0);
#else
    /* AIX backup program */

    sprintf(dumpkeys, "-%d%sf", level, no_record ? "" : "u");
    dumppid = pipespawn(DUMP, &dumpin, dumpout, mesgf, 
			"backup", dumpkeys, "-", device, (char *)0);
#endif

    /* close the write ends of the pipes */

    close(dumpin);
    close(dumpout);
    close(dataf);
    close(mesgf);
}

void parse_backup_messages(mesgin)
int mesgin;
{
    int goterror, size, wpid, retstat;

    goterror = 0;
    *errorstr = '\0';

    do {
	size = read(mesgin, line, MAX_LINE);
	switch(size) {
	case -1:
	    error("error [read mesg pipe: %s]", strerror(errno));
	case 0:
	    close(mesgin);
	    break;
	default:
	    add_msg_data(line, size);
	    break;
	}
    } while(size != 0);

    while((wpid = wait(&retstat)) != -1) {
	/* we know that it exited, so we don't have to check WIFEXITED */
	if((WIFSIGNALED(retstat) || WEXITSTATUS(retstat)) &&
	   check_status(wpid, retstat))
	    goterror = 1;
    }

    if(goterror)
	error("error [%s]", errorstr);
    else if(dump_size == -1)
	error("error [no dump size line]");

    fprintf(stderr, "%s: size %d\n", pname, dump_size);
    fprintf(stderr, "%s: end\n", pname);
}


/*
 * Dump output lines are scanned for two types of regex matches.
 *
 * First, there are some cases, unfortunately, where dump detects an
 * error but does not return an error code.  We would like to bring these
 * errors to the attention of the operators anyway.  
 *
 * Second, we attempt to determine what dump thinks its output size is.
 * This is cheaper than putting a filter between dump and compress just
 * to determine the output size.  The re_size table contains regexes to
 * match the size output by various vendors' dump programs.  Some vendors
 * output the number in Kbytes, some in 512-byte blocks.  Whenever an
 * entry in re_size matches, the first integer in the dump line is
 * multiplied by the scale field to get the dump size.
 */

typedef enum { 
    DMP_NORMAL, DMP_STRANGE, DMP_SIZE, DMP_ERROR,
} dmpline_t;

typedef struct regex_s {
    dmpline_t typ;
    char *regex;
    int scale;                  /* only used in size table */
} regex_t;

regex_t re_table[] = {
  /* the various encodings of dump size */
  { DMP_SIZE, 
	"DUMP: DUMP: [0-9][0-9]* tape blocks",				1024},
  { DMP_SIZE,
	"dump: Actual: [0-9][0-9]* tape blocks",
                                                                        1024},
  { DMP_SIZE,
        "backup: There are [0-9][0-9]* tape blocks on [0-9]* tapes",    1024},

  { DMP_SIZE,
        "backup: [0-9][0-9]* tape blocks on [0-9][0-9]* tape(s)",       1024},

  { DMP_SIZE,
	"DUMP: [0-9][0-9]* blocks ([0-9][0-9]*KB) on [0-9][0-9]* volume", 512},
  { DMP_SIZE,
"DUMP: [0-9][0-9]* blocks ([0-9][0-9]*\\.[0-9][0-9]*MB) on [0-9][0-9]* volume",
                                                                          512},
  { DMP_SIZE, "DUMP: [0-9][0-9]* blocks",                                 512},
  { DMP_SIZE, "DUMP: [0-9][0-9]* bytes were dumped",		            1},

  /* strange dump lines */
  { DMP_STRANGE, "should not happen" },
  { DMP_STRANGE, "Cannot open" },
  /* need to add more ERROR entries here by scanning dump sources */

  /* any blank or non-strange DUMP: lines are marked as normal */
  { DMP_NORMAL, "^  DUMP:" },
  { DMP_NORMAL, "^dump:" },					/* OSF/1 */

  { DMP_NORMAL, "^backup:" },					/* AIX */
  { DMP_NORMAL, "^        Use the umount command to unmount the filesystem" },

  { DMP_NORMAL, "^[ \t]*\\\n" },

  /* catch-all; DMP_STRANGE is returned for all other lines */
  { DMP_STRANGE, NULL, 0}
};

int first_num P((char *str));
dmpline_t parse_dumpline P((char *str));
static void process_dumpline P((char *str));

int first_num(str)
char *str;
/*
 * Returns the value of the first integer in a string.
 */
{
    char tmp[16], *tp;

    tp = tmp;
    while(*str && !isdigit(*str)) str++;
    while(*str && isdigit(*str)) *tp++ = *str++;
    *tp = '\0';

    return atoi(tmp);
}

dmpline_t parse_dumpline(str)
char *str;
/*
 * Checks the dump output line in str against the regex table.
 */
{
    regex_t *rp;
    char *errs, *re_comp();

    /* check for error match */
    for(rp = re_table; rp->regex != NULL; rp++) {
        if((errs = re_comp(rp->regex)) != NULL)
	    error("error in regex: %s", errs);
        if(re_exec(str) == 1) /* got match */
	    break;
    }
    if(rp->typ == DMP_SIZE) dump_size = first_num(str) * rp->scale;
    return rp->typ;	
}


static char msgbuf[MAX_LINE];
static int msgofs = 0;

static void process_dumpline(str)
char *str;
{
    char startchr;
    dmpline_t typ;

    typ = parse_dumpline(str);
    switch(typ) {
    case DMP_NORMAL:
    case DMP_SIZE:
	startchr = '|';
	break;
    default:
    case DMP_STRANGE:
	startchr = '?';
	break;
    }
    fprintf(stderr, "%c %s", startchr, str);
}


void add_msg_data(str, len)
char *str;
int len;
{
    char *nlpos;
    int len1, got_newline;

    while(len) {

	/* find a newline, if any */
	for(nlpos = str; nlpos < str + len; nlpos++)
	    if(*nlpos == '\n') break;

	/* copy up to newline (or whole string if none) into buffer */
	if(nlpos < str+len && *nlpos == '\n') {
	    got_newline = 1;
	    len1 = nlpos - str + 1;
	}
	else {
	    got_newline = 0;
	    len1 = len;
	}

	/* but don't overwrite the buffer */
	if(len1 + msgofs >= MAX_LINE) {
	    len1 = MAX_LINE-1 - msgofs;
	    str[len1-1] = '\n';			/* force newline */
	    got_newline = 1;
	}
	strncpy(msgbuf + msgofs, str, len1);
	msgofs += len1;
	msgbuf[msgofs] = '\0';

	if(got_newline) {
	    process_dumpline(msgbuf);
	    msgofs = 0;
	}
	len -= len1;
	str += len1;
    }
}
